# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the framework for interface configuration.  In
# particular, it provides:
# -  the Intf class
# -  the "config-if" mode
# -  the "[no] interface <name>" command
# -  the "show interfaces [<name>]" command
# -  the "show interfaces [<name>] description" command
# -  the "show interfaces [<name>] counters" command
# -  the "show interfaces [<name>] counters discards" command
# -  the "show interfaces [<name>] counters delta" command
# -  the "clear counters [<name>] [session]" command
# -  the "[no] mtu [value]" command
# -  the "[no] shutdown" command
# -  the "[no] description <desc>" command.
# -  the "[no|default] load-interval <interval>" command (interface level)
# -  the "[no|default] load-interval defaul <interval>" command (global config level)
# -  the "[no] logging event link-status (interface level)
# -  the "[no] logging event link-status global (global config level)
#-------------------------------------------------------------------------------
import copy
import functools
import itertools
import re
import threading
import weakref

import Ark
import Tac
import Arnet
from Arnet.Verify import verify
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
import CliMode.InterfaceDefaults
from CliMode.Intf import IntfMode
from CliModel import UnknownEntityError
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.MacAddr as MacAddr
import CliToken.Clear
import CliToken.Logging
import CliToken.Service
import ConfigMount
import Intf.IntfRange as IntfRange
import Intf.Log as Log
import IntfModel
import LazyMount
import MultiRangeRule
import QuickTrace
import SessionCli
import ShowCommand
import SmashLazyMount
import TacSigint
import Tracing
from VirtualIntfRule import IntfMatcher
import WaitForWarmupCli
import Ethernet

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt8 = QuickTrace.trace8

VERIFY_MAX_INTFS = 100

aclDropConfigDir = None
allIntfConfigDir = None
allIntfStatusDir = None
globalIntfConfig = None
allIntfAclDropCounters = {}
brHwCapabilities = None

# This extension is called when the Cli
# tries to display an attribute of an intf that
# is only meaningful if bridging is enabled, yet
# the forwarding model of an intf is dataLink.
# extension is a function  that takes an eth
# intf name and the Cli mode, and then
# returns the reason (a string) that
# the intf is dataLink.  (E.g. "is in po3" ).
# DataLink Explanation hook has been moved from Ebra
# to here for 2 reasons.
# a.) It does not make sense to add Ebra dependency in packages
#     that want to use the hook.
# b.) Hooks are added by different packages like
# Lag/Mirror etc.. and it can be used by dependent packages.
dataLinkExplanationHook = CliExtensions.CliHook()
# This is the new hook for intfForwardingModelQuietDataLink
quietDataLinkExplanationHook = CliExtensions.CliHook()
# This is the new hook for intfForwardingModelUnauthorized
unauthorizedExplanationHook = CliExtensions.CliHook()
# This is the new hook for intfForwardingModelRecirculation
recirculationExplanationHook = CliExtensions.CliHook()

_forwardingModelMap = { "intfForwardingModelRecirculation" : "recirculation",
                        "intfForwardingModelQuietDataLink" : "quietDataLink",
                        "intfForwardingModelUnauthorized" : "unauthorized",
                        "intfForwardingModelDataLink" : "dataLink",
                        "intfForwardingModelBridged" : "bridged",
                        "intfForwardingModelRouted" : "routed"
                      }

# Have to do all sub-intf related stuff in hook-like style, because
# "interface/config/subintf" is registered in the Ebra package.
subintfMtuHook = []
subintfMtuGuardHook = []

def tryWaitForWarmup( mode ):
   WaitForWarmupCli.tryWaitForWarmupAfterVlanPortChange( mode )

# Helper methods for create and deleting IntfConfig requests from the Cli
def cliCreateIntfConfigRequest( intfConfigDir, intfName ):
   # Enqueue a request for the interface
   intfConfigDir.intfConfigRequest[ Tac.Value(
      "Interface::IntfConfigRequest", intf=intfName, requestor='Cli' ) ] = True

def cliDeleteIntfConfigRequest( intfConfigDir, intfName ):
   # Del the request.
   del intfConfigDir.intfConfigRequest[ Tac.Value(
      "Interface::IntfConfigRequest", intf=intfName, requestor='Cli' ) ]

# Return the requestors for a given interface
def intfConfigRequestors( intfConfigDir, intfName ):
   return intfConfigDir.intfConfig[ intfName ].requestor.keys()

def counterSupportedIntfs( mode, intf=None, mod=None, intfType=None, sort=True ):
   # returns interfaces that support counters
   intfs = Intf.getAll( mode, intf, mod, intfType=intfType )
   if not intfs:
      return None
   intfs = [ i for i in intfs if i.countersSupported() ]
   if not intfs:
      mode.addWarning( "Counters not supported on %s" % (intf or "any interface",) )
      return None
   return intfs

def countersErrorsSupportedIntfs( mode, intf=None, mod=None, sort=True ):
   # returns interfaces that support counters
   intfs = Intf.getAll( mode, intf, mod )
   if not intfs:
      return None
   intfs = [ i for i in intfs if i.countersErrorsSupported() ]
   if not intfs:
      mode.addWarning( "Counters not supported on %s" % (intf or "any interface",) )
      return None
   return intfs

def counterDiscardSupportedIntfs( mode, intf=None, mod=None ):
   # returns interfaces that support counters
   intfs = Intf.getAll( mode, intf, mod )
   if not intfs:
      return None
   intfs = [ i for i in intfs if i.countersDiscardSupported() ]
   if not intfs:
      mode.addWarning( 
            "Counter discards are not supported on %s" % (intf or "any interface",) )
      return None
   return intfs

def countersRateSupportedIntfs( mode, intf=None, mod=None ):
   # returns interfaces that support counters
   intfs = Intf.getAll( mode, intf, mod )
   if not intfs:
      return None
   intfs = [ i for i in intfs if i.countersRateSupported() ]
   if not intfs:
      mode.addWarning( 
            "Counter rates are not supported on %s" % (intf or "any interface",) )
      return None
   return intfs

def _getAclDropConfigKey( intf, aclDropCounters ):
   for i in aclDropCounters:
      if aclDropConfigDir.get( i ):
         keyFound = aclDropConfigDir[ i ].intf.get( intf.name )
         # retVal can be either key or None
         # We return key when the counters should be accessed. Otherwise
         # return None to indicate the key cannot be read at this time
         if keyFound is not None:
            return i
      else:
         return None

def _maybeMountAclDropCounters():
   for name, entry in aclDropConfigDir.iteritems():
      if name not in allIntfAclDropCounters:
         allIntfAclDropCounters[ name ] = SmashLazyMount.mount(
               myEntManager,
               "interface/status/counter/acldrop/" + entry.writerMountName,
               "Smash::Interface::AllDropCounterDir",
               SmashLazyMount.mountInfo( 'reader' ) )

#-------------------------------------------------------------------------------
# Generic interface helper class. This is used as a base class for all 
# dependent classes of Interface that wants to help Intf class to provide 
# certain functionality like lookup for show commands, clean-up interface
# configuration on deletion or setting it to default
#-------------------------------------------------------------------------------
class IntfDependentBase( object ):
   """This class is responsible for cleaning up per-interface configurations
   when the interface is deleted (via "no interface" command) or when the 
   interface is set to its default state (via "default interface" command)
   """
   def __init__( self, intf, mode ):
      self.intf_ = intf

   def destroy( self ):
      # Some tests use only the config state without having the
      # corresponding status and we expect to deal with defaulting
      # those interfaces as well
      if self.intf_.config():
         self.setDefault()

   def showNetworkAddr( self ):
      """If available, display the L3 network address in show interface
      command."""
      pass

   def setDefault( self ):
      pass      

# global storage for session counters
threadLocalData_ = threading.local()

def _sessionCounterDir():
   # TODO: eventually this should just be stored in the mode.session.sessionData
   # like deltaCounterDir is. However it's ok for it to be a per thread attribute
   if getattr( threadLocalData_, 'sessionCounterDir', None ) is None:
      threadLocalData_.sessionCounterDir = {}

   return threadLocalData_.sessionCounterDir

def _deltaCounterDir( mode ):
   deltaCounterDir = mode.session.sessionData( 'intfDeltaCounterDir', None )
   if deltaCounterDir is None:
      deltaCounterDir = {}
      mode.session.sessionDataIs( 'intfDeltaCounterDir', deltaCounterDir )

   return deltaCounterDir

#-------------------------------------------------------------------------------
# Generic interface class.  There is a subclass of this class for every physical
# interface type that is supported by the CLI.
#
# This class should never be instantiated directly; only its subclasses should
# be instantiated.
#
# Note that existence of an instance of this class is independent of existence
# of the corresponding interface and C++ objects.  Whether or not the interface
# exists can be tested by calling the lookup() method.  The interface can be
# created and destroyed by calling the create() and destroy() methods.
#-------------------------------------------------------------------------------
class Intf( object ):

   #----------------------------------------------------------------------------
   # A class for a protocol module that wishes to hook into the base
   # CLI commands for an Intf should use this API to
   # register itself.  The dependent class must have a constructor
   # that takes two parameters: an Intf instance and sysdbRoot.  As of
   # 2007-10-10, IraIpIntfCli.IpIntf is registered to ensure 
   # that the IpIntf gets destroyed in 'no interface' and that the network
   # address gets displayed in 'show interface'
   #----------------------------------------------------------------------------
   @classmethod
   def registerDependentClass( cls, derived, priority=20 ):
      d = cls.dependentClasses_ + [ ( priority, derived ) ]
      d = sorted( d, key=lambda x: x[ 0 ] )
      cls.dependentClasses_ = d
   dependentClasses_ = []

   #----------------------------------------------------------------------------
   # Subclasses of this class must register themselves by calling
   # Intf.addPhysicalIntfType.  This enables this class to query them all to
   # find all existing interfaces, which is needed in order to perform the
   # "show interfaces" command.  This function also modifies the Intf.matcher to
   # support names of particular types of interface.
   #----------------------------------------------------------------------------
   physicalIntfClasses_ = []

   @staticmethod
   def addPhysicalIntfType( clazz, 
                            intfRangeType,
                            withIpSupport=False, 
                            withRoutingProtoSupport=True,
                            matcherWithRoutingProtoSupport=None ):
      # The withRoutingProtoSupport is checked only when withIpSupport is True.
      # When matcherWithRoutingProtoSupport is None, the clazz.matcher is used.
      # In EthPhyIntf case, the clazz.matcher includes both Ethernet and Management
      # interfaces. The former supports routing protocol while the later does 
      # not. In this case, the matcherWithRoutingProtoSupport should be given 
      # explicitly.

      Intf.physicalIntfClasses_.append( clazz )
      Intf.matcher |= clazz.matcher
      Intf.matcherWOSubIntf |= clazz.matcher

      if intfRangeType:
         if not hasattr( intfRangeType, "__iter__" ):
            intfRangeType = ( intfRangeType, )
         for irt in intfRangeType:
            irt.cliIntfClazzIs( clazz )
      if withIpSupport:
         Intf.matcherWithIpSupport |= clazz.matcher
         if withRoutingProtoSupport:
            if matcherWithRoutingProtoSupport:
               Intf.matcherWithRoutingProtoSupport |= \
                  matcherWithRoutingProtoSupport
            else:
               Intf.matcherWithRoutingProtoSupport |= clazz.matcher

   @staticmethod
   def addSubIntf( subMatcher, clazz, withIpSupport=False,
         withRoutingProtoSupport=True ):
      Intf.matcher |= subMatcher
      if clazz not in Intf.physicalIntfClasses_:
         Intf.physicalIntfClasses_.append( clazz )
      if withIpSupport:
         Intf.matcherWithIpSupport |= subMatcher
         if withRoutingProtoSupport:
            Intf.matcherWithRoutingProtoSupport |= subMatcher

   @staticmethod
   def getShortname( longname ):
      return IntfMode.getShortname( longname )

   #----------------------------------------------------------------------------
   # A matcher to match interface names and return a corresponding Intf object.
   # To begin with this matcher is empty, but each call to
   # Intf.addPhysicalIntfType() modifies this matcher to add a pattern matching
   # names of interfaces of that type.
   #----------------------------------------------------------------------------
   matcher = IntfMatcher()
   # interface matchers that exclude the support for sub interface.
   matcherWOSubIntf = IntfMatcher()
   # interface matchers that support IP address configuration
   matcherWithIpSupport = IntfMatcher()
   # interface matcher that support IP and IP routing protocol. 
   # This is a subset of matcherWithIpSupport
   matcherWithRoutingProtoSupport = IntfMatcher()

   # If your command should support interface range, use the following instead of 
   # single-interface matchers.
   rangeMatcher = IntfRange.IntfRangeMatcher()

   # Range matchers for physical interfaces don't support non-existent
   # physical interfaces. So noSingletons is set to make the range
   # matchers match only interface ranges and the single interface matcher,
   # matcherWithIpSupport is used to match non-existent interfaces also.
   # The value of this matcher may be either IntfList or a single Intf
   # object and the CLI command handler needs to handle accordingly.
   rangeMatcherWithIpSupport = IntfRange.IntfRangeMatcher(
      explicitIntfTypes=IntfRange.intfTypesWithIpSupport )

   #----------------------------------------------------------------------------
   # Create a new Intf instance.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      self.name_ = name
      self.mode_ = mode
      self.sysdbRoot_ = self.mode_.sysdbRoot
      self.intfCounter = None
      self.shortname_ = IntfMode.getShortname( name )

      self.counterFreshnessTime = 0

   name = property( lambda self: self.name_ )
   shortname = property( lambda self: self.shortname_ )

   def __str__(self):
      return self.name_
   
   #----------------------------------------------------------------------------
   # This method indicates whether or not interface creation is currently
   # supported.
   #----------------------------------------------------------------------------
   def intfCreationCurrentlySupported( self, mode ):
      return True

   #----------------------------------------------------------------------------
   # Creates the interface corresponding to this object.  This function is
   # called whenever "config-if" mode is entered (i.e. by the "interface"
   # command).
   #----------------------------------------------------------------------------
   def create( self, startupConfig=False ):
      self.createPhysical( startupConfig )

   #----------------------------------------------------------------------------
   # Destroys the interface corresponding to this object.  This function is
   # called by the "no interface" command, for virtual interfaces only.
   #----------------------------------------------------------------------------
   def destroy( self ):
      for ( _priority, cls ) in Intf.dependentClasses_:
         cls( self, self.mode_ ).destroy()
      self.destroyPhysical()

   #----------------------------------------------------------------------------
   # If any comment exists for that interface, it will remove it. 
   # This command is applicable to physical as well as virtual interfaces. 
   #----------------------------------------------------------------------------
   def removeComments( self ):
      commentKey = "if-%s" % self.shortname_
      self.mode_.removeCommentWithKey( commentKey )
      
   #----------------------------------------------------------------------------
   # Sets the interface corresponding to this object to its default state. 
   # This function is called by the "default interface" command. This command
   # is applicable to physical as well as virtual interfaces.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      for ( _priority, cls ) in Intf.dependentClasses_:
         cls( self, self.sysdbRoot_ ).setDefault()

   #----------------------------------------------------------------------------
   # Searches for the interface corresponding to this object, returning True if
   # the interface exists. Also check if the status has a config attached to
   # it because if it doesn't it means it is in a transitive state about to
   # be created or deleted so not visible.
   #----------------------------------------------------------------------------
   def lookup( self ):
      config = self.config() or self.dynConfig()
      return self.lookupPhysical() and config

   #----------------------------------------------------------------------------
   # Returns the module number of the interface, or None if this is not a
   # physical interface.
   #----------------------------------------------------------------------------
   def module( self ):
      # match against interface names with 2 or more levels
      m = re.match( '(\D+)(\d+)/(\d+)(?:/\d+)*$', self.name )
      if m is not None:
         return int( m.group( 2 ) )
      else:
         return None

   #----------------------------------------------------------------------------
   # Returns the module and port number of the interface, or None if this is not
   # a physical interface.  If this is a port in a fixed config system, returns
   # None for the module but the right number for the port number.
   #----------------------------------------------------------------------------
   def moduleAndPort( self ):
      # match against interface names with 2 or more levels
      m = re.match( '(\D+)(\d+)/(\d+)(?:/\d+)*$', self.name )
      if m is not None:
         return (int( m.group( 2 ) ), int( m.group( 3 ) ))
      # match against interface names with 1 or more levels
      m = re.match( '(\D+)(\d+)(?:/\d+)*$', self.name )
      if m is not None:
         return (None, int( m.group( 2 ) ))
      return (None, None)

   #----------------------------------------------------------------------------
   # Returns a sorted list of Intf objects for all existing interfaces of any
   # type.  If intf is specified and it exists, returns a list containing intf.
   # If intf is specified and it does not exist, prints an error message and
   # returns None.  If mod is specified, restricts the output to physical
   # interfaces on that module. If intfType is specified, restricts the output
   # to interfaces of that type or subclasses of that type. If intfType is specified
   # as a list of types, restricts the output to interfaces of at least one type
   # in the list or subclasses of at least one type in the list. If config is
   # specified, only check that the interface config is present, and skip the
   # check entirely if it's a single interface. This allows non-existent single
   # interface to be accepted which is important for startup-config and CliSave
   # tests. It is expected the caller will call intf.create() afterwards. We
   # only do this for single interface to avoid accidentally creating a range
   # of interface by mistake.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAll( mode, intf=None, mod=None, intfType=None, silent=False,
               config=False, exposeInactive=False, exposeUnconnected=False,
               sort=True ):
      def getCliIntfs( intf ):
         '''Create CliIntf objects for each* interface.
         If we are configuring, assume all interfaces are valid.
         Otherwise, cross-reference with current (sub)intf config.
         Yield, instead of building large collections.'''
         CliIntf = functools.partial( intf.type().getCliIntf, mode )
         intfs_ = intf.intfNames()
         if config:
            for i in intfs_:
               yield CliIntf( i )
         else:
            existing = set( allIntfConfigDir ) | set( allIntfStatusDir )
            for i in intfs_:
               if i in existing:
                  yield CliIntf( i )
               else:
                  hasNonexistentIntfs[ 0 ] = True

      if intfType:
         intfTypeList = intfType if type( intfType ) == list else [ intfType ]
         assert all( issubclass( t, Intf ) for t in intfTypeList )
      else:
         intfTypeList = None
      hasNonexistentIntfs = [ False ] # This needs to be a list for pass-by-ref.
      intfs = []
      globalExposeInactive = globalIntfConfig.exposeInactiveLanes
      globalExposeUnconnected = globalIntfConfig.exposeUnconnectedLanes
      if intf is not None:
         if isinstance( intf, MultiRangeRule.IntfList ):
            possibleIntfs = getCliIntfs( intf )

            # We need to find out if this is a single interface.
            # Generators have no `len`, but we can use `tee` to peek.
            possibleIntfs, peek = itertools.tee( possibleIntfs, 2 )
            if next( peek, None ) is None: # No interfaces at all.
               if silent:
                  return []
               raise UnknownEntityError( "Interface does not exist" )
            isSingleIntf = next( peek, None ) is None
         else:
            possibleIntfs = [ intf ]
            isSingleIntf = True

         for i in possibleIntfs:
            intfConfig = i.config() or i.dynConfig()
            if not( intfType is None or  any( isinstance( i, t )
                  for t in intfTypeList ) ):
               mode.addError(
                  'Interface is not of appropriate type for this operation' )
               return []
            if config:
               if not isSingleIntf:
                  # For config command, we only make sure the interface
                  # config exists because we don't care about status.
                  if not intfConfig:
                     hasNonexistentIntfs[ 0 ] = True
                     continue
            elif not i.lookup():
               # check both config and status
               hasNonexistentIntfs[ 0 ] = True
               continue
            if not exposeInactive and not globalExposeInactive and \
                   intfConfig and intfConfig.enabledStateReason == 'inactive':
               continue
            if not exposeUnconnected and not globalExposeUnconnected and \
                   i.name.startswith( 'Un' ) and intfConfig:
               continue
            intfs.append( i )
      else:
         for clazz in Intf.physicalIntfClasses_:
            if intfTypeList is None or any( issubclass( clazz, t )
                  for t in intfTypeList ):
               if ( exposeInactive or globalExposeInactive ) and \
                      ( exposeUnconnected or globalExposeUnconnected ) :
                  intfs.extend( clazz.getAllPhysical( mode ) )
               else:
                  for theIntf in clazz.getAllPhysical( mode ):
                     intfConfig = theIntf.config() or theIntf.dynConfig()
                     if intfConfig is None:
                        intfs.append( theIntf )
                     elif intfConfig.enabledStateReason == 'inactive':
                        if exposeInactive or globalExposeInactive:
                           intfs.append( theIntf )
                     elif intfConfig.prettyName.startswith( ( 'Un', 'Ue' ) ):
                        if exposeUnconnected or globalExposeUnconnected:
                           intfs.append( theIntf )
                     else:
                        intfs.append( theIntf )

      if mod is not None:
         intfs = filter( lambda i: i.module() == mod, intfs )
      
      # we pay a penalty for sorting large nubmer of interfaces, we should only do 
      # if our caller cares about this
      if sort: 
         intfs.sort()
      
      if hasNonexistentIntfs[ 0 ] and not silent:
         if not intfs:
            raise UnknownEntityError( "Interface does not exist" )
         else:
            mode.addWarning( "Some interfaces do not exist" )
      # Set the counterFreshnessTime on all the selected interfaces
      # to the same value. For agents that do batch prefetching of counters
      # this would avoid duplicate prefetches when Cli takes some time to step
      # through the interfaces and finds the already prefetched counter to be 
      # stale
      counterFreshnessTime = Tac.now() - 1 
      for intf in intfs:
         intf.counterFreshnessTime = counterFreshnessTime
      return intfs

   #----------------------------------------------------------------------------
   # Prints information about this interface, or throws an exception if it does
   # not exist.  This function is called by the "show interfaces" command.
   #
   # This function prints a common header but then delegates the rest of the
   # output to the subclass via the pure virtual 'showPhysical' function.
   #----------------------------------------------------------------------------
   def show( self, mode ):
      intfStatusModel = self.getIntfStatusModel()
      self.populateBaseModel( intfStatusModel )
      self.showPhysical( mode, intfStatusModel )
      return intfStatusModel

   #----------------------------------------------------------------------------
   # Prints information about this interface's configuration and any config
   # errors. This is implemented per-interface by overriding the
   # getIntfConfigSanityModel function to return a custom model extending
   # InterfaceConfigSaniy. The default model prints the intf is unsupported.
   #----------------------------------------------------------------------------
   def showConfigSanity( self, mode ):
      intfConfigSanityModel = self.getIntfConfigSanityModel()
      self.populateBaseModel( intfConfigSanityModel )
      return intfConfigSanityModel
   
   def populateBaseModel( self, intfStatusModel ):
      intfStatusModel.forwardingModel = self.forwardingModel()
      ( intfStatusModel.lineProtocolStatus, 
        intfStatusModel.interfaceStatus ) = self.getStatus()
      intfStatusModel.hardware = self.hardware()
      ( intfStatusModel.physicalAddress, 
        intfStatusModel.burnedInAddress ) = self.physicalAddress()
      intfStatusModel.description = self.description().strip()
      intfStatusModel.bandwidth = self.bandwidth() * 1000
      intfStatusModel.l3MtuConfigured = self.l3MtuConfigured()
      intfStatusModel.mtu = self.mtu()
      intfStatusModel.lastStatusChangeTimestamp = self.uptime()
      addrList = self.showNetworkAddr()
      while addrList:
         addr = addrList.pop()
         if isinstance( addr, IntfModel.InterfaceAddressIp6Base ):
            if not intfStatusModel.interfaceAddressIp6:
               intfStatusModel.interfaceAddressIp6 = addr
            else:
               raise RuntimeError( "multiple IPv6 address CLI providers" )
         elif isinstance( addr, IntfModel.InterfaceAddressIp4Base):
            if not intfStatusModel.interfaceAddress:
               intfStatusModel.interfaceAddress = [ addr ]
            else:
               raise RuntimeError( "multiple IPv4 address CLI providers" )

      for hook in IntfModel.interfaceInfoHook.extensions():
         hook( self.name, intfStatusModel )

   def forwardingModel( self ):
      return _forwardingModelMap[ self.status().forwardingModel ]
   
   def getStatus( self ):
      conf = self.config() or self.dynConfig()
      stat = self.status()
      lineProtocolStatus = self.lineProtocolState()
      if not conf:
         interfaceStatus = "uninitialized"
      elif not stat.active:
         interfaceStatus = "inactive"
      elif not conf.adminEnabled:
         interfaceStatus = "disabled"
      elif not conf.enabled or lineProtocolStatus != "up":
         interfaceStatus = conf.enabledStateReason or "notconnect"
      else:
         interfaceStatus = "connected"

      return ( lineProtocolStatus, interfaceStatus.lower() )

   def physicalAddress( self ):
      address = self.addrStr()
      if address is None:
         return ( None, None )
      
      quad = '[0-9a-f]{4}'
      pattern = r'(%s\.%s\.%s)' % ( quad, quad, quad )
      m = re.search( r'address is %s(?: \(bia %s\))?' % ( pattern, pattern ),
                     address )
      physAddr, biAddr = m.groups()
      return ( physAddr, biAddr )

   #----------------------------------------------------------------------------
   # Sets the MAC address of the interface to a specified value or the default
   # value.
   #----------------------------------------------------------------------------
   def setMacAddr( self, addr, routerMacConfigured=False ):
      if addr is None:
         self.config().addr = '00:00:00:00:00:00'
         self.config().routerMacConfigured = False
      elif Ethernet.isUnicast( addr ):
         self.config().addr = addr
         self.config().routerMacConfigured = routerMacConfigured
      else:
         self.mode_.addError( 'Cannot set the interface MAC address to a '
                              'multicast address' )

   def l3MtuConfigured( self ):
      if not self.config():
         return False
      return self.config().l3MtuConfigured

   def mtu( self ):
      return self.status().mtu
   
   def uptime( self ):
      if self.status().operStatusChange.time == 0:
         return None

      return self.status().operStatusChange.time + Tac.utcNow() - Tac.now()

   def showNetworkAddr( self ):
      addresses = []
      for ( _prio, cls ) in Intf.dependentClasses_:
         intfDependent = cls( self, self.sysdbRoot_ )
         addr = intfDependent.showNetworkAddr()
         if addr is not None:
            addresses.append( addr )
      return addresses

   #----------------------------------------------------------------------------
   # Returns the configured interface description.
   #----------------------------------------------------------------------------
   def description( self ):
      config = self.config() or self.dynConfig()
      return config.description if config else ""
   
   def prettyIntfStateStrings( self ):
      """ Return tuple of four values suitable for pretty-printing
      line status in various places in the CLI output. The first
      two items in the tuple fill in the blanks:
         Ethernet1 is ___, line protocol is ___
      The third and fourth items are used in 'show int description'.
      """
      lps = self.lineProtocolState().lower()
      conf = self.config() or self.dynConfig()
      if not conf:
         # There was no Intfconfig object
         return ('down', '%s (uninitialized)' % lps, 'uninit down', lps)
      if not conf.adminEnabled:
         return ('administratively down', '%s (disabled)' % lps, 'admin down', lps)
      elif not conf.enabled or lps != 'up':
         downReasonString = conf.enabledStateReason or 'notconnect'
         return ('down', '%s (%s)' % (lps, downReasonString), 'down', lps)
      else:
         return ('up', '%s (connected)' % lps, 'up', lps)
   
   def getIntfState( self ):
      lps = self.lineProtocolState()
      conf = self.config() or self.dynConfig()
      if not conf:
         # There was no Intfconfig object
         return 'uninitDown'
      elif not conf.adminEnabled:
         return 'adminDown'
      elif not conf.enabled or lps != 'up':
         return 'down'
      else:
         return 'up'

   def lineProtocolState( self ):
      operStatus = self.status().operStatus
      # Strip off 'intfOper' and return the rest of the enum name
      return operStatus[8].lower() + operStatus[9:]

   #----------------------------------------------------------------------------
   # Overloads the comparison operators so that Intf objects have value
   # semantics and are sorted by name.
   #----------------------------------------------------------------------------
   def __cmp__( self, rhs ):
      if isinstance( rhs, Intf ):
         return Arnet.compareIntfName( self.name_, rhs.name_ )
      else:
         return NotImplemented

   #----------------------------------------------------------------------------
   # Hooks/support for "clear counters".
   # See TestIntfCli for examples of what the derived class is supposed to do.
   #----------------------------------------------------------------------------

   #----------------------------------------------------------------------------
   def counter( self ):
      '''Returns the current IntfCounter object for this interface.'''
      if not self.intfCounter:
         if not self.countersSupported():
            return None
         status = self.status()
         if not status:
            return None
         self.intfCounter = status.counter
         if not self.intfCounter:
            qt8( "Intf.counter-", qv( self.name ), "counter is None" )
            return None
         currentTime = Tac.now()
         # For agents that support counter update i.e counterRefreshTime is non-
         # zero, if the counter is older than the counterFreshnessTime then request
         # an update by setting the counterUpdateReqTime and wait till the agent 
         # updates the counter and sets the counterRefreshTime to the time when it
         # is done updating counter
         if self.intfCounter.counterRefreshTime and \
            self.intfCounter.counterRefreshTime < self.counterFreshnessTime:
            self.intfCounter.counterUpdateReqTime = self.counterFreshnessTime
            try:
               Tac.waitFor( 
                      lambda: self.intfCounter.counterRefreshTime >= 
                              self.counterFreshnessTime,
                      timeout=1, warnAfter=0, maxDelay=0.1, description="",
                      sleep=True )
            except Tac.Timeout :
               self.mode_.addWarning(
                  "Timeout getting fresh counters (start: %f, timeout: %f)" %
                  ( currentTime, Tac.now() ) )

      return self.intfCounter

   #----------------------------------------------------------------------------
   # return <intfType>IntfCounterDir must be implemented by subclass
   #---------------------------------------------------------------------------- 
   def getIntfCounterDir( self ):
      '''returns <intfType>IntfCounterDir
      Must be implemented by subclass'''
      raise NotImplementedError()

   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      '''Depending on sessionOnly, return the global or session counter checkpoint.
      If it doesn't exist only create it and the underlying directory structure
      only if a className is passed'''
      if sessionOnly:
         y = _sessionCounterDir().get( self.name )
         if not y:
            if not className:
               return None
            y = Tac.newInstance( "%sDir" % (className), self.name )
            _sessionCounterDir()[ self.name ] = y
      else:
         x = self.getIntfCounterDir()
         if x is None:
            return None
         x = getattr( x, 'checkpointCounterDir', x )
         y = x.intfCounterDir.get( self.name )
         if not y:
            if not className:
               return None
            qt8( "Intf.getCounterCheckpoint-", qv( self.name ), "y is None" )
            y = x.newIntfCounterDir( self.name )

      cc = y.intfCounter.get( 'lastClear' )
      # Get the status of the interface to access it's genId
      status = self.status()

      # Lookup of an existing checkpoint, only return if the genId is the same
      if not className:
         if cc and cc.genId == status.genId:
            return cc
         return None

      # Create new one if it doesn't exist or has wrong genId
      if not cc or cc.genId != status.genId:
         if cc:
            del y.intfCounter[ 'lastClear' ]
         cc = y.newIntfCounter( 'lastClear' )
         cc.genId = status.genId
      return cc

   def getLatestCounterCheckpoint( self ):
      '''Get the latest counter checkpoint wheter global or for this session only'''

      ccGlobal = self.getCounterCheckpoint( sessionOnly=False )
      ccSession = self.getCounterCheckpoint( sessionOnly=True )
      if ccGlobal and ccSession:
         # both there,  compare timestamps
         if ccGlobal.rates.statsUpdateTime > ccSession.rates.statsUpdateTime:
            return ccGlobal
         else:
            return ccSession
      else:
         # at least one not there, return one if any that is there
         if ccGlobal:
            return ccGlobal
         else:
            return ccSession
   
   def clearCounters( self, sessionOnly=False ):
      if self.countersSupported():
         ckpt = self.getCounterCheckpoint( 'Interface::IntfCounter',
                                           sessionOnly=sessionOnly)
         if ckpt is None or self.counter() is None:
            return
         ckpt.statistics = self.counter().statistics
         # Because (current) counters are updated periodically (and
         # statsUpdateTime records latest sample time) and show
         # counter shows counters gathered in latest periodic sample,
         # statsUpdateTime can be saved in ckpt as is (from latest
         # sample). It can still be used to compare global or session
         # chkpt, because: if both session and global clear happen
         # within same sample window, then the chkpt value would be
         # same for both and either can be used later, otherwise (they
         # happens in different sample window), then
         # chkpt.statsUpdateTime can be used correctly. This also make
         # rates calculation to be accurate after clear counters.
         ckpt.rates = self.counter().rates 
         # Update the value of linkStatusChanges.
         ckpt.linkStatusChanges = self.status().operStatusChange.count

   def storeLastCounters( self, mode, className=None ):
      z = _deltaCounterDir( mode ).get( self.name ) 
      if z is None:
         if className is None:
            z = Tac.newInstance( "Interface::IntfCounterDir", self.name )
         else:
            z = Tac.newInstance( "%sDir" % (className), self.name )

      y = z.intfCounter.get( self.name )
      if y is None:
         y = z.newIntfCounter( self.name )
      
      _deltaCounterDir( mode )[ self.name ] = z 

      return y

   def updateLastCounters( self, mode ):
      lastCounter = self.storeLastCounters( mode )
      counter = self.counter()
      if counter:
         lastCounter.statistics = counter.statistics

   def countersSupported( self ):
      return True

   def countersErrorsSupported( self ):
      return False
   
   def countersRateSupported( self ):
      return self.countersSupported()

   def countersDiscardSupported( self ):
      return self.countersSupported()

   def updateInterfaceCountersRateModel( self ):
      return None
   
   def getIntfStatusModel( self ):
      return IntfModel.InterfaceStatus( name=self.status().intfId )

   def getIntfConfigSanityModel ( self ):
      return IntfModel.InterfaceConfigSanity( name=self.status().intfId )

   def getUpdateTime( self, checkpoint ):
      currentTime = self.counter().statistics.lastUpdate

      # when counters have been cleared, return zero until they have been updated
      if checkpoint and checkpoint.statistics.lastUpdate == currentTime:
         return 0.0

      # convert the 'uptime' to UTC
      return Ark.switchTimeToUtc( currentTime )

   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None, 
                                     zeroOut=False ):
      intfCountersModel = IntfModel.InterfaceCounters( _name=self.status().intfId )
      if not zeroOut:
         stat = captureCounterState( counter=self.counter(),
                                     checkpoint=checkpoint,
                                     notFoundString=notFoundString )
         intfCountersModel.inOctets = stat( 'inOctets' )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts' )
         intfCountersModel.inMulticastPkts = stat( 'inMulticastPkts' )
         intfCountersModel.inBroadcastPkts = stat( 'inBroadcastPkts' )
         intfCountersModel.inDiscards = stat( 'inDiscards' )
         intfCountersModel.outOctets = stat( 'outOctets' )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts' )
         intfCountersModel.outMulticastPkts = stat( 'outMulticastPkts' )
         intfCountersModel.outBroadcastPkts = stat( 'outBroadcastPkts' )
         intfCountersModel.outDiscards = stat( 'outDiscards' )
         # NOTE: do NOT use 'stat()' since last update time should never be a delta
         intfCountersModel.lastUpdateTimestamp = self.getUpdateTime( checkpoint )
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.inMulticastPkts = 0
         intfCountersModel.inBroadcastPkts = 0
         intfCountersModel.inDiscards = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0
         intfCountersModel.outMulticastPkts = 0
         intfCountersModel.outBroadcastPkts = 0
         intfCountersModel.outDiscards = 0
         intfCountersModel.lastUpdateTimestamp = 0.0

      return intfCountersModel
   
   def routingSupported( self ):
      # This method indicates whether or not routing may be supported
      # on the interface i.e whether to allow routing related commands
      # on the interface or not.
      return True

   def routingCurrentlySupported( self ):
      # This method indicates whether or not routing is currently supported
      # on the interface i.e whether or not the interface appears in the output
      # of "show" commands.
      intfStatus = allIntfStatusDir.intfStatus.get( self.name )
      if intfStatus and intfStatus.forwardingModel == "intfForwardingModelRouted":
         return True
      return False

   def vrfSupported( self ):
      # Whether we allow 'vrf forwarding' command on the interface.
      return self.routingSupported()

   def isUnicastInetAddress( self, a1 ):
      # Check for non-unicast.

      if IpAddrMatcher.isReservedIpAddr( a1.address ) or\
             IpAddrMatcher.validateMulticastIpAddr( a1.address ) is None:
         return (False, "IP address must be unicast" )

      elif  Arnet.Subnet( a1 ).isBroadcastAddress():
         return (False, "IP address must not be broadcast" )

      elif Arnet.Subnet( a1 ).isAllZerosAddress():
         
         return (False, "IP address must not have a zero host number" )
      
      elif IpAddrMatcher.isMartianIpAddr( a1.address ):

         return ( False, "Not a valid host address" )

      else:
         return ( True, "" )

   def isValidHostInetAddress( self, a1 ):
      if a1.len >= 32:
         return ( False, "Prefix length must be less than 32" )
      elif IpAddrMatcher.isLoopbackIpAddr( a1.address ):
         return ( False, "IP address must not be loopback" )      
      else:
         return self.isUnicastInetAddress( a1 )

   def isUnicastInet6Address( self, a1 ):
      # Check for non-unicast.
      if a1.address.isMulticast:
         return ( False, "Address must be unicast" )
      if a1.address.isUnspecified:
         return ( False, "Invalid unicast address" )
      if a1.address.isLoopback:
         # Loopback address can be configured ONLY on lo. But lo is
         # not configurable thru EOS CLI. Reject it.
         return ( False, "Invalid address for this interface" )
      return ( True, "" )

   def isValidHostInet6Address( self, a1 ):
      if a1.len >= 128:
         return ( False, "Prefix length must be less than 128" )

      return self.isUnicastInet6Address( a1 )

   #----------------------------------------------------------------------------
   # Only a subinterface will return True
   #----------------------------------------------------------------------------
   def isSubIntf( self ):
      return False

   #----------------------------------------------------------------------------
   # Only an interface that can be created dynamically will return a dyn config
   #----------------------------------------------------------------------------
   def dynConfig( self ):
      return None

   #----------------------------------------------------------------------------
   # Method stubs to be defined in child classes
   #----------------------------------------------------------------------------

   def createPhysical( self, startupConfig=False ):
      raise NotImplementedError()

   def destroyPhysical( self ):
      raise NotImplementedError()

   def lookupPhysical( self ):
      raise NotImplementedError()

   def showPhysical( self, mode, intfStatusModel ):
      raise NotImplementedError()
   
   def config( self ):
      raise NotImplementedError()

   def status( self ):
      raise NotImplementedError()
   
   def bandwidth( self ):
      raise NotImplementedError()
   
   def hardware( self ):
      raise NotImplementedError()
   
   def addrStr( self ):
      raise NotImplementedError()

class VirtualIntf( Intf ):
   # This is an abstract class, so don't warn about abstract inherited
   # methods
   # pylint: disable-msg=W0223
   def noInterface( self ):
      self.destroy()

class VxlanVirtualIntf( VirtualIntf ):
   # This is an abstract class for vxlan interface cli
   # pylint: disable-msg=W0223
   def noInterface( self ):
      self.destroy()

# canShutdownIfHook extensions accept one argument: a list of
# IntfConfigMode objects on which 'shutdown' is being called, and
# returns True if the interfaces can be shutdown and False otherwise.
canShutdownIfHook = CliExtensions.CliHook()

# canNoShutdownIfHook extensions accept one argument: a list of
# IntfConfigMode objects on which 'no shutdown' is being called, and
# returns True if the interfaces can be no shutdown and False otherwise.
canNoShutdownIfHook = CliExtensions.CliHook()

# shutdownIfHook extensions accept one argument: a list of
# IntfConfigMode objects on which 'shutdown' is being called. This 
# extension provides a callback to VirtualIntfs so that they can do
# housekeeping when interface is being shutdown or brought back up
shutdownIfHook = CliExtensions.CliHook()

class HiddenIntfFilterMatcher( CliMatcher.Matcher ):
   """This is a rule that filters Intf.matcher with specified tags.
   It's not for general use. Instead, it's used as a Hidden matcher
   by intfRangeWithSingleExpression() so it can parse an interface with
   specified tags at startup-config time when the interface has
   not been created."""
   __slots__ = ( 'matcher_', 'tags_' )
   def __init__( self, matcher, tags, priority=None ):
      # by default use the underlying matcher's priority and value function
      self.matcher_ = matcher
      self.tags_ = tags
      CliMatcher.Matcher.__init__( self, priority=CliParser.PRIO_LOW )

   def match( self, mode, context, token ):
      if self.tags_ and context.state is None:
         # first token
         m = IntfRange.intfTagRe_.match( token )
         if m and m.group( 1 ) not in self.tags_:
            # We have a tag that is not allowed
            return CliParser.noMatch
      return self.matcher_.match( mode, context, token )

   def completions( self, mode, context, token ):
      # We are hidden, so no completion
      return []

   def valueFunction( self, context ):
      return self.matcher_.valueFunction( context )

def intfRangeWithSingleExpression( name, explicitIntfTypes=None ):
   """
   WARNING: it's probably NOT what you want to use!

   One weakness of IntfRangeMatcher is that cannot match physical interfaces that
   do not have an interface config, such as Ethernet. If you have a config command
   with interface range for physical interfaces, it needs to come after interface
   modes.

   There are a couple of commands that do not do exactly this.
   1. The interface command itself. We use this expression so we could parse single
      physical interface and create the interface config.
   2. Some IGMP snooping commands take an interface range but outputs individual
      interfaces in running-config, and it comes before interface mode (not sure if
      it's a good idea to change it now).

   In these cases, we add a hidden single-interface matcher (IntfCli.Intf.matcher)
   so we could parse it.

   The value of this rule is either a MultiRangeRule.IntfList or a single
   Intf object. The Cli command handler should pass the value to
   Intf.getAll() to return a list of interfaces. If getAll() returns None,
   no interfaces are available.
   """
   tags = ( set( [ x.tagLong for x in explicitIntfTypes ] ) if explicitIntfTypes
            else None )
   nameSingle = name + '_SINGLE'
   nameRange = name + '_RANGE'
   class IntfRangeConfigExpression( CliCommand.CliExpression ):
      expression = nameSingle + " | " + nameRange
      data = {
         nameSingle : HiddenIntfFilterMatcher( Intf.matcher, tags ),
         nameRange : IntfRange.IntfRangeMatcher(
            explicitIntfTypes=explicitIntfTypes,
            earlySingletonReject=True )
      }
      @staticmethod
      def adapter( mode, args, argsList ):
         intf = args.get( nameSingle ) or args.get( nameRange )
         if intf:
            args[ name ] = intf
   return IntfRangeConfigExpression

#-------------------------------------------------------------------------------
# The "config-if" mode.
#-------------------------------------------------------------------------------
class IntfConfigModeDependent( IntfDependentBase ):
   def setDefault( self ):
      IntfConfigMode.setDefault( self.intf_ )

class IntfConfigMode( IntfMode, BasicCli.ConfigModeBase ):

   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Interface configuration'
   modeParseTree = CliParser.ModeParseTree()
   nameRe = re.compile( '([A-Za-z]+)' )

   #----------------------------------------------------------------------------
   # Constructs a new IntfConfigMode instance for a particular Intf object.
   #
   # Note that the intf attribute must be set up before calling the superclass
   # constructor, since the Modelet instances created by the superclass
   # constructor need to use it.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, intf ):
      self.intf = intf
      self.intfRangeRef = None
      IntfMode.__init__( self, self.intf.name, '%s' % self.intf.shortname )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      # Share instance rules for interfaces of the same type
      #
      # We use interface tag + class name, because
      # 1. Same class name can refer to different types (EthPhyIntf)
      # 2. Same tag can refer to different types (ma0 and ma1)
      m = self.nameRe.match( self.intf.name )
      tag = m.group( 1 )
      return self.intf.__class__.__name__ + ':' + tag

   @staticmethod
   def setDefault( intf ):
      intf.config().adminEnabled = True
      if intf.name.startswith( "Ethernet" ):
         if globalIntfConfig.defaultEthernetShutdown == 'shutdown':
            intf.config().adminEnabled = False
      intf.config().description = ''
      intf.config().linkStatusLogging = 'useGlobal'
      intf.config().loadInterval = defaultLoadInterval()
      intf.removeComments()

   def maybeIntfRangeCliHook( self, hook, hookName, hookArgsCallback=None ):
      ''' If this intfMode is part of a range, call
      IntfRangeConfigMode.intfRangeHook and return the
      result. Otherwise, just call the hook and return the result. See
      the IntfRangeConfigMode.intfRangeHook docstring for a
      description of the arguments.'''
      intfRange = self.intfRange()
      if intfRange is not None:
         return intfRange.intfRangeCliHook( self, hook, hookName,
                                            hookArgsCallback )
      else:
         hookArgs = [ [self] ]
         if hookArgsCallback:
            hookArgsCallback( hookArgs )
         return hook( *hookArgs )

   def setShutdown( self, no=False ):
      if no:
         hooks = canNoShutdownIfHook.extensions()
         hookName = 'canNoShutdownHook'
      else:
         hooks = canShutdownIfHook.extensions()
         hookName = 'canShutdownHook'

      for hook in hooks:
         if not self.maybeIntfRangeCliHook( hook, hookName ):
            return

      self.intf.config().adminEnabled = no

      for shutdownHook in shutdownIfHook.extensions():
         self.maybeIntfRangeCliHook( shutdownHook, 'shutdownHook' )

   def setDescription( self, desc ):
      self.intf.config().description = desc

   def noDescription( self ):
      self.intf.config().description = ''

   def setLoadInterval( self, interval ):
      self.intf.config().loadInterval = newLoadInterval( interval )

   def noLoadInterval( self ):
      self.intf.config().loadInterval = defaultLoadInterval()

   def setLoggingLinkStatus( self ):
      self.intf.config().linkStatusLogging = 'on'

   def setLoggingLinkStatusUseGlobal( self ):
      self.intf.config().linkStatusLogging = 'useGlobal'

   def noLoggingLinkStatus( self ):
      self.intf.config().linkStatusLogging = 'off'

   def intfRangeIs( self, obj ):
      self.intfRangeRef = weakref.ref(obj)

   def intfRange( self ):
      return self.intfRangeRef() if self.intfRangeRef else None


interfaceKwMatcher = CliMatcher.KeywordMatcher( 'interface',
                                   helpdesc='Interface Configuration' )
intfAfterServiceKwMatcher = CliMatcher.KeywordMatcher( 'interface',
                                   helpdesc='Change interface related parameters' )

class InterfaceDefaultsConfigMode( CliMode.InterfaceDefaults.InterfaceDefaultsMode,
                                   BasicCli.ConfigModeBase ):
   name = "Interface Defaults Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.InterfaceDefaults.InterfaceDefaultsMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class IntfDefaults( CliCommand.CliCommandClass ):
   syntax = "interface defaults"
   data = {
      "interface" : interfaceKwMatcher,
      "defaults" : ( "Set attribute values to use in absence of specific "
                     "configuration (user configurable defaults)" )
      }
   @staticmethod
   def handler( mode, args ):
      newMode = mode.childMode( InterfaceDefaultsConfigMode )
      mode.session_.gotoChildMode( newMode )

# FIXME: We need a "no/default interface defaults" command
BasicCli.GlobalConfigMode.addCommandClass( IntfDefaults )

#-------------------------------------------------------------------------------
# The interface-defaults mode "[no] mtu" command.
#-------------------------------------------------------------------------------
class GlobalL3MtuCmd( CliCommand.CliCommandClass ):
   syntax = "mtu MTU"
   noOrDefaultSyntax = "mtu ..."
   data = { 
           'mtu' : 'Set default MTU for all Layer 3 interfaces',
           'MTU' : CliMatcher.IntegerMatcher( 68, 65535,
              helpdesc='Maximum transmission unit in bytes' ),
   }

   @staticmethod
   def handler( mode, args ):
      globalL3Mtu = args[ 'MTU' ]
      globalIntfConfig.l3Mtu = globalL3Mtu

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      globalIntfConfig.l3Mtu = globalIntfConfig.l3MtuDefault

InterfaceDefaultsConfigMode.addCommandClass( GlobalL3MtuCmd )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>]" command, in enable mode.
#  show interfaces [<name>]
#  show interfaces module <modnum>
#-------------------------------------------------------------------------------
moduleExpressionFactory_ = None

def registerModularExpression( moduleExpr ):
   global moduleExpressionFactory_
   moduleExpressionFactory_ = moduleExpr

# This is a wrapper expression factory that can be used in ShowIntfCommand
# as moduleExpressionFactory_ is None initially and cannot be used directly.
class ModuleWrapperFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      if moduleExpressionFactory_:
         return moduleExpressionFactory_.generate( name )
      else:
         # Nobody set moduleExpressionFactory. Just return an invalid expression.
         class NullExpr( CliCommand.CliExpression ):
            expression = "INVALID"
            data = dict( INVALID=CliCommand.Node( CliMatcher.KeywordMatcher(
               "<INVALID>", helpdesc="INVALID" ), hidden=True ) )
         return NullExpr

interfacesShowKw = CliMatcher.KeywordMatcher(
   'interfaces', helpdesc='Details related to interfaces' )

class ShowIntfCommand( ShowCommand.ShowCliCommandClass ):
   # syntax: show interfaces [ INTF | module MOD ] EXTRA_SYNTAX
   # if moduleAtEnd is True, also add: show interfaces EXTRA_SYNTAX module MOD
   ALLOWED_FIELDS = tuple( list( ShowCommand.ShowCliCommandClass.ALLOWED_FIELDS ) +
                           [ 'moduleAtEnd' ] )
   baseSyntax_ = "show interfaces [ INTF | MOD ] %s"
   baseSyntaxWithModuleAtEnd_ = "show interfaces ( [ INTF | MOD ] %s ) " \
                                "| ( %s MOD )"
   baseData_ = {
      'interfaces' : interfacesShowKw,
      'INTF' : Intf.rangeMatcher,
      'MOD' : ModuleWrapperFactory()
   }
   dynamicSyntax_ = True
   moduleAtEnd = False

   @classmethod
   def _generateSyntaxAndData( cls ):
      cls._assert( cls.syntax.startswith( "show interfaces " ),
                   "must define syntax with 'show interfaces ...'" )
      cls._assert( 'INTF' not in cls.syntax and 'MOD' not in cls.syntax,
                   "INTF/MOD syntax is automatically generated" )
      for k in cls.baseData_:
         cls._assert( k not in cls.data, "'%s' should not be defined in data" % k )
      extraSyntax = cls.syntax[ 16: ]
      if cls.moduleAtEnd:
         cls.syntax = cls.baseSyntaxWithModuleAtEnd_ % ( extraSyntax,
                                                         extraSyntax )
      else:
         cls.syntax = cls.baseSyntax_ % extraSyntax

      cls.userData_ = copy.copy( cls.data )
      cls.data.update( cls.baseData_ )

   @classmethod
   def _expandData( cls ):
      data = super( ShowIntfCommand, cls )._expandData()
      for k, v in data.iteritems():
         if k not in cls.userData_:
            if isinstance( v, CliMatcher.Matcher ):
               data[ k ] = CliCommand.Node( matcher=v, overrideCanMerge=True ) 
            else:
               v.overrideCanMerge_ = True
      return data

def showInterfaces( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfStatuses = IntfModel.InterfaceStatuses()
   intfs = Intf.getAll( mode, intf, mod, sort=False )
   if not intfs:
      return intfStatuses
   intfs = [ i for i in intfs if i.lookup() ] # See bug 9124
   if not intfs:
      return intfStatuses

   for x in intfs:
      # pylint: disable-msg=E1101
      intfStatuses.interfaces[ x.status().intfId ] = x.show( mode )
      TacSigint.check()

   return intfStatuses

class ShowInterfaces( ShowIntfCommand ):
   syntax = "show interfaces "
   data = {}
   cliModel = IntfModel.InterfaceStatuses
   handler = showInterfaces

BasicCli.addShowCommandClass( ShowInterfaces )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] description" command, in enable mode.
#  show interfaces [<name>] description
#  show interfaces module <modnum> description
#-------------------------------------------------------------------------------
def showInterfacesDescription( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfDescriptions = IntfModel.InterfacesDescriptions() 
   intfs = Intf.getAll( mode, intf, mod )
   if not intfs:
      return intfDescriptions

   intfs = [ i for i in intfs if i.lookup() ] # See bug 9124
   if not intfs:
      return intfDescriptions

   for x in intfs:
      intfDescription = IntfModel.InterfacesDescriptions.InterfaceDescription()
      intfDescription.description = x.description()
      intfDescription.lineProtocolStatus = x.lineProtocolState()
      intfDescription.interfaceStatus = x.getIntfState()
      intfDescriptions.interfaceDescriptions[ x.name ] = intfDescription

   return intfDescriptions

class ShowIntfDescription( ShowIntfCommand ):
   syntax = 'show interfaces description'
   data = dict( description='Details on description strings of interfaces' )
   cliModel = IntfModel.InterfacesDescriptions
   handler = showInterfacesDescription

BasicCli.addShowCommandClass( ShowIntfDescription )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] config-sanity" command, in enable mode.
#  show interfaces [<name>] config-sanity
#  show interfaces module <modnum> config-sanity
#-------------------------------------------------------------------------------

def showInterfacesConfigSanity( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfConfigSanities = IntfModel.InterfaceStatuses()
   intfs = Intf.getAll( mode, intf, mod )
   
   for i in intfs:
      if i.lookup():
         intfConfigSanities.interfaces[ i.name ] = i.showConfigSanity( mode )

   return intfConfigSanities

class ShowIntfConfigSanity( ShowIntfCommand ):
   syntax = 'show interfaces config-sanity'
   data = {
      'config-sanity' : 'Check for interface configuration errors',
   }
   cliModel = IntfModel.InterfaceStatuses
   handler = showInterfacesConfigSanity

BasicCli.addShowCommandClass( ShowIntfConfigSanity )

def captureCounterState( counter, checkpoint, notFoundString='n/a' ):
   """
   Returns a function which takes a specific counter name as an
   argument (such as "inOctets") and returns the current value of that
   specific counter (or the 'notFoundString' text if the counter is
   not defined) given an 'IntfCounter' object and a counter
   checkpoint.

   Note that for counters defined as 'Ark::MaybeU63', a value of
   'None' in Python corresponds to Ark::MaybeU63::null being
   true. 'notFoundString' will be returned in these cases.
   """
   def stat( attr ):
      if counter is None:
         # Unders some circumstances (e.g. a linecard being hotswapped), the counter
         # argument may be None.  In theory, the right thing to do in this case
         # would be to return notFoundString.  This would however cause some non-
         # optional counter attributes in the interface counters model (e.g.,
         # inOctets, inUcastPkts, etc.) to be missing.  Because of that we for
         # now simply return 0.  We have BUG90662 to track a more complete solution.
         return 0

      sysdbValue = getattr( counter.statistics, attr, None )
      
      if sysdbValue is None:
         sysdbValue = getattr( counter.rates, attr, None )

      if sysdbValue is None:
         return notFoundString
  
      checkpointValue = 0
      if checkpoint:
         checkpointValue = getattr( checkpoint.statistics, attr, None )

         if checkpointValue is None:
            checkpointValue = getattr( checkpoint.rates, attr, 0 )

      return sysdbValue - checkpointValue

   return stat

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters delta" command, in enable mode.
#  show interfaces [<name>] counters delta
#  show interfaces module <modnum> counters delta
#  show interfaces counters delta module <modnum>
#-------------------------------------------------------------------------------
countersDeltaKw = CliMatcher.KeywordMatcher( 'delta',
                                       helpdesc='Interface counters delta' )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters" command, in enable mode.
#  show interfaces [<name>] counters
#  show interfaces module <modnum> counters
#  show interfaces counters module <modnum>
#-------------------------------------------------------------------------------
countersKw = CliMatcher.KeywordMatcher( 'counters',
                                       helpdesc='Interface counters' )
def clearLastCounters( mode ):
   _deltaCounterDir( mode ).clear()

def showInterfacesCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   direction = args.get( 'incoming' ) or args.get( 'outgoing' )

   intfsCounters = IntfModel.InterfacesCounters()
   if direction is not None:
      intfsCounters._outputType = direction

   # Intf.getAll  may return None
   intfs = counterSupportedIntfs( mode, intf, mod, sort=False )
   if not intfs:
      return intfsCounters

   for x in intfs:
      if 'delta' in args:
         intf = _deltaCounterDir( mode ).get( x.name )
         zeroOut = intf is None or intf.intfCounter.get( x.name ) is None
         checkpoint = None if zeroOut else intf.intfCounter.get( x.name )

         intfCountersModel = x.updateInterfaceCountersModel( checkpoint=checkpoint,
                                                             zeroOut=zeroOut )
         intfsCounters.interfaces[ x.status().intfId ] = intfCountersModel
      else:
         checkpoint = x.getLatestCounterCheckpoint()
         intfCountersModel = x.updateInterfaceCountersModel( checkpoint=checkpoint )
         intfsCounters.interfaces[ x.status().intfId ] = intfCountersModel
         x.updateLastCounters( mode )

   return intfsCounters


#-------------------------------------------------------------------------------
# The "show interfaces counters incoming", "show interfaces counters outgoing",
# "show interfaces counters delta incoming", and
# "show interfaces counters delta outgoing" commands, in enable mode.
# show interfaces [module <modnum>|<name>] counters [delta] [incoming|outgoing]
# show interfaces counters [delta] [incoming|outgoing] module <modnum>
#-------------------------------------------------------------------------------
class ShowIntfCounters( ShowIntfCommand ):
   syntax = "show interfaces counters [ delta ] [ incoming | outgoing ]"
   data = dict( counters=countersKw,
                delta='Interface counters delta',
                incoming='Incoming counters',
                outgoing='Outgoing counters' )
   cliModel = IntfModel.InterfacesCounters
   handler = showInterfacesCounters
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCounters )

#-------------------------------------------------------------------------------
# The "show interfaces counters discards" command, in enable mode.
#  show interfaces [<name>] counters discards
#  show interfaces module <modnum> counters discards
#  show interfaces counters discards module <modnum>
#-------------------------------------------------------------------------------
def showInterfacesCountersDiscards( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfsCountersDiscards = IntfModel.InterfacesCountersDiscards()
   intfs = counterDiscardSupportedIntfs( mode, intf, mod )
   if not intfs:
      return intfsCountersDiscards
   
   for x in intfs:
      intfId = x.status().intfId 
      intfCountersDiscardsModel = IntfModel.InterfaceCounterDiscards( _name=intfId )
      stat = captureCounterState( counter=x.counter(),
                                  checkpoint=x.getLatestCounterCheckpoint(),
                                  notFoundString='N/A' )

      inDiscards = stat( 'inDiscards' )
      intfCountersDiscardsModel.inDiscards = inDiscards

      # 'inDiscards' should allways be updated, but leave this check
      # in just in case to avoid a run-time error.
      intfsCountersDiscards.inDiscardsTotal += ( inDiscards if inDiscards
                                                    is not 'N/A' else 0 )

      outDiscards = stat( 'outDiscards' )

      # If 'outDiscards' is not being updated for this interface, then
      # don't update any totals.
      if outDiscards != 'N/A':
         intfCountersDiscardsModel.outDiscards = outDiscards
         if intfsCountersDiscards.outDiscardsTotal is not None:
            intfsCountersDiscards.outDiscardsTotal += outDiscards
         else:
            intfsCountersDiscards.outDiscardsTotal = outDiscards

      intfsCountersDiscards.interfaces[ intfId ] = intfCountersDiscardsModel

   return intfsCountersDiscards

class ShowIntfCountersDiscards( ShowIntfCommand ):
   syntax = "show interfaces counters discards"
   data = dict( counters=countersKw,
                discards='Packet discard counters' )
   cliModel = IntfModel.InterfacesCountersDiscards
   handler = showInterfacesCountersDiscards

BasicCli.addShowCommandClass( ShowIntfCountersDiscards )

#-------------------------------------------------------------------------------
# The "show interfaces counters ingress acl drop" command, in enable mode.
#  show interfaces [<name>] counters ingress acl drop
#-------------------------------------------------------------------------------
def showInterfacesCountersAclDrops( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   _maybeMountAclDropCounters()

   intfsCountersAclDrop = IntfModel.InterfacesCountersAclDrop()
   intfs = counterSupportedIntfs( mode, intf, mod, sort=False )

   if not intfs:
      return intfsCountersAclDrop

   for x in intfs:
      intfId = x.status().intfId
      intfCountersAclDropModel = IntfModel.InterfacesCounterAclDrop( _name=intfId )
      aclDropKey =  _getAclDropConfigKey( x, allIntfAclDropCounters )
      if aclDropKey is None:
         continue
      aclDropCounterDir = allIntfAclDropCounters[ aclDropKey ].counter.get( intfId )
      if aclDropCounterDir:
         intfCountersAclDropModel.aclDrops = aclDropCounterDir.inAclDrops
         intfsCountersAclDrop.interfaces[ intfId ] = intfCountersAclDropModel

   return intfsCountersAclDrop

ingressDirectionKw = CliMatcher.KeywordMatcher( 'ingress',
                                          helpdesc='Ingress counter information' )

class ShowIntfCountersIngressAclDrop( ShowIntfCommand ):
   syntax = 'show interfaces counters ingress acl drop'
   data = dict( counters=countersKw,
                ingress=ingressDirectionKw,
                acl='Ingress ACL information',
                drop='Ingress ACL drop information' )
   cliModel = IntfModel.InterfacesCountersAclDrop
   handler = showInterfacesCountersAclDrops
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCountersIngressAclDrop )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters rates" command.
#   show interfaces [<name>] counters rates
#   show interfaces module <modnum> counters rates
#-------------------------------------------------------------------------------
countersRatesKw = CliMatcher.KeywordMatcher( 'rates',
                                             helpdesc='Input/Output rate counters' )

def showInterfacesCountersRates( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfsCountersRates = IntfModel.InterfaceCountersRates()
   # Intf.getAll  may return None
   intfs = countersRateSupportedIntfs( mode, intf, mod )
   if not intfs:
      return intfsCountersRates

   for x in intfs:
      if x.addrStr():
         intfsCountersRates.interfaces[ x.name_ ] = \
               x.updateInterfaceCountersRateModel()

   return intfsCountersRates

class ShowIntfCountersRates( ShowIntfCommand ):
   syntax = "show interfaces counters rates"
   data = dict( counters=countersKw,
                rates=countersRatesKw )
   cliModel = IntfModel.InterfaceCountersRates
   handler = showInterfacesCountersRates

BasicCli.addShowCommandClass( ShowIntfCountersRates )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters errors" command.
#-------------------------------------------------------------------------------
countersErrorsKw = CliMatcher.KeywordMatcher( 'errors',
                                              helpdesc='Error counters' )

def showInterfacesCountersErrors( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   allInterfacesCountersErrors = IntfModel.InterfacesErrorCounters()
   allIntf = countersErrorsSupportedIntfs( mode, intf=intf, mod=mod) 
   if not allIntf:
      return allInterfacesCountersErrors
   for oneIntf in allIntf:
      intfModel = oneIntf.getCountersErrorsModel()
      allInterfacesCountersErrors.interfaceErrorCounters[ oneIntf.name ] = intfModel
   return allInterfacesCountersErrors

class ShowIntfCountersErrors( ShowIntfCommand ):
   syntax = "show interfaces counters errors"
   data = dict( counters=countersKw,
                errors=countersErrorsKw )
   cliModel = IntfModel.InterfacesErrorCounters
   handler = showInterfacesCountersErrors
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCountersErrors )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters half-duplex" command.
#-------------------------------------------------------------------------------

countersHalfDuplexKw = CliMatcher.KeywordMatcher( 'half-duplex',
                                                  helpdesc='Half-duplex error '
                                                           'counters' )

def showInterfacesCountersHalfDuplexErrors( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   allIntfsCountersHalfDuplexErrors = IntfModel.InterfacesHalfDuplexErrorCounters()
   allIntf = countersErrorsSupportedIntfs( mode, intf=intf, mod=mod )
   # allIntf may be None.
   if not allIntf:
      return allIntfsCountersHalfDuplexErrors
   halfDuplexCounters = allIntfsCountersHalfDuplexErrors.intfHalfDuplexErrorCounters
   for oneIntf in allIntf:
      intfModel = oneIntf.getCountersHalfDuplexErrorsModel()
      halfDuplexCounters[ oneIntf.name ] = intfModel
   return allIntfsCountersHalfDuplexErrors

class ShowIntfCountersHalfDuplexErrors( ShowIntfCommand ):
   syntax = "show interfaces counters half-duplex"
   data = {
      "counters": countersKw,
      "half-duplex": countersHalfDuplexKw,
   }
   cliModel = IntfModel.InterfacesHalfDuplexErrorCounters
   handler = showInterfacesCountersHalfDuplexErrors
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfCountersHalfDuplexErrors )

#-------------------------------------------------------------------------------
# The "clear counters [interface] [session]" command, in "privileged exec" mode.
#-------------------------------------------------------------------------------

# Allow clear counters callbacks
clearCountersHook = []

def registerClearCountersHook( hook ):
   clearCountersHook.append( hook )

def clearCounters( mode, intf=None, mod=None, sessionOnly=False):
   intfs = Intf.getAll( mode, intf, mod )
   if not intfs:
      return

   if not sessionOnly:
      if intf is None:
         Log.logClearCounters( "", "all interfaces" )
      else:
         if len( intfs ) > 1:
            Log.logClearCounters( "", "interfaces %s to %s" %
                                        ( intfs[0].shortname,
                                          intfs[-1].shortname ) )
         else:
            Log.logClearCounters( "", "interface " + intfs[ 0 ].shortname )

   for i in intfs:
      i.clearCounters( sessionOnly=sessionOnly )
   clearLastCounters( mode ) 

   for hook in clearCountersHook:
      hook( mode, intfs, sessionOnly, intf is None and mod is None )

counterSessionKw = CliMatcher.KeywordMatcher(
   'session',
   helpdesc='Clear for this CLI session only' )

class ClearCountersCmd( CliCommand.CliCommandClass ):
   syntax = "clear counters [ INTF ] [ session ]"
   data = dict( clear=CliToken.Clear.clearKwNode,
                counters=countersKw,
                INTF=Intf.rangeMatcher,
                session=counterSessionKw )
   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      clearCounters( mode, intf, mod=None, sessionOnly='session' in args )

BasicCli.EnableMode.addCommandClass( ClearCountersCmd )

#-------------------------------------------------------------------------------
# The "[no|default] shutdown" command, in "config-if" mode.
#-------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = "shutdown"
   noSyntax = syntax
   defaultSyntax = syntax
   data = dict( shutdown="Administratively shut off the interface" )

   @staticmethod
   def handler( mode, args ):
      mode.setShutdown( no=False )

   @staticmethod
   def noHandler( mode, args ):
      mode.setShutdown( no=True )

   @staticmethod
   def defaultHandler( mode, args ):
      # if default state is shutdown, pass no=False (shutdown)
      no = ( globalIntfConfig.defaultEthernetShutdown != 'shutdown' )
      mode.setShutdown( no=no )

IntfConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# The "[no] description <desc>" command, in "config-if" mode.
#-------------------------------------------------------------------------------
class DescriptionCmd( CliCommand.CliCommandClass ):
   syntax = "description DESC"
   noOrDefaultSyntax = "description ..."
   data = dict( description='Description string to associate with the interface',
                DESC=CliMatcher.StringMatcher(
                   helpname='LINE',
                   helpdesc='Description for this interface' ) )
   @staticmethod
   def handler( mode, args ):
      mode.setDescription( args[ 'DESC' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noDescription()

IntfConfigMode.addCommandClass( DescriptionCmd )

#-------------------------------------------------------------------------------
# The "[no|default] load-interval <interval>" command, in "config-if" mode.
#-------------------------------------------------------------------------------

# instantiate temp LoadInterval object so that we can get the min/max values
LoadInterval = Tac.Type( 'Interface::LoadInterval' )

intervalMatcher = CliMatcher.IntegerMatcher(
   LoadInterval.minVal,
   LoadInterval.maxVal,
   helpdesc='Number of seconds' )

class LoadIntervalCmd( CliCommand.CliCommandClass ):
   syntax = "load-interval INTERVAL"
   noOrDefaultSyntax = "load-interval ..."
   data = {
      "load-interval" : 'Time used in calculating interface utilization',
      "INTERVAL" : intervalMatcher
      }
   @staticmethod
   def handler( mode, args ):
      mode.setLoadInterval( args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noLoadInterval()

IntfConfigMode.addCommandClass( LoadIntervalCmd )

#-------------------------------------------------------------------------------
# The "[no|default] load-interval default <interval>" command, in "config" mode.
#-------------------------------------------------------------------------------
class LoadIntervalDefaultCmd( CliCommand.CliCommandClass ):
   syntax = "load-interval _default_ INTERVAL"
   noOrDefaultSyntax = "load-interval _default_ ..."
   data = {
      "load-interval" : 'Specify global interval for load calculation on interfaces',
      "_default_" : CliMatcher.KeywordMatcher(
         'default',
         helpdesc='Default load-interval configuration' ),
      "INTERVAL" : intervalMatcher
      }
   @staticmethod
   def handler( mode, args ):
      globalIntfConfig.loadInterval = newLoadInterval( args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      globalIntfConfig.loadInterval = defaultLoadInterval()

BasicCli.GlobalConfigMode.addCommandClass( LoadIntervalDefaultCmd )

# helper function to get actual load-interval value:
#   if default at interface level
#      use global one (default or user-defined)
#   else
#      use user-defined at interface level
def getActualLoadIntervalValue( intfLoadInterv ):
   if intfLoadInterv.useDefault:
      globLoadInterv = globalIntfConfig.loadInterval
      if globLoadInterv.useDefault:
         return globLoadInterv.defaultVal
      else:
         return globLoadInterv.val
   else:
      return intfLoadInterv.val

# helper function to get printable string for load interval value
# (convert to minutes / seconds as appropriate)
def getLoadIntervalPrintableString( value ):
   value = int( round( value ) )
   minutes = value / 60
   seconds = value % 60
   intervalStr = ''
   if minutes > 0:
      intervalStr = '%d minute' % minutes
      if minutes > 1:
         intervalStr += 's'
      if seconds != 0:
         intervalStr += ', '
   if seconds != 0:
      intervalStr += '%d second' % seconds
      if seconds > 1:
         intervalStr += 's'
   return intervalStr

# return LoadInterval object w/ default value
def defaultLoadInterval():
   li = Tac.newInstance( "Interface::LoadInterval" ) 
   li.useDefault = True
   return li

# return LoadInterval object w/ non-default value
def newLoadInterval( value ):
   li = Tac.newInstance( "Interface::LoadInterval" ) 
   li.useDefault = False
   li.val = value
   return li

#-------------------------------------------------------------------------------
# Provide global 'logging event FEATURE' and interface mode
# 'logging event FEATURE [use-global]' base classes.
#-------------------------------------------------------------------------------
class LoggingEventGlobalCmd( CliCommand.CliCommandClass ):
   # Subclass should define:
   # syntax = "logging event FEATURE"
   # noOrDefaultSyntax
   # data = { "FEATURE" : ... }
   #
   # and implement handler/noOrDefaultHandler
   baseData_ = {
      'logging' : CliToken.Logging.loggingForConfig,
      'event' : CliToken.Logging.eventForConfig,
   }
   data = {}

   dynamicSyntax_ = True

   @classmethod
   def _generateSyntaxAndData( cls ):
      cls._assert( cls.syntax.startswith( "logging event " ),
                   "must define syntax with 'logging event ...'" )
      for k in cls.baseData_:
         cls._assert( k not in cls.data, "'%s' should not be defined in data" % k )
      cls.data.update( cls.baseData_ )

class LoggingEventIntfCmd( CliCommand.CliCommandClass ):
   # Subclass should define:
   # syntax = "logging event FEATURE"
   # data = { "FEATURE" : ... }
   #
   # and implement handler/noOrDefaultHandler
   #
   # and the following handlers:
   #
   # 1. _enableHandler( cls, mode, args ) ( handler without using 'use-global' )
   # 2. _disableHandler( cls, mode, args ) ( no-handler )
   # 3. _useGlobalHandler( cls, mode, args ) (default-handler)
   baseData_ = {
      'logging' : CliToken.Logging.loggingForConfigIf,
      'event' : CliToken.Logging.eventForConfigIf,
      'use-global' : CliToken.Logging.useGlobalForConfigIf
   }
   data = {}

   @classmethod
   def _mustImplementHandler( cls, mode, args ):
      assert False, "Subclass must implement _enable/_disable/_useGlobalHandler"

   _enableHandler = _disableHandler = _useGlobalHandler = _mustImplementHandler

   dynamicSyntax_ = True

   @classmethod
   def _generateSyntaxAndData( cls ):
      cls._assert( cls.syntax.startswith( "logging event " ),
                   "must define syntax with 'logging event ...'" )
      for k in cls.baseData_:
         cls._assert( k not in cls.data, "'%s' should not be defined in data" % k )
      cls.syntax += ' [ use-global ]'
      cls.noOrDefaultSyntax = cls.syntax + ' ...'
      cls.data.update( cls.baseData_ )

   @classmethod
   def handler( cls, mode, args ):
      if 'use-global' in args:
         cls._useGlobalHandler( mode, args )
      else:
         cls._enableHandler( mode, args )

   @classmethod
   def noHandler( cls, mode, args ):
      cls._disableHandler( mode, args )

   @classmethod
   def defaultHandler( cls, mode, args ):
      cls._useGlobalHandler( mode, args )

#-------------------------------------------------------------------------------
# The "[no|default] logging event link-status global" command, in "config" mode.
#-------------------------------------------------------------------------------
class LoggingEventLinkStatusGlobal( LoggingEventGlobalCmd ):
   syntax = "logging event link-status global"
   noOrDefaultSyntax = syntax
   data = {
      'link-status' : 'UPDOWN messages',
      'global' : 'Global link status configuration'
      }

   @staticmethod
   def handler( mode, args ):
      globalIntfConfig.linkStatusLogging = 'on'

   @staticmethod
   def noHandler( mode, args ):
      globalIntfConfig.linkStatusLogging = 'off'

   # default is 'on'
   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventLinkStatusGlobal )

#-------------------------------------------------------------------------------
# The "[no] logging event link-status" command.
# The "logging event link-status use-global" command.
#-------------------------------------------------------------------------------
class LoggingEventLinkStatusIntf( LoggingEventIntfCmd ):
   syntax = "logging event link-status"
   data = {
      'link-status' : 'UPDOWN messages',
   }
   @classmethod
   def _enableHandler( cls, mode, args ):
      mode.setLoggingLinkStatus()

   @classmethod
   def _disableHandler( cls, mode, args ):
      mode.noLoggingLinkStatus()

   @classmethod
   def _useGlobalHandler( cls, mode, args ):
      mode.setLoggingLinkStatusUseGlobal()

IntfConfigMode.addCommandClass( LoggingEventLinkStatusIntf )

inactiveKwMatcher = CliMatcher.KeywordMatcher( 'inactive',
   helpdesc='Inactive subordinate lanes in multi-lane mode' )
exposeKwMatcher = CliMatcher.KeywordMatcher( 'expose',
                                             helpdesc='Expose information' )
unconnectedKwMatcher = CliMatcher.KeywordMatcher( 'unconnected',
   helpdesc='Unconnected/Internal ethernet ports' )

class ExposeLanesCommand( CliCommand.CliCommandClass ):
   syntax = "service interface inactive | unconnected expose"
   noOrDefaultSyntax = syntax
   data = {
      "service": CliToken.Service.serviceKw,
      "interface": intfAfterServiceKwMatcher,
      "inactive": inactiveKwMatcher,
      "unconnected": unconnectedKwMatcher,
      "expose": exposeKwMatcher
      }

   @staticmethod
   def handler( mode, args ):
      if 'inactive' in args: # pylint: disable=simplifiable-if-statement
         globalIntfConfig.exposeInactiveLanes = True
      else:
         globalIntfConfig.exposeUnconnectedLanes = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'inactive' in args: # pylint: disable=simplifiable-if-statement
         globalIntfConfig.exposeInactiveLanes = False
      else:
         globalIntfConfig.exposeUnconnectedLanes = False

BasicCli.GlobalConfigMode.addCommandClass( ExposeLanesCommand )

################################################################################
#-------------------------------------------------------------------------------
# Adds routed interface specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class RoutedIntfDependent( IntfDependentBase ):
   def setDefault( self ):
      RoutedIntfModelet.setDefaultMtu( self.intf_ )

class RoutedIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ):
      CliParser.Modelet.__init__( self )
      self.intfConfigMode = intfConfigMode

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.routingSupported()

   @staticmethod    
   def setDefaultMtu( intf ):
      intf.config().l3MtuConfigured = False
      intf.config().mtu = globalIntfConfig.l3MtuDefault
      # Copy parent's mtu to its sub interfaces.
      for hook in subintfMtuHook:
         hook( intf.name, globalIntfConfig.l3MtuDefault )
      Tracing.trace0( "Modelet noMtu set", intf.name, "MTU to", \
              globalIntfConfig.l3MtuDefault )

   def setMtu( self, args ):
      mtu = args[ 'MTU' ]
      intf = self.intfConfigMode.intf
      if intf.lookup():
         intfStatus = intf.status()
         if ( intfStatus.maxMtu != 0 and
              intfStatus.maxMtu < mtu ):
            Tracing.trace0(   "Invalid MTU", mtu,
                              "for", intf.name,
                              "max is", intfStatus.maxMtu )
            self.intfConfigMode.addError( "Max MTU for %s is %d" %
                                          ( intf.name, intfStatus.maxMtu ) )
            return

      intf.config().l3MtuConfigured = True
      intf.config().mtu = mtu
      # Copy parent's mtu to its sub interfaces.
      for hook in subintfMtuHook:
         hook( intf.name, mtu )
      Tracing.trace0( "Modelet Set", intf.name, "MTU to", mtu )

   def noMtu( self, args ):
      # Clear the configured mtu (use default)
      intf = self.intfConfigMode.intf
      RoutedIntfModelet.setDefaultMtu( intf )

#-------------------------------------------------------------------------------
# Associate the RoutedIntfModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfConfigMode.addModelet( RoutedIntfModelet )

#-------------------------------------------------------------------------------
# The "[no] mtu" command.
# Also aliased to from the hidden "[no] ip mtu" command.
#-------------------------------------------------------------------------------
def mtuGuard( mode, token ):
   if mode.intf.isSubIntf():
      for hook in subintfMtuGuardHook:
         rc = hook( mode, token )
         return rc
   else:
      return None

mtuKw = CliCommand.Node(
      CliMatcher.KeywordMatcher( 'mtu',
         helpdesc='Set IP Maximum Transmission Unit size in bytes' ),
      guard=mtuGuard )

class MtuCmd( CliCommand.CliCommandClass ):
   syntax = "mtu MTU"
   noOrDefaultSyntax = "mtu ..."
   data = dict( mtu=mtuKw,
                MTU=CliMatcher.IntegerMatcher( 68, 65535,
                                    helpdesc='Maximum transmission unit in bytes' ) )
   handler = RoutedIntfModelet.setMtu
   noOrDefaultHandler = RoutedIntfModelet.noMtu

RoutedIntfModelet.addCommandClass( MtuCmd )

myEntManager = None

class VerifyInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'verify interface ( all | INTF )'
   data = {
      'verify': CliCommand.Node( CliMatcher.KeywordMatcher(
         'verify', helpdesc='Verify the state of an object in the system' ),
                                 hidden=True ),
      'interface': 'Verify the state of an interface',
      'all': 'Verify the state for all interfaces in the system',
      'INTF': Intf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      def ignoreIntf( intf ):
         intf = str( intf ).lower()
         return 'management' in intf or 'internal' in intf

      actors = [ 'Ebra', 'Lag', 'Ira', 'KernelFib' ]
      if 'all' in args:
         intfs = [ str( intf ) for intf in allIntfConfigDir.intfConfig.keys()
                               if not ignoreIntf( intf ) ]
      else:
         intf = str( args[ 'INTF' ] )
         intfs = [ intf ] if not ignoreIntf( intf ) else []

      # If total number of intfs exceeds VERIFY_MAX_INTFS limit, skip the
      # verification
      if len( intfs ) > VERIFY_MAX_INTFS:
         print "Skipping verification as total number of intfs exceeds %d" \
               % VERIFY_MAX_INTFS
      else:
         verify( 'Intf', intfs, actors, myEntManager )

BasicCli.EnableMode.addCommandClass( VerifyInterfaceCmd )

def getDefaultEthernetShutdown():
   return globalIntfConfig.defaultEthernetShutdown

def setDefaultEthernetShutdown( value ):
   globalIntfConfig.defaultEthernetShutdown = value

#--------------------------------------------------------------------------------
# [ no | default ] mac-address MAC_ADDR
# [ no | default ] mac-address router MAC_ADDR
#-------------------------------------------------------------------------------
def macAddressRouterSupportedGuard( mode, token ):
   if brHwCapabilities.staticMacOnRoutedIntfSupported:
      return None
   return CliParser.guardNotThisPlatform

matcherMacAddress = CliMatcher.KeywordMatcher( 'mac-address',
      helpdesc='Set interface MAC address' )

class MacAddressRouterAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-address router MAC_ADDR'
   noOrDefaultSyntax = 'mac-address [ router ] ...'
   data = {
      'mac-address' : matcherMacAddress,
      'router' : CliCommand.guardedKeyword(
         'router',
         helpdesc='Set interface MAC address to use for Routing',
         guard=macAddressRouterSupportedGuard ),
      'MAC_ADDR' : MacAddr.macAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      macAddr = args[ 'MAC_ADDR' ]
      mode.intf.setMacAddr( macAddr, routerMacConfigured=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setMacAddr( None )

#-------------------------------------------------------------------------------
# Mount the GlobalIntfConfig and other necessary Sysdb objects.
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global aclDropConfigDir
   global allIntfConfigDir
   global allIntfStatusDir
   global globalIntfConfig
   global myEntManager
   global brHwCapabilities

   myEntManager = entityManager

   globalIntfConfig = ConfigMount.mount( entityManager, "interface/config/global",
                                         "Interface::GlobalIntfConfig", "w" )
   allIntfConfigDir = LazyMount.mount( entityManager, 'interface/config/all',
                                       'Interface::AllIntfConfigDir', 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   aclDropConfigDir = LazyMount.mount( entityManager,
                                       "interface/aclDropCounter/aclDropConfigDir",
                                       "Tac::Dir", "ri" )
   brHwCapabilities = LazyMount.mount( entityManager, "bridging/hwcapabilities",
                                       "Bridging::HwCapabilities", "r" )

   Intf.registerDependentClass( RoutedIntfDependent )
   Intf.registerDependentClass( IntfConfigModeDependent )

   # register user configurable defaults with CliSession's rollback clean-config
   SessionCli.registerCustomizableDefaults( getDefaultEthernetShutdown,
                                            setDefaultEthernetShutdown )
