#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from cStringIO import StringIO
import math
import re
import textwrap

import AgentCommandRequest
import BasicCli
import CliCommand
from CliPlugin.KernelFibCli import RouterKernelConfigMode
import CliMatcher
import CliParser
import ConfigMount
import Sfa
import ShowCommand

entMan = None
interruptConfig = None
maxWidth = 100
maxIntSize = int( math.pow( 2, 32 ) - 1 )

#--------------------------------------------------------------------------------
# show router kernel counters interrupts [ EPOCH1 EPOCH2 rate ]
#--------------------------------------------------------------------------------
class RouterKernelCountersInterruptsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show router kernel counters interrupts [ EPOCH1 EPOCH2 rate ]'
   data = {
      'router' : 'Routing protocol commands',
      'kernel' : 'Kernel statistics and counters',
      'counters' : 'Kernel statistics and counters',
      'interrupts' : 'Interrupt statistics per CPU',
      'EPOCH1' : CliMatcher.PatternMatcher( pattern='\\d+',
         helpdesc='Show statistics at this epoch', helpname='<1-4294967295>' ),
      'EPOCH2' : CliMatcher.PatternMatcher( pattern='\\d+',
         helpdesc='Show statistics at this epoch', helpname='<1-4294967295>' ),
      'rate' : 'Display rate',
   }

   @staticmethod
   def handler( mode, args ):
      epoch1 = args.get( 'EPOCH1' )
      epoch2 = args.get( 'EPOCH2' )
      if epoch1 is None:
         showCurInterrupts( mode )
      else:
         showInterruptsHistory ( mode, int( epoch1 ), int( epoch2 ) )

BasicCli.addShowCommandClass( RouterKernelCountersInterruptsCmd )

# format the header and stats and print to CLI.
def formatPrint( formatStr, *args, **kwargs ):
   sp0 = maxWidth / 5
   sp1 = 6
   sp2 = 30
   sp3 = 13
   sp5 = 13
   sp6 = 7
   sp4 = sp5 + sp6
   sp7 = 3
   sp8 = 30
   # The caller of this function provides a string argument formatStr which
   # describes how the output is to be formatted, they are listed below.
   formatStrings = {
      'hisHeader1' : '{:^{sp1}}' + '{:^{sp2}}' + '{:>{sp3}}' + '{:^{sp7}}' + \
         ( '{:^{sp4}}' + '{:^{sp7}}' ) * 2,
      'hisHeader2' : '{:<{sp1}}' + '{:<{sp2}}' + '{:<{sp3}}' + '{:^{sp7}}' + \
         ( '{:^{sp5}}' + '{:>{sp6}}' + '{:^{sp7}}' ) * 2,
      'hisBody' : '{:<{sp1}}' + '{:>{sp2}}' + '{:>{sp3}}' + '{:^{sp7}}' + \
         ( '{:>{sp5}}' + '{:>{sp6}}' + '{:^{sp7}}' ) * 2,
      'curFormat' :  '{:<{sp0}}' + '{:>{sp8}}' + '{:>{sp8}}',
      'curSubHeader' :  '{:<{sp0}}' + '{:<{sp8}}' + '{:<{sp8}}',
   }

   string = formatStrings[ formatStr ].format( *args, **locals() )
   print textwrap.fill( string, width=maxWidth, subsequent_indent=' '*sp0 )

def connectToSfa( cmd ): 
   outputBuffer = StringIO()
   response = None
   AgentCommandRequest.runSocketCommand( entMan, Sfa.name, "CollectSamples",
                                         cmd, keepalive=True, timeout=10,
                                         stringBuff=outputBuffer )
   if outputBuffer is None:
      print "Error: Could not obtain statistics"
      return response
   if re.search( 'error', outputBuffer.getvalue() ):
      print "Error: Could not connect to Sfa agent"
      return response
   if re.search( 'Error:', outputBuffer.getvalue() ):
      print outputBuffer.getvalue()
      return response
   return outputBuffer.getvalue()

def showInterruptsHistory( mode, epoch1=3, epoch2=30 ):
   if epoch1 > maxIntSize or epoch2 > maxIntSize:
      print "Error: Maximum supported configuration is " + str( maxIntSize )
      return
   # send show history command to Sfa agent to process
   cmd = "interrupts history " + str( epoch1 ) + " " + str( epoch2 )
   response = connectToSfa( cmd )
   if response is None:
      return
   # read the stats from the socket and store them into lists
   getStat = lambda x : eval( re.search( x + '(.+?)epoch',
                              response ).group( 1 ).rstrip() )
   sample0 = getStat( 'epoch0' )
   sample1 = getStat( 'epoch1' )
   sample2 = getStat( 'epoch2' )

   # print headers to be displayed on the CLI
   formatPrint( 'hisHeader1', 'CPU', 'Statistic', 'Now', '',
      str( epoch1 ) + ' sec(s) ago', '', str( epoch2 ) + ' sec(s) ago', '' )
   formatPrint( 'hisHeader2', '', '', '', '', 'count', 'rate', '', 'count',
      'rate', '' )
   print "-" * maxWidth
   # compute the rate, format the output and print the stats
   for cpuStat0, cpuStat1, cpuStat2 in zip( sample0, sample1, sample2 ):
      formatPrint( 'hisBody', 'CPU' + str( cpuStat0[ 0 ] ), *[ '' ] * 9 )
      for fields0, fields1, fields2 in \
         zip( cpuStat0[ 1 ], cpuStat1[ 1 ], cpuStat2[ 1 ] ):
         if fields0[ 0 ] == '':
            continue
         if fields0[ 0 ] != "subheader":
            stat0 = fields0[ 1 ]
            stat1 = fields1[ 1 ]
            stat2 = fields2[ 1 ]
            rate1 = ( stat0 - stat1 ) / epoch1
            rate2 = ( stat0 - stat2 ) / epoch2
            formatPrint( 'hisBody', '', ' ' + fields0[ 0 ] , stat0, '',
               stat1, rate1, '', stat2, rate2, '' )
         else:
            formatPrint( 'hisHeader2', '', fields0[ 1 ],  *( [ '' ] * 8 ) )

def showCurInterrupts( mode ):
   # send show current command to Sfa agent to process
   cmd = "interrupts current"
   response = connectToSfa( cmd )
   if response is None:
      return
   # read the stats from the socket and store them into lists
   getStat = lambda x : eval( re.search( x + '(.+?)epoch',
                              response ).group( 1 ).rstrip() )
   sample0 = getStat( 'epoch0' )
   # print headers to be displayed on the CLI
   formatPrint( 'curFormat', "CPU", "Statistic" , "value" )
   print "-" * 80
   # format the output and print the stats
   for cpustat in sample0:
      formatPrint( 'curFormat', "CPU" + str( cpustat[ 0 ] ), "", "" )
      for fields in cpustat[ 1 ]:
         fieldName = fields[ 0 ]
         fieldValue = fields[ 1 ]
         if fieldName == '':
            continue
         if fieldName != "subheader":
            formatPrint( 'curFormat', "", "  " + fieldName , fieldValue )
         else:
            formatPrint( 'curSubHeader', "", "  " + fieldValue , "" )
      
#--------------------------------------------------------------------------------
# [ no | default ] counters interrupts interval INTERVAL secs samples SAMPLES
#--------------------------------------------------------------------------------
class RouterKernelStatConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'counters interrupts interval INTERVAL secs samples SAMPLES'
   noOrDefaultSyntax = 'counters interrupts'
   data = {
      'counters' : 'Kernel statistics and counters',
      'interrupts' : 'Interrupt statistics per CPU',
      'interval' : 'Statistics collection time interval',
      'INTERVAL' : CliMatcher.PatternMatcher( pattern='\\d+',
         helpdesc='Specify an interval at which statstics will be collected',
         helpname='<interval>' ),
      'secs' : 'Seconds',
      'samples' : 'Number of samples to be collected',
      'SAMPLES' : CliMatcher.PatternMatcher( pattern='\\d+',
         helpdesc='Specify maximum number of samples to be collected',
         helpname='<samples>' ),
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RouterKernelStatConfigMode )
      noOrDefault = None
      if CliCommand.isNoCmd( args ):
         noOrDefault = True
      elif CliCommand.isDefaultCmd( args ):
         noOrDefault = 'default'

      childMode.modifyStatConfig( noOrDefault,
            args.get( 'INTERVAL' ), args.get( 'SAMPLES' ) )

   noOrDefaultHandler = handler

RouterKernelConfigMode.addCommandClass( RouterKernelStatConfigModeCmd )

class RouterKernelStatConfigMode( BasicCli.ConfigModeBase ):
   name = 'Router Kernel Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.modeKey = "router-kernel-stat"
      self.longModeKey = "router-kernel-stat"
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def modifyStatConfig( self, noOrDefault=None, interval=None, samples=None ):
      if noOrDefault is True:
         # disable history collection
         interruptConfig.interval = 0
      elif noOrDefault is None:
         # enable history collection
         if int( interval ) > maxIntSize or int( samples ) > maxIntSize:
            print "Error: Maximum supported configuration is " + str( maxIntSize )
            return
         interruptConfig.totSamples = int( samples )
         interruptConfig.interval = int( interval )
      else:
         # revert history collection to interval 1 sec and 256 samples
         interruptConfig.totSamples = 256
         interruptConfig.interval = 1

def Plugin( entityManager ):
   global entMan
   global interruptConfig
   entMan = entityManager

   interruptConfig = ConfigMount.mount(
      entMan,
      "sfa/config/interrupt",
      "Sfa::InterruptStat::Config", "w" )
