#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import random, sys
import Tac
import BasicCliUtil
import Fru
import FruCli
import LazyMount, ReloadCli
import FileReplicationCmds, Url, SimpleConfigFile, Cell
import FileUrl

#------------------------------------------------------
# The "install" command, in "enable" mode
#
# The full syntax of this command is:
#
#   install image_src [image_dest] [reload] [now]
#------------------------------------------------------

class RevertChangesException( Exception ):
   def __init__( self, error='', aborted=False ):
      self.error = error
      self.aborted = aborted

class CommandFailedException( Exception ):
   def __init__( self, error='', aborted=False ):
      self.error = error
      self.aborted = aborted

class StandbyTransferException( Exception ):
   def __init__( self, error, isMessage=False, standbyMissing=False ):
      self.error = error
      self.isMessage = isMessage
      self.standbyMissing = standbyMissing

entityManager_ = None
isModularSystem_ = False
redundancyStatus_ = None

def printAndFlush( msg ):
   print msg,
   sys.stdout.flush()

class Installer( object ):
   # NOTE: Tac.run(...) uses stderr=Tac.CAPTURE because without it
   # Tac.SystemCommandError.output is an empty string in case the
   # command fails and throws the exception

   def __init__( self ):
      self.currId_ = Fru.slotId()
      self.bootConfFile_ = None
      self.swiFile_ = None
      self.tmpBootConfFile_ = None
      self.tmpSwiFile_ = None
      self.isModularSystem_ = False
      self.mode_ = None
      self.now_ = False
      self.skipDownload_ = False
   
   def _promptUser( self, question ):
      if self.now_:
         return True
      
      return BasicCliUtil.confirm( self.mode_, question )

   def _deleteTempFile( self, filename, onActive=True ):
      assert 'tmp' in filename

      if onActive:
         cmd = [ 'rm', '-f', filename ]
      else:
         cmd = FileReplicationCmds.deleteFile( self.currId_, useKey=True,
                                               target=filename )
      Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )

   def _revertChanges( self ):
      assert self.tmpSwiFile_ and self.tmpBootConfFile_

      printAndFlush( 'Trying to revert changes...' )
      try:
         if not self.skipDownload_:
            Tac.run( [ 'rm', '-f', self.tmpSwiFile_ ], stdout=Tac.DISCARD,
                     stderr=Tac.CAPTURE, asRoot=True )
         Tac.run( [ 'rm', '-f', self.tmpBootConfFile_ ], stdout=Tac.DISCARD,
                  stderr=Tac.CAPTURE, asRoot=True )
         if self.isModularSystem_:
            cmd = FileReplicationCmds.deleteFile( self.currId_, useKey=True,
                                                  target=self.tmpSwiFile_ )
            Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
            cmd = FileReplicationCmds.deleteFile( self.currId_, useKey=True,
                                                  target=self.tmpBootConfFile_ )
            Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
         print 'done.'
      except Tac.SystemCommandError as e:
         print ''
         # should never get here, but this is an attempt to die gracefully
         # incase something unexpected happens
         self.mode_.addError( 'Unexpected error: %s' % e )
      

   def _commitChanges( self, onActive=True, onStandby=True ):
      try:
         if onActive:
            s = ' on this supervisor' if self.isModularSystem_ else ''
            printAndFlush( 'Committing changes%s...' % s )
            if not self.skipDownload_:
               Tac.run( [ 'mv', self.tmpSwiFile_, self.swiFile_ ], asRoot=True )
            Tac.run( [ 'mv', self.tmpBootConfFile_, self.bootConfFile_ ], 
                     asRoot=True )
            print 'done.'
         if onStandby and self.isModularSystem_:
            printAndFlush( 'Committing changes on standby supervisor...' )
            if self.tmpSwiFile_ != self.swiFile_:
               cmd = FileReplicationCmds.rename( self.currId_, self.swiFile_, 
                                                 self.tmpSwiFile_, useKey=True )
               Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )

            cmd = FileReplicationCmds.rename( self.currId_, self.bootConfFile_, 
                                              self.tmpBootConfFile_, useKey=True )
            Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
            print 'done.'
      except Tac.SystemCommandError as e:
         print ''
         # should never get here, but this is an attempt to wind up changes
         # and die gracefully in case something unexpected happens
         raise RevertChangesException( error='Unexpected error: %s' % e )

   def _transferToStandby( self, srcAddr, destAddr ):
      idx = destAddr.rfind( '/' )
      # if transfering file to a specific directory, create it first
      if idx > 0:
         cmd = FileReplicationCmds.createDir( self.currId_, destAddr[:idx], 
                                              useKey=True )
         Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
      cmd = FileReplicationCmds.copyFile( self.currId_, destAddr, srcAddr, 
                                          useKey=True )
      Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )  

      # sync file to make sure the data are reached to the disk
      cmd = FileReplicationCmds.syncFile( self.currId_, destAddr, useKey=True )
      try:
         Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
      except Tac.SystemCommandError, e:
         if 'command not found' not in e.output:
            # The standby might be running an older version, ignore this error
            raise
         print "syncFile: command not found, continue..."

   def _initializeFilenames( self, srcUrl, destUrl ):
      try:
         f = str.split( srcUrl.url, '/' )[-1]
         if not f.endswith( '.swi' ):
            question = 'Extension does not match \'.swi\'. Continue? [confirm]'
            if not self._promptUser( question ):
               raise CommandFailedException( aborted=True )
         # if destUrl not provided AND the source url is on flash:/, make sure that
         # the file exists, and then skip download
         if destUrl is None:
            if srcUrl.fs.scheme in [ 'flash:', 'file:' ]:
               srcUrl.open()
               if srcUrl.fs.scheme == 'flash:':
                  self.skipDownload_ = True
                  destUrl = srcUrl
            else:
               context = Url.Context( *Url.urlArgsFromMode( self.mode_ ) )
               destUrl = Url.parseUrl( 'flash:/' + f, context )
         elif destUrl.isdir():
            # Destination is a directory, so add the file name taken
            # from src url.
            # Note: isdir() is also True for scheme only urls (e.g. "flash:",
            #       "file:")
            context = Url.Context( *Url.urlArgsFromMode( self.mode_ ) )
            destUrl = Url.parseUrl( destUrl.url + '/' + f, context )
      except ValueError as e:
         raise CommandFailedException( error='Error: %s' % e )
      except IOError as ( errno, errmsg ):
         raise CommandFailedException( error='Error: %s' % errmsg )

      bootConfig = FileUrl.localBootConfig( *Url.urlArgsFromMode( self.mode_ ) )
      self.bootConfFile_ = bootConfig.localFilename()
      self.swiFile_ = destUrl.localFilename()

      # the operations are performed in two steps: 
      # 1) perform all actions on temporary files
      # 2) override original files with temporary files
      # this makes recovery easier in case it is necessary
      rSeq = '_tmp%04x' % random.randrange( 0, 65536 )
      self.tmpSwiFile_ = self.swiFile_ + rSeq
      self.tmpBootConfFile_ = self.bootConfFile_ + rSeq
      
      return destUrl

   # acquire SWI from source path
   def _downloadFromSource( self, srcUrl ):
      try:
         s = ' to this supervisor' if self.isModularSystem_ else ''
         printAndFlush( 'Downloading image%s...' % s )
         srcUrl.writeLocalFile(srcUrl, self.tmpSwiFile_ )
         print 'done.'
      except EnvironmentError as ( errno, errmsg ):
         print ''
         raise CommandFailedException( 'Error: %s' % errmsg )

   
   def _validateSwi( self ):
      # Doing a lazy import to only load zipfile when this command gets
      # executed. This way, even if there are multiple instances of Cli
      # running, zipfile will not act as a memory hog for each of them
      import zipfile
      try:
         f = self.swiFile_ if self.skipDownload_ else self.tmpSwiFile_
         if not zipfile.is_zipfile( f ):
            raise zipfile.BadZipfile()
         z = zipfile.ZipFile( f )
         if 'version' not in z.namelist():
            raise zipfile.BadZipfile()
      except zipfile.BadZipfile:
         error = 'Error: Software image seems to be corrupted'
         raise RevertChangesException( error=error )

   def _verifyImageCompatibility( self, imgSrcUrl ):
      # Check if the image being installed is compatible (ReloadPolicy checks
      # pass, swEpoch >= hwEpoch and if 2Gb image, it supports the hardware)
      # with the hardware
      import CliPlugin.ReloadImageEpochCheckCli as epochHook
      import CliPlugin.Reload2GbImageCheckCli as twoGbHook
      import CliPlugin.ReloadPolicyCheckCli as rpHook

      f = self.swiFile_ if self.skipDownload_ else self.tmpSwiFile_

      # ReloadPolicy check includes SWI signature check and factors in whether
      # or not SecureBoot is enabled in that check
      rpComp, rpResult = rpHook.checkReloadPolicy( self.mode_, f )
      for rpWarning in rpResult.warnings:
         self.mode_.addWarning( rpWarning )
      if not rpComp:
         rpError = '\n'.join( rpResult.errors )
         raise RevertChangesException( error=rpError )

      epochComp, epochErr = epochHook.checkImageEpochAllowed( self.mode_, f,
                                                              imgSrcUrl )
      if not epochComp :
         raise RevertChangesException( error=epochErr )
      twoGbComp, twoGbErr = twoGbHook.checkImage2GbCompatible( self.mode_, f,
                                                               imgSrcUrl )
      if not twoGbComp:
         raise RevertChangesException( error=twoGbErr )

   # modify boot-config with new SWI filename
   def _modifyBootConfig( self, destUrl ):
      try:
         printAndFlush( 'Preparing new boot-config...' )
         cf = SimpleConfigFile.SimpleConfigFileDict( self.bootConfFile_, True )
         tmpCf = SimpleConfigFile.SimpleConfigFileDict( self.tmpBootConfFile_, True )
         for key in cf.keys():
            tmpCf[key] = cf[key]
         tmpCf['SWI'] = str( destUrl )
         print 'done.'
      except (IOError, SimpleConfigFile.ParseError) as (errno, errmsg):
         print ''
         # if the boot-config file could not be modified, revert changes, as the
         # file would now just end up consuming space without being used
         raise RevertChangesException( error='Error: %s' % errmsg )


   def _transferChangesToStandby( self ):
      if redundancyStatus_.peerMode != 'standby':
         e = 'Standby supervisor is disabled and will not be upgraded.'
         raise StandbyTransferException( error=e, isMessage=True, 
                                         standbyMissing=True )
      else:
         try:
            printAndFlush( 
               'Copying new software image to standby supervisor...' )
            f = self.swiFile_ if self.skipDownload_ else self.tmpSwiFile_
            self._transferToStandby( f, self.tmpSwiFile_ )
            print 'done.'
            printAndFlush( 'Copying new boot-config to standby supervisor...' )
            self._transferToStandby( self.tmpBootConfFile_, self.tmpBootConfFile_ )
            print 'done.'
         except Tac.SystemCommandError as e:
            print ''
            errMsg = e.output
            if 'No space left on device' in e.output:
               errMsg = 'Error: No space left on standby supervisor'
            elif 'Network is unreachable' in e.output:
               errMsg = 'Error: Unable to communicate with standby supervisor'
            raise StandbyTransferException( error=errMsg )

   def run( self, mode, srcUrl, doReload, now, destUrl ):
      self.mode_ = mode
      self.now_ = now
      self.skipDownload_ = False
      self.isModularSystem_ = isModularSystem_

      try:
         if self.isModularSystem_:
            if( redundancyStatus_.mode != 'active' and 
                redundancyStatus_.mode != 'switchover' ):
               e = 'Error: This command is only available on the active supervisor'
               raise CommandFailedException( e )
            self.isModularSystem_ = ( redundancyStatus_.peerState != 'notInserted' )

         destUrl = self._initializeFilenames( srcUrl, destUrl )

         if not self.skipDownload_:
            self._downloadFromSource( srcUrl )

         self._validateSwi()
         self._verifyImageCompatibility( srcUrl )
         self._modifyBootConfig( destUrl )

         if self.isModularSystem_:
            try:
               self._transferChangesToStandby()
            except StandbyTransferException as e:
               if e.isMessage:
                  self.mode_.addWarning( e.error )
               else:
                  self.mode_.addError( e.error )
               if not e.standbyMissing:
                  self._deleteTempFile( self.tmpSwiFile_, onActive=False )
                  self._deleteTempFile( self.tmpBootConfFile_, onActive=False )
               self.isModularSystem_ = False    
               question = 'Commit changes on this supervisor? [confirm]'
               if not self._promptUser( question ):
                  raise RevertChangesException( aborted=True )

         self._commitChanges( onActive=False, onStandby=True )

         # (if needed) reload both supes
         if doReload:
            if ( self.isModularSystem_ and 
                 redundancyStatus_.peerMode == 'standby' ):

               printAndFlush( 'Reloading standby supervisor...' )
               try:
                  ReloadCli.reloadPeerSupervisor( quiet=True )
                  print 'done.'
                  self._commitChanges( onActive=True, onStandby=False )
                  # this is set so that we don't warn the user that the
                  # peer supervisor is disabled, since we just reloaded it
                  setattr( mode, 'disableElectionCheckBeforeReload', True )
               except Tac.Timeout:
                  error = 'Error: Failed to reload standby supervisor.' \
                      'The SWI may be corrupt.'
                  # we cannot revert changes on the modular system anymore
                  self.isModularSystem_ = False
                  raise RevertChangesException( error=error )
            else:
               self._commitChanges( onActive=True, onStandby=False )
            print 'Reloading%s...' % ( ' this supervisor' if self.isModularSystem_
                                       else '' )
            sys.stdout.flush()
            ReloadCli.doReload( mode, power=True, now=self.now_ )
            # set this back to false in the event that it was not needed 
            setattr( mode, 'disableElectionCheckBeforeReload', False )
         else:
            self._commitChanges( onActive=True, onStandby=False )

      except RevertChangesException as e:
         if not e.aborted:
            self.mode_.addError( e.error )
         self._revertChanges()
         return None if e.aborted else False
      except CommandFailedException as e:
         if not e.aborted:
            self.mode_.addError( e.error )
         return None if e.aborted else False
      return True

def doInstall( mode, args ):
   srcUrl = args[ 'SOURCE' ]
   destUrl = args.get( 'DESTINATION' )
   doReload = args.get( 'reload' )
   now = args.get( 'now' )

   installer = Installer()
   success = installer.run( mode, srcUrl, doReload, now, destUrl )
   if success is None:
      print 'Installation aborted.'
   elif success:
      print 'Installation succeeded.'
   else:
      print 'Installation failed.'

def Plugin( entityManager ):
   global entityManager_, redundancyStatus_
   entityManager_ = entityManager
   redundancyStatus_ = LazyMount.mount( entityManager,
      Cell.path( 'redundancy/status' ), 'Redundancy::RedundancyStatus', 'r' )

def _isModular():
   global isModularSystem_
   isModularSystem_ = True

FruCli.registerModularSystemCallback( _isModular )
