# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import ConfigMount
import LazyMount
import RecircIntfCli
from SamplePolicyCliLib import ( SamplePolicyContext, SamplePolicyMatchRuleContext )
from CliMode.InbandTelemetry import ( IntSamplePolicyConfigMode,
                                      IntSamplePolicyMatchRuleConfigMode,
                                      IntSamplePolicyActionRuleConfigMode )

from TypeFuture import TacLazyType
RecircFeature = Tac.Type( "Lag::Recirc::RecircFeature" )
intMonitorSessionName = "_InternalInbandTelemetry"

TruncationSize = TacLazyType( 'Mirroring::TruncationSize' )

#------------------------------------------------------------------------------------
# Global variables
#------------------------------------------------------------------------------------
lagIntfConfigDir = None
lagConfigCli = None
inbandTelemetryConfig = None
mirroringConfig = None
mirroringHwCapability = None

IP_PROTOCOL_RESERVED = [ '6', '17' ]

def intUseInternalLoopback():
   version2 = Tac.Type( "InbandTelemetry::IntVersion" ).Version2
   if inbandTelemetryConfig.intParam.intVersion == version2:
      return True
   return False

def isValidDestination( cfg, sessionName, destIntf, ):
   # Destination is invalid, if it is present as a source or a destination in any
   # other session, or is present as a source interface in the current session
   # cpu port cannot serve as a destination.
   if destIntf == "Cpu":
      return 'CPU'

   # For ports other than cpu port, it cannot serve as the mirror destination
   # if it is already a source interface in a session or a destination in another
   # session
   for name, session in cfg.session.items():
      if name != sessionName and ( destIntf in session.srcIntf or
                                   destIntf in session.targetIntf ):
         return 'Duplicate'
      if name == sessionName and destIntf in session.srcIntf:
         return( False, 'Duplicate' )

   return ''

def intMirrorSessionPrereq( mode, recircLagIntf=None ):
   recircIntFeatureSet = False
   intf = 0
   elic = 0

   # Cannot create mirror session if mirroring is not supported
   # or mirroring to a lag is not supported
   if ( not ( mirroringHwCapability.maxSupportedSessions > 0 and
              mirroringHwCapability.lagDestSupported ) and
      ( mirroringHwCapability.initialized or
        mode.session_.guardsEnabled() ) ):
      return None

   if intUseInternalLoopback():
      intf = "Loopback0"
   elif recircLagIntf and recircLagIntf in lagIntfConfigDir.intfConfig:
      elic = lagIntfConfigDir.intfConfig[ recircLagIntf ]
      if RecircFeature.telemetryinband in elic.recircFeature:
         recircIntFeatureSet = True
         intf = recircLagIntf
   else:
      for intf, elic in lagIntfConfigDir.intfConfig.iteritems():
         if RecircFeature.telemetryinband in elic.recircFeature:
            recircIntFeatureSet = True
            break

   if not intUseInternalLoopback() and not recircIntFeatureSet:
      return None

   # Do not create mirror session if inband telemetry has not been configured
   if not inbandTelemetryConfig.enable:
      return None

   # Do not create mirror session if no port has been configured as an
   # inband telemetry edge port
   if not inbandTelemetryConfig.intfToEdgePortProfiles:
      return None

   return intf

def maybeUpdateIntMirrorSession( mode, recircLagIntf=None ):
   intf = intMirrorSessionPrereq( mode, recircLagIntf )
   cfg = mirroringConfig
   sessionCfg = cfg.session.get( intMonitorSessionName )

   if intf:
      # The check for max mirror sessions reached is not done,
      # leaving it to Mirroring to handle such boundary conditions.
      reason = isValidDestination( mirroringConfig,
                                   intMonitorSessionName,
                                   intf )
      if reason:
         mode.addWarning( 'Mirror destination %s was rejected '
                          'with reason %s' % ( intf, reason ) )
         return False

      if not sessionCfg:
         sessionCfg = cfg.session.newMember( intMonitorSessionName )
         sessionCfg.isInternal = True
         sessionCfg.owner = "InbandTelemetry"
         sessionCfg.truncate = True
         sessionCfg.truncationSize = TruncationSize.null

      if ( mirroringHwCapability.multiDestSupported or
           ( not mirroringHwCapability.initialized and
             not mode.session_.guardsEnabled() ) ):
         sessionCfg.targetIntf.newMember( intf )
      else:
         sessionCfg.targetIntf.clear()
         sessionCfg.targetIntf.newMember( intf )
         sessionCfg.destIntf = intf
   elif sessionCfg:
      if intUseInternalLoopback():
         intf = "Loopback0"
      else:
         intf = recircLagIntf
         if not intf:
            for intf, elic in lagIntfConfigDir.intfConfig.iteritems():
               if RecircFeature.telemetryinband in elic.recircFeature:
                  break
      sessionCfg.destIntf = ""
      sessionCfg.truncate = False
      del sessionCfg.targetIntf[ intf ]
      del cfg.session[ intMonitorSessionName ]

   return True

class IntSamplePolicyContext( SamplePolicyContext ):
   def childMode( self ):
      return IntSamplePolicyConfigMode

class IntSamplePolicyMatchRuleContext( SamplePolicyMatchRuleContext ):
   def childMode( self, matchRuleName, matchOption ):
      return IntSamplePolicyMatchRuleConfigMode

   def actionMode( self ):
      return IntSamplePolicyActionRuleConfigMode

def Plugin( entityManager ):
   global lagIntfConfigDir, lagConfigCli
   global inbandTelemetryConfig
   global mirroringConfig, mirroringHwCapability

   inbandTelemetryConfig = ConfigMount.mount( entityManager,
                                              "inbandtelemetry/config",
                                              "InbandTelemetry::Config", "w" )
   lagIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/eth/lag",
                                       "Interface::EthLagIntfConfigDir", "w" )
   lagConfigCli = ConfigMount.mount( entityManager, "lag/input/config/cli",
                                       "Lag::Input::Config", "w" )
   mirroringConfig = ConfigMount.mount( entityManager, "mirroring/config",
                                        "Mirroring::Config", "w" )
   mirroringHwCapability = LazyMount.mount( entityManager, "mirroring/hwcapability",
                                            "Mirroring::HwCapability", "r" )
   RecircIntfCli.intMirrorSessionHook.addExtension( maybeUpdateIntMirrorSession )
