#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import CliParser
import ConfigMount
import LazyMount
import SmashLazyMount
import Plugins
import ShowCommand
import CliCommand
import CliMatcher
import BasicCli
import IntfCli
import Tac
import Tracing
from CliMode.InbandTelemetry import ( FEATURE,
                                      InbandTelemetryConfigMode,
                                      ProfileConfigMode, IntSamplePolicyConfigMode,
                                      IntSamplePolicyMatchRuleConfigMode,
                                      IntSamplePolicyActionRuleConfigMode )
from InbandTelemetryTypes import ( tacCollectorLogBufferSize,
                                   tacPortProfileType,
                                   tacCollectorType,
                                   tacOperState, )
import CliToken
from TrafficPolicyCliModel import Rule, Matches, NumericalRange, Actions
from InbandTelemetryCliModel import (
   Profile,
   IntfList,
   ModelIntProfileSummary,
   InbandTelemetry,
   InbandTelemetryProfiles,
   IntTrackingModel,
   IntFlowDetailModel,
   IntFlowDetailNodeInfo,
   IntFlowDetailIntervalStats,
   SamplePolicyModel,
)
import Toggles.InbandTelemetryCommonToggleLib
from SftCliShow import (
   ShowFlowTable,
   ShowCounters,
   generateShowTrackingFilter,
)
from SftModel import FtrCounters
from FlowTrackingCliLib import (
   ftrTypeKwStr,
   trackingShowKw,
   flowTableKw,
   trackerKw,
   trackerNameMatcher,
   exporterKw,
   exporterNameMatcher,
   telemetryShowKw,
   inbandShowKw,
   isInbandTelemetryAgentRunning,
)
from FlowTrackerCliUtil import (
   ftrTypeInbandTelemetry,
)
from PolicyMapCliLib import PolicyOpChkr
from ClassificationCliLib import CommitAbortModelet
from TrafficPolicyCli import ( ShowPendingCmd, matchRuleName,
                               DscpConfigCmd, ActionsConfigCmdBase,
                               MatchRuleConfigCmdBase )
import InbandTelemetryCliLib
from InbandTelemetryCliLib import ( IP_PROTOCOL_RESERVED, IntSamplePolicyContext,
                                    IntSamplePolicyMatchRuleContext )
from SamplePolicyCliLib import ( SamplePolicyConfigCmd,
                                 SampleActionCmd )
import Smash

__defaultTraceHandle__ = Tracing.Handle( 'InbandTelemetryCli' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

#------------------------------------------------------------------------------------
# Global variables
#------------------------------------------------------------------------------------
inbandTelemetryConfig = None
inbandTelemetryStatus = None
policiesCliConfig = None
policiesStatusRequestDir = None
policiesStatus = None
hwCapability = None
em = None
activeAgentDir = None
intFtCounters = None

minSampleRate = 1
defSampleRate = 1 << 20
maxSampleRate = 1 << 24
minLogNum = 64
maxLogNum = 4 * 1024
deflogNum = 1024
maxConfigVersion = Tac.Value(
      "InbandTelemetry::CommandMaskType" ).cliCommandMask

#------------------------------------------------------------------------------------
# Utility functions
#------------------------------------------------------------------------------------
# pylint: disable-msg=R1703

def copyIntParam( oldParam, newParam ):
   newParam.deviceId = oldParam.deviceId
   newParam.probeMarker = oldParam.probeMarker
   newParam.probeProtocol = oldParam.probeProtocol
   newParam.intDetectionMethod = oldParam.intDetectionMethod
   newParam.intVersion = oldParam.intVersion

def isDefaultIntParam( param ):
   if param.deviceId == param.defaultDeviceId and \
      param.probeMarker == param.defaultProbeMarker and \
      param.probeProtocol == param.defaultProbeProtocol and \
      param.intDetectionMethod == param.defaultIntDetectionMethod and \
      param.intVersion == param.defaultIntVersion:
      return True
   else:
      return False

def identicalIntParam( oldParam, newParam ):
   if oldParam.deviceId == newParam.deviceId and \
      oldParam.probeMarker == newParam.probeMarker and \
      oldParam.probeProtocol == newParam.probeProtocol and \
      oldParam.intDetectionMethod == newParam.intDetectionMethod and \
      oldParam.intVersion == newParam.intVersion:
      return True
   return False

# For correctly initializing intVersion and detectionMethod in IntParam
# TODO: to be removed when cli for setting intVersion is ready
def rectifyIntParam( param ):
   if Toggles.InbandTelemetryCommonToggleLib.\
         toggleFeatureIfa2BasedInbandTelemetryEnabled():
      param.intVersion = Tac.Type( "InbandTelemetry::IntVersion" ).Version2
      param.intDetectionMethod = \
            Tac.Type( "InbandTelemetry::IntDetectionMethod" ).ProbeIpProtocolBased

def identicalProfile( profile, currentEntry ):
   if not profile or not currentEntry:
      return False
   if profile.type != currentEntry.type or \
           profile.collectorName != currentEntry.collectorName or \
           profile.collectorType != currentEntry.collectorType or \
           profile.sampleRate != currentEntry.sampleRate or \
           profile.samplePolicy != currentEntry.samplePolicy or \
           profile.truncate != currentEntry.truncate or \
           profile.truncationSize != currentEntry.truncationSize or \
           profile.egressDrop != currentEntry.egressDrop:
      return False
   return True

def identicalTelemetryConfig( oldConfig, newConfig ):
   if not identicalIntParam( oldConfig.intParam, newConfig.intParam ):
      return False
   if oldConfig.enable != newConfig.enable:
      return False
   oldCorePortProfiles = oldConfig.corePortProfiles.keys()
   newCorePortProfiles = newConfig.corePortProfiles.keys()
   oldEdgePortProfiles = oldConfig.edgePortProfiles.keys()
   newEdgePortProfiles = newConfig.edgePortProfiles.keys()
   oldIntfToEdgePortProfiles = oldConfig.intfToEdgePortProfiles.values()
   newIntfToEdgePortProfiles = newConfig.intfToEdgePortProfiles.values()
   oldIntfToCorePortProfiles = oldConfig.intfToCorePortProfiles.values()
   newIntfToCorePortProfiles = newConfig.intfToCorePortProfiles.values()
   if oldCorePortProfiles != newCorePortProfiles:
      oldSet = set( oldCorePortProfiles )
      newSet = set( newCorePortProfiles )
      if list( oldSet - newSet ) != [ 'default' ] or \
         list( newSet - oldSet ) != []:
         return False
   for coreProfile in oldCorePortProfiles:
      if coreProfile == 'default':
         continue
      oldCoreProfile = oldConfig.corePortProfiles[ coreProfile ]
      newCoreProfile = newConfig.corePortProfiles[ coreProfile ]
      if not identicalProfile( oldCoreProfile, newCoreProfile ):
         return False
   if oldEdgePortProfiles != newEdgePortProfiles:
      oldSet = set( oldEdgePortProfiles )
      newSet = set( newEdgePortProfiles )
      if list( oldSet - newSet ) != [ 'default' ] or \
         list( newSet - oldSet ) != []:
         return False
   for edgeProfile in oldEdgePortProfiles:
      if edgeProfile == 'default':
         continue
      oldEdgeProfile = oldConfig.edgePortProfiles[ edgeProfile ]
      newEdgeProfile = newConfig.edgePortProfiles[ edgeProfile ]
      if not identicalProfile( oldEdgeProfile, newEdgeProfile ):
         return False
   if oldIntfToEdgePortProfiles != newIntfToEdgePortProfiles or \
      oldIntfToCorePortProfiles != newIntfToCorePortProfiles:
      return False
   if oldConfig.collectors.keys() != newConfig.collectors.keys():
      return False
   # TODO: when collector log is supported uncomment the below
   # if oldConfig.collectors[ 'DefaultLog' ].maxPktsToBuffer != \
   #   newConfig.collectors[ 'DefaultLog' ].maxPktsToBuffer:
   #   return False
   return True

def isDefaultTelemetryConfig( config ):
   if not isDefaultIntParam( config.intParam ):
      return False
   if config.corePortProfiles.keys() != [ 'default' ]:
      return False
   if config.edgePortProfiles.keys() != [ 'default' ]:
      return False
   if config.collectors.keys():
      return False
   if policiesCliConfig.pmapType.pmapInput.keys():
      return False
   # TODO: when collector log is supported uncomment the below
   # if config.collectors.keys() != [ 'DefaultLog' ]:
   #   return False
   # if 'DefaultLog' in config.collectors and \
   #   config.collectors[ 'DefaultLog' ].maxPktsToBuffer != 64:
   #   return False
   if config.intfToCorePortProfiles or config.intfToEdgePortProfiles:
      return False
   return True

def copyProfile( newProfile, oldProfile ):
   newProfile.collectorName = oldProfile.collectorName
   newProfile.collectorType = oldProfile.collectorType
   newProfile.sampleRate = oldProfile.sampleRate
   newProfile.samplePolicy = oldProfile.samplePolicy
   newProfile.truncate = oldProfile.truncate
   newProfile.truncationSize = oldProfile.truncationSize
   newProfile.egressDrop = oldProfile.egressDrop

def getPortProfileNames( profileType ):
   if profileType == 'core':
      return inbandTelemetryConfig.corePortProfiles.keys()
   else:
      return inbandTelemetryConfig.edgePortProfiles.keys()

def getFlowTrackerNames():
   flowTrackerNames = [ dest.name for dest in
                        inbandTelemetryConfig.collectors.values()
                        if dest.type ==
                        tacCollectorType.CollectorFlowTracker ]
   return flowTrackerNames

def defaultProfileGuard( mode, token ):
   if token == 'default':
      return CliParser.guardNotThisPlatform
   return None

def edgeProfileGuard( mode, token ):
   if mode.profileContext.profileType_ == tacPortProfileType.IntEdge:
      return None
   return CliParser.guardNotThisPlatform

def telemetryGuard( mode, token ):
   if hwCapability.inbandTelemetrySupported:
      return None
   return CliParser.guardNotThisPlatform

def ifa2Guard( mode, token ):
   intVersions = Tac.Type( "InbandTelemetry::IntVersion" )
   if hwCapability.defaultIntVersion == intVersions.Version2:
      return None
   return CliParser.guardNotThisPlatform

def getProfileTypeFromArgs( args ):
   if 'core' in args:
      profileType = args[ 'core' ]
   else:
      profileType = args[ 'edge' ]
   return profileType

def getProfileNamesFromContext( mode, context ):
   profileType = getProfileTypeFromArgs( context.sharedResult )
   return getPortProfileNames( profileType )

def handleIntMirrorSession( mode ):
   InbandTelemetryCliLib.maybeUpdateIntMirrorSession( mode )

def incrementConfigVersion( config ):
   if config.version == maxConfigVersion:
      config.version = 0
   else:
      config.version += 1

coreProfileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'core' ),
   "Inband Telemetry Core Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

edgeProfileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'edge' ),
   "Inband Telemetry Edge Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

nonDefaultCoreProfileNameMatcher = CliCommand.Node( CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'core' ),
   "Inband Telemetry Core Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' ), guard=defaultProfileGuard )

nonDefaultEdgeProfileNameMatcher = CliCommand.Node( CliMatcher.DynamicNameMatcher(
   lambda mode: getPortProfileNames( 'edge' ),
   "Inband Telemetry Edge Profile Name",
   pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' ), guard=defaultProfileGuard )

flowTrackerNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: getFlowTrackerNames(),
   "Flow Tracker Name" )

profileNameMatcher = CliCommand.Node( matcher=CliMatcher.DynamicNameMatcher(
      getProfileNamesFromContext, "Profile name",
      passContext=True ), storeSharedResult=True )

coreKeyword = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'core', helpdesc='core profile' ), storeSharedResult=True )

edgeKeyword = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'edge', helpdesc='edge profile' ), storeSharedResult=True )

profileKeyword = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
   'profile', helpdesc='Show inband telemetry profiles' ) )

probeMarkerMatcherUpper = CliMatcher.PatternMatcher(
   pattern='0x?[a-fA-F0-9]{1,8}', helpname='0x0-0xffffffff',
   helpdesc="Upper 32 bits of the probe marker" )

probeMarkerMatcherLower = CliMatcher.PatternMatcher(
   pattern='0x?[a-fA-F0-9]{1,8}', helpname='0x0-0xffffffff',
   helpdesc="Lower 32 bits of the probe marker" )

hexMatcher = CliMatcher.PatternMatcher( '0x?[a-fA-F0-9]{1,8}',
                                        helpname='0x0-0xffffffff',
                                        helpdesc="32 bit hexadecimal" )

class IpProtocolMatcher( CliMatcher.IntegerMatcher ):
   '''Extends IntegerMatcher, but filters out reserved values'''
   def match( self, mode, context, token ):
      if token in IP_PROTOCOL_RESERVED:
         return CliMatcher.noMatch
      return super( IpProtocolMatcher, self ).match(
            mode, context, token )

#------------------------------------------------------------------------------------
# Modelet under Intf mode for inband telemetry command
#------------------------------------------------------------------------------------

class IntfTelemetryModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ( 'Ethernet', 'Port-Channel' ) )

IntfCli.IntfConfigMode.addModelet( IntfTelemetryModelet )

#------------------------------------------------------------------------------------
# The "monitor telemetry inband" mode context
#------------------------------------------------------------------------------------

class InbandTelemetryContext( object ):
   def __init__( self, mode ):
      self.mode = mode
      self.config = inbandTelemetryConfig
      self.currentEntry_ = None
      self.previousEntry_ = Tac.newInstance( 'InbandTelemetry::Config', "intConfig" )
      if not isDefaultIntParam( inbandTelemetryConfig.intParam ):
         intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
         copyIntParam( inbandTelemetryConfig.intParam, intParam )
         self.previousEntry_.intParam = intParam
      # TODO: when collector log is supported uncomment the below
      # dftLog = self.previousEntry_.collectors.newMember(
      #   'DefaultLog', tacCollectorType.CollectorLog )
      # dftLog.maxPktsToBuffer = inbandTelemetryConfig.collectors[
      #   'DefaultLog' ].maxPktsToBuffer
      self.previousEntry_.enable = inbandTelemetryConfig.enable

   def copyEditEntry( self ):
      newEntry = Tac.newInstance( 'InbandTelemetry::Config', "intConfig" )
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      if not isDefaultIntParam( self.config.intParam ):
         copyIntParam( self.config.intParam, intParam )
      rectifyIntParam( intParam )
      newEntry.intParam = intParam
      for collector in self.config.collectors:
         collectorType = self.config.collectors[ collector ].type
         newEntry.collectors.newMember( collector, collectorType )
         newEntry.collectors[ collector ].maxPktsToBuffer = self.config.collectors[
            collector ].maxPktsToBuffer
      for coreProfile in self.config.corePortProfiles:
         newEntry.corePortProfiles.newMember( coreProfile,
                                              tacPortProfileType.IntCore )
         newProfile = newEntry.corePortProfiles[ coreProfile ]
         newProfile.egressDrop = self.config.corePortProfiles[
            coreProfile ].egressDrop
         newProfile.sampleRate = self.config.corePortProfiles[
            coreProfile ].sampleRate
         newProfile.samplePolicy = self.config.corePortProfiles[
            coreProfile ].samplePolicy
         newProfile.collectorName = self.config.corePortProfiles[
            coreProfile ].collectorName
         newProfile.collectorType = self.config.corePortProfiles[
            coreProfile ].collectorType
      for edgeProfile in self.config.edgePortProfiles:
         newEntry.edgePortProfiles.newMember( edgeProfile,
                                              tacPortProfileType.IntEdge )
         newProfile = newEntry.edgePortProfiles[ edgeProfile ]
         newProfile.egressDrop = self.config.edgePortProfiles[
            edgeProfile ].egressDrop
         newProfile.sampleRate = self.config.edgePortProfiles[
            edgeProfile ].sampleRate
         newProfile.samplePolicy = self.config.edgePortProfiles[
            edgeProfile ].samplePolicy
         newProfile.collectorName = self.config.edgePortProfiles[
            edgeProfile ].collectorName
         newProfile.collectorType = self.config.edgePortProfiles[
            edgeProfile ].collectorType
      for intf, profile in self.config.intfToCorePortProfiles.items():
         newEntry.intfToCorePortProfiles[ intf ] = profile
      for intf, profile in self.config.intfToEdgePortProfiles.items():
         newEntry.intfToEdgePortProfiles[ intf ] = profile
      newEntry.enable = self.config.enable
      self.currentEntry_ = newEntry
      return newEntry

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      if self.currentEntry_:
         if identicalTelemetryConfig( self.config, self.currentEntry_ ):
            # no need to commit anything from currentEntry_ as it has not changed
            return
         if not identicalIntParam( self.currentEntry_.intParam,
                                   self.config.intParam ):
            intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
            copyIntParam( self.currentEntry_.intParam, intParam )
            self.config.intParam = intParam
         # TODO: when collector log is supported uncomment the below
         # self.config.collectors[ 'DefaultLog' ].\
         #   maxPktsToBuffer = self.currentEntry_.collectors[
         #      'DefaultLog' ].maxPktsToBuffer

         # Remove collectors that are no longer used
         collectorsNotInUse = []
         for collector in self.config.collectors:
            collectorInUse = False
            # TODO: when collector log is supported uncomment the below
            # if collector == 'DefaultLog':
            #   continue
            for coreProfile in self.config.corePortProfiles:
               if collector == self.config.corePortProfiles[
                     coreProfile ].collectorName:
                  collectorInUse = True
                  break
            for edgeProfile in self.config.edgePortProfiles:
               if collector == self.config.edgePortProfiles[
                     edgeProfile ].collectorName:
                  collectorInUse = True
                  break
            if not collectorInUse:
               collectorsNotInUse.append( collector )

         for collector in collectorsNotInUse:
            del self.config.collectors[ collector ]

         # Set enable upon successfully committing from inband telemetry mode's
         # context. If config.enable was already false and currentEntry_.enable
         # is also false, no need to increment version and handle int mirror session
         versionIncNeeded = self.config.enable or self.currentEntry_.enable
         self.config.enable = self.currentEntry_.enable
         if versionIncNeeded:
            incrementConfigVersion( self.config )
            # Attempt to create an internal inband telemetry mirror
            # session when the feature is enabled.
            # This must be gated through HW capabilities for future
            # implementations that don't require such an operation.
            handleIntMirrorSession( self.mode )

#------------------------------------------------------------------------------------
# The "monitor telemetry MODE" mode command
#------------------------------------------------------------------------------------

class MonitorTelemetryInband( CliCommand.CliCommandClass ):
   syntax = '''monitor telemetry inband'''
   noOrDefaultSyntax = syntax
   data = {
      'monitor': CliToken.Monitor.monitorMatcher,
      'telemetry': 'Telemetry configuration',
      'inband': CliCommand.guardedKeyword( 'inband',
                                           helpdesc='Inband telemetry',
                                           guard=telemetryGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      context = InbandTelemetryContext( mode )
      context.copyEditEntry()
      childMode = mode.childMode( InbandTelemetryConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for name in policiesCliConfig.pmapType.pmap:
         # pylint: disable=protected-access
         IntSamplePolicyConfigCmd._removePolicy( mode, name )
      coreProfiles = inbandTelemetryConfig.corePortProfiles.keys()
      for profile in coreProfiles:
         if profile != 'default':
            del inbandTelemetryConfig.corePortProfiles[ profile ]
      edgeProfiles = inbandTelemetryConfig.edgePortProfiles.keys()
      for profile in edgeProfiles:
         if profile != 'default':
            del inbandTelemetryConfig.edgePortProfiles[ profile ]
      collectors = inbandTelemetryConfig.collectors.keys()
      # TODO: when collector log is supported uncomment the below
      #inbandTelemetryConfig.collectors[ 'DefaultLog' ].maxPktsToBuffer = 64
      for collector in collectors:
         # TODO: when collector log is supported uncomment the below
         # if collector != 'DefaultLog':
         #   del inbandTelemetryConfig.collectors[ collector ]
         del inbandTelemetryConfig.collectors[ collector ]
      # Set version to 0, enable to False and reset int param when setting
      # inbandTelemetryConfig to default
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      inbandTelemetryConfig.intParam = intParam
      if inbandTelemetryConfig.enable:
         inbandTelemetryConfig.enable = False
         incrementConfigVersion( inbandTelemetryConfig )
         # Delete the internal inband telemetry mirror
         # session when the feature is disabled.
         # This must be gated through HW capabilities for future
         # implementations that don't require such an operation.
         handleIntMirrorSession( mode )

#--------------------------------------------------------------------------------
# [ no | default ] disabled
#--------------------------------------------------------------------------------
class DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Enable the inband telemetry feature'
   }

   @staticmethod
   def handler( mode, args ):
      mode.telemetryContext.currentEntry_.enable = False

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      mode.telemetryContext.currentEntry_.enable = True

def getSamplePolicyNames( mode ):
   if not policiesCliConfig:
      return []
   return policiesCliConfig.pmapType.pmapInput

samplePolicyNameMatcher = CliMatcher.DynamicNameMatcher(
   getSamplePolicyNames, "Sample Policy Name",
   pattern=r'[A-Za-z0-9_:{}\[\]-]+' )


#[ no ] sample policy POLICY_NAME
class IntSamplePolicyConfigCmd( SamplePolicyConfigCmd ):
   data = {
      'POLICY_NAME': samplePolicyNameMatcher
   }
   data.update( SamplePolicyConfigCmd._baseData )

   @staticmethod
   def _feature():
      return FEATURE

   @classmethod
   def _context( cls, name ):
      return IntSamplePolicyContext( policiesCliConfig, policiesStatusRequestDir,
                                     policiesStatus, name )

#[ no ]  match  <match-rule-name> ipv4|ipv6
class IntSamplePolicyMatchRuleConfigCmd( MatchRuleConfigCmdBase ):
   syntax = 'match RULE_NAME ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'match RULE_NAME [ ipv4 | ipv6 ]'
   data = {
      'match': 'Configure match rule',
      'RULE_NAME': matchRuleName,
      'ipv4': 'IPv4 match rule',
      'ipv6': 'IPv6 match rule',
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      return IntSamplePolicyMatchRuleContext( trafficPolicyContext, ruleName,
                                              matchOption )

class IntSamplePolicyActionsConfigCmd( ActionsConfigCmdBase ):
   @staticmethod
   def _feature():
      return FEATURE

class DeviceIdCmd( CliCommand.CliCommandClass ):
   syntax = '''device-id HEX_VALUE'''
   noOrDefaultSyntax = '''device-id'''
   data = {
      'device-id' : 'Set the device id for inband telemetry',
      'HEX_VALUE' : hexMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      deviceId = args[ 'HEX_VALUE' ]
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.deviceId = deviceId
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      if not currentEntry.intParam:
         return
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.deviceId = '0x0'
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

class ProbeMarkerCmd( CliCommand.CliCommandClass ):
   syntax = '''probe marker <markerA> <markerB>'''
   noOrDefaultSyntax = '''probe marker'''
   data = {
      'probe' : 'Configure probe marker for inband telemetry',
      'marker': 'Set the probe marker value',
      '<markerA>' : probeMarkerMatcherUpper,
      '<markerB>' : probeMarkerMatcherLower
      }

   @staticmethod
   def handler( mode, args ):
      markerA = args[ '<markerA>' ]
      markerB = args[ '<markerB>' ]
      currentEntry = mode.telemetryContext.currentEntry_
      probeMarker = ( int( markerA, 16 ) << 32 ) + \
                    int( markerB, 16 )
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeMarker = probeMarker
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      if not currentEntry.intParam:
         return
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeMarker = intParam.defaultProbeMarker
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

#--------------------------------------------------------------------------------
# [ no | default ] probe marker ip protocol PROTOCOL_VAL
#--------------------------------------------------------------------------------
class ProbeProtocolCmd( CliCommand.CliCommandClass ):
   syntax = '''probe marker ip protocol PROTOCOL_VAL'''
   noOrDefaultSyntax = '''probe marker ip protocol'''
   data = {
      'probe': 'Configure probe marker for inband telemetry',
      'marker': 'Set the probe marker value',
      'ip': CliCommand.guardedKeyword( 'ip',
         helpdesc='Configure IP header to probe InbandTelemetry packets',
         guard=ifa2Guard ),
      'protocol': 'Set IP protocol/next-header field to probe '
            'InbandTelemetry packets',
      'PROTOCOL_VAL': IpProtocolMatcher( 0, 254,
         helpdesc='0-254 except 6 ( reserved for TCP ) and 17 ( reserved for UDP )' )
   }

   @staticmethod
   def handler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      probeProtocol = args[ 'PROTOCOL_VAL' ]
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeProtocol = probeProtocol
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      if not currentEntry.intParam:
         return
      intParam = Tac.newInstance( 'InbandTelemetry::InbandTelemetryParam' )
      copyIntParam( currentEntry.intParam, intParam )
      intParam.probeProtocol = intParam.defaultProbeProtocol
      if not identicalIntParam( currentEntry.intParam, intParam ):
         currentEntry.intParam = intParam

class EgressCollectionLogNumCmd( CliCommand.CliCommandClass ):
   syntax = '''egress collection log <num>'''
   noOrDefaultSyntax = '''egress collection log'''
   data = {
      'egress' : 'Configure egress parameters',
      'collection' : 'Set collector configuration',
      'log' : 'Set collector as log',
      '<num>' : CliMatcher.IntegerMatcher( minLogNum, maxLogNum,
                                           helpdesc='Max packets to buffer' )
      }

   @staticmethod
   def handler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      collectorCfg = currentEntry.collectors[ 'DefaultLog' ]
      maxPktsToBuffer = args[ '<num>' ]
      collectorCfg.maxPktsToBuffer = maxPktsToBuffer

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentEntry = mode.telemetryContext.currentEntry_
      collectorCfg = currentEntry.collectors[ 'DefaultLog' ]
      collectorCfg.maxPktsToBuffer = tacCollectorLogBufferSize.minSize

#------------------------------------------------------------------------------------
# The profile  mode context
#------------------------------------------------------------------------------------

class ProfileModeContext( object ):
   def __init__( self, mode, profileName, profileType ):
      self.mode = mode
      self.profile_ = None
      self.profileName_ = profileName
      self.profileType_ = profileType
      self.currentEntry_ = None
      self.previousEntry_ = None
      portProfileConfig = inbandTelemetryConfig.edgePortProfiles if profileType == \
                          tacPortProfileType.IntEdge else \
                          inbandTelemetryConfig.corePortProfiles
      if profileName in portProfileConfig:
         self.profile_ = portProfileConfig[ profileName ]
         prevProfile = Tac.newInstance( 'InbandTelemetry::PortProfileConfig',
                                        profileName, profileType )
         copyProfile( prevProfile, self.profile_ )
         self.previousEntry_ = prevProfile

   def copyEditEntry( self ):
      newProfile = Tac.newInstance( 'InbandTelemetry::PortProfileConfig',
                                    self.profileName_, self.profileType_ )
      copyProfile( newProfile, self.profile_ )
      self.currentEntry_ = newProfile
      return newProfile

   def newEditEntry( self ):
      newProfile = Tac.newInstance( 'InbandTelemetry::PortProfileConfig',
                                    self.profileName_, self.profileType_ )
      self.currentEntry_ = newProfile

   def profileName( self ):
      return self.profileName_

   def profileType( self ):
      return self.profileType_

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      chkr = PolicyOpChkr( policiesStatusRequestDir, policiesStatus )
      if identicalProfile( self.profile_, self.currentEntry_ ):
         return
      if self.profile_ is None:
         if self.profileType_ == tacPortProfileType.IntEdge:
            self.profile_ = inbandTelemetryConfig.edgePortProfiles.newMember(
               self.profileName_, self.profileType_ )
         else:
            self.profile_ = inbandTelemetryConfig.corePortProfiles.newMember(
               self.profileName_, self.profileType_ )
      if self.profileType_ == tacPortProfileType.IntEdge:
         for intf in inbandTelemetryConfig.intfToEdgePortProfiles:
            if inbandTelemetryConfig.intfToEdgePortProfiles[ intf ] != \
               self.profileName_:
               continue
            result = chkr.verify( 'intf', intf )
            commitStatus = result[ 0 ]
            if not commitStatus:
               reason = result[ 1 ].error if result[ 1 ] else 'unknown'
               self.mode.addError( 'Failed to commit edge profile : %s' %
                                   reason )
      self.profile_.collectorName = self.currentEntry_.collectorName
      self.profile_.collectorType = self.currentEntry_.collectorType
      self.profile_.sampleRate = self.currentEntry_.sampleRate
      self.profile_.samplePolicy = self.currentEntry_.samplePolicy
      self.profile_.truncate = self.currentEntry_.truncate
      self.profile_.truncationSize = self.currentEntry_.truncationSize
      self.profile_.egressDrop = self.currentEntry_.egressDrop
      self.profile_.version += 1


#------------------------------------------------------------------------------------
# The "profile <profile-type> <profile-name>" mode command
#------------------------------------------------------------------------------------
class ProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''profile ( ( core CORE_PROFILE ) | ( edge EDGE_PROFILE ) )'''
   noOrDefaultSyntax = syntax
   data = {
      'profile' : 'Configure inband telemetry profile',
      'core' : 'Configure core profile',
      'CORE_PROFILE': nonDefaultCoreProfileNameMatcher,
      'edge' : 'Configure edge profile',
      'EDGE_PROFILE': nonDefaultEdgeProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if "edge" in args:
         profileType = tacPortProfileType.IntEdge
         profileName = args[ 'EDGE_PROFILE' ]
      else:
         profileType = tacPortProfileType.IntCore
         profileName = args[ 'CORE_PROFILE' ]
      context = ProfileModeContext( mode, profileName, profileType )

      if ( profileType == tacPortProfileType.IntEdge and
            profileName in inbandTelemetryConfig.edgePortProfiles ) or \
         ( profileType == tacPortProfileType.IntCore and
           profileName in inbandTelemetryConfig.corePortProfiles ):
         context.copyEditEntry()
      else:
         context.newEditEntry()
      childMode = mode.childMode( ProfileConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if "edge" in args:
         profileType = 'edge'
         profileName = args[ 'EDGE_PROFILE' ]
      else:
         profileType = 'core'
         profileName = args[ 'CORE_PROFILE' ]
      if profileType == 'edge':
         if profileName in inbandTelemetryConfig.edgePortProfiles:
            del inbandTelemetryConfig.edgePortProfiles[ profileName ]
      else:
         if profileName in inbandTelemetryConfig.corePortProfiles:
            del inbandTelemetryConfig.corePortProfiles[ profileName ]

class SampleRateCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress sample rate <rate-value>'''
   noOrDefaultSyntax = '''ingress sample rate'''
   data = {
      'ingress' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'ingress', helpdesc='Configure ingress parameters' ),
         guard=edgeProfileGuard ),
      'sample'  : 'Configure sampling parameters',
      'rate'    : 'Configure sampling rate',
      '<rate-value>' :
                 CliMatcher.IntegerMatcher( minSampleRate, maxSampleRate,
                    helpdesc='Sampling Rate Value' )
   }

   @staticmethod
   def handler( mode, args ):
      sampleRate = args[ '<rate-value>' ]
      profile = mode.profileContext.currentEntry_
      profile.sampleRate = sampleRate

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      profile.sampleRate = defSampleRate

class EgressCollectionLogCmd( CliCommand.CliCommandClass ):
   syntax = '''egress collection log'''
   noOrDefaultSyntax = syntax
   data = {
      'egress' : 'Configure egress parameters',
      'collection' : 'Set collector configuration',
      'log' : 'Set collector as log',
      }

   @staticmethod
   def handler( mode, args ):
      collectionName = 'DefaultLog'
      profile = mode.profileContext.currentEntry_
      profile.collectorName = collectionName
      profile.collectorType = tacCollectorType.CollectorLog

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      if profile.collectorName != 'DefaultLog':
         return
      profile.collectorName = ""
      profile.collectorType = tacCollectorType.CollectorNone

class IngressSamplePolicyCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress sample policy POLICY_NAME'''
   noOrDefaultSyntax = '''ingress sample policy [ POLICY_NAME ]'''
   data = {
      'ingress': CliCommand.guardedKeyword(
         'ingress', helpdesc='Configure ingress parameters',
         guard=edgeProfileGuard ),
      'sample': 'Configure sampling parameters',
      'policy': 'Configure sample policy',
      'POLICY_NAME': samplePolicyNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      profile = mode.profileContext.currentEntry_
      profile.samplePolicy = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      policyName = args.get( 'POLICY_NAME' )
      if not policyName or policyName == profile.samplePolicy:
         profile.samplePolicy = ""

class EgressCollectionFlowTrackerCmd( CliCommand.CliCommandClass ):
   syntax = '''egress collection flow tracker <flow-tracker-name>'''
   noOrDefaultSyntax = '''egress collection flow tracker'''
   data = {
      'egress' : 'Configure egress parameters',
      'collection' : 'Set collector configuration',
      'flow' : 'Flow configuration commands',
      'tracker' : 'Set collector as flow tracker',
      '<flow-tracker-name>' : flowTrackerNameMatcher
      }

   @staticmethod
   def handler( mode, args ):
      collectionName = args[ '<flow-tracker-name>' ]
      collectionType = tacCollectorType.CollectorFlowTracker
      profile = mode.profileContext.currentEntry_
      if collectionName not in inbandTelemetryConfig.collectors:
         inbandTelemetryConfig.collectors.newMember( collectionName, collectionType )
      profile.collectorName = collectionName
      profile.collectorType = collectionType

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      if profile.collectorName == 'DefaultLog':
         return
      profile.collectorName = ""
      profile.collectorType = tacCollectorType.CollectorNone

class EgressDropDisabledCmd( CliCommand.CliCommandClass ):
   syntax = '''egress drop disabled'''
   noOrDefaultSyntax = syntax
   data = {
      'egress': 'Configure egress parameters',
      'drop': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'drop', helpdesc='Set egress drop configuration' ),
         guard=edgeProfileGuard ),
      'disabled': 'Disable egress drop',
      }

   @staticmethod
   def handler( mode, args ):
      profile = mode.profileContext.currentEntry_
      profile.egressDrop = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      profile.egressDrop = True

class InbandTelemetryInterface( CliCommand.CliCommandClass ):
   syntax = '''telemetry inband profile ( ( core [ CORE_PROFILE ] ) |'''  \
            ''' ( edge [ EDGE_PROFILE ] ) )'''
   noOrDefaultSyntax = syntax
   data = {
      'inband' : 'Apply inband variant',
      'telemetry' : 'Apply telemetry feature',
      'profile' : 'Apply inband telemetry profile',
      'core': 'Apply core profile',
      'CORE_PROFILE' : coreProfileNameMatcher,
      'edge': 'Apply edge profile',
      'EDGE_PROFILE' : edgeProfileNameMatcher
      }

   @staticmethod
   def handler( mode, args ):
      if "edge" in args:
         profileType = 'edge'
         profileName = args[ 'EDGE_PROFILE' ] if 'EDGE_PROFILE' in args \
                       else 'default'
      else:
         profileType = 'core'
         profileName = args[ 'CORE_PROFILE' ] if 'CORE_PROFILE' in args \
                       else 'default'
      intfName = mode.intf.name
      if profileType == 'edge':
         intfToPortProfiles = inbandTelemetryConfig.intfToEdgePortProfiles
         if intfName in inbandTelemetryConfig.intfToCorePortProfiles:
            del inbandTelemetryConfig.intfToCorePortProfiles[ intfName ]
      else:
         intfToPortProfiles = inbandTelemetryConfig.intfToCorePortProfiles
         if intfName in inbandTelemetryConfig.intfToEdgePortProfiles:
            del inbandTelemetryConfig.intfToEdgePortProfiles[ intfName ]
      if intfName in intfToPortProfiles and intfToPortProfiles[
            intfName ] == profileName:
         return
      intfToPortProfiles[ intfName ] = profileName
      if inbandTelemetryConfig.enable:
         incrementConfigVersion( inbandTelemetryConfig )
      # Attempt to create an internal inband telemetry mirror
      # session when an interface is associated with an edge
      # profile when such an association does not already exist.
      # This must be gated through HW capabilities for future
      # implementations that don't require such an operation.
      handleIntMirrorSession( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'CORE_PROFILE' in args:
         profileName = args[ 'CORE_PROFILE' ]
      elif 'EDGE_PROFILE' in args:
         profileName = args[ 'EDGE_PROFILE' ]
      else:
         profileName = None
      if "edge" in args:
         profileType = 'edge'
      else:
         profileType = 'core'
      intfName = mode.intf.name
      if profileType == 'edge':
         intfToPortProfiles = inbandTelemetryConfig.intfToEdgePortProfiles
      else:
         intfToPortProfiles = inbandTelemetryConfig.intfToCorePortProfiles
      if intfName in intfToPortProfiles:
         if not profileName or ( profileName and intfToPortProfiles[ intfName ]
                                 == profileName ):
            del intfToPortProfiles[ intfName ]
            if inbandTelemetryConfig.enable:
               incrementConfigVersion( inbandTelemetryConfig )
            # Delete the internal inband telemetry mirror session
            # when no interface is associated with an edge profile.
            # This must be gated through HW capabilities for future
            # implementations that don't require such an operation.
            handleIntMirrorSession( mode )

#------------------------------------------------------------------------------------
# "show monitor telemetry inband profile [ ( edge | core) [ NAME ] ]" summary
#------------------------------------------------------------------------------------

def nameInterfaceLinker( profileType, profileName ):
   profileNameToIntfCore = {}
   profileNameToIntfEdge = {}
   modelProfileNameToIntfCore = {}
   modelProfileNameToIntfEdge = {}
   if profileType is None or profileType == 'Core':
      for intf, profName in\
        sorted( inbandTelemetryConfig.intfToCorePortProfiles.iteritems() ):
         if profName not in profileNameToIntfCore:
            profileNameToIntfCore.setdefault( profName, [] ).append( intf )
         else:
            profileNameToIntfCore[ profName ].append( intf )
      if profileName:
         reqList = []
         if profileName in profileNameToIntfCore:
            reqList = profileNameToIntfCore[ profileName ]
         profileNameToIntfCore.clear()
         profileNameToIntfCore[ profileName ] = reqList

   if profileType is None or profileType == 'Edge':
      for intf, profName in\
        sorted( inbandTelemetryConfig.intfToEdgePortProfiles.iteritems() ):
         if profName not in profileNameToIntfEdge:
            profileNameToIntfEdge.setdefault( profName, [] ).append( intf )
         else:
            profileNameToIntfEdge[ profName ].append( intf )
      if profileName:
         reqList = []
         if profileName in profileNameToIntfEdge:
            reqList = profileNameToIntfEdge[ profileName ]
         profileNameToIntfEdge.clear()
         profileNameToIntfEdge[ profileName ] = reqList

   if profileNameToIntfCore:
      for profName in sorted( profileNameToIntfCore ):
         modelProfileNameToIntfCore[ profName ] = IntfList(
               Intfs=profileNameToIntfCore[ profName ] )
   else:
      modelProfileNameToIntfCore = None

   if profileNameToIntfEdge:
      for profName in sorted( profileNameToIntfEdge ):
         modelProfileNameToIntfEdge[ profName ] = IntfList(
               Intfs=profileNameToIntfEdge[ profName ] )
   else:
      modelProfileNameToIntfEdge = None

   return modelProfileNameToIntfCore, modelProfileNameToIntfEdge

def showMonitorTelemetryInbandSummary( mode, args ):
   profileNameToIntfCore = {}
   profileNameToIntfEdge = {}
   profileType = None
   profileName = args.get( 'NAME' )
   if "core" in args:
      profileType = 'Core'
   elif "edge" in args:
      profileType = 'Edge'

   profileNameToIntfCore, profileNameToIntfEdge = nameInterfaceLinker(
         profileType, profileName )
   return ModelIntProfileSummary(
            coreIntfList=profileNameToIntfCore, edgeIntfList=profileNameToIntfEdge )

class ShowMonitorTelemetryInbandSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry inband profile\
         [ ( core | edge ) [ NAME ] ] summary'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'telemetry': CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher(
            'telemetry', helpdesc='Show monitor telemetry information' ),
            guard=telemetryGuard ),
         'inband': 'Show monitor inband telemetry',
         'profile': profileKeyword,
         'core': coreKeyword,
         'edge': edgeKeyword,
         'NAME': profileNameMatcher,
         'summary': 'Show inband telemetry profile summary',
         }
   cliModel = ModelIntProfileSummary
   handler = showMonitorTelemetryInbandSummary
   privileged = True

#------------------------------------------------------------------------------------
# The "show monitor telemetry inband profile [ ( edge | core) [ NAME ] ]" command
#------------------------------------------------------------------------------------

def getEgressDrop( profileName ):
   egressDrop = 'enabled'
   portProfileConfig = inbandTelemetryConfig.edgePortProfiles
   if not portProfileConfig[ profileName ].egressDrop:
      egressDrop = 'disabled'
   return egressDrop

def getProfileStateAndReason( profileType, profileName ):
   if profileType == 'edge':
      profileStatus = inbandTelemetryStatus.edgePortProfileStatus
   else:
      profileStatus = inbandTelemetryStatus.corePortProfileStatus
   if profileName not in profileStatus:
      return 'unknown', ''
   portProfileState = profileStatus[ profileName ].portProfileOperStatus.state
   if portProfileState == 'operStateUnknown':
      profileState = 'unknown'
   elif portProfileState == 'operStateActive':
      profileState = 'active'
   else:
      profileState = 'inactive'
   profileErrorReason = profileStatus[ profileName ].portProfileOperStatus.reason
   return profileState, profileErrorReason

def getPortProfileConfig( profileType ):
   if profileType == 'edge':
      portProfileConfig = inbandTelemetryConfig.edgePortProfiles
   else:
      portProfileConfig = inbandTelemetryConfig.corePortProfiles
   return portProfileConfig

def getProfileModel( profileType, profileName ):
   portProfileConfig = getPortProfileConfig( profileType )
   egressCollection = portProfileConfig[ profileName ].collectorName
   profileState, profileErrorReason = getProfileStateAndReason( profileType,
                                                                profileName )
   extraArgs = {}

   if profileType == 'edge':
      extraArgs[ "sampleRate" ] = inbandTelemetryConfig.\
            edgePortProfiles[ profileName ].sampleRate

      if Toggles.InbandTelemetryCommonToggleLib.\
            toggleFeatureInbandTelemetrySamplePolicyEnabled():
         extraArgs[ "samplePolicy" ] = inbandTelemetryConfig.\
               edgePortProfiles[ profileName ].samplePolicy

      extraArgs[ "egressDrop" ] = getEgressDrop( profileName )

   return Profile( profileType=profileType,
                   egressCollection=egressCollection,
                   profileStatus=profileState,
                   profileErrorReason=profileErrorReason,
                   **extraArgs )

def getProfiles( profileType, profileName ):
   config = {}
   portProfileConfig = getPortProfileConfig( profileType )
   if profileName == '':
      for portProfileName in portProfileConfig:
         config[ portProfileName ] = getProfileModel( profileType, portProfileName )
   else:
      if profileName in portProfileConfig:
         config[ profileName ] = getProfileModel( profileType, profileName )
   return config

def getPolicyModel( policyName ):
   rules = []
   pmapConfig = policiesCliConfig.pmapType.pmapInput[ policyName ]
   ruleNames = pmapConfig.currCfg.classPrio.values()
   for ruleName in ruleNames:
      dscps = []
      rule = pmapConfig.currCfg.rawClassMap[ ruleName ]
      match = rule.match.items()[ 0 ][ 1 ]
      matchOpt = "ipv4" if match.option == 'matchIpAccessGroup' else "ipv6"
      for dscpRange in match.structuredFilter.dscp:
         dscps.append( NumericalRange( low=dscpRange.rangeStart,
                                       high=dscpRange.rangeEnd ) )
      dscpList = sorted( dscps, key=lambda x: x.low )
      policyActions = pmapConfig.currCfg.classAction[ ruleName ].policyAction
      actions = Actions( sample='sample' in policyActions,
                         sampleAll='sampleAll' in policyActions )
      rules.append( Rule( ruleString=ruleName, matchOption=matchOpt,
                          matches=Matches( dscps=dscpList ), actions=actions ) )
   return SamplePolicyModel( rules=rules )

def getPolicies():
   config = {}
   for policyName in policiesCliConfig.pmapType.pmapInput:
      config[ policyName ] = getPolicyModel( policyName )
      return config

def showInbandTelemetryProfileHandler( mode, args ):
   if 'NAME' in args:
      profileName = args[ 'NAME' ]
   else:
      profileName = ''
   if 'edge' in args:
      profileType = 'edge'
   elif 'core' in args:
      profileType = 'core'
   else:
      profileType = ''
   coreProfiles = None
   edgeProfiles = None
   if profileType == 'core':
      coreProfiles = getProfiles( profileType, profileName )
   elif profileType == 'edge':
      edgeProfiles = getProfiles( profileType, profileName )
   else:
      coreProfiles = getProfiles( 'core', profileName )
      edgeProfiles = getProfiles( 'edge', profileName )
   return InbandTelemetryProfiles( coreProfiles=coreProfiles,
                                   edgeProfiles=edgeProfiles )

class ShowInbandTelemetryProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry inband profile [ ( edge | core ) [ NAME ] ]'
   data = { 'monitor': CliToken.Monitor.monitorMatcherForShow,
            'telemetry': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'telemetry', helpdesc='Show monitor telemetry information' ),
               guard=telemetryGuard ),
            'inband': 'Show monitor inband telemetry',
            'profile': profileKeyword,
            'edge': edgeKeyword,
            'core': coreKeyword,
            'NAME': profileNameMatcher
   }
   cliModel = InbandTelemetryProfiles
   handler = showInbandTelemetryProfileHandler
   privileged = True

class AbortCmd( CliCommand.CliCommandClass ):
   syntax = '''abort'''
   data = {
      'abort': CliToken.Cli.abortMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

#------------------------------------------------------------------------------------
# The "show monitor telemetry inband" command
#------------------------------------------------------------------------------------

def showInbandTelemetryHandler( mode, args ):
   probeMarker = inbandTelemetryConfig.intParam.probeMarker
   coreProfiles = getProfiles( 'core', '' )
   edgeProfiles = getProfiles( 'edge', '' )

   profiles = InbandTelemetryProfiles( coreProfiles=coreProfiles,
                                       edgeProfiles=edgeProfiles )
   if not inbandTelemetryStatus.aggHwStatus or inbandTelemetryStatus.version != \
      inbandTelemetryStatus.aggHwStatus.version:
      operStatus = "In Progress"
   else:
      if inbandTelemetryStatus.intOperStatus.state == tacOperState.operStateActive:
         operStatus = "Success"
      elif inbandTelemetryStatus.intOperStatus.state == \
           tacOperState.operStateInactive:
         operStatus = "Error, %s" % inbandTelemetryStatus.intOperStatus.reason
      else:
         operStatus = "In Progress"

   policies = getPolicies()
   return InbandTelemetry( enabled=inbandTelemetryConfig.enable,
         deviceId=inbandTelemetryConfig.intParam.deviceId,
         probeMarker=probeMarker,
         probeProtocol=inbandTelemetryConfig.intParam.probeProtocol,
         operStatus=operStatus,
         profiles=profiles, policies=policies,
         intDetectionMethod=inbandTelemetryConfig.intParam.intDetectionMethod )

class ShowInbandTelemetry( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry inband'
   data = { 'monitor': CliToken.Monitor.monitorMatcherForShow,
            'telemetry': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'telemetry', helpdesc='Show monitor telemetry information' ),
               guard=telemetryGuard ),
            'inband': 'Show monitor inband telemetry',
   }
   cliModel = InbandTelemetry
   handler = showInbandTelemetryHandler
   privileged = True


# SHOW COMMANDS
#------------------------------------------------------------
# show flow tracking inband telemetry flow-table [detail]
# [ tracker <tracker-name> ] [ group <group-name> ]
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ interface <interface-range> ]
#------------------------------------------------------------
class ShowIntFlowTable( ShowFlowTable ):

   def getFlowTable( self, sMount, trackerName ):
      mountPath = 'flowtracking/%s/flowTable/%s' % ( ftrTypeInbandTelemetry,
                                                     trackerName )
      entityType = 'InbandTelemetry::IntFlowTable'
      smashFlowTable = sMount.doMount( mountPath, entityType,
                                       Smash.mountInfo( 'reader' ) )
      return smashFlowTable

   def fetchTrackingModel( self ):
      trackingModel = IntTrackingModel()
      trackingModel.running = isInbandTelemetryAgentRunning( em, activeAgentDir )
      trackingModel.ftrType = ftrTypeInbandTelemetry
      return trackingModel

   def showIntFlowDetail( self, flowDetailModel, entry ):
      flowDetailModel.pathTransistions = entry.intPathId
      flowDetailModel.pathPackets = entry.intPathNumPackets
      flowDetailModel.hopCountExceeded = bool( entry.intPathOverflowed )
      flowDetailModel.devicesInPath = entry.intPathNumNodes
      flowDetailModel.flowIntervals = entry.intPathNumIntervals
      for nodeIdx in range( entry.intPathNumNodes ):
         nodeInfo = IntFlowDetailNodeInfo()
         nodeInfo.deviceId = entry.intNodeInfo[ nodeIdx ].deviceId
         nodeInfo.ingressPortId = entry.intNodeInfo[ nodeIdx ].ingressPort
         nodeInfo.egressPortId = entry.intNodeInfo[ nodeIdx ].egressPort
         nodeInfo.egressQueueId = entry.intNodeInfo[ nodeIdx ].egressQueueId
         nodeInfo.ttl = entry.intNodeInfo[ nodeIdx ].intTTL
         nodeInfo.lastPacketCongestion = entry.intNodeInfo[ nodeIdx ].congestion
         nodeInfo.lastPacketLatency = entry.intNodeInfo[ nodeIdx ].latency
         flowDetailModel.devicesInformation.append( nodeInfo )
      for statIdx in range( entry.intPathNumIntervals ):
         intrvlStats = IntFlowDetailIntervalStats()
         for nodeIdx in range( entry.intPathNumNodes ):
            globalStatEntry = entry.intGlobalIntervalStats[ statIdx ]
            statEntry = entry.intIntervalStats[
                  ( nodeIdx * hwCapability.intFlowTrackingIntervalsPerNode ) +
                        statIdx ]
            intrvlStats.timestamp = globalStatEntry.intervalStartTime
            intrvlStats.pkts = globalStatEntry.numPackets
            intrvlStats.congestions.append( statEntry.congestion )
            intrvlStats.avgLatencies.append( long( statEntry.avgLatency ) )
            intrvlStats.maxLatencies.append( long( statEntry.maxLatency ) )
            intrvlStats.minLatencies.append( long( statEntry.minLatency ) )
         flowDetailModel.flowIntervalStats.append( intrvlStats )

   def showFlowEntry( self, flowTable, ipGroup, key, entry, bgp=None,
         ifIndexMap=None, displayType=None ):

      flowModel = self.populateFlowModel( ipGroup, key, entry )

      if displayType == "detail":
         flowDetailModel = IntFlowDetailModel()
         intFlowEntry = \
               flowTable.intIpFlowEntry if ipGroup else flowTable.intIp6FlowEntry
         self.showFlowDetail( flowDetailModel, ipGroup, entry, bgp, ifIndexMap )
         self.showIntFlowDetail( flowDetailModel, intFlowEntry.get( key ) )
         flowModel.flowDetail = flowDetailModel

      return flowModel

def showIntFlowTable( mode, args ):
   t = ShowIntFlowTable()
   return t.showFlowTable( mode, args )

ftrTypeInbandTelemetryKwStrs = ftrTypeKwStr[ ftrTypeInbandTelemetry ].split()

class ShowIntFlowTrackingFlowTable( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ' + ftrTypeKwStr[ ftrTypeInbandTelemetry ] + \
         ' flow-table [ SHOW_TRACKING_FILTER ] '
   data = {
      'flow': CliToken.Flow.flowMatcherForShow,
      'tracking': trackingShowKw,
      ftrTypeInbandTelemetryKwStrs[ 0 ]: telemetryShowKw,
      ftrTypeInbandTelemetryKwStrs[ 1 ]: inbandShowKw,
      'flow-table': flowTableKw,
      'SHOW_TRACKING_FILTER': generateShowTrackingFilter()
   }

   handler = showIntFlowTable
   cliModel = IntTrackingModel

class ShowIntFlowCounters( ShowCounters ):
   def getFlowTable( self, sMount, trackerName ):
      mountPath = 'flowtracking/%s/flowTable/%s' % ( ftrTypeInbandTelemetry,
                                                     trackerName )
      entityType = 'InbandTelemetry::IntFlowTable'
      smashFlowTable = sMount.doMount( mountPath, entityType,
                                       Smash.mountInfo( 'reader' ) )
      return smashFlowTable

   def showCounters( self, mode, args ):
      model = FtrCounters()
      model.ftrType = ftrTypeInbandTelemetry
      model.running = isInbandTelemetryAgentRunning( em, activeAgentDir )
      if model.running:
         trFilter = args.get( 'TRACKER_NAME' )
         expFilter = args.get( 'EXPORTER_NAME' )
         self.addTrackers( model, intFtCounters, trFilter, expFilter )
      return model

def showIntFlowCounters( mode, args ):
   t = ShowIntFlowCounters()
   return t.showCounters( mode, args )

class ShowIntCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ' + ftrTypeKwStr[ ftrTypeInbandTelemetry ] + \
            ' counters [ tracker TRACKER_NAME [ exporter EXPORTER_NAME ] ]'

   data = {
      'flow': CliToken.Flow.flowMatcherForShow,
      'tracking': trackingShowKw,
      ftrTypeInbandTelemetryKwStrs[ 0 ]: telemetryShowKw,
      ftrTypeInbandTelemetryKwStrs[ 1 ]: inbandShowKw,
      'counters': 'Show flow tracking counters',
      'tracker': trackerKw,
      'TRACKER_NAME': trackerNameMatcher,
      'exporter': exporterKw,
      'EXPORTER_NAME': exporterNameMatcher,
   }
   handler = showIntFlowCounters
   cliModel = FtrCounters

# Commands guarded by Toggle. Also protecting exclusively
if Toggles.InbandTelemetryCommonToggleLib.toggleFeatureInbandTelemetryEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( MonitorTelemetryInband )
   # TODO: when collector log is supported uncomment the below
   #InbandTelemetryConfigMode.addCommandClass( EgressCollectionLogNumCmd )
   InbandTelemetryConfigMode.addCommandClass( DeviceIdCmd )
   InbandTelemetryConfigMode.addCommandClass( ProbeMarkerCmd )
   InbandTelemetryConfigMode.addCommandClass( ProfileConfigCmd )
   InbandTelemetryConfigMode.addCommandClass( DisabledCmd )
   InbandTelemetryConfigMode.addModelet( CommitAbortModelet )
   IntfTelemetryModelet.addCommandClass( InbandTelemetryInterface )
   ProfileConfigMode.addCommandClass( SampleRateCmd )
   # TODO: when collector log is supported uncomment the below
   #ProfileConfigMode.addCommandClass( EgressCollectionLogCmd )
   ProfileConfigMode.addCommandClass( EgressCollectionFlowTrackerCmd )
   ProfileConfigMode.addCommandClass( EgressDropDisabledCmd )
   ProfileConfigMode.addModelet( CommitAbortModelet )
   BasicCli.addShowCommandClass( ShowInbandTelemetry )
   BasicCli.addShowCommandClass( ShowInbandTelemetryProfile )
   BasicCli.addShowCommandClass( ShowMonitorTelemetryInbandSummary )
   BasicCli.addShowCommandClass( ShowIntFlowTrackingFlowTable )
   BasicCli.addShowCommandClass( ShowIntCounters )
   if Toggles.InbandTelemetryCommonToggleLib.\
      toggleFeatureIfa2BasedInbandTelemetryEnabled():
      InbandTelemetryConfigMode.addCommandClass( ProbeProtocolCmd )
   if Toggles.InbandTelemetryCommonToggleLib.\
      toggleFeatureInbandTelemetrySamplePolicyEnabled():
      InbandTelemetryConfigMode.addCommandClass( IntSamplePolicyConfigCmd )
      IntSamplePolicyConfigMode.addCommandClass( IntSamplePolicyMatchRuleConfigCmd )
      IntSamplePolicyConfigMode.addModelet( CommitAbortModelet )
      IntSamplePolicyMatchRuleConfigMode.addCommandClass( DscpConfigCmd )
      IntSamplePolicyMatchRuleConfigMode.addCommandClass(
         IntSamplePolicyActionsConfigCmd )
      IntSamplePolicyActionRuleConfigMode.addCommandClass( SampleActionCmd )
      IntSamplePolicyMatchRuleConfigMode.addModelet( CommitAbortModelet )
      IntSamplePolicyActionRuleConfigMode.addModelet( CommitAbortModelet )
      IntSamplePolicyConfigMode.addShowCommandClass( ShowPendingCmd )
      IntSamplePolicyMatchRuleConfigMode.addShowCommandClass( ShowPendingCmd )
      IntSamplePolicyActionRuleConfigMode.addShowCommandClass( ShowPendingCmd )
      ProfileConfigMode.addCommandClass( IngressSamplePolicyCmd )

#-------------------------------------------------------------------------------
# Support for default interface command.
# Clean up all the configuration set by cmds in interface config mode.
#-------------------------------------------------------------------------------
class InbandTelemetryIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      if intfName in inbandTelemetryConfig.intfToEdgePortProfiles:
         del inbandTelemetryConfig.intfToEdgePortProfiles[ intfName ]
      elif intfName in inbandTelemetryConfig.intfToCorePortProfiles:
         del inbandTelemetryConfig.intfToCorePortProfiles[ intfName ]

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
@Plugins.plugin( provides=( "InbandTelemetryCli", ) )
def Plugin( entityManager ):
   global inbandTelemetryConfig, inbandTelemetryStatus, hwCapability, em
   global policiesCliConfig, policiesStatus, policiesStatusRequestDir
   global activeAgentDir, intFtCounters

   em = entityManager

   samplePoliciesConfigPath = 'inbandtelemetry/samplePolicies/input/cli'
   samplePoliciesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesStatusPath = 'cell/%d/inbandtelemetry/samplePolicies/status' % \
                        Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesStatusRequestDirPath = 'inbandtelemetry/samplePolicies/statusRequest'
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'

   mountGroup = entityManager.mountGroup()
   policiesStatus = mountGroup.mount( policiesStatusPath, policiesStatusType, 'ri' )
   mountGroup.close( callback=None, blocking=False )

   inbandTelemetryConfig = ConfigMount.mount( entityManager,
                                              "inbandtelemetry/config",
                                              "InbandTelemetry::Config", "w" )
   inbandTelemetryStatus = LazyMount.mount( entityManager, "inbandtelemetry/status",
                                            "InbandTelemetry::Status", "r" )
   hwCapability = LazyMount.mount( entityManager, "inbandtelemetry/hwCapability",
                                   "InbandTelemetry::HwCapability", "r" )
   policiesCliConfig = ConfigMount.mount( entityManager, samplePoliciesConfigPath,
                                          samplePoliciesCliConfigType, 'wi' )
   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestDirPath,
                                               policiesStatusRequestDirType, 'w' )
   intFtCounters = SmashLazyMount.mount( entityManager,
                                         "flowtracking/inbandTelemetry/counters",
                                         "Smash::FlowTracker::FtCounters",
                                         SmashLazyMount.mountInfo( 'reader' ) )
   IntfCli.Intf.registerDependentClass( InbandTelemetryIntfJanitor )
   activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                     'Tac::Dir', 'ri' )
