#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import cStringIO, os
from itertools import chain

# pylint: disable-msg=F0401
import AgentCommandRequest
import AgentDirectory
import Arnet
from Arnet import IpGenAddr
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.BridgingCli as BridgingCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.MacAddr as MacAddr
from CliPlugin.SwitchIntfCli import SwitchIntf
import CliPlugin.VlanCli as VlanCli
import CliPlugin.WaitForWarmupCli as WaitForWarmupCli
import CliToken.Clear
import CliToken.Ip
import CliToken.IgmpSnooping
from CliToken.IgmpSnooping import * # pylint: disable-msg=W0401
import CliModel
import ConfigMount
import Ethernet
from EthIntfCli import EthPhyIntf, EthPhyAutoIntfType
import IgmpSnoopingCliLib
import IgmpSnoopingLib
from IgmpSnoopingLib import agentName
from IgmpSnoopingModel import IgmpSnoopingVlanInfo, IgmpSnoopingInfo, \
      IgmpSnoopingVlanReportFlooding, IgmpSnoopingReportFlooding, \
      IgmpSnoopingQuerier, IgmpSnoopingVlanQuerier, IgmpQuerierStatus, \
      IgmpQuerierStatusVlan, IgmpSnoopingQuerierMembership, MulticastGroupInfo, \
      IgmpSnoopingVlanQuerierMembership, GroupSourceInfo, \
      IgmpSnoopingQuerierCounters, IgmpSnoopingVlanQuerierCounters,\
      IgmpSnoopingMrouterDetail, IgmpSnoopingMrouterVlan, IgmpSnoopingMrouter, \
      IgmpQuerierMembershipGroup, IgmpSnoopingCounters, IgmpSnoopingTableEntry, \
      IgmpSnoopingCountersInterface, MacAddressTableIgmpSnooping, \
      GmpMembershipInfo, GmpMembershipDetailInfo, GmpMembershipCountInfo
import Intf.IntfRange
import Intf.Log
from LagIntfCli import EthLagIntf, LagAutoIntfType
import LazyMount
import ShowCommand
import Tac
import Tracing
import Toggles.IgmpSnoopingToggleLib


t0 = Tracing.trace0
bridgingHwCapabilities = None

def igmpSnoopingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.igmpSnoopingSupported:
      return None
   return CliParser.guardNotThisPlatform

def isSnoopingAgentRunning( sysname, agent ):
   if 'AGENT_COMMAND_REQUEST_DRY_RUN' in os.environ:
      # Running from breadth tests
      return True

   return AgentDirectory.agentIsRunning( sysname, agent )

def runAgentCommand( mode, agent, command, model=None, **kwargs ):
   """
   @param mode: Cli Mode
   @param command: Command string to be send to agent
   @param model: Cli model used for Capi. None if there is no Capi model.
   @param **kwargs: Arbitrary keyword arguments, passed to runSocketCommand function

   @return: None when error occurs else model object having value set

   @exception: Handles RunSocketCommandException while other exception will be raised
   """

   # check if the agent is running
   if not isSnoopingAgentRunning( mode.entityManager.sysname(), agent ):
      mode.addError( '%s agent is not running' % agent )
      return

   errors = [ '%s Agent is not initialized\n' % agent ]
   try:
      outputFormat = mode.session_.outputFormat_
      AgentCommandRequest.runSocketCommand(
            mode.entityManager, agent, "IgmpSnoopingWithFormat",
            command, stringBuff=None, outputFormat=outputFormat,
            throwException=True, errors=errors, **kwargs )
      return CliModel.cliPrinted( model )
   except AgentCommandRequest.RunSocketCommandException, e:
      mode.addError( str( e ) )

def runOldAgentCommand( em, agent, *args, **kwargs ):
   if AgentDirectory.agentIsRunning( em.sysname(), agent ):
      AgentCommandRequest.runSocketCommand( em, agent, *args, **kwargs )

# Any vlan/port change should warmup for IgmpSnooping agent.
WaitForWarmupCli.warmupAgentsOnVlanPortChange.append( lambda m: [ 'IgmpSnooping' ] )

igmpNode.guard_ = igmpSnoopingSupportedGuard

mrouterNodeDeprecated = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'mrouter',
                                      helpdesc='IGMP router related configuration' ),
   deprecatedByCmd='ip igmp snooping vlan <vlan-range> '\
                    'multicast-router interface <interface-range>' )
multicastRouterMatcher = CliMatcher.KeywordMatcher( 'multicast-router',
   helpdesc='IGMP router related configuration' )

reportFloodingMatcher = CliMatcher.KeywordMatcher(
   'report-flooding',
   helpdesc='IGMP Membership Report Flooding feature' )
interfaceMatcher = CliMatcher.KeywordMatcher(
   'interface',
   helpdesc='Interface configuration' )
switchPortMatcher = CliMatcher.KeywordMatcher(
   'switch-port',
   helpdesc='Specify switch port' )
staticNodeDeprecated = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      'static',
      helpdesc="Specify static multicast groups" ),
   deprecatedByCmd='ip igmp snooping vlan <vlan-id> member <ip-addr>'
                   ' interface <interface-range>' )
memberMatcher = CliMatcher.KeywordMatcher(
   'member',
   helpdesc="Specify multicast group interface members" )

numGroupsMatcher = CliMatcher.IntegerMatcher(
   1, 4096, helpdesc='The number of multicast groups to be configured' )

matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='Counters' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Detail' )
matcherGroup = CliMatcher.KeywordMatcher( 'group',
      helpdesc='Group' )
matcherMembership = CliMatcher.KeywordMatcher( 'membership',
      helpdesc='Membership' )
matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='Show multicast information' )
matcherQuerier = CliMatcher.KeywordMatcher( 'querier',
      helpdesc='IGMP querier related status and configuration' )
matcherStatus = CliMatcher.KeywordMatcher( 'status',
      helpdesc='Status' )
matcherImmediateLeave = CliMatcher.KeywordMatcher( 'immediate-leave',
      helpdesc='Leave groups immediately with a V2 leave from last known host' )
immediateLeaveNode = CliCommand.Node( matcher=matcherImmediateLeave,
      deprecatedByCmd='ip igmp snooping [vlan <vlan-range>] fast-leave' )
matcherFastLeave = CliMatcher.KeywordMatcher( 'fast-leave',
      helpdesc='Leave groups immediately with a V2 leave from last known host' )

class IpIgmpSnoopingForConfig( CliCommand.CliExpression ):
   expression = "ip igmp snooping"
   data = {
      "ip" : CliToken.Ip.ipMatcherForConfig,
      "igmp" : CliToken.IgmpSnooping.igmpNode,
      "snooping" : CliToken.IgmpSnooping.snoopingMatcher
      }

# similar to VlanCli.vlanId, however it doesnt return Vlan object,
# which has overloaded __cmp__ operator
vlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Identifier for a Virtual LAN' )

intfTypes = ( EthPhyAutoIntfType, LagAutoIntfType )

profileConfig = None

igmpSnoopingConfig = None
swForceConfig = None
igmpSnoopingCounterConfig = None
forwardingStatus = None
protocolConfig = None
protocolStatus = None
querierConfigDir = None
querierStatusDir = None
counterDir = None
counterCheckpointDir = None
bridgingConfig = None
bridgingInputCli = None
entityManager = None
peerSyncStatus = None

#####################################################################################
# helper functions
#####################################################################################
ipAddrZero = Arnet.IpAddr( "0.0.0.0" )
MulticastRouter = Tac.Type( "Bridging::IgmpSnooping::MulticastRouter" )

def getSysdbNodes():
   '''Return frequently used sysdb directories/objects.'''
   vlanConfig = igmpSnoopingConfig.vlanConfig
   contextStatus = protocolStatus.contextStatus
   vlanStatus = forwardingStatus.vlanStatus
   return ( igmpSnoopingConfig, vlanConfig, contextStatus, vlanStatus )

def getRouterIntfs( vlanId, configVlan, contextStatusVlan,
                     statusVlan, detail = False ):
   '''The function returns list of static and dynamic mrouter interfaces.
      if detail is True, it also returns a dict containing details of all
      routers registered via igmp.
   '''
   dynamicIntfs = []
   staticIntfs = []
   mlagPeerIntfs = []
   mrouterDetails = []
   fwdRtrIntfs = []

   if statusVlan:
      fwdRtrIntfs = statusVlan.routerIntf.keys()
   if configVlan and statusVlan and statusVlan.routerIntf:
      staticIntfs = [ i for i in configVlan.routerIntf.keys() if i in fwdRtrIntfs ]
   if contextStatusVlan:
      routerInfo = contextStatusVlan.router.itervalues()

      for mrouter in routerInfo:
         # The core has to keep track of router since it wants to forward
         # leaves/joins to the router port. Hence we have to skip it in cli
         # if we don't want to show it in cli
         # its better to not show the l2 querier as a router/group member
         # since we know its not a router anyway.
         if mrouter.intf == 'Switch':
            continue
         # two routers with diff addresses can be on same intf
         if mrouter.intf not in dynamicIntfs:
            dynamicIntfs.append( mrouter.intf )
         if detail:
            router = {}
            router[ 'addr' ] = mrouter.addr
            router[ 'intf' ] = mrouter.intf
            router[ 'firstHeard' ] = mrouter.firstHeard + Tac.utcNow() - Tac.now()
            router[ 'lastHeard' ] =  mrouter.lastHeard + Tac.utcNow() - Tac.now()
            router[ 'expire' ] = mrouter.expire + Tac.utcNow() - Tac.now()
            router[ 'type' ] = mrouter.routerType
            mrouterDetails.append( router )

      #Are there any router interfaces we need to add from peer
      #These are interfaces that are in forwarding status but not in
      #our local context and not statically configured. Consider them as mlag
      #peer interfaces
      mlagPeerIntfs = \
         list( set( fwdRtrIntfs ) - set( staticIntfs + dynamicIntfs ) )
         
   return ( staticIntfs, dynamicIntfs, mlagPeerIntfs, mrouterDetails )


def getVlanSnoopingState( config, vlanId ):
   state = 'enabled' if config.vlanDefaultEnabled else 'disabled'
   vlan = config.vlanConfig.get( vlanId )
   if vlan:
      if vlan.enabled == 'vlanStateDisabled':
         state = 'disabled'
      # else it is default as inherited from global setting.
   return state

igmpVersionTable = {
   'igmpVersionUnknown': 'unknown',
   'igmpVersion1': 'v1',
   'igmpVersion2': 'v2',
   'igmpVersion3': 'v3',
   'mldVersion1':  'v1',
   'mldVersion2':  'v2'
   }
filterModePrintingTable = {
   'filterModeExclude': 'exclude',
   'filterModeInclude': 'include'
   }
routerTypePrintingTable = {
   'multicastRouterTypeUnknown': 'unknown',
   'multicastRouterTypePim': 'pim',
   'multicastRouterTypeDvmrp': 'dvmrp',
   'multicastRouterTypeMrd': 'mrd',
   'multicastRouterTypeIgmpQuerier' : 'querier'
   }

#####################################################################################
# show ip igmp snooping [ vlan <vlanid> ]
#####################################################################################

def doShowIpIgmpSnooping( mode, args ):
   ''' the function shows singleton attributes for the global settings,
       and per-vlan settings. Only the active vlans are shown here, as per
       industry-standard.
   '''
   vlanId = args.get( 'VLAN_ID' )
   igmpSnooping = IgmpSnoopingInfo()
   if vlanId != None:
      if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
         mode.addWarning( "Vlan %s is not active." % vlanId )
         return igmpSnooping
      vlanIds = [ vlanId ]
   else:
      vlanIds = IgmpSnoopingLib.activeVlans( bridgingInputCli )

   if igmpSnoopingConfig.vlanDefaultEnabled is True:
      igmpSnooping.igmpSnoopingState = 'enabled'
   else:
      igmpSnooping.igmpSnoopingState = 'disabled'

   if igmpSnoopingConfig.immediateLeave:
      igmpSnooping.immediateLeave = 'enabled'
   else:
      igmpSnooping.immediateLeave = 'disabled'
   igmpSnooping.robustness = igmpSnoopingConfig.robustness
   if igmpSnoopingConfig.globalReportFloodingEnabled:
      igmpSnooping.reportFlooding = 'enabled'
   else:
      igmpSnooping.reportFlooding = 'disabled'
   igmpSnooping.reportFloodingSwitchPorts = getReportFloodingSwitchPorts()
   for vlanId in vlanIds:
      igmpSnooping.vlans[ vlanId ] = \
            doShowIpIgmpSnoopingVlan( igmpSnoopingConfig, swForceConfig, vlanId )
   return igmpSnooping

# XXX...what about ????
# immediate leave specifically?

def doShowIpIgmpSnoopingVlan( config, forceConfig, vlanId ):
   ''' show singleton attributes for a given vlan. '''

   vlans = IgmpSnoopingVlanInfo()
   vlans.igmpSnoopingState = getVlanSnoopingState( config, vlanId )
   vlanConfig = config.vlanConfig.get( vlanId )

   if vlanConfig:
      if vlanConfig.immediateLeave == "immediateLeaveEnabled":
         vlans.immediateLeave = 'enabled'
      elif vlanConfig.immediateLeave == "immediateLeaveDisabled":
         vlans.immediateLeave = 'disabled'
      else:
         vlans.immediateLeave = 'default'
   else:
      vlans.immediateLeave = 'default'

   vlans.multicastRouterLearningMode = 'pim-dvmrp'
   contextStatus = protocolStatus.contextStatus.get( vlanId )


   if vlanConfig and (vlanConfig.maxGroups < vlanConfig.unlimitedGroups):
      vlans.maxGroups = vlanConfig.maxGroups

   if contextStatus and contextStatus.groupLimitExceeded:
      vlans.groupsOverrun = True

   if vlanConfig and vlanConfig.reportFloodingConfig \
         == 'vlanReportFloodingEnabled':
      vlans.reportFlooding = 'enabled'
   elif vlanConfig and vlanConfig.reportFloodingConfig \
         == 'vlanReportFloodingDefault':
      vlans.reportFlooding = 'default'
   else:
      vlans.reportFlooding = 'disabled'
   (_, _, _, statuses ) = getSysdbNodes()
   statusVlan = statuses.get( vlanId )
   if contextStatus:
      vlans.pruningActive = contextStatus.confident
   if statusVlan:
      vlans.floodingTraffic = statusVlan.useVlanFloodset

   contextConfig = protocolConfig.contextConfig.get( vlanId )
   if contextConfig:
      vlans.proxyActive =  contextConfig.proxyEnabled
   return vlans

#--------------------------------------------------------------------------------
# show ip igmp snooping [ vlan VLAN_ID ]
#--------------------------------------------------------------------------------
class ShowIpIgmpSnoopingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping [ vlan VLAN_ID ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
   }

   handler = doShowIpIgmpSnooping
   cliModel = IgmpSnoopingInfo

BasicCli.addShowCommandClass( ShowIpIgmpSnoopingCmd )

#####################################################################################
# show ip igmp snooping mrouter [vlan <vlanid>]
#####################################################################################

def doShowGmpSnoopingMrouter( mode, vlanId, detail, sysdbNodes ):
   ''' show multicast router entries. '''
   ( _, vlanConfig, contextStatuses, statuses ) = sysdbNodes
   igmpSnoopingMrouter = IgmpSnoopingMrouter()
   if vlanId != None:
      if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
         return igmpSnoopingMrouter
      vlanIds = [ vlanId ]
   else:
      vlanIds = contextStatuses.keys()
      vlanIds.sort( key=int  )

   detailInfo = ( True if detail else False )
   vlans = {}
   for vlanId in vlanIds:
      contextStatusVlan = contextStatuses.get( vlanId )
      configVlan = vlanConfig.get( vlanId )
      statusVlan = statuses.get( vlanId )
      vlanInfo = doShowIpIgmpSnoopingMrouterVlan(
           configVlan, contextStatusVlan, statusVlan, vlanId, detail )
      if vlanInfo:
         vlans[ vlanId ] = vlanInfo
   return IgmpSnoopingMrouter( vlans=vlans, _detailInfo=detailInfo )

def doShowIpIgmpSnoopingMrouter ( mode, args ):
   detail = 'detail' in args
   return doShowGmpSnoopingMrouter( mode, args.get( 'VLAN_ID' ), detail,
                                    getSysdbNodes() )

def doShowIpIgmpSnoopingMrouterVlan( configVlan, contextStatusVlan, statusVlan,
                                     vlanId, detail ):
   ''' show mrouter entries for a particular vlan.
   '''
   (staticIntfs, dynamicIntfs, mlagPeerIntfs, routerDetails) = getRouterIntfs(
         vlanId, configVlan, contextStatusVlan, statusVlan, detail=detail )
   if not staticIntfs and not dynamicIntfs and not mlagPeerIntfs:
      return None

   igmpSnoopingMrouterVlan = IgmpSnoopingMrouterVlan()
   if not detail:
      intfnameset = set( staticIntfs )
      intfnameset.update( dynamicIntfs )
      intfnameset.update( mlagPeerIntfs )
      intfnames = sorted( intfnameset )
      for intf in intfnames:
         interfaces = IgmpSnoopingMrouterDetail()
         interfaces.interfaceName = intf
         interfaces.routerType = ( 'static' if intf in staticIntfs
               else 'dynamic' )
         igmpSnoopingMrouterVlan.interfaces.append( interfaces )
      return igmpSnoopingMrouterVlan

   routerDetails.sort( key=lambda item: Arnet.intfNameKey( item[ 'intf' ] ) )
   for rd in routerDetails:
      interfaces = IgmpSnoopingMrouterDetail()
      interfaces.interfaceName = rd[ 'intf' ]
      interfaces.address = rd[ 'addr' ]
      interfaces.firstHeardTime = rd ['firstHeard' ]
      interfaces.lastHeardTime = rd[ 'lastHeard' ]
      interfaces.expireTime = rd[ 'expire' ]
      interfaces.routerType = routerTypePrintingTable[ rd[ 'type' ] ]
      igmpSnoopingMrouterVlan.interfaces.append( interfaces )
   for intfName in staticIntfs:
      interfaces = IgmpSnoopingMrouterDetail()
      interfaces.interfaceName = intfName
      interfaces.address = '0.0.0.0'
      interfaces.routerType = 'static'
      igmpSnoopingMrouterVlan.interfaces.append( interfaces )
   for intfName in mlagPeerIntfs:
      interfaces = IgmpSnoopingMrouterDetail()
      interfaces.interfaceName = intfName
      interfaces.address = '0.0.0.0'
      igmpSnoopingMrouterVlan.interfaces.append( interfaces )
   return igmpSnoopingMrouterVlan

#--------------------------------------------------------------------------------
# show ip igmp snooping mrouter [ vlan VLAN_ID ] [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpSnoopingMrouterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping mrouter [ vlan VLAN_ID ] [ detail ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'mrouter' : 'IGMP router related status and configuration',
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'detail' : matcherDetail,
   }

   handler = doShowIpIgmpSnoopingMrouter
   cliModel = IgmpSnoopingMrouter

BasicCli.addShowCommandClass( IpIgmpSnoopingMrouterCmd )

#####################################################################################
# show ip igmp snooping counter [errors] [detail]
#####################################################################################

def syncCounterCheckpointDir( toDir, fromDir ):
   fromIntfNames = set( fromDir.intfCounter.keys() )
   toIntfNames = set( toDir.intfCounter.keys() )

   for intf in toIntfNames:
      if intf not in fromIntfNames:
         del toDir.intfCounter[ intf ]

   toIntfNames = set( toDir.intfCounter.keys() )

   for intf in fromIntfNames:
      if intf not in toIntfNames:
         toDir.intfCounter.newMember( intf )

def updatePortCounters( mode, args ):
   igmpSnoopingCounterConfig.counterUpdateRequestTime = Tac.now() # monotonic time
   def countersUpdated():
      return forwardingStatus.counterUpdateTime >= \
             igmpSnoopingCounterConfig.counterUpdateRequestTime
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.2,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def doShowIpIgmpSnoopingCounter( mode, args ):
   ''' show multicast counter entries. '''
   errors = 'errors' in args
   syncCounterCheckpointDir( counterCheckpointDir, counterDir )

   if not counterCheckpointDir.intfCounter:
      return IgmpSnoopingCounters()

   c = Tac.newInstance( "Bridging::IgmpSnooping::IgmpContextIntfCounter", "" )
   interfaces = {}
   for intf in Arnet.sortIntf( counterCheckpointDir.intfCounter ):
      counter = counterDir.intfCounter.get( intf )
      counterCheckpoint = counterCheckpointDir.intfCounter.get( intf )
      if counter is None or counterCheckpoint is None:
         continue
      # print difference since last checkpointed counter.
      c.doCopy( counter )
      c.doSubtract( counterCheckpoint )
      interfaceCounters = IgmpSnoopingCountersInterface()
      interfaceCounters.shortPacketsReceived = c.inPktTooShort
      interfaceCounters.nonIpPacketsReceived = c.inPktNonIp
      interfaceCounters.badChecksumIpPacketsReceived = c.inPktBadIpChecksum
      interfaceCounters.unknownIpPacketsReceived = c.inPktUnknown
      interfaceCounters.badChecksumIgmpPacketsReceived = c.inPktBadIgmpChecksum
      interfaceCounters.badChecksumPimPacketsReceived = c.inPktBadPimChecksum
      if not errors:
         interfaceCounters.igmpV1QueryReceived = c.inPktIgmpV1Query
         interfaceCounters.igmpV2QueryReceived = c.inPktIgmpV2Query
         interfaceCounters.igmpV3QueryReceived = c.inPktIgmpV3Query
         interfaceCounters.igmpV1ReportReceived = c.inPktIgmpV1Report
         interfaceCounters.igmpV2ReportReceived = c.inPktIgmpV2Report
         interfaceCounters.igmpV3ReportReceived = c.inPktIgmpV3Report
         interfaceCounters.igmpV2LeaveReceived = c.inPktIgmpV2Leave
         interfaceCounters.dvmrpPacketsReceived = c.inPktDvmrp
         interfaceCounters.mrdPacketsReceived = c.inPktMrd
         interfaceCounters.otherIgmpPacketsReceived = c.inPktIgmpOther
         interfaceCounters.pimPacketsReceived = c.inPktPim
         interfaceCounters.igmpQueriesSent = c.outPktGmpQuery
         interfaceCounters.igmpReportSent = c.outPktGmpReport
         interfaceCounters.igmpLeaveSent = c.outPktIgmpLeave
         interfaceCounters.otherPacketsSent = c.outPktOther
      interfaces[ intf ] = interfaceCounters
   return IgmpSnoopingCounters( interfaces=interfaces,
         _errorSpecific=errors )

#--------------------------------------------------------------------------------
# show ip igmp snooping counters [ errors ] [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpSnoopingCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping counters [ errors ] [ detail ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'counters' : matcherCounters,
      'errors' : 'Error counters',
      'detail' : matcherDetail,
   }

   handler = doShowIpIgmpSnoopingCounter
   prepareFunction = updatePortCounters
   cliModel = IgmpSnoopingCounters

BasicCli.addShowCommandClass( IpIgmpSnoopingCountersCmd )

#####################################################################################
# show igmp snooping querier [ vlan <vlanid> ]
#
# Legacy:
# show ip igmp snooping querier [ vlan <vlanid> ]
#####################################################################################

def printQuerierTitle( ):
   print "%-5s %-16s %-8s %s" % (
      'Vlan', 'IP Address', 'Version', 'Port')
   print "-" * 40

def getReportFloodingSwitchPorts( ):
   return Arnet.sortIntf( igmpSnoopingConfig.switchPort )

#####################################################################################
# show ip igmp snooping [ vlan <vlanid> ] report-flooding
#####################################################################################
def doShowIpIgmpSnoopingReportFlooding( mode, args ):
   # pylint: disable-msg=W0612
   vlanId = args.get( 'VLAN_ID' )
   detail = 'detail' in args
   ( config, vlanConfig, contextStatuses, statuses ) = getSysdbNodes()

   reportFlooding = IgmpSnoopingReportFlooding()
   if vlanId is not None:
      if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
         mode.addWarning( "Vlan %s is not active." % vlanId )
         return reportFlooding

   reportFlooding.reportFloodingEnabled = config.globalReportFloodingEnabled
   reportFlooding.reportFloodingSwitchPorts = getReportFloodingSwitchPorts()

   def getConfigString( inputString ):
      if inputString == 'vlanReportFloodingDefault':
         return 'default'
      elif inputString == 'vlanReportFloodingEnabled':
         return 'enabled'
      elif inputString == 'vlanReportFloodingDisabled':
         return 'disabled'
      elif inputString == 'True':
         return 'enabled'
      else:
         return 'disabled'

   def doShowVlanIpIgmpSnoopingReportFlooding(
               vlan, config, vlanConfig, vlanStatus, forceShow ):

      # should not happen, but check anyways
      if config is None or vlanConfig is None:
         return
      if vlanConfig.reportFloodingConfig == 'vlanReportFloodingDefault' \
            and not vlanConfig.switchPort.keys() and \
            config.globalReportFloodingEnabled is False and vlan != 1 \
            and forceShow is False:
         return
      if vlanConfig.reportFloodingConfig == 'vlanReportFloodingDisabled' \
            and forceShow is False:
         return

      vlanReportFlooding = IgmpSnoopingVlanReportFlooding()
      vlanReportFlooding.reportFloodingOperationalState = getConfigString(
                                       vlanConfig.reportFloodingConfig )
      if vlanStatus:
         vlanReportFlooding.reportFloodingEnabled = vlanStatus.reportFloodingEnabled

      # go through all configured switch-ports
      ports = set( chain( vlanConfig.switchPort, config.switchPort ) )
      for vcPort in Arnet.sortIntf( ports ):
         # verify against the switch-ports in status
         if vlanStatus and vlanStatus.switchPort.get(vcPort):
            vlanReportFlooding.switchPorts[ vcPort ] = 'OK'
         else:
            vlanReportFlooding.switchPorts[ vcPort ] = 'Not OK'

      # go through all switch-ports in status for sanity
      if vlanStatus:
         for port in Arnet.sortIntf( vlanStatus.switchPort ):
            if port not in ports:
               vlanReportFlooding.sanityCheckFailedPorts.append( port )

      reportFlooding.vlans[ vlan ] = vlanReportFlooding

   if vlanId != None:
      vc = config.vlanConfig.get( vlanId )
      contextStatus = contextStatuses.get( vlanId )
      doShowVlanIpIgmpSnoopingReportFlooding( vlanId, config, vc,
                                                contextStatus, True )
   else:
      vlanLists = vlanConfig.keys()
      vlanLists.sort()
      for vlan in vlanLists:
         vc = config.vlanConfig.get( vlan )
         contextStatus = contextStatuses.get( vlan )
         doShowVlanIpIgmpSnoopingReportFlooding( vlan, config, vc,
                                                   contextStatus, False )
   return reportFlooding

#--------------------------------------------------------------------------------
# show ip igmp snooping report-flooding [ vlan VLAN_ID ] [ detail ]
#--------------------------------------------------------------------------------
class ShowIpIgmpSnoopingReportFloodingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping report-flooding [ vlan VLAN_ID ] [ detail ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'report-flooding' : reportFloodingMatcher,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'detail' : matcherDetail,
   }

   handler = doShowIpIgmpSnoopingReportFlooding
   cliModel = IgmpSnoopingReportFlooding

BasicCli.addShowCommandClass( ShowIpIgmpSnoopingReportFloodingCmd )

#####################################################################################
# show igmp snooping querier [ vlan <vlanid> ]
#
# Legacy:
# show ip igmp snooping querier [ vlan <vlanid> ]
#####################################################################################

def doShowIpIgmpSnoopingQuerier( mode, args ):
   # pylint: disable-msg=W0612
   vlanId = args.get( 'VLAN_ID' )
   detail = 'detail' in args
   (config, vlanConfig, contextStatuses, statuses) = getSysdbNodes()
   igmpSnoopingQuerier = IgmpSnoopingQuerier()
   vlans = {}
   if vlanId != None:
      contextStatus = contextStatuses.get( vlanId )
      if contextStatus:
         vlanInfo = showVlanQuerierHelper( contextStatus )
         if vlanInfo:
            vlans[ vlanId ] = vlanInfo
      return IgmpSnoopingQuerier( vlans = vlans, _vlanSpecific = True )
   else:
      vlanIds = contextStatuses.keys()
      vlanIds.sort( key=int )
      for vlanId in vlanIds:
         contextStatus = contextStatuses.get( vlanId )
         if contextStatus:
            vlanInfo = showVlanQuerierHelper( contextStatus )
            if vlanInfo:
               vlans[ vlanId ] = vlanInfo
      return IgmpSnoopingQuerier( vlans = vlans )

def showVlanQuerierHelper( status ):
   #pylint: disable-msg=E1101
   helper = IgmpSnoopingCliLib.IgmpSnoopingCliHelper()
   vlanQuerierInfo = IgmpSnoopingVlanQuerier()

   def getIntfForDisplay( vlanId, addr, intf ):
      intfForDisplay = intf
      peerVlanStatus = peerSyncStatus.vlanStatus.get( vlanId )
      if peerVlanStatus and peerVlanStatus.querierStatus:
         if IpGenAddr( peerVlanStatus.querierStatus.addr ) == addr:
            localIntf = helper.getLocalIntf( peerVlanStatus.querierStatus.intfId )
            if localIntf and helper.getLinkStatus( localIntf ) == 'linkUp':
               intfForDisplay = localIntf
      return intfForDisplay

   querier = status.querier
   if querier is None:
      # Watching for race conditions with entity log updates.
      return None
   querierIntf = querier.intf
   if querierIntf is None:
      # Watching for race conditions with entity log updates.
      return None
   elif helper.isSupported( "Mlag" ) and helper.isPeerIntf( querierIntf ):
      querierIntf = getIntfForDisplay( status.vlanId, querier.addr, querierIntf )

   vlanQuerierInfo.querierAddress = str( querier.addr )
   vlanQuerierInfo.igmpVersion = \
               igmpVersionTable[ status.querierVersion ]
   vlanQuerierInfo.querierInterface = querierIntf
   vlanQuerierInfo.queryResponseInterval = status.queryResponseInterval
   return vlanQuerierInfo

#--------------------------------------------------------------------------------
# show igmp snooping querier [ vlan VLAN_ID ] [ detail ]
#--------------------------------------------------------------------------------
class IgmpSnoopingQuerierCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show igmp snooping querier [ vlan VLAN_ID ] [ detail ]'
   data = {
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : matcherQuerier,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'detail' : matcherDetail,
   }

   handler = doShowIpIgmpSnoopingQuerier
   cliModel = IgmpSnoopingQuerier

BasicCli.addShowCommandClass( IgmpSnoopingQuerierCmd )

#--------------------------------------------------------------------------------
# show ip igmp snooping querier [ vlan VLAN_ID ] [ detail ]
#--------------------------------------------------------------------------------
class ShowIpIgmpSnoopingQuerierCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping querier [ vlan VLAN_ID ] [ detail ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : CliCommand.Node( matcher=matcherQuerier,
      deprecatedByCmd='show igmp snooping querier ...' ),
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'detail' : matcherDetail,
   }

   handler = doShowIpIgmpSnoopingQuerier
   cliModel = IgmpSnoopingQuerier

BasicCli.addShowCommandClass( ShowIpIgmpSnoopingQuerierCmd )

def showQuerierStatusVlan( vlanId, config, gqc, vqc, qs ):
   assert gqc

   vlanInfo = IgmpQuerierStatusVlan()
   opState = 'non-querier'
   if qs:
      # other states are querierStateInitial, querierStateNonQuerier
      opState = 'querier' if qs.gmpQuerierStatus.state == 'querierStateQuerier' \
                          else 'non-querier'
   if vqc:
      if vqc.enabledConfigured == 'vlanQuerierDefault':
         enabled = gqc.enabled
      elif vqc.enabledConfigured == 'vlanQuerierEnabled':
         enabled = True
      else:
         enabled = False
      if vqc.ipAddrConfigured != '0.0.0.0':
         ipAddr = vqc.ipAddrConfigured
      else:
         ipAddr = gqc.ipAddr
      if vqc.queryIntervalConfigured:
         qi = vqc.queryIntervalConfigured
      else:
         qi = gqc.queryInterval
      if vqc.queryResponseIntervalConfigured:
         qri = vqc.queryResponseIntervalConfigured
      else:
         qri = gqc.queryResponseInterval
      if vqc.querierVersionConfigured != 'igmpVersionUnknown':
         querierVersion = vqc.querierVersionConfigured
      else:
         querierVersion = gqc.querierVersion
      if vqc.lastMemberQueryIntervalConfigured:
         vlanInfo.lastMemberQueryInterval = vqc.lastMemberQueryIntervalConfigured
      elif gqc.lastMemberQueryInterval:
         vlanInfo.lastMemberQueryInterval = gqc.lastMemberQueryInterval
      if vqc.lastMemberQueryCountConfigured:
         vlanInfo.lastMemberQueryCount = vqc.lastMemberQueryCountConfigured
      elif gqc.lastMemberQueryCount:
         vlanInfo.lastMemberQueryCount = gqc.lastMemberQueryCount
      if vqc.startupQueryIntervalConfigured:
         vlanInfo.startupQueryInterval = vqc.startupQueryIntervalConfigured
      elif gqc.startupQueryInterval:
         vlanInfo.startupQueryInterval = gqc.startupQueryInterval
      if vqc.startupQueryCountConfigured:
         vlanInfo.startupQueryCount = vqc.startupQueryCountConfigured
      elif gqc.startupQueryCount:
         vlanInfo.startupQueryCount = gqc.startupQueryCount
   else:
      enabled = gqc.enabled
      ipAddr = gqc.ipAddr
      qi = gqc.queryInterval
      qri = gqc.queryResponseInterval
      querierVersion = gqc.querierVersion

   vlanInfo.adminState = ( 'enabled' if enabled else 'disabled' )
   vlanInfo.querierAddress = ipAddr
   vlanInfo.queryInterval = qi
   vlanInfo.queryResponseInterval = qri
   vlanInfo.querierExpiryTime = ( config.robustness * qi + qri/2.0 )
   vlanInfo.operationalState = opState
   vlanInfo.version = igmpVersionTable[ querierVersion ]
   return vlanInfo

def showIpIgmpQuerierStatus( mode, args ):
   # we show all active vlans for querier status if vlan is not provided.
   vlanId = args.get( 'VLAN_ID' )

   igmpQuerierStatusInfo = IgmpQuerierStatus()
   if vlanId != None:
      if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
         mode.addWarning( "Vlan %s is not active." % vlanId )
         return igmpQuerierStatusInfo
      vlanIds = [ vlanId ]
   else:
      vlanIds = IgmpSnoopingLib.activeVlans( bridgingInputCli )

   if not igmpSnoopingConfig.globalQuerierConfig:
      mode.addWarning( "Global querier config found to be None" )
      return igmpQuerierStatusInfo

   gqc = igmpSnoopingConfig.globalQuerierConfig

   igmpQuerierStatusInfo.adminState = ( 'enabled' if gqc.enabled else 'disabled' )
   igmpQuerierStatusInfo.querierAddress = str( gqc.ipAddr )
   igmpQuerierStatusInfo.queryInterval = gqc.queryInterval
   igmpQuerierStatusInfo.queryResponseInterval = gqc.queryResponseInterval
   igmpQuerierStatusInfo.robustness = igmpSnoopingConfig.robustness
   igmpQuerierStatusInfo.querierExpiryTime = ( igmpSnoopingConfig.robustness *
         gqc.queryInterval + gqc.queryResponseInterval/2.0 )
   igmpQuerierStatusInfo.lastMemberQueryInterval = gqc.lastMemberQueryInterval
   if gqc.lastMemberQueryCount:
      igmpQuerierStatusInfo.lastMemberQueryCount = gqc.lastMemberQueryCount
   if gqc.startupQueryInterval:
      igmpQuerierStatusInfo.startupQueryInterval = gqc.startupQueryInterval
   if gqc.startupQueryCount:
      igmpQuerierStatusInfo.startupQueryCount = gqc.startupQueryCount

   vlanInfo = {}
   for vlanId in vlanIds:
      vlanConfig = igmpSnoopingConfig.vlanConfig.get( vlanId )
      querierName = str( vlanId )
      status = showQuerierStatusVlan( vlanId, igmpSnoopingConfig,
                             igmpSnoopingConfig.globalQuerierConfig,
                             vlanConfig.vlanQuerierConfig if vlanConfig else None,
                             querierStatusDir.querierStatus.get( querierName ) )
      if status:
         vlanInfo[ vlanId ] = status
   igmpQuerierStatusInfo.vlans = vlanInfo
   return igmpQuerierStatusInfo

#--------------------------------------------------------------------------------
# show igmp snooping querier status [ vlan VLAN_ID ] [ detail ]
#--------------------------------------------------------------------------------
class IgmpSnoopingQuerierStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show igmp snooping querier status [ vlan VLAN_ID ] [ detail ]'
   data = {
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : matcherQuerier,
      'status' : matcherStatus,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'detail' : matcherDetail,
   }

   handler = showIpIgmpQuerierStatus
   cliModel = IgmpQuerierStatus

BasicCli.addShowCommandClass( IgmpSnoopingQuerierStatusCmd )

#--------------------------------------------------------------------------------
# show ip igmp snooping querier status [ vlan VLAN_ID ] [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpSnoopingQuerierStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping querier status [ vlan VLAN_ID ] [ detail ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : CliCommand.Node( matcher=matcherQuerier,
         deprecatedByCmd='show igmp snooping querier ...' ),
      'status' : matcherStatus,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'detail' : matcherDetail,
   }

   handler = showIpIgmpQuerierStatus
   cliModel = IgmpQuerierStatus

BasicCli.addShowCommandClass( IpIgmpSnoopingQuerierStatusCmd )

# -----------------------------------------------------------------------------------
# show igmp snooping querier counters [ vlan <vlanid> ]
#
# Legacy:
# show ip igmp snooping querier counters [ vlan <vlanid> ]
# -----------------------------------------------------------------------------------

def showQuerierCounterVlan( vlanId, config, gqc, vqc, qc, qs ):
   assert gqc

   igmpVlanQuerierCounters = IgmpSnoopingVlanQuerierCounters()

   if not qc or not qs:
      return None

   gqc = qc.gmpQuerierConfig
   gqs = qs.gmpQuerierStatus
   if not gqc or not gqs:
      return None

   # other states are querierStateInitial, querierStateNonQuerier
   igmpVlanQuerierCounters.operationalState = ( 'querier'
           if gqs.state == 'querierStateQuerier' else 'non-querier' )
   igmpVlanQuerierCounters.querierAddress =  str( qc.ipAddr )
   igmpVlanQuerierCounters.igmpVersion = \
           igmpVersionTable[ gqc.querierVersion ]
   igmpVlanQuerierCounters.v1QueriesSent = gqs.v1QueriesSent
   igmpVlanQuerierCounters.v1QueriesReceived = gqs.v1QueriesReceived
   igmpVlanQuerierCounters.v1ReportReceived = gqs.v1ReportReceived
   igmpVlanQuerierCounters.v2QueriesSent = gqs.v2QueriesSent
   igmpVlanQuerierCounters.v2QueriesReceived = gqs.v2QueriesReceived
   igmpVlanQuerierCounters.v2ReportReceived = gqs.v2ReportReceived
   igmpVlanQuerierCounters.v2LeaveReceived = gqs.v2LeaveReceived
   igmpVlanQuerierCounters.v3QueriesSent = gqs.v3QueriesSent
   igmpVlanQuerierCounters.v3GSQueriesSent = gqs.v3GSQueriesSent
   igmpVlanQuerierCounters.v3GSSQueriesSent = gqs.v3GSSQueriesSent
   igmpVlanQuerierCounters.v3QueriesReceived = gqs.v3QueriesReceived
   igmpVlanQuerierCounters.v3ReportReceived = gqs.v3ReportReceived
   igmpVlanQuerierCounters.totalGeneralQueriesSent = gqs.totalGeneralQueriesSent
   igmpVlanQuerierCounters.errorPacketReceived = gqs.errorPacketReceived
   igmpVlanQuerierCounters.otherPacketReceived = gqs.otherPacketReceived
   return igmpVlanQuerierCounters

def showIpIgmpQuerierCounter( mode, args ):
   vlanId = args.get( 'VLAN_ID' )
   # we show all active vlans for querier status if vlan is not provided.

   igmpQuerierCounters = IgmpSnoopingQuerierCounters()
   if vlanId != None:
      if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
         mode.addWarning( "Vlan %s is not active." % vlanId )
         return igmpQuerierCounters
      vlanIds = [ vlanId ]
   else:
      vlanIds = IgmpSnoopingLib.activeVlans( bridgingInputCli )

   if not vlanIds:
      return igmpQuerierCounters

   for vlanId in vlanIds:
      vlanConfig = igmpSnoopingConfig.vlanConfig.get( vlanId )
      querierName = str( vlanId )
      vlanInfo = showQuerierCounterVlan( vlanId, igmpSnoopingConfig,
                             igmpSnoopingConfig.globalQuerierConfig,
                             vlanConfig.vlanQuerierConfig if vlanConfig else None,
                             querierConfigDir.querierConfig.get( querierName ),
                             querierStatusDir.querierStatus.get( querierName ) )
      if vlanInfo:
         igmpQuerierCounters.vlans[ vlanId ] = vlanInfo

   return igmpQuerierCounters

#--------------------------------------------------------------------------------
# show igmp snooping querier counters [ vlan VLAN_ID ]
#--------------------------------------------------------------------------------
class IgmpSnoopingQuerierCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show igmp snooping querier counters [ vlan VLAN_ID ]'
   data = {
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : matcherQuerier,
      'counters' : matcherCounters,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
   }

   handler = showIpIgmpQuerierCounter
   cliModel = IgmpSnoopingQuerierCounters

BasicCli.addShowCommandClass( IgmpSnoopingQuerierCountersCmd )

#--------------------------------------------------------------------------------
# show ip igmp snooping querier counters [ vlan VLAN_ID ]
#--------------------------------------------------------------------------------
class IpIgmpSnoopingQuerierCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping querier counters [ vlan VLAN_ID ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : CliCommand.Node( matcher=matcherQuerier,
         deprecatedByCmd='show igmp snooping querier ...' ),
      'counters' : matcherCounters,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
   }

   handler = showIpIgmpQuerierCounter
   cliModel = IgmpSnoopingQuerierCounters

BasicCli.addShowCommandClass( IpIgmpSnoopingQuerierCountersCmd )

# -----------------------------------------------------------------------------------
# show igmp snooping querier membership [ vlan <vlanid> ]
# 
# Legacy:
# show ip igmp snooping querier membership [ vlan <vlanid> ]
# -----------------------------------------------------------------------------------

def showQuerierMembershipVlan( vlanId, config, gqc, vqc, qc, qs ):
   assert gqc

   if not qc or not qs:
      return None

   gqs = qs.gmpQuerierStatus
   gmpqc = qc.gmpQuerierConfig
   if not gqs:
      return None

   allGroups = qs.gmpQuerierStatus.group.keys()
   if allGroups == []:
      return None

   querierStatus = IgmpQuerierStatusVlan()
   igmpVlanQuerierMembership = IgmpSnoopingVlanQuerierMembership()
   if qs:
      querierStatus.operationalState = (
            'querier' if gqs.state == 'querierStateQuerier'
            else 'non-querier' )
   else:
      querierStatus.operationalState = 'non-querier'
   if vqc:
      if vqc.enabledConfigured == 'vlanQuerierDefault':
         enabled = gqc.enabled
      elif vqc.enabledConfigured == 'vlanQuerierEnabled':
         enabled = True
      else:
         enabled = False
   else:
      enabled = gqc.enabled
   if gqs.querierVersion != 'igmpVersionUnknown':
      querierStatus.version = \
            igmpVersionTable[ gqs.querierVersion ]
   else:
      querierStatus.version = igmpVersionTable[ gqc.querierVersion ]
   if gmpqc.lastMemberQueryInterval:
      querierStatus.lastMemberQueryInterval = gmpqc.lastMemberQueryInterval
   elif gqc.lastMemberQueryInterval:
      querierStatus.lastMemberQueryInterval = gqc.lastMemberQueryInterval
   if gmpqc.lastMemberQueryCount:
      querierStatus.lastMemberQueryCount = gmpqc.lastMemberQueryCount
   elif gqc.lastMemberQueryCount:
      querierStatus.lastMemberQueryCount = gqc.lastMemberQueryCount
   if gmpqc.startupQueryInterval:
      querierStatus.startupQueryInterval = gmpqc.startupQueryInterval
   elif gqc.startupQueryInterval:
      querierStatus.startupQueryInterval = gqc.startupQueryInterval
   if gmpqc.startupQueryCount:
      querierStatus.startupQueryCount = gmpqc.startupQueryCount
   elif gqc.startupQueryCount:
      querierStatus.startupQueryCount = gqc.startupQueryCount
   if gmpqc.lastMemberQueryTime:
      igmpVlanQuerierMembership.lastMemberQueryTime = gmpqc.lastMemberQueryTime
   querierStatus.adminState = ( 'enabled' if enabled else 'disabled' )
   querierStatus.querierAddress = str( gqs.electedQuerierAddr )
   querierStatus.queryInterval = gqs.queryInterval
   igmpVlanQuerierMembership.robustness = gqs.robustness
   querierStatus.queryResponseInterval = gqs.queryResponseInterval
   querierStatus.querierExpiryTime = (
         config.robustness * gqs.queryInterval + gqs.queryResponseInterval / 2.0 )
   if gqs.otherQuerierPresentInterval:
      igmpVlanQuerierMembership.otherQuerierPresentInterval = \
            gqs.otherQuerierPresentInterval
   if gqs.olderHostPresentInterval:
      igmpVlanQuerierMembership.olderHostPresentInterval = \
         gqs.olderHostPresentInterval
   if gqs.generalQueryExpiryTime:
      igmpVlanQuerierMembership.generalQueryExpiryTime = gqs.generalQueryExpiryTime
   if gqs.otherQuerierExpiryTime:
      igmpVlanQuerierMembership.otherQuerierExpiryTime = gqs.otherQuerierExpiryTime
   igmpVlanQuerierMembership.addRandomDelayToTimer = gmpqc.addRandomDelayToTimer
   igmpVlanQuerierMembership.querierStatus = querierStatus
   if gqs.groupMembershipInterval:
      igmpVlanQuerierMembership.groupMembershipInterval = \
            gqs.groupMembershipInterval
   # Get all the groups

   allGroups.sort( key=lambda x: x )
   for group in allGroups:
      # Try to get the group again, use get function in case it's gone
      memberGroup = qs.gmpQuerierStatus.group.get( group )
      if memberGroup is None:
         continue
      groupInfo = MulticastGroupInfo()
      groupInfo.groupAddress = str( group )
      for sourceAddr, info in memberGroup.source.items():
         source = GroupSourceInfo()
         source.sourceAddress = str( sourceAddr )
         source.creationTime = ( info.creationTime + Tac.utcNow() - Tac.now()
               if info.creationTime else None )
         source.expiryTime = ( info.expiryTime + Tac.utcNow() - Tac.now()
               if info.expiryTime else None )
         source.expired = info.expired
         groupInfo.sources.append( source )
      groupInfo.filterMode =  filterModePrintingTable[ memberGroup.filterMode ]
      groupInfo.igmpVersion = \
            igmpVersionTable[ memberGroup.compatibilityMode ]
      groupInfo.creationTime = ( memberGroup.creationTime + Tac.utcNow() - Tac.now()
            if memberGroup.creationTime else None )
      groupInfo.expiryTime = ( memberGroup.expiryTime + Tac.utcNow() - Tac.now()
            if memberGroup.expiryTime else None )
      groupInfo.lastReporter = memberGroup.lastReporter
      igmpVlanQuerierMembership.groups.append( groupInfo )
   return igmpVlanQuerierMembership

def showIpIgmpQuerierMembership( mode, args ):
   # we show all active vlans for querier membership if vlan is not provided.
   vlanId = args.get( 'VLAN_ID' )
   igmpQuerierMembership = IgmpSnoopingQuerierMembership()
   if vlanId != None:
      if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
         mode.addWarning( "Vlan %s is not active." % vlanId )
         return igmpQuerierMembership
      vlanIds = [ vlanId ]
   else:
      vlanIds = IgmpSnoopingLib.activeVlans( bridgingInputCli )

   if not vlanIds:
      return igmpQuerierMembership

   for vlanId in vlanIds:
      vlanConfig = igmpSnoopingConfig.vlanConfig.get( vlanId )
      querierName = str( vlanId )
      vlanInfo = showQuerierMembershipVlan( vlanId, igmpSnoopingConfig,
                             igmpSnoopingConfig.globalQuerierConfig,
                             vlanConfig.vlanQuerierConfig if vlanConfig else None,
                             querierConfigDir.querierConfig.get( querierName ),
                             querierStatusDir.querierStatus.get( querierName ) )
      if vlanInfo:
         igmpQuerierMembership.vlans[ vlanId ] = vlanInfo
   return igmpQuerierMembership

#--------------------------------------------------------------------------------
# show igmp snooping querier membership [ vlan VLAN_ID ]
#--------------------------------------------------------------------------------
class IgmpSnoopingQuerierMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show igmp snooping querier membership [ vlan VLAN_ID ]'
   data = {
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : matcherQuerier,
      'membership' : matcherMembership,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
   }

   handler = showIpIgmpQuerierMembership
   cliModel = IgmpSnoopingQuerierMembership

BasicCli.addShowCommandClass( IgmpSnoopingQuerierMembershipCmd )

#--------------------------------------------------------------------------------
# show ip igmp snooping querier membership [ vlan VLAN_ID ]
#--------------------------------------------------------------------------------
class IpIgmpSnoopingQuerierMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping querier membership [ vlan VLAN_ID ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : CliCommand.Node( matcher=matcherQuerier,
         deprecatedByCmd='show igmp snooping querier ...' ),
      'membership' : matcherMembership,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
   }

   handler = showIpIgmpQuerierMembership
   cliModel = IgmpSnoopingQuerierMembership

BasicCli.addShowCommandClass( IpIgmpSnoopingQuerierMembershipCmd )

# -----------------------------------------------------------------------------------
# show igmp snooping querier membership vlan <vlanid> group <ip-addr>
#
# Legacy:
# show ip igmp snooping querier membership vlan <vlanid> group <ip-addr>
# -----------------------------------------------------------------------------------
def showIpIgmpQuerierMembershipGroup( mode, args ):
   vlanId = args[ 'VLAN_ID' ]
   groupIpAddr = args[ 'IPADDR' ]
   igmpQuerierMembershipGroup = IgmpQuerierMembershipGroup()
   if vlanId is None or groupIpAddr is None:
      return igmpQuerierMembershipGroup

   if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
      mode.addWarning( "Vlan %s is not active." % vlanId )
      return igmpQuerierMembershipGroup

   # Get the querierConfig and querierStatus
   querierName = str( vlanId )
   querierConfig = querierConfigDir.querierConfig.get( querierName )
   querierStatus = querierStatusDir.querierStatus.get( querierName )

   if querierConfig is None or querierStatus is None:
      return igmpQuerierMembershipGroup

   # Get the group
   groupStatus = querierStatus.gmpQuerierStatus.group.get( IpGenAddr( groupIpAddr ) )
   if groupStatus is None:
      return igmpQuerierMembershipGroup

   allSources = groupStatus.source.keys()
   allSources.sort( key=lambda x: x )

   for sourceIp in allSources:
      sourceInfo = GroupSourceInfo()
      sourceStatus = groupStatus.source.get( sourceIp )
      if sourceStatus is None:
         continue
      sourceInfo.sourceAddress = str( sourceIp )
      sourceInfo.expired = sourceStatus.expired
      sourceInfo.creationTime = ( sourceStatus.creationTime + Tac.utcNow()
            - Tac.now() if sourceStatus.creationTime else None )
      sourceInfo.expiryTime = ( sourceStatus.expiryTime + Tac.utcNow()
            - Tac.now() if sourceStatus.expiryTime else None )
      igmpQuerierMembershipGroup.sources.append( sourceInfo )
   return igmpQuerierMembershipGroup

#--------------------------------------------------------------------------------
# show igmp snooping querier membership vlan VLAN_ID group IPADDR
#--------------------------------------------------------------------------------
class ShowIgmpSnoopingQuerierMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show igmp snooping querier membership vlan VLAN_ID group IPADDR'
   data = {
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : matcherQuerier,
      'membership' : matcherMembership,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'group' : matcherGroup,
      'IPADDR' : IpAddrMatcher.ipAddrMatcher,
   }

   handler = showIpIgmpQuerierMembershipGroup
   cliModel = IgmpQuerierMembershipGroup

BasicCli.addShowCommandClass( ShowIgmpSnoopingQuerierMembershipCmd )

#--------------------------------------------------------------------------------
# show ip igmp snooping querier membership vlan VLAN_ID group IPADDR
#--------------------------------------------------------------------------------
class ShowIgmpSnoopingQuerierMembershipDepCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp snooping querier membership vlan VLAN_ID group IPADDR'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : CliCommand.Node( matcher=matcherQuerier,
         deprecatedByCmd='show igmp snooping querier ...' ),
      'membership' : matcherMembership,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'group' : matcherGroup,
      'IPADDR' : IpAddrMatcher.ipAddrMatcher,
   }

   handler = showIpIgmpQuerierMembershipGroup
   cliModel = IgmpQuerierMembershipGroup

BasicCli.addShowCommandClass( ShowIgmpSnoopingQuerierMembershipDepCmd )

def doShowGmpSnoopingGroupsVlansHelper (
   mode, vlanConfig, contextStatuses, statuses, agent, warningSentence,
   vlanId = None, intfs = None, showWhat=None, detail=None ):
   ''' this function prints group information as per arguments passed.
       We somewhat diverge from industry standards here. We show only
       those interfaces which are consistent with the topology map both
       for static and dynamic interfaces.
   '''
   whatMAddress = None
   if showWhat != 'count' and showWhat != 'dynamic' and showWhat != 'user' and \
          showWhat != None:
      # its multicast address based filter.
      whatMAddress = showWhat
      invalidMulticast = IpAddrMatcher.validateMulticastIpAddr( whatMAddress )
      if invalidMulticast:
         mode.addError( "Group filter must be a valid multicast address" )
         return
      showWhat = 'address'

   numMGroups = 0
   titlePrinted = [ ] # The list is used as a bool flag.
   if vlanId is None:
      # if snooping is active, then it definitely will show up in
      # the contextstatus. In other case, we don't bother about any
      # config.
      vlanIds = contextStatuses.keys()
      # if contextStatus filters info based on
      vlanIds.sort( key=int )
   else:
      vlanIds = [ vlanId ]

   intfNames = []
   if intfs:
      intfNames += list( intfs.intfNames() )

   printWarning = False
   warning = ''.join( ['-'*80, '\n', warningSentence] )
   for vlanId in vlanIds:
      configVlan = vlanConfig.get( vlanId )
      contextStatusVlan = contextStatuses.get( vlanId )
      statusVlan = statuses.get( vlanId )
      if not contextStatusVlan:
         continue
      numGroups = showGroupHelper( vlanId, configVlan, contextStatusVlan,
                                   statusVlan, 'count', whatMAddress,
                                   detail, titlePrinted, intfNames, agent )
      if not contextStatusVlan.confident and numGroups:
         printWarning = True
      if showWhat == 'count':
         numMGroups += numGroups
         # pylint: disable-msg=W0105
         '''if intfs:
            # FIXME XXX need to do agentCommandRequest
            # numMGroups += groupsOnIntfInVlan( intfNames, vlanId )
            numMGroups = 0
         else:
            numMGroups += len( contextStatusVlan.groupList )'''

      else:
         showGroupHelper( vlanId, configVlan, contextStatusVlan, statusVlan,
                          showWhat, whatMAddress, detail, titlePrinted, intfNames,
                          agent )
   if titlePrinted and printWarning:
      print warning
   if showWhat == 'count':
      print 'Total number of multicast groups: %d' % numMGroups,
      if printWarning:
         print ' *'
         print warningSentence
      else:
         print

# -----------------------------------------------------------------------------------
# show ip igmp snooping groups [ vlan <vlanid> ] [ interface <intfs> ] \
#      [ count | dynamic | user | A.B.C.D ]
# -----------------------------------------------------------------------------------

#Vlan      Group                    Type        Version     Port List
#-----------------------------------------------------------------------
#1         224.12.22.33             user                    Gi0/24, Po1,
#                                                           Po2, Po3
#1         225.0.0.1                user                    Te0/2, Po1,
#                                                           Po2, Po3,
#                                                           Po6
#2         224.12.22.33             user                    Gi0/24, Te0/1
def doShowIpIgmpSnoopingGroupsVlans( mode, args ):
   # pylint: disable-msg=W0612
   vlanId = args.get( 'VLAN_ID' )
   intfs = args.get( 'INTF' )
   if 'IPADDR' in args:
      showWhat = args[ 'IPADDR' ]
   else:
      showWhat = args.get( 'OPTION' )
   detail = 'detail' in args
   ( _, vlanConfig, contextStatuses, statuses ) = getSysdbNodes()
   warningSentence = \
      ("* Warning: IGMP Snooping Pruning is not active. Flooding traffic"
       " to vlan.\nPlease check for IGMP Querier.")
   doShowGmpSnoopingGroupsVlansHelper( mode, vlanConfig, contextStatuses, statuses,
                                       agentName(), warningSentence, vlanId, intfs,
                                       showWhat, detail )

def getGroups( configVlan, contextStatusVlan, statusVlan, showWhat=None ):
   if configVlan:
      sGAs = configVlan.ipGroup.keys()
   else:
      sGAs = []
   if statusVlan:
      # is status vlan exists, then it keeps both static and user entries.
      allGAs = statusVlan.group.keys()
   else:
      # this means that vlan or snopping is not enabled.
      allGAs = sGAs[ : ]
   return ( sGAs, allGAs )

# XXX: currently we print the lowest version field from the interface type.
# does this version actually make sense?
def showGroupHelper (
   vlanId, configVlan, contextStatusVlan, statusVlan, showWhat,
   whatMAddress, detail, titlePrinted, intfNamesFilter, agent ):
   ''' This function takes a vlan, and prints the group entries for that vlan.
       FIXME XXX Update comment
   '''
   count = showWhat == 'count'
   assert contextStatusVlan
   # pylint: disable-msg=W0612
   if showWhat == 'user' or showWhat == 'dynamic':
      intfType = showWhat
   else:
      intfType = None

   if showWhat == 'address':
      allAddrs = [ whatMAddress ]
   else:
      allAddrs = list( set( list( contextStatusVlan.groupList ) +
         list( contextStatusVlan.peerGroupList ) ) )

      # FIXME XXX need to sort allAddrs.sort( IpUtils.compareIpAddress )

   def printTitleIfNeeded():
      if titlePrinted:
         return
      if count:
         return
      if not detail:
         print ( '%-5s %-16s %-8s %-19s %s' % (
                 'Vlan', 'Group', 'Type', 'Version', 'Port-List' ) )
         print "-" * 80
      else:
         print "%-4s %-15s %-15s %-11s %-11s %-8s %-3s %-6s %s" % (
            'Vlan', 'Group', 'IP', 'First', 'Last', 'Expire',
               'Ver', 'Filter', 'Port' )
         print "%37s%-12s%-11s%14s%-10s" % ("", "Heard", "Heard", "", "Mode")
         print "-" * 85
      titlePrinted.append( True )

   numMGroups = 0

   # No filter is applied, so if all we want is a group address, just ad the number
   # of all addresses here.
   if count and not intfNamesFilter:
      numMGroups += len( allAddrs )

   # Otherwise, we are either counting with a filter or actually printing group
   # information. We will need to AgentCommandRequest into IgmpSnooping to get it.
   else:
      nofGroupsToProcess = 64
      groups = ''
      for groupAddrs in map( None, *( iter( allAddrs), ) * nofGroupsToProcess ):
         printTitleIfNeeded()
         groups = \
            ' '.join( ['-g{0}'.format(i) for i in filter( None, groupAddrs ) ] )

         # format of command:
         # showGroups -d=detail -u=user -y=dynamic
         # -v<vlanId> -i<intf full name> -g<group>
         # -c=count
         cmd = "showGroups "
         if detail:
            cmd += '-d '
         cmd += ('-v%d ' % vlanId )
         cmd += groups + ' '
         if intfNamesFilter:
            for intf in intfNamesFilter:
               cmd += '-i%s ' % intf
         if showWhat == 'user':
            cmd += '-u '
         elif showWhat == 'dynamic':
            cmd += '-y '
         if count:
            cmd += '-c '

         # The count command
         if count:
            # We handeled the other case above
            assert showWhat or intfNamesFilter
            # A filter is applied, so we need to use AgentCommandRequest
            # The agent will return 'y' if the group should be printed and 'n'
            # otherwise.
            stringBuff = cStringIO.StringIO()
            runOldAgentCommand( entityManager,
                                agent, "igmpsnoopingdebug", cmd,
                                stringBuff=stringBuff )
            data = stringBuff.getvalue()
            stringBuff.close()
            numMGroups += data.count( 'y' )

         else:
            runOldAgentCommand( entityManager, agent,
                                "igmpsnoopingdebug", cmd )

   if count:
      return numMGroups

#--------------------------------------------------------------------------------
# show ip igmp snooping groups [ vlan VLAN_ID ] [ interface INTF ]
#                              [ OPTION | IPADDR ] [ detail ]
#--------------------------------------------------------------------------------
class IpIgmpSnoopingGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip igmp snooping groups [ vlan VLAN_ID ] [ interface INTF ] '
                                           '[ OPTION | IPADDR ] [ detail ]' )
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'groups' : 'Show IGMP group related information',
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'interface' : interfaceMatcher,
      'INTF' : Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes ),
      'OPTION': CliMatcher.EnumMatcher( {
         'count' : 'Count number of groups',
         'dynamic' : 'Groups learnt via IGMP',
         'user' : 'Groups configured by user',
      } ),
      'IPADDR' : IpAddrMatcher.ipAddrMatcher,
      'detail' : matcherDetail,
   }

   handler = doShowIpIgmpSnoopingGroupsVlans

if not Toggles.IgmpSnoopingToggleLib.toggleGmpSnoopingRefactorEnabled():
   BasicCli.addShowCommandClass( IpIgmpSnoopingGroupsCmd )

# -----------------------------------------------------------------------------------
# Display MembershipJoinStatus from different clients - EVPN, user, local, mlag.
# If no filter is provided, display the merged state.
#
# show ip igmp snooping groups [ local | mlag | mcs | evpn | user ] \
#                              [ vlan <vlanid> ] [ A.B.C.D ] [ detail ]
#
# Example of output:
# IGMP Snooping Group Membership
# EX: Filter mode Exclude
# IN: Filter mode Include
# IR: Ingress Replication
# VLAN  Group            Members
#----  ---------------  ----------------------------------------
#10    225.0.0.1        Et1, Et2, Et3, Et4, PIM-Tunnel
#                       10.1.1.1(IR), 10.1.1.2(IR), 10.1.1.3(IR)
#10    225.0.0.2        Et1, Et2, Et3, Et4, 10.1.1.1(IR),
#                       10.1.1.2(IR), 10.1.1.3(IR)
#10    *                Et3, Et4
matcherGroups = CliMatcher.KeywordMatcher( 'groups',
                                    helpdesc='Show IGMP group related information' )

def showClientSnoopingsGroupsHandler( mode, agent, args, cmdName, model ):
   command = cmdName

   client = args.get( 'CLIENT' )
   if client:
      if client == 'local':
         command += "#client#igmpsnooping"
      elif client == 'user':
         command += "#client#igmpsnoopingStatic"
      else:
         command += "#client#%s" % client

   vlanId = args.get( 'VLANID' )
   if vlanId:
      command += "#vlan#%d" % vlanId

   if cmdName == 'showIpIgmpSnoopingGroups':
      grpAddr = args.get( 'GRPADDR' )
      if grpAddr:
         command += "#group#%s" % grpAddr

      detail = 'detail' in args
      if detail:
         command += "#detail"

   return runAgentCommand( mode, agent, command, model )

clientMatcher = CliMatcher.EnumMatcher( {
   'evpn' : 'Show groups learnt via EVPN',
   'local' : 'Show groups learnt locally via IGMP',
   'mcs' : 'Show groups learnt via Media Control Service',
   'mlag' : 'Show groups learnt via MLAG peer',
   'user' : 'Show groups configured by user' } )

class ShowClientSnoopingGroups( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show ip igmp snooping groups [ CLIENT ] [ vlan VLANID ] [ GRPADDR ]" )
   data = {
         'ip' : CliToken.Ip.ipMatcherForShow,
         'igmp' : CliToken.IgmpSnooping.igmpNode,
         'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
         'groups' : matcherGroups,
         'CLIENT' : clientMatcher,
         'vlan' : CliToken.IgmpSnooping.vlanMatcher,
         'VLANID' : vlanIdMatcher,
         'GRPADDR' : IpAddrMatcher.ipAddrMatcher,
         }
   cliModel = GmpMembershipInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args,
                                               'showIpIgmpSnoopingGroups',
                                               GmpMembershipInfo )

class ShowClientSnoopingGroupsDetail( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show ip igmp snooping groups [ CLIENT ] [ vlan VLANID ] [ GRPADDR ] "
              "detail" )
   data = {
         'ip' : CliToken.Ip.ipMatcherForShow,
         'igmp' : CliToken.IgmpSnooping.igmpNode,
         'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
         'groups' : matcherGroups,
         'CLIENT' : clientMatcher,
         'vlan' : CliToken.IgmpSnooping.vlanMatcher,
         'VLANID' : vlanIdMatcher,
         'GRPADDR' : IpAddrMatcher.ipAddrMatcher,
         'detail' : matcherDetail,
         }
   cliModel = GmpMembershipInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler(
            mode, agentName(), args, 'showIpIgmpSnoopingGroups',
            GmpMembershipDetailInfo )

class ShowClientSnoopingGroupsCount( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show ip igmp snooping groups [ CLIENT ] [ vlan VLANID ] count" )
   data = {
         'ip' : CliToken.Ip.ipMatcherForShow,
         'igmp' : CliToken.IgmpSnooping.igmpNode,
         'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
         'groups' : matcherGroups,
         'CLIENT' : clientMatcher,
         'vlan' : CliToken.IgmpSnooping.vlanMatcher,
         'VLANID' : vlanIdMatcher,
         'count' : CliMatcher.KeywordMatcher( 'count',
                                              helpdesc='Show membership count' ),
         }
   cliModel = GmpMembershipCountInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args, 
                                               'showIpIgmpSnoopingGroupsCount',
                                               GmpMembershipCountInfo )

if Toggles.IgmpSnoopingToggleLib.toggleGmpSnoopingRefactorEnabled():
   BasicCli.addShowCommandClass( ShowClientSnoopingGroups )
   BasicCli.addShowCommandClass( ShowClientSnoopingGroupsDetail )
   BasicCli.addShowCommandClass( ShowClientSnoopingGroupsCount )


# -----------------------------------------------------------------------------------
# show mac address-table multicast [ address <multicast mac address> \
#                             interface <interface-range> vlan <vlanid> ]
# show mac address-table multicast [ vlan <vlanid> ] brief
# -----------------------------------------------------------------------------------

def doShowMacAddressTableMulticast( mode, args ):
   # pylint: disable-msg=W0612
   macAddress = args.get( 'MAC_ADDR' )
   intfs = args.get( 'INTF' )
   vlanId = args.get( 'VLAN_ID' )
   brief = 'brief' in args
   ( config, vlanConfig, contextStatuses, statuses ) = getSysdbNodes()

   intfNames = []
   if intfs:
      intfNames += list( intfs.intfNames() )

   if macAddress and not Ethernet.isMulticast( macAddress ):
      mode.addError( "Address %s not a valid usable multicast mac address."
                     % macAddress )
      return None

   vlanIdForMulticast = vlanId
   vlanIds = []
   if vlanId is None:
      vlanIds = statuses.keys()
      vlanIds.sort( key=int )
   else:
      if vlanId in statuses:
         vlanIds = [ vlanId ]        
   macEntry = []
   for vlanId in vlanIds:
      statusVlan = statuses.get( vlanId )
      if statusVlan and not statusVlan.useVlanFloodset:
         macEntry += showEthGroupHelper( 
            vlanId, statusVlan, intfNames, macAddress, brief )

   if vlanIdForMulticast is not None:
      # Simulates vlanId.lookup, which is not available since vlan is not
      # parsed as a Vlan object
      if bridgingConfig.vlanConfig.has_key( vlanIdForMulticast ):
         vlans = [ vlanIdForMulticast ]
      else:
         vlans = []
   else:
      vlans = VlanCli.Vlan.getAll( mode )
   vlans.sort()

   vlanConfigs = bridgingConfig.vlanConfig
   fdbConfigs = bridgingConfig.fdbConfig

   multicastTable = BridgingCli.getStaticMulticastTable( mode, vlans, macAddress,
                                                         intfs, fdbConfigs,
                                                         vlanConfigs )
   return MacAddressTableIgmpSnooping( igmpMacEntries=macEntry, 
         multicastMacTable=multicastTable, _brief=brief )

def showEthGroupHelper ( vlanId, statusVlan, intfs, macAddress, brief ):
   ''' This function takes a vlan, and prints the ethGroup entries for that vlan.
       EthGroup entries show all host and router interfaces.
   '''
   assert statusVlan

   if macAddress:
      if macAddress not in statusVlan.ethGroup:
         return []
      allAddrs = [ macAddress ]
   else:
      allAddrs = statusVlan.ethGroup.keys()
  
   entryList = []
   for groupAddr in allAddrs:
      macTable = IgmpSnoopingTableEntry()
      ethGroup = statusVlan.ethGroup.get( groupAddr )
      if not ethGroup:
         continue
      intfNames = ethGroup.intf.keys()
      printIntfs = set( intfNames )
      # add router ports to port list
      printIntfs.update( statusVlan.routerIntf.keys() )
      if intfs and not set( intfs ).intersection( printIntfs ):
         continue
      
      macTable.vlanId = vlanId
      macTable.macAddr = groupAddr
      macTable.macType = 'igmp'
      macTable.interfaces = list( printIntfs )
      entryList.append( macTable )
   return entryList

#------------------------------------------------------------------------------------
# show ( ( mac address-table ) | mac-address-table ) multicast [ vlan VLAN_ID ] brief
#------------------------------------------------------------------------------------
class MulticastBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show MAC_ADDR_TABLE multicast [ vlan VLAN_ID ] brief'
   data = {
      'MAC_ADDR_TABLE': BridgingCli.MacAddrTableExprForShow,
      'multicast' : matcherMulticast,
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
      'brief' : 'Brief',
   }

   handler = doShowMacAddressTableMulticast
   cliModel = MacAddressTableIgmpSnooping

BasicCli.addShowCommandClass( MulticastBriefCmd )

#--------------------------------------------------------------------------------
# show ( ( mac address-table ) | mac-address-table ) multicast
#                            [ address MAC_ADDR ] [ interface INTF ] [ vlan VLAN_ID ]
#--------------------------------------------------------------------------------
class MulticastCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show MAC_ADDR_TABLE multicast
              [ address MAC_ADDR ] [ interface INTF ] [ vlan VLAN_ID ]'''
   data = {
      'MAC_ADDR_TABLE': BridgingCli.MacAddrTableExprForShow,
      'multicast' : matcherMulticast,
      'address' : 'Address keyword',
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'interface' : interfaceMatcher,
      'INTF' : Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes ),
      'vlan' : vlanMatcher,
      'VLAN_ID' : vlanIdMatcher,
   }

   handler = doShowMacAddressTableMulticast
   cliModel = MacAddressTableIgmpSnooping

BasicCli.addShowCommandClass( MulticastCmd )

# ===================================================================================
# Configuration Commands
# ===================================================================================

# _______ :vlan enable/disable config command: _____
# If vlan snooping state is disabled, then snoopin on vlan is disabled irrespective
# of the global state. If it is enabled, then it follows the global enable/disable
# state.
# _______ :show command: _______
# Show vlan state if disabled, else show the global state.

# -----------------------------------------------------------------------------------
# [no|default] ip igmp snooping
# -----------------------------------------------------------------------------------
class IpIgmpSnoopingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      igmpSnoopingConfig.vlanDefaultEnabled = True

   @staticmethod
   def noHandler( mode, args ):
      igmpSnoopingConfig.vlanDefaultEnabled = False

   @staticmethod
   def defaultHandler( mode, args ):
      default = igmpSnoopingConfig.vlanDefaultEnabledDefault
      igmpSnoopingConfig.vlanDefaultEnabled = default


BasicCli.GlobalConfigMode.addCommandClass( IpIgmpSnoopingCmd )

# ----------------------------------------------------------------------
# [ no | default] ip igmp snooping vlan <vlan range>
# ----------------------------------------------------------------------
def doSetIgmpSnoopingVlan( mode, vlanSet, no ):
   # The value of no passed from cli is a bit unintuitive.
   # Its is True for no, 'default' for default, and None otherwise.
   vlanConfig = igmpSnoopingConfig.vlanConfig
   for vlanId in vlanSet.ids:
      vlan = vlanConfig.get( vlanId )
      if vlan:
         if no is True:
            vlan.enabled = 'vlanStateDisabled'
         else:
            vlan.enabled = 'vlanStateDefault'
      else:
         # create an entry only if non-default values being set.
         if no is True: # no form of the command
            vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
            vlan.enabled = 'vlanStateDisabled'

class IpIgmpSnoopingVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping vlan VLANSET'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      doSetIgmpSnoopingVlan( mode, args[ 'VLANSET' ], no=None )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      no = True if CliCommand.isNoCmd( args ) else 'default'
      doSetIgmpSnoopingVlan( mode, args[ 'VLANSET' ], no=no )

BasicCli.GlobalConfigMode.addCommandClass( IpIgmpSnoopingVlanVlansetCmd )


# ------------------------------------------------------------------------------
# [no|default] ip igmp snooping vlan <vlan range> multicast-router interface
# <intface-range>
#
# # legacy:
# [no|default] ip igmp snooping vlan <vlan range> mrouter interface
# <intface-range> 
# ------------------------------------------------------------------------------
def setSnoopingVlanMrouter( mode, snoopingConfig, vlanIds, intfs, no=False,
                                deprecatedCmd=False ):
   def setIgmpSnoopingVlanMrouterIntf( intf ):
      intfname = intf.name
      for vlanId in vlanIds:
         vlan = snoopingConfig.newVlanConfig( vlanId )
         if no:
            # The boolean value is just a dummy. The dict entry has to be deleted if
            # the entry needs to be invalidated.
            del vlan.routerIntf[ intfname ] # deletion is idempotent.
         else:
            intf.create()
            vlan.routerIntf[ intfname ] = True

   interfaces = IntfCli.Intf.getAll( mode, intfs, config=True )
   if not interfaces:
      return
   for i in interfaces:
      setIgmpSnoopingVlanMrouterIntf( i )

intfRangeConfigMatcher = IntfCli.intfRangeWithSingleExpression(
   'INTF', explicitIntfTypes=intfTypes )

class VlanMulticastRouterCommand( CliCommand.CliCommandClass ):
   syntax = "ip-igmp-snooping vlan VLANSET mrouter | multicast-router " \
            "interface INTF"
   noOrDefaultSyntax = syntax
   data = {
      "ip-igmp-snooping" : IpIgmpSnoopingForConfig,
      "vlan" : vlanMatcher,
      "VLANSET" : VlanCli.vlanSetMatcher,
      "mrouter" : mrouterNodeDeprecated,
      "multicast-router" : multicastRouterMatcher,
      "interface" : interfaceMatcher,
      "INTF" : intfRangeConfigMatcher
   }
   @staticmethod
   def handler( mode, args ):
      vlanSet = args[ 'VLANSET' ]
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      setSnoopingVlanMrouter( mode, igmpSnoopingConfig, vlanSet.ids, intfs,
                                  no=no )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( VlanMulticastRouterCommand )

# ------------------------------------------------------------------------------
# [no|default] ip igmp snooping vlan <vlan range> report-flooding switch-port
# ethernet <switch-port-range>
# ------------------------------------------------------------------------------
def setIgmpSnoopingVlanReportFloodingSwitchPort( mode, vlanSet, intfs, no=False ):
   def setIgmpSnoopingVlanReportFloodingSwitchPortIntf ( intf ):
      intfname = intf.name
      for vlanId in vlanSet.ids:
         vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
         if no:
            # The boolean value is just a dummy. The dict entry has to be deleted if
            # the entry needs to be invalidated.
            del vlan.switchPort [ intfname ] # deletion is idempotent.
         else:
            intf.create()
            vlan.switchPort [ intfname ] = True

   interfaces = IntfCli.Intf.getAll( mode, intfs, config=True )
   if not interfaces:
      return
   for i in interfaces:
      setIgmpSnoopingVlanReportFloodingSwitchPortIntf ( i )

class VlanReportFloodingCommand( CliCommand.CliCommandClass ):
   syntax = "ip-igmp-snooping vlan VLANSET report-flooding switch-port INTF"
   noOrDefaultSyntax = syntax
   data = {
      "ip-igmp-snooping" : IpIgmpSnoopingForConfig,
      "vlan" : vlanMatcher,
      "VLANSET" : VlanCli.vlanSetMatcher,
      "report-flooding" : reportFloodingMatcher,
      "switch-port" : switchPortMatcher,
      "INTF" : intfRangeConfigMatcher
   }
   @staticmethod
   def handler( mode, args ):
      vlanSet = args[ 'VLANSET' ]
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      setIgmpSnoopingVlanReportFloodingSwitchPort( mode, vlanSet, intfs, no=no )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( VlanReportFloodingCommand )

 #--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping vlan VLANSET max-groups MAX_GROUPS
#--------------------------------------------------------------------------------
class IgmpSnoopingVlanMaxGroupsCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping vlan VLANSET max-groups MAX_GROUPS'
   noOrDefaultSyntax = 'ip igmp snooping vlan VLANSET max-groups ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'max-groups' : 'Configures the maximum number of IGMP groups allowed per VLAN',
      'MAX_GROUPS' : CliMatcher.IntegerMatcher( 0, 65534,
         helpdesc='Maximum number of allowable groups for this VLAN' ),
   }

   @staticmethod
   def handler( mode, args ):
      for vlanId in args[ 'VLANSET' ].ids:
         vlanConfig = igmpSnoopingConfig.newVlanConfig( vlanId )
         vlanConfig.maxGroups = args[ 'MAX_GROUPS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for vlanId in args[ 'VLANSET' ].ids:
         vlanConfig = igmpSnoopingConfig.newVlanConfig( vlanId )
         vlanConfig.maxGroups = vlanConfig.unlimitedGroups

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingVlanMaxGroupsCmd )

# ------------------------------------------------------------------------------
# [no|default] ip igmp snooping vlan <vlan range> fast-leave
# 
# legacy:
# [no|default] ip igmp snooping vlan <vlan range> immediate-leave
# ------------------------------------------------------------------------------
class IgmpSnoopingImmediateLeaveCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping vlan VLANSET ( fast-leave | immediate-leave )'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'fast-leave' : matcherFastLeave,
      'immediate-leave' : immediateLeaveNode,
   }

   @staticmethod
   def _setValue( args, value ):
      deprecatedCmd = 'immediate-leave' in args
      for vlanId in args[ 'VLANSET' ].ids:
         vlanConfig = igmpSnoopingConfig.newVlanConfig( vlanId )
         vlanConfig.immediateLeave = value
         # If the user uses default, then the command will not be saved unless save
         # all. If the user explicitly enables or disables immediate leave, we will
         # save it in the format it was entered.
         # Because we have a single flag for all commands, whatever user does last
         # changes the flag.
         igmpSnoopingConfig.deprecatedCmd = deprecatedCmd and \
                                            ( value == "immediateLeaveDisabled" or
                                             value == "immediateLeaveEnabled" )

   @staticmethod
   def handler( mode, args ):
      IgmpSnoopingImmediateLeaveCmd._setValue( args, 'immediateLeaveEnabled' )

   @staticmethod
   def noHandler( mode, args ):
      IgmpSnoopingImmediateLeaveCmd._setValue( args, 'immediateLeaveDisabled' )

   @staticmethod
   def defaultHandler( mode, args ):
      IgmpSnoopingImmediateLeaveCmd._setValue( args, 'immediateLeaveDefaultState' )

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingImmediateLeaveCmd )

# ------------------------------------------------------------------------------
# [no|default] ip igmp snooping fast-leave
# 
# legacy:
# [no|default] ip igmp snooping immediate-leave
# ------------------------------------------------------------------------------
class IpIgmpSnoopingFastLeaveCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping ( fast-leave | immediate-leave )'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'fast-leave' : matcherFastLeave,
      'immediate-leave' : immediateLeaveNode,
   }

   @staticmethod
   def _setValue( args, value ):
      igmpSnoopingConfig.immediateLeave = value
      # By default, immediate leave is true. So deprecated flag should only
      # be set if user sets immediate leave to false
      # Because we have a single flag for all commands, whatever user does last
      # changes the flag.
      igmpSnoopingConfig.deprecatedCmd = ( 'immediate-leave' in args
                                           and value is False )

   @staticmethod
   def handler( mode, args ):
      IpIgmpSnoopingFastLeaveCmd._setValue( args, True )

   @staticmethod
   def noHandler( mode, args ):
      IpIgmpSnoopingFastLeaveCmd._setValue( args, False )

   @staticmethod
   def defaultHandler( mode, args ):
      IpIgmpSnoopingFastLeaveCmd._setValue( args,
            igmpSnoopingConfig.immediateLeaveDefault )

BasicCli.GlobalConfigMode.addCommandClass( IpIgmpSnoopingFastLeaveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping report-flooding
#--------------------------------------------------------------------------------
def setIgmpSnoopingReportFlooding( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if( ( not no ) and igmpSnoopingConfig.vlanDefaultEnabled is False ):
      mode.addWarning(
         "Igmp snooping needs to be enabled for report-flooding feature" )
   # In the "no" form of the command, cli passes 'no = True'
   igmpSnoopingConfig.globalReportFloodingEnabled = not no

class IpIgmpSnoopingReportFloodingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping report-flooding'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'report-flooding' : reportFloodingMatcher,
   }

   handler = setIgmpSnoopingReportFlooding
   noOrDefaultHandler = setIgmpSnoopingReportFlooding

BasicCli.GlobalConfigMode.addCommandClass( IpIgmpSnoopingReportFloodingCmd )

# ------------------------------------------------------------------------------
# [no|default] ip igmp snooping report-flooding switch-port
# ethernet <switch-port-range>
# ------------------------------------------------------------------------------
def setIgmpSnoopingReportFloodingSwitchPort ( mode, intfs, no=False ):
   def setIgmpSnoopingReportFloodingSwitchPortIntf ( intf ):
      intfname = intf.name
      if no:
         del igmpSnoopingConfig.switchPort [ intfname ] # deletion is idempotent.
      else:
         intf.create()
         igmpSnoopingConfig.switchPort [ intfname ] = True

   interfaces = IntfCli.Intf.getAll( mode, intfs, config=True )
   if not interfaces:
      return
   for i in interfaces:
      setIgmpSnoopingReportFloodingSwitchPortIntf ( i )

class ReportFloodingCommand( CliCommand.CliCommandClass ):
   syntax = "ip-igmp-snooping report-flooding switch-port INTF"
   noOrDefaultSyntax = syntax
   data = {
      "ip-igmp-snooping" : IpIgmpSnoopingForConfig,
      "report-flooding" : reportFloodingMatcher,
      "switch-port" : switchPortMatcher,
      "INTF" : intfRangeConfigMatcher
   }
   @staticmethod
   def handler( mode, args ):
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      setIgmpSnoopingReportFloodingSwitchPort( mode, intfs, no=no )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( ReportFloodingCommand )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping querier
#--------------------------------------------------------------------------------
class IpIgmpSnoopingQuerierCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping querier'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'querier' : matcherQuerier,
   }

   @staticmethod
   def handler( mode, args ):
      igmpSnoopingConfig.globalQuerierConfig.enabled = True

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      igmpSnoopingConfig.globalQuerierConfig.enabled = False

BasicCli.GlobalConfigMode.addCommandClass( IpIgmpSnoopingQuerierCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping vlan VLANSET report-flooding
#--------------------------------------------------------------------------------
def setIgmpSnoopingVlanReportFlooding( mode, args ):
   vlanSet = args[ 'VLANSET' ]
   if CliCommand.isNoCmd( args ):
      no = True
   elif CliCommand.isDefaultCmd( args ):
      no = 'default'
   else:
      no = None

   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if( igmpSnoopingConfig.vlanDefaultEnabled is False or \
                              vlan.enabled == 'vlanStateDisabled' ):
         mode.addWarning( "Report flooding on VLAN %d will not be enabled "
                          "until IGMP snooping is enabled." % vlanId )

      if no is True:
         vlan.reportFloodingConfig = 'vlanReportFloodingDisabled'
      elif no == 'default':
         vlan.reportFloodingConfig = 'vlanReportFloodingDefault'
      else:
         if not igmpSnoopingConfig.globalReportFloodingEnabled:
            mode.addWarning( "Report flooding on VLAN %d will not be enabled "
                             "until report flooding is enabled globally."
                             % vlanId )
         vlan.reportFloodingConfig = 'vlanReportFloodingEnabled'

class IgmpSnoopingVlanReportFloodingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping vlan VLANSET report-flooding'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'report-flooding' : reportFloodingMatcher,
   }

   handler = setIgmpSnoopingVlanReportFlooding
   noOrDefaultHandler = setIgmpSnoopingVlanReportFlooding

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingVlanReportFloodingCmd )


# ----------------------------------------------------------------------
# [ no | default ] ip igmp snooping vlan VLANSET querier
# ----------------------------------------------------------------------
def setIgmpSnoopingVlanQuerier( mode, vlanSet, no ):
   # The value of no passed from cli is a bit unintuitive.
   # Its is True for no, 'default' for default, and None otherwise.

   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ()
      if no is True:
         vlan.vlanQuerierConfig.enabledConfigured = 'vlanQuerierDisabled'
      elif no == 'default':
         vlan.vlanQuerierConfig.enabledConfigured = 'vlanQuerierDefault'
      else: # no == None
         vlan.vlanQuerierConfig.enabledConfigured = 'vlanQuerierEnabled'

class IgmpSnoopingVlanQuerierCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping vlan VLANSET querier'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
   }

   @staticmethod
   def handler( mode, args ):
      setIgmpSnoopingVlanQuerier( mode, args.get( 'VLANSET' ), no=None )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      no = True if CliCommand.isNoCmd( args ) else 'default'
      setIgmpSnoopingVlanQuerier( mode, args.get( 'VLANSET' ), no=no )

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingVlanQuerierCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier address IPADDR
#--------------------------------------------------------------------------------
def setIgmpSnoopingQuerierAddress( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   ipAddr = args.get( 'IPADDR', '0.0.0.0' )
   isIpAddrConfigured = True
   if CliCommand.isNoOrDefaultCmd( args ):
      ipAddr = '0.0.0.0'
      isIpAddrConfigured = False
   else:
      try:
         ipAddress = Arnet.IpAddress( ipAddr )
         if Arnet.Subnet( '224.0.0.0/4' ).containsAddr( ipAddress ):
            raise ValueError, "Multicast address not allowed."
      except ValueError, e:
         mode.addError( str( e ) )
         return
   if not vlanSet:
      igmpSnoopingConfig.globalQuerierConfig.isIpAddrConfigured = isIpAddrConfigured
      igmpSnoopingConfig.globalQuerierConfig.ipAddr = ipAddr
      return
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ()
      vlan.vlanQuerierConfig.isIpAddrConfigured = isIpAddrConfigured
      vlan.vlanQuerierConfig.ipAddrConfigured = ipAddr

class IgmpSnoopingQuerierAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping [ vlan VLANSET ] querier address IPADDR'
   noOrDefaultSyntax = 'ip igmp snooping [ vlan VLANSET ] querier address ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'address' : 'IP address',
      'IPADDR' : IpAddrMatcher.ipAddrMatcher,
   }

   handler = setIgmpSnoopingQuerierAddress
   noOrDefaultHandler = setIgmpSnoopingQuerierAddress

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingQuerierAddressCmd )

#------------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier query-interval INTERVAL
#------------------------------------------------------------------------------------
def setIgmpSnoopingQuerierQueryInterval( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   queryInterval = args.get( 'INTERVAL', 0.0 )
   no = CliCommand.isNoOrDefaultCmd( args )
   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.queryInterval = \
            igmpSnoopingConfig.globalQuerierConfig.queryIntervalDefault
      else:
         igmpSnoopingConfig.globalQuerierConfig.queryInterval = queryInterval
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ()
      if no:
         vlan.vlanQuerierConfig.queryIntervalConfigured = 0.0
      else:
         vlan.vlanQuerierConfig.queryIntervalConfigured = queryInterval

class IgmpSnoopingQuerierQueryIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping [ vlan VLANSET ] querier query-interval INTERVAL'
   noOrDefaultSyntax = 'ip igmp snooping [ vlan VLANSET ] querier query-interval ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'query-interval' : CliToken.IgmpSnooping.queryIntervalMatcher,
      'INTERVAL' : CliMatcher.IntegerMatcher( 5, 3600,
         helpdesc='Time between queries in units of seconds' ),
   }

   handler = setIgmpSnoopingQuerierQueryInterval
   noOrDefaultHandler = setIgmpSnoopingQuerierQueryInterval

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingQuerierQueryIntervalCmd )

#------------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [vlan VLANSET] querier max-response-time INTERVAL
#------------------------------------------------------------------------------------
def setIgmpSnoopingQuerierQueryResponseInterval( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   queryResponseInterval = args.get( 'INTERVAL', 0.0 )
   no = CliCommand.isNoOrDefaultCmd( args )
   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.queryResponseInterval = \
             igmpSnoopingConfig.globalQuerierConfig.queryResponseIntervalDefault
      else:
         igmpSnoopingConfig.globalQuerierConfig.queryResponseInterval = \
             queryResponseInterval
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ()
      if no:
         vlan.vlanQuerierConfig.queryResponseIntervalConfigured = 0.0
      else:
         vlan.vlanQuerierConfig.queryResponseIntervalConfigured = \
             queryResponseInterval

class IgmpSnoopingQuerierQueryResponseIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping [ vlan VLANSET ] querier max-response-time INTERVAL'
   noOrDefaultSyntax = ( 'ip igmp snooping [ vlan VLANSET ] querier '
                                                            'max-response-time ...' )
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'max-response-time' : CliToken.IgmpSnooping.maxResponseTimeMatcher,
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 25,
         helpdesc='Query response Interval in seconds' ),
   }

   handler = setIgmpSnoopingQuerierQueryResponseInterval
   noOrDefaultHandler = setIgmpSnoopingQuerierQueryResponseInterval

BasicCli.GlobalConfigMode.addCommandClass(
      IgmpSnoopingQuerierQueryResponseIntervalCmd )

#------------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier
#                                                 last-member-query-interval INTERVAL
#------------------------------------------------------------------------------------
def setIgmpSnoopingQuerierLastMemberInterval( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   lastMemberQueryInterval = args.get( 'INTERVAL', 0.0 )
   no = CliCommand.isNoOrDefaultCmd( args )
   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.lastMemberQueryInterval = \
             igmpSnoopingConfig.globalQuerierConfig.lastMemberQueryIntervalDefault
      else:
         igmpSnoopingConfig.globalQuerierConfig.lastMemberQueryInterval = \
             lastMemberQueryInterval
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ()
      if no:
         vlan.vlanQuerierConfig.lastMemberQueryIntervalConfigured = 0.0
      else:
         vlan.vlanQuerierConfig.lastMemberQueryIntervalConfigured = \
             lastMemberQueryInterval

class IgmpSnoopingQuerierLastMemberIntervalCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip igmp snooping [ vlan VLANSET ] querier last-member-query-interval '
                                                                        'INTERVAL' )
   noOrDefaultSyntax = ( 'ip igmp snooping [ vlan VLANSET ] querier '
                                                   'last-member-query-interval ...' )
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'last-member-query-interval' : \
            CliToken.IgmpSnooping.lastMemberQueryIntervalMatcher,
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 25,
         helpdesc='Last member query interval in seconds' ),
   }

   handler = setIgmpSnoopingQuerierLastMemberInterval
   noOrDefaultHandler = setIgmpSnoopingQuerierLastMemberInterval

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingQuerierLastMemberIntervalCmd )

#------------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier last-member-query-count
#                                                                               COUNT
#------------------------------------------------------------------------------------
def setIgmpSnoopingQuerierLastMemberCount( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   lastMemberQueryCount = args.get( 'COUNT', 0 )
   no = CliCommand.isNoOrDefaultCmd( args )
   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.lastMemberQueryCount = 0
      else:
         igmpSnoopingConfig.globalQuerierConfig.lastMemberQueryCount = \
             lastMemberQueryCount
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ( )
      if no:
         vlan.vlanQuerierConfig.lastMemberQueryCountConfigured = 0
      else:
         vlan.vlanQuerierConfig.lastMemberQueryCountConfigured = \
             lastMemberQueryCount

class IgmpSnoopingQuerierLastMemberCountCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping [ vlan VLANSET ] querier last-member-query-count COUNT'
   noOrDefaultSyntax = ( 'ip igmp snooping [ vlan VLANSET ] querier '
                                                      'last-member-query-count ...' )
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'last-member-query-count' : CliToken.IgmpSnooping.lastMemberQueryCountMatcher,
      'COUNT' : CliMatcher.IntegerMatcher( 1, 3, helpdesc='Last member query count' )
   }

   handler = setIgmpSnoopingQuerierLastMemberCount
   noOrDefaultHandler = setIgmpSnoopingQuerierLastMemberCount

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingQuerierLastMemberCountCmd )

#------------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier
#                                                     startup-query-interval INTERVAL
#------------------------------------------------------------------------------------
def setIgmpSnoopingQuerierStartupInterval( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   startupQueryInterval = args.get( 'INTERVAL', 0.0 )
   no = CliCommand.isNoOrDefaultCmd( args )

   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.startupQueryInterval = 0.0
      else:
         igmpSnoopingConfig.globalQuerierConfig.startupQueryInterval = \
             startupQueryInterval
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ( )
      if no:
         vlan.vlanQuerierConfig.startupQueryIntervalConfigured = 0.0
      else:
         vlan.vlanQuerierConfig.startupQueryIntervalConfigured = \
             startupQueryInterval

class IgmpSnoopingQuerierStartupIntervalCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip igmp snooping [ vlan VLANSET ] querier startup-query-interval '
                                                                        'INTERVAL' )
   noOrDefaultSyntax = ( 'ip igmp snooping [ vlan VLANSET ] querier '
                                                      'startup-query-interval ...' )
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'startup-query-interval' : CliToken.IgmpSnooping.startupQueryIntervalMatcher,
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 3600,
         helpdesc='Startup query interval in seconds' ),
   }

   handler = setIgmpSnoopingQuerierStartupInterval
   noOrDefaultHandler = setIgmpSnoopingQuerierStartupInterval

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingQuerierStartupIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier
#                                                        startup-query-count COUNT
#--------------------------------------------------------------------------------
def setIgmpSnoopingQuerierStartupCount( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   startupQueryCount = args.get( 'COUNT', 0 )
   no = CliCommand.isNoOrDefaultCmd( args )
   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.startupQueryCount = 0
      else:
         igmpSnoopingConfig.globalQuerierConfig.startupQueryCount = \
             startupQueryCount
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ( igmpSnoopingConfig.globalQuerierConfig, )
      if no:
         vlan.vlanQuerierConfig.startupQueryCountConfigured = 0
      else:
         vlan.vlanQuerierConfig.startupQueryCountConfigured = \
             startupQueryCount

class IgmpSnoopingQuerierStartupCountCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping [ vlan VLANSET ] querier startup-query-count COUNT'
   noOrDefaultSyntax = ( 'ip igmp snooping [ vlan VLANSET ] querier '
                                                         'startup-query-count ...' )
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'startup-query-count' : CliToken.IgmpSnooping.startupQueryCountMatcher,
      'COUNT' : CliMatcher.IntegerMatcher( 1, 3,
         helpdesc='Startup query count' ),
   }

   handler = setIgmpSnoopingQuerierStartupCount
   noOrDefaultHandler = setIgmpSnoopingQuerierStartupCount

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingQuerierStartupCountCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier version VERSION
#--------------------------------------------------------------------------------
def setIgmpSnoopingQuerierVersion( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   querierVersion = args.get( 'VERSION', 2 )
   no = CliCommand.isNoOrDefaultCmd( args )
   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.querierVersion = \
             igmpSnoopingConfig.globalQuerierConfig.querierVersionDefault
      else:
         igmpSnoopingConfig.globalQuerierConfig.querierVersion = \
             'igmpVersion%d' % querierVersion
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ()
      if no:
         vlan.vlanQuerierConfig.querierVersionConfigured = 'igmpVersionUnknown'
      else:
         vlan.vlanQuerierConfig.querierVersionConfigured = \
             'igmpVersion%d' % querierVersion

class IgmpSnoopingQuerierVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping [ vlan VLANSET ] querier version VERSION'
   noOrDefaultSyntax = 'ip igmp snooping [ vlan VLANSET ] querier version ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'version' : CliToken.IgmpSnooping.versionMatcher,
      'VERSION' : CliMatcher.IntegerMatcher( 1, 3, helpdesc='Querier Version' ),
   }

   handler = setIgmpSnoopingQuerierVersion
   noOrDefaultHandler = setIgmpSnoopingQuerierVersion

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingQuerierVersionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping [ vlan VLANSET ] querier mtu MTU
#--------------------------------------------------------------------------------
def setIgmpSnoopingMtu( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   mtu = args.get( 'MTU', 0 )
   no = CliCommand.isNoOrDefaultCmd( args )

   if not vlanSet: # global configuration
      if no:
         igmpSnoopingConfig.globalQuerierConfig.mtu = \
             igmpSnoopingConfig.globalQuerierConfig.mtuDefault
      else:
         igmpSnoopingConfig.globalQuerierConfig.mtu = mtu
      return
   # configure for vlans
   for vlanId in vlanSet.ids:
      vlan = igmpSnoopingConfig.newVlanConfig( vlanId )
      if not vlan.vlanQuerierConfig:
         vlan.vlanQuerierConfig = ()
      if no:
         vlan.vlanQuerierConfig.mtuConfigured = 0
      else:
         vlan.vlanQuerierConfig.mtuConfigured = mtu

class IgmpSnoopingMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping [ vlan VLANSET ] querier mtu MTU'
   noOrDefaultSyntax = 'ip igmp snooping [ vlan VLANSET ] querier mtu ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'vlan' : vlanMatcher,
      'VLANSET' : VlanCli.vlanSetMatcher,
      'querier' : matcherQuerier,
      'mtu' : 'Configure maximum query size',
      'MTU' : CliMatcher.IntegerMatcher( 36, 9212,
         helpdesc='Largest packet size that can be sent by the querier (in bytes)' ),
   }

   hidden = True
   handler = setIgmpSnoopingMtu
   noOrDefaultHandler = setIgmpSnoopingMtu

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingMtuCmd )

# -------------------------------------------------------------------------------
# [no|default] ip igmp snooping vlan <vlan id> member <ip-addr>
#              interface <interface-range>
#
# Legacy:
# [no|default] ip igmp snooping vlan <vlan id> static <ip-addr>
#              interface <interface-range>
#
# Note: only a single vlanid as opposed to vlanset is allowed in this command.
# -------------------------------------------------------------------------------
def doSetSnoopingVlanMember( mode, snoopingConfig, vlanId, ipAddr, intfs, no=False,
                             deprecatedCmd=False ):
   genAddr = IpGenAddr( str( ipAddr ) )
   if genAddr.af == "ipv4":
      error = IpAddrMatcher.validateMulticastIpAddr( ipAddr, False )
      if error != None:
         mode.addError( "Enter a valid usable multicast address - %s" % error )
         return
   else:
      assert genAddr.af == "ipv6"
      error = None
      if not genAddr.isMulticast:
         error = "Invalid multicast range"
      elif genAddr.isLinkLocalMulticast:
         error = "Link local multicast range"
      if error:
         mode.addError( "Enter a valid usable multicast address - %s" % error )
         return
   interfaces = IntfCli.Intf.getAll( mode, intfs, config=True )
   if not interfaces:
      return
   vlan = snoopingConfig.newVlanConfig( vlanId )
   ipGroup = vlan.newIpGroup( genAddr )

   def doSetIgmpSnoopingVlanStaticIntf ( intf ):
      intfname = intf.name
      if no:
         # The boolean value is just a dummy. The dict entry has to be deleted if
         # the entry needs to be invalidated.
         del ipGroup.intf[ intfname ] # deletion is idempotent
      else:
         intf.create()
         ipGroup.intf[ intfname ] = True

   for i in interfaces:
      doSetIgmpSnoopingVlanStaticIntf( i )

   if not ipGroup.intf:
      del vlan.ipGroup[ genAddr ]

   igmpSnoopingConfig.deprecatedCmd = deprecatedCmd and ( not no )

class VlanMemberCommand( CliCommand.CliCommandClass ):
   syntax = "ip-igmp-snooping vlan VLAN_ID member | static IPADDR " \
            "interface INTF"
   noOrDefaultSyntax = syntax
   data = {
      "ip-igmp-snooping" : IpIgmpSnoopingForConfig,
      "vlan" : vlanMatcher,
      "VLAN_ID" : vlanIdMatcher,
      "static" : staticNodeDeprecated,
      "member" : memberMatcher,
      "IPADDR" : IpAddrMatcher.ipAddrMatcher,
      "interface" : interfaceMatcher,
      "INTF" : intfRangeConfigMatcher
   }
   @staticmethod
   def handler( mode, args ):
      vlanId = args[ 'VLAN_ID' ]
      ipAddr = args[ 'IPADDR' ]
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      doSetSnoopingVlanMember( mode, igmpSnoopingConfig, vlanId, ipAddr, intfs,
                               no=no )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( VlanMemberCommand )

# -------------------------------------------------------------------------------
# [no|default] ip igmp snooping vlan <vlan id> member <ip-addr> <num-groups>
#                                             interface <intface-range>
# Note: only a single vlanid as opposed to vlanset is allowed in this command.
# -------------------------------------------------------------------------------
def doSetSnoopingVlanGroupMembers( mode, snoopingConfig, vlanId,
                                   ipAddr, numGroups, intfs, no=None ):
   for group in Arnet.getMcastGroupAddresses( ipAddr, numGroups ):
      doSetSnoopingVlanMember( mode, snoopingConfig, vlanId, group, intfs, no=no )

class VlanMemberGroupCommand( CliCommand.CliCommandClass ):
   syntax = "ip-igmp-snooping vlan VLAN_ID member IPADDR GROUPS interface INTF"
   noOrDefaultSyntax = syntax
   data = {
      "ip-igmp-snooping" : IpIgmpSnoopingForConfig,
      "vlan" : vlanMatcher,
      "VLAN_ID" : vlanIdMatcher,
      "member" : memberMatcher,
      "IPADDR" : IpAddrMatcher.ipAddrMatcher,
      "GROUPS" : numGroupsMatcher,
      "interface" : interfaceMatcher,
      "INTF" : intfRangeConfigMatcher
   }
   @staticmethod
   def handler( mode, args ):
      vlanId = args[ 'VLAN_ID' ]
      ipAddr = args[ 'IPADDR' ]
      numGroups = args[ 'GROUPS' ]
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      doSetSnoopingVlanGroupMembers( mode, igmpSnoopingConfig, vlanId, ipAddr,
                                     numGroups, intfs, no=no )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( VlanMemberGroupCommand )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping robustness-variable ROBUSTNESS
#--------------------------------------------------------------------------------
class IgmpSnoopingRobustnessCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping robustness-variable ROBUSTNESS'
   noOrDefaultSyntax = 'ip igmp snooping robustness-variable ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'robustness-variable' : CliToken.IgmpSnooping.robustnessVariableMatcher,
      'ROBUSTNESS' : CliMatcher.IntegerMatcher( 1, 3, helpdesc='Robustness value' ),
   }

   @staticmethod
   def handler( mode, args ):
      igmpSnoopingConfig.robustness = args[ 'ROBUSTNESS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      igmpSnoopingConfig.robustness = 2

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingRobustnessCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping restart query-interval QUERY_INTERVAL
#--------------------------------------------------------------------------------
class IgmpSnoopingRestartIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping restart query-interval QUERY_INTERVAL'
   noOrDefaultSyntax = 'ip igmp snooping restart query-interval ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'restart' : 'Configure IgmpSnooping agent restart',
      'query-interval' : CliToken.IgmpSnooping.queryIntervalMatcher,
      'QUERY_INTERVAL' : CliMatcher.IntegerMatcher( 2, 400,
         helpdesc='Query Interval during agent restart in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      igmpSnoopingConfig.restartQueryInterval = args[ 'QUERY_INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      igmpSnoopingConfig.restartQueryInterval = 0

BasicCli.GlobalConfigMode.addCommandClass( IgmpSnoopingRestartIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping interface-restart-query { QUERY_TIME }
#--------------------------------------------------------------------------------
class IpIgmpSnoopingInterfaceRestartQueryCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping interface-restart-query { QUERY_TIME }'
   noOrDefaultSyntax = 'ip igmp snooping interface-restart-query ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'interface-restart-query' : ( 'Configure interface startup initial query '
                                    'times in milliseconds' ),
      'QUERY_TIME' : CliMatcher.IntegerMatcher( 100, 50000,
         helpdesc='Time in milliseconds at which initial query is sent' ),
   }

   @staticmethod
   def handler( mode, args ):
      queryTimeList = args[ 'QUERY_TIME' ]
      igmpSnoopingConfig.intfRestartQueryTime.clear()
      index = 0
      for t in sorted( queryTimeList ):
         igmpSnoopingConfig.intfRestartQueryTime[ index ] = t
         index = index + 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      igmpSnoopingConfig.intfRestartQueryTime.clear()

BasicCli.GlobalConfigMode.addCommandClass( IpIgmpSnoopingInterfaceRestartQueryCmd )

def clearSnoopingCounters( mode, snoopingCounterCheckpointDir, snoopingCounterDir,
                           igmp=True, intfs=None ):
   syncCounterCheckpointDir( snoopingCounterCheckpointDir, snoopingCounterDir )

   logStr = 'igmp' if igmp else 'mld'
   if intfs: # clear only given interfaces
      intfNames = []
      for intf in intfs:
         Intf.Log.logClearCounters( "%s snooping" % logStr,
                                    "interface " + intf.name )
         intfNames.append( intf.name )
   else:
      Intf.Log.logClearCounters( "%s snooping" % logStr, "all interfaces" )
      intfNames = snoopingCounterCheckpointDir.intfCounter.keys()

   for intf in intfNames:
      counterCheckpoint =  snoopingCounterCheckpointDir.intfCounter.get( intf )
      counter = snoopingCounterDir.intfCounter.get( intf )
      if counter is None or counterCheckpoint is None:
         continue
      # checkpoint latest counter value.
      counterCheckpoint.doCopy( counter )

#--------------------------------------------------------------------------------
# clear ip igmp snooping counters [ LAG_INTF | PHY_INTF | SWTICH_INTF ]
#--------------------------------------------------------------------------------
class ClearIpIgmpSnoopingCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip igmp snooping counters [ LAG_INTF | PHY_INTF | SWTICH_INTF ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'counters' : 'Counters',
      'LAG_INTF' : EthLagIntf.matcher,
      'PHY_INTF': EthPhyIntf.ethMatcher,
      'SWTICH_INTF': SwitchIntf.matcher
   }

   @staticmethod
   def handler( mode, args ):
      intfs = None
      if 'LAG_INTF' in args:
         intfs = [ args[ 'LAG_INTF' ] ]
      elif 'PHY_INTF' in args:
         intfs = [ args[ 'PHY_INTF' ] ]
      elif 'SWTICH_INTF' in args:
         intfs = [ args[ 'SWTICH_INTF' ] ]
      clearSnoopingCounters( mode, counterCheckpointDir, counterDir, True, intfs )

BasicCli.EnableMode.addCommandClass( ClearIpIgmpSnoopingCountersCmd )

#-------------------------------------------------------------------------------
# Add Physical Interface ( Ethernet and Port-Channel ) specific CLI commands to
# the "config-if" mode.
#-------------------------------------------------------------------------------
class PhysIntfModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( ( mode.intf.name.startswith( 'Ethernet' ) or
                 mode.intf.name.startswith( 'Po' ) ) and
               not mode.intf.isSubIntf() )

   modeletParseTree = CliParser.ModeletParseTree()

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp snooping filter PROFILE_NAME
#--------------------------------------------------------------------------------
def setIgmpSnoopingFilter( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   if mode.intf.name in igmpSnoopingConfig.intfConfig:
      intfConfig = igmpSnoopingConfig.intfConfig[ mode.intf.name ]
   else:
      intfConfig = igmpSnoopingConfig.intfConfig.newMember( mode.intf.name )
   intfConfig.profileName = profileName

def noIgmpSnoopingFilter( mode, args ):
   profileName = args.get( 'PROFILE_NAME' )
   intfConfig = igmpSnoopingConfig.intfConfig.get( mode.intf.name )
   if profileName and ( not intfConfig or intfConfig.profileName != profileName ):
      mode.addError( 'This profile is not configured on the interface' )
      return
   if intfConfig:
      intfConfig.profileName = ""

class IpIgmpSnoopingFilterProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp snooping filter PROFILE_NAME'
   noOrDefaultSyntax = 'ip igmp snooping filter [ PROFILE_NAME ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'snooping' : CliToken.IgmpSnooping.snoopingMatcher,
      'filter' : 'Set IGMP Snooping filter profile',
      'PROFILE_NAME' : CliMatcher.DynamicNameMatcher(
         lambda mode: profileConfig.profiles.keys(), helpdesc='Profile name' ),
   }

   handler = setIgmpSnoopingFilter
   noOrDefaultHandler = noIgmpSnoopingFilter

PhysIntfModelet.addCommandClass( IpIgmpSnoopingFilterProfilenameCmd )

#-------------------------------------------------------------------------------
# Associate the PhysIntfModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( PhysIntfModelet )

#-------------------------------------------------------------------------------
# Cleanup per-interface configuration
#-------------------------------------------------------------------------------
class IntfIgmpSnoopingConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface IgmpSnooping config
   when the interface is deleted."""
   # pylint: disable-msg=E1101
   def setDefault( self ):
      if self.intf_.name in igmpSnoopingConfig.intfConfig:
         del igmpSnoopingConfig.intfConfig[ self.intf_.name ]

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
def Plugin ( em ):
   global igmpSnoopingConfig, forwardingStatus, protocolStatus, querierConfigDir
   global querierStatusDir, counterDir, counterCheckpointDir, bridgingConfig
   global bridgingInputCli, swForceConfig, protocolConfig
   global entityManager
   global profileConfig
   global igmpSnoopingCounterConfig
   global peerSyncStatus
   global bridgingHwCapabilities

   entityManager = em

   igmpSnoopingConfig = ConfigMount.mount(
      entityManager,
      "bridging/igmpsnooping/config", "Bridging::IgmpSnooping::Config", "w" )
   swForceConfig = LazyMount.mount( entityManager,
                                    "bridging/igmpsnooping/swForceConfig",
                                    "Bridging::IgmpSnooping::SwForceConfig", "r" )
   igmpSnoopingCounterConfig = LazyMount.mount(
      entityManager,
      "bridging/igmpsnooping/counterConfig",
      "Bridging::IgmpSnooping::CounterConfig", "w" )
   forwardingStatus = LazyMount.mount( entityManager,
                                       "bridging/igmpsnooping/forwarding/status",
                                       "Bridging::IgmpSnooping::Status", "r" )
   protocolConfig = LazyMount.mount(
      entityManager,
      "bridging/igmpsnooping/protocol/config",
      "Bridging::IgmpSnooping::IgmpProtocolConfig", "r" )

   protocolStatus = LazyMount.mount(
      entityManager,
      "bridging/igmpsnooping/protocol/status",
      "Bridging::IgmpSnooping::IgmpProtocolStatus", "r" )
   querierConfigDir = LazyMount.mount( entityManager,
                                    'bridging/igmpsnooping/querier/config',
                                    'Igmp::QuerierConfigDir', "r" )
   querierStatusDir = LazyMount.mount( entityManager,
                                       'bridging/igmpsnooping/querier/status',
                                       'Igmp::QuerierStatusDir', "r" )
   counterDir = LazyMount.mount( entityManager,
                                 'bridging/igmpsnooping/counterDir',
                                 'Bridging::IgmpSnooping::IgmpCounterDir', 'r' )
   counterCheckpointDir = LazyMount.mount(
      entityManager,
      'bridging/igmpsnooping/counterCheckpointDir',
      'Bridging::IgmpSnooping::IgmpCounterDir', 'w' )
   bridgingConfig = LazyMount.mount( entityManager,
                                     'bridging/config', 'Bridging::Config', 'r' )
   bridgingInputCli = LazyMount.mount( entityManager,
                                       'bridging/input/config/cli',
                                       'Bridging::Input::CliConfig', 'r' )
   profileConfig = LazyMount.mount( entityManager,
      "igmpprofile/profileconfig", "IgmpProfile::ProfileConfig", "r" )
   peerSyncStatus = LazyMount.mount( entityManager,
      "bridging/igmpsnooping/input/mlag", "Bridging::IgmpSnooping::SyncStatus", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
      "bridging/hwcapabilities", "Bridging::HwCapabilities", "r" )

   IntfCli.Intf.registerDependentClass( IntfIgmpSnoopingConfigCleaner, priority=10 )
