#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import CliCommand
import CliMatcher
from CliMode.IgmpProfile import IgmpProfileMode
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliToken.Ip
import CliToken.IgmpSnooping
import ConfigMount
from IgmpSnoopingModel import IgmpProfileInfo
import LazyMount
import ShowCommand
import Tac

bridgingHwCapabilities = None
profileConfig = None

def igmpSnoopingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.igmpSnoopingSupported:
      return None
   return CliParser.guardNotThisPlatform

profileMatcher = CliCommand.guardedKeyword( 'profile',
      helpdesc='IGMP Profile',
      guard=igmpSnoopingSupportedGuard )
profileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: profileConfig.profiles.keys(), 'Profile name' )

# ------------------------------------------------------------------------------
# show ip igmp profile [ PROFILE_NAME ]
# If no profile name is specified, all profiles are shown. Example:
# show ip igmp profile
# IGMP Profile 1
#      range 224.2.0.0 224.5.0.0
# IGMP Profile 2
#      permit
#      range 224.5.0.0 224.10.0.0
# if a profile name is specified, only that profile is shown. Example:
# show ip igmp profile 1
# IGMP Profile 1
#      range 224.2.0.0 224.5.0.0
# ------------------------------------------------------------------------------
def showIgmpProfile( profileName=None ):
   igmpProfile = IgmpProfileInfo()
   profiles = profileConfig.profiles
   for name, profile in profiles.iteritems():
      if profileName is None or name == profileName:
         igmpProfile.profiles[ name ] = igmpProfile.ProfileInfo()
         igmpProfile.profiles[ name ].action = profile.action
         for ip in profile.ranges:
            igmpProfile.profiles[ name ].ranges.append( 
                  igmpProfile.profiles[ name ].Ranges( min = ip.lowAddr,
                                                          max = ip.highAddr ) )
   return igmpProfile

class IpIgmpProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp profile [ PROFILE_NAME ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'profile' : profileMatcher,
      'PROFILE_NAME' : profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return showIgmpProfile( args.get( 'PROFILE_NAME' ) )

   cliModel = IgmpProfileInfo

BasicCli.addShowCommandClass( IpIgmpProfileCmd )

# ------------------------------------------------------------------------------
# config-igmp-profile mode
# ------------------------------------------------------------------------------
class igmpProfileConfigMode( IgmpProfileMode, BasicCli.ConfigModeBase ):
   name = "IGMP Profile Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, profileName ):
      self.profileName = profileName
      # We already have a previous implementation for 'show active'.
      self.__class__.showActiveCmdRegistered_ = True
      IgmpProfileMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.updateAction()

   def profile( self ):
      return profileConfig.profiles.newMember( self.profileName )

   def updateAction( self ):
      self.action = self.profile().action
      self.setLongModeKey()

# ------------------------------------------------------------------------------
# The "show active" command, in "config-igmp-profile" mode.
# "show active all detail" will be the default version.
# ------------------------------------------------------------------------------
class ShowActiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show active'
   data = {
      'active' : 'Display IGMP Profile information',
   }
   cliModel = IgmpProfileInfo

   @staticmethod
   def handler( mode, args ):
      return showIgmpProfile( mode.profileName )

igmpProfileConfigMode.addShowCommandClass( ShowActiveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp profile PROFILE_NAME
#--------------------------------------------------------------------------------
class IpIgmpProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'igmp' : CliToken.IgmpSnooping.igmpNode,
      'profile' : profileMatcher,
      'PROFILE_NAME' : profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( igmpProfileConfigMode,
                                  profileName=args[ 'PROFILE_NAME' ] )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName in profileConfig.profiles:
         del profileConfig.profiles[ profileName ]

BasicCli.GlobalConfigMode.addCommandClass( IpIgmpProfileProfilenameCmd )

# ===================================================================================
# IGMP Profile Configuration Commands
# ===================================================================================
# the following commands run in an IGMP Profile configuration mode
class ActionCmd( CliCommand.CliCommandClass ):
   syntax = 'ACTION'
   data = {
      'ACTION': CliMatcher.EnumMatcher( {
         'permit' : 'Matching addresses are permitted',
         'deny' : 'Matching addresses are denied',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      profile = mode.profile()
      profile.action = args[ 'ACTION' ]
      mode.updateAction()

igmpProfileConfigMode.addCommandClass( ActionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] range LOW_ADDR [ HIGH_ADDR ]
#--------------------------------------------------------------------------------
def validateRange( mode, lowAddr, highAddr ):
   lowAddr = Arnet.IpAddress( lowAddr )
   highAddr = Arnet.IpAddress( highAddr )
   if IpAddrMatcher.validateMulticastIpAddr( str( lowAddr ) ) or \
          IpAddrMatcher.validateMulticastIpAddr( str( highAddr ) ):
      mode.addError( "Invalid multicast address" )
      return False
   if lowAddr > highAddr:
      mode.addError( "Invalid range order" )
      return False
   return True

class RangeCmd( CliCommand.CliCommandClass ):
   syntax = 'range LOW_ADDR [ HIGH_ADDR ]'
   noOrDefaultSyntax = syntax
   data = {
      'range' : 'Add a range to the set',
      'LOW_ADDR' : IpAddrMatcher.IpAddrMatcher(
         helpdesc='Low IP multicast address' ),
      'HIGH_ADDR' : IpAddrMatcher.IpAddrMatcher(
         helpdesc='High IP multicast address' ),
   }

   # the input range is merged with any existing overlapping ranges
   @staticmethod
   def handler( mode, args ):
      lowAddr = args[ 'LOW_ADDR' ]
      highAddr = args.get( 'HIGH_ADDR' )
      if not highAddr:
         highAddr = lowAddr
      if not validateRange( mode, lowAddr, highAddr ):
         return
      r = Tac.newInstance( "IgmpProfile::Range", lowAddr, highAddr )
      mode.profile().addRange( r )
      
   # the inputed range is removed from the existing ranges
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      lowAddr = args[ 'LOW_ADDR' ]
      highAddr = args.get( 'HIGH_ADDR' )
      if not highAddr:
         highAddr = lowAddr
      if not validateRange( mode, lowAddr, highAddr ):
         return
      r = Tac.newInstance( "IgmpProfile::Range", lowAddr, highAddr )
      mode.profile().removeRange( r )

igmpProfileConfigMode.addCommandClass( RangeCmd )

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
def Plugin ( entityManager ):
   global profileConfig
   global bridgingHwCapabilities
   profileConfig = ConfigMount.mount( entityManager,
      "igmpprofile/profileconfig", "IgmpProfile::ProfileConfig", "w" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
      "bridging/hwcapabilities", "Bridging::HwCapabilities", "r" )
