#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
from CliPlugin.AclCli import ipAclNameMatcher, inKwMatcher, accessGroupKwMatcher
import CliPlugin.IgmpCli as IgmpCli
from CliPlugin.IgmpShowCommands import nodeIgmp
from CliPlugin.IpAddrMatcher import IpAddrMatcher, ipPrefixExpr
from CliPlugin.IraIpIntfCli import RoutingProtocolIntfConfigModelet
from CliToken.Ip import ipMatcherForConfig, ipMatcherForConfigIf

matcherCount = CliMatcher.IntegerMatcher( 0, 3,
      helpdesc='Last member query count' )
matcherInterval = CliMatcher.IntegerMatcher( 1, 3175,
      helpdesc='Time between queries in units of seconds' )
matcherQueryResponseTime = CliMatcher.KeywordMatcher( 'query-max-response-time',
      helpdesc='Time to wait for IGMP query responses' )
matcherQueryResponseTimeDeprecated = CliCommand.Node(
      matcher=matcherQueryResponseTime,
      deprecatedByCmd='igmp query-max-response-time <value in deciseconds>' )
matcherQueryCount = CliMatcher.IntegerMatcher( 1, 65535, 
      helpdesc='Startup query count' )
matcherQueryInterval = CliMatcher.IntegerMatcher( 10, 317440,
      helpdesc='query interval in deciseconds' )
matcherTime = CliMatcher.IntegerMatcher( 10, 31744,
      helpdesc='query response value in deciseconds' )
matcherSourceRange = ipPrefixExpr( 'Source range prefix',
      'Source range prefix mask', 'Source address range with prefix', partial=True )
matcherGroupRange = ipPrefixExpr( 'Group range prefix',
      'Group range prefix mask', 'Group address range with prefix', partial=True )
matcherNumGroups = CliMatcher.IntegerMatcher( 1, 4096, 
      helpdesc='the number of multicast groups to be configured' )
#--------------------------------------------------------------------------------
# ( no | default ) ip igmp access-group [ ACLNAME ] [ in ]
#--------------------------------------------------------------------------------
class IpIgmpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp access-group ACLNAME [ in ]'
   noOrDefaultSyntax = 'ip igmp access-group [ ACLNAME ] [ in ]'
   data = {
      'ip' : ipMatcherForConfig,
      'igmp' : nodeIgmp,
      'access-group' : accessGroupKwMatcher,
      'ACLNAME' : ipAclNameMatcher,
      'in' : inKwMatcher,
   }
   handler = IgmpCli.setIgmpServiceAcl
   noOrDefaultHandler = IgmpCli.noIgmpServiceAcl

IgmpCli.RouterIgmpSharedModelet.addCommandClass( IpIgmpAccessGroupCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp
#--------------------------------------------------------------------------------
class IpIgmpCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
   }
   handler = IgmpCli.cmdIgmp
   noOrDefaultHandler = IgmpCli.cmdNoIgmp

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp version
#--------------------------------------------------------------------------------
class IpIgmpVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp version VERSION'
   noOrDefaultSyntax = 'ip igmp version ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'version' : 'IGMP version',
      'VERSION' : CliMatcher.IntegerMatcher( 1, 3, helpdesc='Version' ),
   }
   handler = IgmpCli.cmdIgmpVersion
   noOrDefaultHandler = IgmpCli.cmdNoIgmpVersion

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpVersionCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp last-member-query-count COUNT
#--------------------------------------------------------------------------------
class IpIgmpQueryCountCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp last-member-query-count COUNT'
   noOrDefaultSyntax = 'ip igmp last-member-query-count ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'last-member-query-count' : 'IGMP last member query count',
      'COUNT' : matcherCount,
   }
   handler = IgmpCli.cmdIgmpLastMemberQueryCount
   noOrDefaultHandler = IgmpCli.cmdNoIgmpLastMemberQueryCount

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpQueryCountCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp last-member-query-interval QUERYINTERVAL
#--------------------------------------------------------------------------------
class IpIgmpQueryIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp last-member-query-interval QUERYINTERVAL'
   noOrDefaultSyntax = 'ip igmp last-member-query-interval ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'last-member-query-interval' : 'IGMP last member query interval',
      'QUERYINTERVAL' : matcherQueryInterval,
   }
   handler = IgmpCli.cmdIgmpLastMemberQueryInterval
   noOrDefaultHandler = IgmpCli.cmdNoIgmpLastMemberQueryInterval

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpQueryIntervalCmd )

#--------------------------------------------------------------------------------
# [no|default] igmp query-max-response-time TIME
#--------------------------------------------------------------------------------
class IpIgmpQueryResponseTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'igmp query-max-response-time TIME'
   noOrDefaultSyntax = 'igmp query-max-response-time ...'
   data = {
      'igmp' : nodeIgmp,
      'query-max-response-time' : matcherQueryResponseTime,
      'TIME' : matcherTime,
   }
   handler = IgmpCli.cmdIgmpQueryMaxResponseTime
   noOrDefaultHandler = IgmpCli.cmdNoIgmpQueryMaxResponseTime

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpQueryResponseTimeCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp query-max-response-time TIME
#--------------------------------------------------------------------------------
class IpIgmpQueryResponseTimeDepCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp query-max-response-time TIME'
   noOrDefaultSyntax = 'ip igmp query-max-response-time ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'query-max-response-time' : matcherQueryResponseTimeDeprecated,
      'TIME' : matcherTime,
   }
   handler = IgmpCli.cmdIgmpQueryMaxResponseTime
   noOrDefaultHandler = IgmpCli.cmdNoIgmpQueryMaxResponseTime

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpQueryResponseTimeDepCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp startup-query-count
#--------------------------------------------------------------------------------
class IpIgmpStartupQueryCountCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp startup-query-count QUERYCOUNT'
   noOrDefaultSyntax = 'ip igmp startup-query-count ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'startup-query-count' : 'IGMP startup query count',
      'QUERYCOUNT' : matcherQueryCount,
   }
   handler = IgmpCli.cmdIgmpStartupQueryCount
   noOrDefaultHandler = IgmpCli.cmdNoIgmpStartupQueryCount

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpStartupQueryCountCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp query-interval INTERVAL
#--------------------------------------------------------------------------------
class IpIgmpQueryIntervalcmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp query-interval INTERVAL'
   noOrDefaultSyntax = 'ip igmp query-interval ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'query-interval' : 'Time between IGMP queries',
      'INTERVAL' : matcherInterval,
   }
   handler = IgmpCli.cmdIgmpQueryInterval
   noOrDefaultHandler = IgmpCli.cmdNoIgmpQueryInterval

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpQueryIntervalcmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp startup-query-interval QUERYINTERVAL
#--------------------------------------------------------------------------------
class IpIgmpStartupQueryIntervalcmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp startup-query-interval QUERYINTERVAL'
   noOrDefaultSyntax = 'ip igmp startup-query-interval ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'startup-query-interval' : 'IGMP startup query interval',
      'QUERYINTERVAL' : matcherQueryInterval,
   }
   handler = IgmpCli.cmdIgmpStartupQueryInterval
   noOrDefaultHandler = IgmpCli.cmdNoIgmpStartupQueryInterval

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpStartupQueryIntervalcmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp static-group GROUPADDR [ SOURCEADDR ]
#--------------------------------------------------------------------------------
class IpIgmpStaticGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp static-group GROUPADDR [ SOURCEADDR ]'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'static-group' : 'Configured IGMP group membership',
      'GROUPADDR' : IpAddrMatcher( 'Multicast group address' ),
      'SOURCEADDR' : IpAddrMatcher( 'IPv4 address used as a source' ),
   }
   handler = IgmpCli.cmdIgmpStaticGroup
   noOrDefaultHandler = IgmpCli.cmdNoIgmpStaticGroup

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpStaticGroupCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp static-group acl ACLNAME
#--------------------------------------------------------------------------------
class IpIgmpStaticGroupAclCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp static-group acl ACLNAME'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'static-group' : 'Configured IGMP group membership',
      'acl' : 'IP ACL for use as static group list',
      'ACLNAME' : ipAclNameMatcher,
   }
   handler = IgmpCli.cmdIgmpStaticGroupAcl
   noOrDefaultHandler = IgmpCli.cmdNoIgmpStaticGroupAcl

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpStaticGroupAclCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp router-alert ( mandatory ) | ( optional [ connected ] )
#--------------------------------------------------------------------------------
class IpIgmpRouterAlertCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp router-alert ( mandatory ) | ( optional [ connected ] )'
   noOrDefaultSyntax = 'ip igmp router-alert ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'router-alert' : 'IGMP router-alert options',
      'mandatory' : 'Packets without router-alert would be dropped',
      'optional' : 'Packets with or without router-alert will be processed',
      'connected' : 'Processed only if they are from local subnet',
   }
   handler = IgmpCli.cmdIgmpRouterAlertConfigMandatory
   noOrDefaultHandler = IgmpCli.cmdNoIgmpRouterAlertConfig

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpRouterAlertCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp static-group GROUP num-groups GROUPNUM SOURCE 
#--------------------------------------------------------------------------------
class IpIgmpStaticNumGroupsCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp static-group GROUP num-groups GROUPNUM SOURCE'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'static-group' : 'Configured IGMP group membership',
      'GROUP' : IpAddrMatcher( 'Multicast group address' ),
      'num-groups' : 'Number of multicast groups',
      'GROUPNUM' : matcherNumGroups,
      'SOURCE' : IpAddrMatcher( 'IPv4 address used as a source' ),
   }
   hidden = True
   handler = IgmpCli.cmdIpIgmpStaticNumGroups
   noOrDefaultHandler = IgmpCli.cmdNoIpIgmpStaticNumGroups

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpStaticNumGroupsCmd )

#--------------------------------------------------------------------------------
# [no|default] ip igmp static-group range ( GROUP | GROUPADDR ) 
# [ source ( SOURCE | SOURCEADDR ) ]
#--------------------------------------------------------------------------------
class IpIgmpStaticGroupsCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip igmp static-group range ( GROUP | GROUPADDR ) '
              '[ source ( SOURCE | SOURCEADDR ) ]' )
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'static-group' : 'Configured IGMP group membership',
      'range' : 'Range of multicast groups',
      'GROUP' : IpAddrMatcher( 'Multicast group address' ),
      'GROUPADDR' : matcherGroupRange,
      'source' : 'IP address of the source',
      'SOURCE' : IpAddrMatcher( 'IPv4 address used as a source' ),
      'SOURCEADDR' : matcherSourceRange,
   }
   handler = IgmpCli.cmdIpIgmpStaticGroupRange
   noOrDefaultHandler = IgmpCli.cmdNoIpIgmpStaticGroupRange

RoutingProtocolIntfConfigModelet.addCommandClass( IpIgmpStaticGroupsCmd )
