#/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=C0321
import IgmpCliModel, GmpCli
import CliParser, BasicCli, IraIpCli, IraIpRouteCliLib, IntfCli
import CliParserCommon
import Tac, LazyMount
import ConfigMount
import McastCommonCliLib
import Arnet
from IgmpCliModel import IgmpInterfaces, IgmpGroups, IgmpGroupsCount
from IgmpCliModel import IgmpStatistics
from IgmpCliModel import IgmpMembership
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import vrfExists, DEFAULT_VRF, RESERVED_VRF_NAMES
import MrouteCli
import SharedMem
import Smash
from CliMode.Igmp import RoutingIgmpMode, RoutingIgmpVrfMode
import AclLib, AclCliLib, AclCli
import sys
import Toggles.IgmpToggleLib
import CliCommand
import CliMatcher

IpAddress = IpAddrMatcher.Arnet.IpAddress

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
SourceGroup = Tac.Type( 'Routing::Igmp::SourceGroup' )
GroupIntfKey = Tac.Type( 'Igmp::Smash::GroupIntfKey' )
AclAction = Tac.Type( 'Acl::Action' )

igmpConfig = None
igmpClearConfigColl = None
igmpStatus = None
gmpStatusColl = {}
igmpClearStatusColl = None
mfibVrfConfig = None
ipConfig = None
ipStatus = None
_entityManager = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None

def ipIgmpSupportedGuard( mode=None, token=None ):
   if McastCommonCliLib.mcastRoutingSupported( _entityManager.root(),
                                               routingHardwareStatus=None ):
      return None
   return CliParser.guardNotThisPlatform

def _igmpVrfDefinitionHook( vrfName ):
   if vrfName not in igmpClearConfigColl.vrfClearConfig:
      c = igmpClearConfigColl.newVrfClearConfig( vrfName )
      assert c is not None

def _igmpVrfDeletionHook( vrfName ):
   if vrfName in igmpClearConfigColl.vrfClearConfig and \
         vrfName != DEFAULT_VRF:
      del igmpClearConfigColl.vrfClearConfig[ vrfName ]

def applyMask( ipAddr, mask, base=True ):
   """Returns a specific IP Address resulting from applying the
   32-bit mask to the given IP Address ipAddr.
   If base is true, returns the all-zeros address of the mask,
   otherwise returns the all ones address of the mask."""
   if base:
      return IpAddress( 0xFFFFFFFF & ( ipAddr.value & mask ) )
   else:
      return IpAddress( 0xFFFFFFFF & ( ipAddr.value | ~mask ) )

def getSpecifiedOrSessionVrf( mode, args ):
   vrf = args.get( 'VRF' )
   if vrf:
      if not vrfExists( vrf ):
         mode.addError( 'Invalid vrf name %s ' % vrf )
         raise CliParserCommon.AlreadyHandledError
   else:
      # User didn't specify a VRF. If session has no VRF, DEFAULT_VRF is returned.
      vrf = VrfCli.vrfMap.getCliSessVrf( mode.session )

   return vrf

def isMulticast( addr ):
   addrStr = addr if type( addr ) is str else str( addr )
   return IpAddrMatcher.validateMulticastIpAddr( addrStr , False ) is None

def isUnicast( addr ):
   addrStr = addr if type( addr ) is str else str( addr )
   return IpAddrMatcher.isValidIpAddr( addrStr ) and \
         not isMulticast( addrStr )

def forceMountIfNeed( mnt ):
   if type( mnt ) == LazyMount._Proxy:
      LazyMount.force( mnt )

def createIgmpIntfConfig( _igmpConfig, intfName ):
   intfConfig = _igmpConfig.newIntfConfig( intfName )
   # Set the defaults explicitly as they may be different in Igmp vs in Gmp
   intfConfig.gmpQuerierConfig = ( intfName, )
   intfConfig.gmpQuerierConfig.querierVersion = intfConfig.versionDefault
   intfConfig.gmpQuerierConfig.lastMemberQueryCount = \
         intfConfig.lastMemberQueryCountDefault
   intfConfig.gmpQuerierConfig.lastMemberQueryInterval = \
         intfConfig.lastMemberQueryIntervalDefault
   intfConfig.gmpQuerierConfig.queryResponseInterval = \
         intfConfig.queryResponseIntervalDefault
   intfConfig.gmpQuerierConfig.queryInterval = \
         intfConfig.queryIntervalDefault
   intfConfig.gmpQuerierConfig.startupQueryCount = \
         intfConfig.startupQueryCountDefault
   intfConfig.gmpQuerierConfig.startupQueryInterval = \
         intfConfig.startupQueryIntervalDefault
   intfConfig.gmpQuerierConfig.addRandomDelayToTimer = True
   return intfConfig

def _getOrCreateIgmpIntfConfig( mode, printError=True ):
   if mode.intf.name not in igmpConfig.intfConfig:
      def createFcn():
         return createIgmpIntfConfig( igmpConfig, mode.intf.name )

      McastCommonCliLib.createMcastIntfConfig( mode,
            VrfCli.getVrfNameFromIntfConfig( ipConfig, mode.intf.name ),
            AddressFamily.ipv4, mode.intf.name, createFcn, printError )
   return igmpConfig.intfConfig.get( mode.intf.name )

def _deleteInfoConfigIfAllDefault( mode ):
   intfName = mode.intf.name
   ic = igmpConfig.intfConfig.get( intfName )
   if not ic:
      return
   if( ( ic.enabled == ic.enabledDefault ) and
       ( ic.implicitlyEnabled == ic.enabledDefault ) and
       ( ic.gmpQuerierConfig.querierVersion == ic.versionDefault ) and
       ( ic.gmpQuerierConfig.lastMemberQueryCount == \
             ic.lastMemberQueryCountDefault ) and
       ( ic.gmpQuerierConfig.lastMemberQueryInterval == \
             ic.lastMemberQueryIntervalDefault ) and
       ( ic.gmpQuerierConfig.queryResponseInterval == \
             ic.queryResponseIntervalDefault ) and
       ( ic.gmpQuerierConfig.queryInterval == ic.queryIntervalDefault ) and
       ( ic.gmpQuerierConfig.startupQueryCount == ic.startupQueryCountDefault ) and
       ( ic.gmpQuerierConfig.startupQueryInterval == \
             ic.startupQueryIntervalDefault ) and
       ( ic.routerAlertConfigOption == ic.routerAlertConfigOptionDefault ) and
       ( len( ic.staticJoinSourceGroup ) == 0 ) and
       ( len( ic.staticJoinAcl ) == 0 ) and
       ( ic.localIntfId == '' ) ):
      del igmpConfig.intfConfig[ intfName ]

def _getGmpStatus( vrfName ):
   if vrfName not in gmpStatusColl:
      if vrfExists( vrfName, igmpClearConfigColl.vrfClearConfig ):
         shmemEm = SharedMem.entityManager( sysdbEm=_entityManager )
         gmpStatusColl[ vrfName ] = \
               shmemEm.doMount( 'routing/gmp/status/' + vrfName,
                                'Routing::Gmp::Smash::Status',
                                Smash.mountInfo( 'reader' ) )

   return gmpStatusColl.get( vrfName )

##### # show ip igmp  #####
#
# show ip igmp groups
#
# show ip igmp interface
#
# show ip igmp static-groups [detail] [interface <intf>]
#
# show ip igmp static-groups acl <name>
#
# show ip igmp statistics
#
# show ip igmp membership
#
###########################

#------------------------------------------------------------------------------------
# show ip igmp groups
#------------------------------------------------------------------------------------

def _getIgmpSmashStatus():
   shmemEm =  SharedMem.entityManager( sysdbEm=_entityManager, )
   return shmemEm.doMount( 'routing/igmp/status', 'Igmp::Smash::Status',
                           Smash.mountInfo( 'reader' ) )

def cmdShowIpIgmpGroups( mode, args ):
   ''' Implementation for show ip igmp groups '''
   vrfName = getSpecifiedOrSessionVrf( mode, args )
   detail = 'detail' in args
   igmpGroups = IgmpGroups()

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return igmpGroups

   igmpSmashStatus = _getIgmpSmashStatus()
   if igmpSmashStatus is None:
      return igmpGroups

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   forceMountIfNeed( ipStatus )
   showGroups = Tac.newInstance( "IgmpCapiModel::ShowIpIgmpGroups",
                                    gmpStatus, igmpSmashStatus,
                                    ipStatus )
   showGroups.render( fd, fmt, detail, Tac.Value( "Arnet::IntfId" ),
         Tac.Value( "Arnet::IpAddr" ) )
   return IgmpGroups

def cmdShowIpIgmpGroupsCount( mode, args ):
   ''' Implementation for show ip igmp groups count '''
   vrfName = getSpecifiedOrSessionVrf( mode, args )
   countModel = IgmpGroupsCount()
   count = 0
   countModel.count = count

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return countModel

   igmpSmashStatus = _getIgmpSmashStatus()
   if igmpSmashStatus is None:
      return countModel

   for giKey, _ in igmpSmashStatus.groupRecord.iteritems():
      if giKey.intfId in gmpStatus.intfStatus:
         count += 1
   countModel.count = count
   return countModel

def cmdShowIpIgmpGroupsCountIntf( mode, args ):
   ''' Implementation for show ip igmp groups count interface <intf>'''
   intf = args[ 'INTF' ]
   vrfName = getSpecifiedOrSessionVrf( mode, args )
   countModel = IgmpGroupsCount()
   count = 0
   countModel.count = count

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return countModel

   igmpSmashStatus = _getIgmpSmashStatus()
   if igmpSmashStatus is None:
      return countModel

   intfId = intf.name

   if intf and gmpStatus.intfStatus.get( intf.name ):
      if intfId in igmpStatus.intfStatus.keys():
         for giKey in igmpSmashStatus.groupRecord.keys():
            if giKey.intfId == intfId:
               count += 1
   countModel.count = count
   countModel.interfaceName = intf.name
   return countModel

def cmdShowIpIgmpGroupsInterface( mode, args ):
   ''' Implementation for show ip igmp groups interface '''
   intf = args[ 'INTF' ]
   detail = 'detail' in args
   vrfName = getSpecifiedOrSessionVrf( mode, args )
   igmpGroups = IgmpGroups()
   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return igmpGroups

   igmpSmashStatus = _getIgmpSmashStatus()
   if igmpSmashStatus is None:
      return igmpGroups

   igmpGroups._detail = detail
   igmpGroups.groupList = []
   intfId = intf.name

   igmpIntfStatus = None
   igmpIntfStatus = igmpStatus.intfStatus.get( intf.name )

   if not igmpIntfStatus:
      return igmpGroups

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   forceMountIfNeed( ipStatus )
   showGroups = Tac.newInstance( "IgmpCapiModel::ShowIpIgmpGroups",
                                    gmpStatus, igmpSmashStatus,
                                    ipStatus )
   showGroups.render( fd, fmt, detail, intfId, Tac.Value( "Arnet::IpAddr" ))
   return IgmpGroups

def cmdShowIpIgmpGroupsAddress( mode, args ):
   ''' Implementation for show ip igmp groups '''
   groupAddr = args[ 'GROUPADDR' ]
   detail = 'detail' in args
   vrfName = getSpecifiedOrSessionVrf( mode, args )
   igmpGroups = IgmpGroups()

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return igmpGroups

   igmpSmashStatus = _getIgmpSmashStatus()
   if igmpSmashStatus is None:
      return igmpGroups

   igmpGroups._detail = detail
   igmpGroups.groupList = []

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   forceMountIfNeed( ipStatus )

   showGroups = Tac.newInstance( "IgmpCapiModel::ShowIpIgmpGroups",
                                    gmpStatus, igmpSmashStatus,
                                    ipStatus )
   showGroups.render( fd, fmt, detail,
         Tac.Value( "Arnet::IntfId" ), groupAddr )
   return IgmpGroups

#------------------------------------------------------------------------------------
# show ip igmp interface
#------------------------------------------------------------------------------------
def cmdShowIpIgmpInterface( mode, args ):
   ''' This implements show ip igmp interface '''
   intf = args.get( 'INTF' )
   vrfName = getSpecifiedOrSessionVrf( mode, args )
   igmpInterfaces = IgmpInterfaces()

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return igmpInterfaces

   igmpSmashStatus = _getIgmpSmashStatus()
   # Error handling
   if intf:
      intfId = intf.name
      config = igmpConfig.intfConfig.get( intfId )
      status = igmpStatus.intfStatus.get( intfId )
      if intfId not in gmpStatus.intfStatus.keys() or not config or not status :
         mode.addError( "%s: No IGMP configuration found in VRF %s"
                        % ( intf.name, vrfName ) )
      elif not status.enabled :
         mode.addWarning( "%s: IGMP is configured or implicitly enabled by another "
                        "agent but is effectively disabled" % intf.name )
   else:
      for intfId in gmpStatus.intfStatus.keys():
         if intfId in igmpStatus.intfStatus.keys():
            if not igmpStatus.intfStatus[ intfId ].enabled :
               mode.addWarning( "%s: IGMP is configured or implicitly enabled by "
                              "another agent but is effectively disabled" % intfId )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   forceMountIfNeed( igmpStatus )
   forceMountIfNeed( igmpConfig )
   forceMountIfNeed( ipStatus )
   forceMountIfNeed( mfibVrfConfig )

   showInterface = Tac.newInstance( "IgmpCapiModel::ShowIpIgmpInterface",
                                    gmpStatus, igmpSmashStatus, igmpConfig,
                                    igmpStatus, ipStatus, mfibVrfConfig )

   if intf:
      intfId = Tac.Value( "Arnet::IntfId", intf.name )
   else:
      intfId = Tac.Value( "Arnet::IntfId", "" )
   showInterface.render( fd, fmt, intfId )
   return IgmpInterfaces

#------------------------------------------------------------------------------------
# show ip igmp static-groups acl
#------------------------------------------------------------------------------------

# returns the allowed number of statically-configured groups through igmp
# keeping this separate, so that it can be changed at this single place in future
def allowedNumberOfStaticGroups():
   return 65536

# examines the fitness of an acl IpRuleConfig as a SourceGroup range.
def checkSourceGroupRule( rule ):
   if rule.action == AclAction.permit:
      # Ip addr/prefix len
      destination = rule.filter.destination
      destAddr = IpAddress( destination.address )
      destmask = 0xFFFFFFFF & destination.mask
      destmin = applyMask( destAddr, destmask )
      destmax = applyMask( destAddr, destmask, base=False )
      if not( isMulticast( destmin ) and isMulticast( destmax )  ) :
         return ( False, "group address range contains invalid multicast addresses" )
      # check if the range is too large i.e. if the number of groups > allowedNumber
      if ( abs( destmax.value - destmin.value ) > allowedNumberOfStaticGroups() ):
         return ( False, "range for static-group too big," +
               " should be less than %d" % allowedNumberOfStaticGroups() )
      source = rule.filter.source
      srcAddr =  IpAddress( source.address )
      srcmask =  0xFFFFFFFF & source.mask
      if srcmask == 0:
         if srcAddr.value != 0:
            return ( False, "source address is not a valid unicast address" )
         else: return ( True, "( 0.0.0.0, %s )" % destination.stringValue )
      elif srcmask == 0xFFFFFFFF:
         if not isUnicast( srcAddr ):
            return ( False, "source address is not a valid unicast address" )
         else: return ( True, "( %s, %s )" %
                       ( srcAddr, destination.stringValue ) )
      else:
         return ( False, "source address must be a single host or *, not a range" )
   else:
      return ( False, "action must be 'permit'" )

# examines if total range of groups < allowedNumber in all rules of acl combined
def checkNumberOfGroupsInAcl( aclName ):
   totalGroups = 0
   acl = aclConfig.config['ip'].acl[ aclName ].currCfg
   for item in acl.ruleBySequence.iteritems():
      uid = item[ 1 ]
      rule = acl.ipRuleById[ uid ]
      destination = rule.filter.destination
      destAddr = IpAddress( destination.address )
      destmask = 0xFFFFFFFF & destination.mask
      destmin = applyMask( destAddr, destmask )
      destmax = applyMask( destAddr, destmask, base=False )
      countInCurrentRule = abs( destmax.value - destmin.value )
      totalGroups += countInCurrentRule

   if( totalGroups > allowedNumberOfStaticGroups() ):
      return ( False, "total number of static-groups in acl too big," \
                       " should be less than %d" % allowedNumberOfStaticGroups() )

   return ( True, "%s has a valid number of groups" % aclName )

def formatStaticGroupAcl( acl, check=True ):
   acls = IgmpCliModel.IgmpAclInfo()
   for seq, uid in acl.currCfg.ruleBySequence.iteritems():
      rule = acl.currCfg.ipRuleById[ uid ]
      valid, res = checkSourceGroupRule( rule )
      aclInfo = IgmpCliModel.AclInfo()
      if check and not valid:
         aclInfo.sequence = seq
         aclInfo.sourceGroupInfo = res
      elif valid:
         aclInfo.sourceGroupInfo = res
      acls.aclDetails.append( aclInfo )
   return acls

def cmdShowIpIgmpStaticGroupsAcl( mode, args ):
   name = args[ 'ACLNAME' ]
   vrfName = getSpecifiedOrSessionVrf( mode, args )

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return 

   if not name or name not in aclConfig.config['ip'].acl:
      print "ACL '%s' not configured" % name
      return
   valid = checkNumberOfGroupsInAcl( name )[ 0 ]
   if not valid:
      print "\tTotal number of static groups in %s too big" % name
   acl = aclConfig.config['ip'].acl[ name ]
   print '\n'.join( formatStaticGroupAcl( acl ).getFormat() )
   print 'Interfaces using this ACL for static groups:'
   intfsConfigured = False
   for intfName in gmpStatus.intfStatus.keys():
      if intfName in igmpConfig.intfConfig.keys():
         intf = igmpConfig.intfConfig[ intfName ]
         if name in intf.staticJoinAcl:
            print '\t%s' % intfName
            intfsConfigured = True
   if not intfsConfigured:
      print '\t none'

def cmdShowIpIgmpStaticGroupsGroup( mode, args ):
   group = args[ 'group' ]
   grpAddress = args.get( 'GROUPADDR' )
   vrfName = getSpecifiedOrSessionVrf( mode, args )

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return 

   groupHash = IgmpCliModel.IgmpSourceGroups()
   # if 'group' is specified, then iterate over all interfaces and
   # build a hash of various interfaces in source groups
   if group:
      # Build the mapping
      for intfName in gmpStatus.intfStatus:
         if intfName in igmpConfig.intfConfig:
            intf = igmpConfig.intfConfig[ intfName ]
            for sourceGroup in intf.staticJoinSourceGroup:
               sGroupAddr = str( sourceGroup.groupAddr )

               if grpAddress and not grpAddress in sGroupAddr:
                  continue

               if sGroupAddr not in groupHash.sourceGroups.keys():
                  groupHash.sourceGroups[ sGroupAddr ] = \
                        IgmpCliModel.IgmpInterfacesList()

               groupHash.sourceGroups[ sGroupAddr ].interfaces.append( intf.intfId )
   return groupHash

#------------------------------------------------------------------------------------
# show ip igmp static-groups [detail] [interface <intf>]
# show ip igmp static-groups [group [ipaddr]]
#------------------------------------------------------------------------------------
def cmdShowIpIgmpStaticGroups( mode, args ):
   detail = 'detail' in args
   interface = args.get( 'INTF' )
   vrfName = getSpecifiedOrSessionVrf( mode, args )

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return 

   staticGroupIntfAddrs = IgmpCliModel.IgmpStaticGroups()
   intfs = None
   if interface:
      if interface.name in gmpStatus.intfStatus.keys() and \
         interface.name in igmpConfig.intfConfig:
         intfs = [ igmpConfig.intfConfig[ interface.name ] ]
      else:
         mode.addError( "Interface %s: no IGMP static groups configured"
                        % interface.name )
         return staticGroupIntfAddrs
   elif interface is None:
      intfs = []
      for intfName in gmpStatus.intfStatus.keys():
         if intfName in igmpConfig.intfConfig.keys():
            intfs.append( igmpConfig.intfConfig[ intfName ] )

   # Filter out the disabled interfaces
   for intf in intfs:
      igmpIntfStatus = igmpStatus.intfStatus.get( intf.intfId )
      if not igmpIntfStatus or not igmpIntfStatus.enabled:
         intfs.remove( intf )

   for intf in intfs:
      # create list for this interface
      intfInfo = IgmpCliModel.IgmpStaticGroupsInterface()
      if detail:
         for aclName in intf.staticJoinAcl:
            if not aclName in aclConfig.config[ 'ip' ].acl:
               intfInfo.aclInfo[ aclName ] = IgmpCliModel.IgmpAclInfo(
                        errorMessage="acl %s not configured" % aclName )
               continue
            acl = aclConfig.config[ 'ip' ].acl[ aclName ]
            aclInfo = formatStaticGroupAcl( acl, check=False )
            intfInfo.aclInfo[ aclName ] = aclInfo
      else:
         for aclName in intf.staticJoinAcl:
            if not aclName in aclConfig.config[ 'ip' ].acl:
               intfInfo.aclInfo[ aclName ] = IgmpCliModel.IgmpAclInfo(
                        errorMessage="acl %s not configured" % aclName )
               continue
            intfInfo.aclInfo[ aclName ] = IgmpCliModel.IgmpAclInfo()

      # create list for this interface
      for sourceGroup in intf.staticJoinSourceGroup:
         # store source-group address pair for interface
         addrs = IgmpCliModel.InterfaceAddrs( sourceAddr=sourceGroup.sourceAddr,
                                              groupAddr=sourceGroup.groupAddr )
         intfInfo.groupAddrsList.append( addrs )
      # add intfInfo for this interface to dictionary
      staticGroupIntfAddrs.intfAddrs[ intf.intfId ] = intfInfo
   return staticGroupIntfAddrs

#------------------------------------------------------------------------------------
# show ip igmp statistics
#------------------------------------------------------------------------------------
def igmpStatisticsFromStatus( intfStatus, intfId ):
   ''' This utility creates a IgmpStatisticsInterface Model from intfStatus '''
   intf = IgmpStatistics.IgmpStatisticsInterface()
   
   igmpSmashStatus = _getIgmpSmashStatus()
   if igmpSmashStatus is None:
      return intf
   igmpSmashQuerierStatistics = igmpSmashStatus.packetStatisticsRecord.get( intfId )
   if not igmpSmashQuerierStatistics:
      return intf

   intf.v1QueriesSent = igmpSmashQuerierStatistics.v1QueriesSent
   intf.v2QueriesSent = igmpSmashQuerierStatistics.v2QueriesSent
   intf.v3QueriesSent = igmpSmashQuerierStatistics.v3QueriesSent
   intf.v3GSQueriesSent = igmpSmashQuerierStatistics.v3GSQueriesSent
   intf.v3GSSQueriesSent = igmpSmashQuerierStatistics.v3GSSQueriesSent
   intf.v1QueriesReceived = igmpSmashQuerierStatistics.v1QueriesReceived
   intf.v2QueriesReceived = igmpSmashQuerierStatistics.v2QueriesReceived
   intf.v3QueriesReceived = igmpSmashQuerierStatistics.v3QueriesReceived
   intf.v1ReportReceived = igmpSmashQuerierStatistics.v1ReportReceived
   intf.v2ReportReceived = igmpSmashQuerierStatistics.v2ReportReceived
   intf.v2LeaveReceived = igmpSmashQuerierStatistics.v2LeaveReceived
   intf.v3ReportReceived = igmpSmashQuerierStatistics.v3ReportReceived
   intf.totalGeneralQueriesSent = igmpSmashQuerierStatistics.totalGeneralQueriesSent
   intf.errorPacketReceived = igmpSmashQuerierStatistics.errorPacketReceived
   intf.otherPacketReceived = igmpSmashQuerierStatistics.otherPacketReceived
   return intf

def cmdShowIpIgmpStatistics( mode, args ):
   ''' This implements show ip igmp interface '''
   intf = args.get( 'INTF' )
   vrfName = getSpecifiedOrSessionVrf( mode, args )
   igmpStatistics = IgmpStatistics()

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return igmpStatistics

   igmpSmashStatus = _getIgmpSmashStatus()
   if igmpSmashStatus is None:
      return igmpStatistics

   if intf:
      intfId = intf.name
      config = igmpConfig.intfConfig.get( intfId )
      status = igmpStatus.intfStatus.get( intfId )

      if intfId not in gmpStatus.intfStatus.keys() or not config or not status \
            or not status.enabled or \
            ( intfId not in igmpSmashStatus.packetStatisticsRecord.keys() ):
         mode.addError( "%s: No IGMP statistics found" % intf.name )
      else :
         igmpStatisticsModel = igmpStatisticsFromStatus(
               igmpStatus.intfStatus[ intfId ], intfId )
         igmpStatistics.IgmpStatistics = { intfId : igmpStatisticsModel }
   else:
      for intfId in gmpStatus.intfStatus.keys():
         if intfId in igmpStatus.intfStatus.keys():
            if igmpStatus.intfStatus[ intfId ].enabled and \
                  intfId in igmpSmashStatus.packetStatisticsRecord.keys() :
               igmpStatisticsModel = igmpStatisticsFromStatus(
                     igmpStatus.intfStatus[ intfId ], intfId )
               igmpStatistics.IgmpStatistics[ intfId ] = igmpStatisticsModel
   return igmpStatistics

#------------------------------------------------------------------------------------
# show ip igmp membership
#------------------------------------------------------------------------------------
def gmpGroupFromStatus( intfId, groupStatus, includeSrcs, excludeSrcs ):
   ''' Generate a model based on status information for group and interface '''
   gmpGroupModel = IgmpCliModel.IgmpMemGrp()
   gmpGroupModel.interfaceName = intfId
   gmpGroupModel.groupAddress = groupStatus.groupAddr
   gmpGroupModel.includeSrcs = includeSrcs
   gmpGroupModel.excludeSrcs = excludeSrcs
   return gmpGroupModel

def cmdShowIpIgmpMembership( mode, args ):
   ''' This implements show ip igmp membership \
         [interface <intf>] [group <groupAddr>] '''
   intf = args.get( 'INTF' )
   groupAddr = args.get( 'GROUPADDR' )
   vrfName = getSpecifiedOrSessionVrf( mode, args )

   gmpStatus = _getGmpStatus( vrfName )
   if not gmpStatus:
      return

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   showMembership = Tac.newInstance(
         "IgmpCapiModel::ShowIpIgmpMembership", gmpStatus );
   if intf:
      intfId = Tac.Value( "Arnet::IntfId", intf.name )
   else:
      intfId = Tac.Value( "Arnet::IntfId", "" )
   if groupAddr:
      grp = Tac.Value( "Arnet::IpGenAddr", groupAddr )
   else:
      grp = Tac.Value( "Arnet::IpGenAddr", "" )
   showMembership.render( fd, fmt, intfId, grp )
   return IgmpMembership

##### # clear ip igmp ... #####
#
# clear ip igmp group
#
# clear ip igmp statistics [ interface <intf> ]
#
###############################

#------------------------------------------------------------------------------------
# clear ip igmp group
#------------------------------------------------------------------------------------
def waitForGroupsToClear( mode, vrfName, igmpClearConfig ):
   try:
      Tac.waitFor(
         lambda:
            igmpClearStatusColl.vrfClearStatus[ vrfName ].lastGrpClearTime \
                  >= igmpClearConfig.clearGrpRequestTime,
         description='Groups to get cleared.',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=20 )
   except Tac.Timeout:
      mode.addWarning(
            "Igmp Agent did not respond within 20 seconds. It could \
                  either be busy clearing groups or has stoppped running." )

def cmdClearIpIgmpGroup( mode, args ):
   ''' Implementation for clear ip igmp group <grpAddr> interface <intf>'''
   groupAddr = args.get( 'GROUPADDR' )
   intf = args.get( 'INTF' )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if type( igmpClearConfigColl ) == LazyMount._Proxy:
      # force a mount, Tac.newInstance does not like unmounted proxies
      LazyMount.force( igmpClearConfigColl )
   if type( igmpClearStatusColl ) == LazyMount._Proxy:
      LazyMount.force( igmpClearStatusColl )
   elif not vrfExists( vrfName, igmpClearConfigColl.vrfClearConfig ):
      mode.addError( "Invalid vrf name %s " % vrfName )
      return

   if groupAddr and not isMulticast( groupAddr ):
      mode.addError( "Group range contains invalid multicast addresses" )
      return

   if vrfName in igmpClearConfigColl.vrfClearConfig and \
      vrfName in igmpClearStatusColl.vrfClearStatus:
      igmpClearConfig = igmpClearConfigColl.vrfClearConfig[ vrfName ]
      igmpClearConfig.gAddrToClear = groupAddr or '0.0.0.0'
      if intf:
         igmpClearConfig.intfIdToClear = intf.name
      else:
         igmpClearConfig.intfIdToClear = Tac.newInstance( "Arnet::IntfId" )

      igmpClearConfig.clearGrpRequestTime = Tac.now()
      waitForGroupsToClear( mode, vrfName, igmpClearConfig )

def waitForStatisticsToClear( mode, vrfName, igmpClearConfig ):
   try:
      Tac.waitFor(
         lambda:
            igmpClearStatusColl.vrfClearStatus[ vrfName ].lastStatsClearTime \
                  >= igmpClearConfig.clearStatsRequestTime,
         description='Counters to get cleared.',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      mode.addWarning(
            "Igmp Counters may not have been cleared yet. Is Igmp Agent running?" )

def cmdClearIpIgmpStatistics( mode, args ):
   ''' Implementation for clear ip igmp group '''
   intf = args.get( 'INTF' )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   # pylint: disable-msg=W0212
   if type( igmpClearConfigColl ) == LazyMount._Proxy:
      # force a mount, Tac.newInstance does not like unmounted proxies
      LazyMount.force( igmpClearConfigColl )
   # pylint: disable-msg=W0212
   if type( igmpStatus ) == LazyMount._Proxy:
      LazyMount.force( igmpStatus )
   elif not vrfExists( vrfName, igmpClearConfigColl.vrfClearConfig ):
      mode.addError( "Invalid vrf name %s " % vrfName )
      return

   if igmpClearConfigColl.vrfClearConfig[ vrfName ] is not None and \
      vrfName in igmpClearStatusColl.vrfClearStatus:
      igmpClearConfig = igmpClearConfigColl.vrfClearConfig[ vrfName ]
      if intf:
         igmpClearConfig.intfIdToClear = intf.name
      else:
         igmpClearConfig.intfIdToClear = Tac.newInstance( "Arnet::IntfId" )
      igmpClearConfig.clearStatsRequestTime = Tac.now()
      waitForStatisticsToClear( mode, vrfName, igmpClearConfig )

#------------------------------------------------------------------------------------
# clear ip igmp access-list counters
#------------------------------------------------------------------------------------
def clearIpAclCounters( mode, args ):
   AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, 'ip' )

##### (config-if)# ip igmp ... #####
#
# [no|default] ip igmp
#
# [no|default] ip igmp version { 1 | 2 | 3 }
#
# [no|default] ip igmp last-member-query-count < count >
#
# [no|default] ip igmp last-member-query-interval < time >
#
# [no|default] ip igmp query-max-response-time < time >
#
# [no|default] ip igmp query-interval < time >
#
# [no|default] ip igmp startup-query-count < count >
#
# [no|default] ip igmp startup-query-interval < time >
#
# [no|default] ip igmp static-group < group > { source }
#
# [no|default] ip igmp static-group acl < name >
#
# [no|default] ip igmp static-group range < group/prefix > [ source < source > ]
#
# [no|default] ip igmp static-group range < group > source < source/prefix >
#
####################################

#------------------------------------------------------------------------------------
# helper function to implicitly enable igmp from other multicast protocol
#------------------------------------------------------------------------------------
def implicitlyEnabledIs( mode, on ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if on:
      intfConfig.implicitlyEnabled = True
   else:
      intfConfig.implicitlyEnabled = False
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# helper function to implicitly use local-interface from PIM
#------------------------------------------------------------------------------------
def igmpLocalInterfaceIs( mode, localIntfId ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if localIntfId == '':
      intfConfig.localIntfId = localIntfId
      _deleteInfoConfigIfAllDefault( mode )
   else:
      intfConfig.localIntfId = localIntfId

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp
#------------------------------------------------------------------------------------
def cmdIgmp( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.enabled = True

def cmdNoIgmp( mode, args ):
   intfConfig = igmpConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      intfConfig.enabled = False
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp version { 2 | 3 }
#------------------------------------------------------------------------------------
def cmdIgmpVersion( mode, args ):
   version = args[ 'VERSION' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.gmpQuerierConfig.querierVersion = 'igmpVersion%d' % version
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpVersion( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      intfConfig.gmpQuerierConfig.querierVersion = intfConfig.versionDefault
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp last-member-query-count
#------------------------------------------------------------------------------------
def cmdIgmpLastMemberQueryCount( mode, args ):
   count = args[ 'COUNT' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.gmpQuerierConfig.lastMemberQueryCount = count
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpLastMemberQueryCount( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      intfConfig.gmpQuerierConfig.lastMemberQueryCount = \
            intfConfig.lastMemberQueryCountDefault
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp last-member-query-interval { deciseconds }
#------------------------------------------------------------------------------------

# There is a factor of 10 between the config cli input ( deciseconds ) to the
# value stored in the Igmp tac model ( seconds )
def cmdIgmpLastMemberQueryInterval( mode, args ):
   interval = args[ 'QUERYINTERVAL' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.gmpQuerierConfig.lastMemberQueryInterval = float( interval ) / 10.0
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpLastMemberQueryInterval( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      intfConfig.gmpQuerierConfig.lastMemberQueryInterval = \
            intfConfig.lastMemberQueryIntervalDefault
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp query-max-response-time { deciseconds }
#------------------------------------------------------------------------------------

# There is a factor of 10 between the config cli input ( deciseconds ) to the value
# stored in the Igmp tac model ( seconds )
def cmdIgmpQueryMaxResponseTime( mode, args ):
   time = args[ 'TIME' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.gmpQuerierConfig.queryResponseInterval = float( time ) / 10.0
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpQueryMaxResponseTime( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      intfConfig.gmpQuerierConfig.queryResponseInterval = \
            intfConfig.queryResponseIntervalDefault
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp query-interval { seconds }
#------------------------------------------------------------------------------------
def cmdIgmpQueryInterval( mode, args ):
   interval = args[ 'INTERVAL' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.gmpQuerierConfig.queryInterval = interval
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpQueryInterval( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      intfConfig.gmpQuerierConfig.queryInterval = intfConfig.queryIntervalDefault
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp startup-query-count
#------------------------------------------------------------------------------------
def cmdIgmpStartupQueryCount( mode, args ):
   count = args[ 'QUERYCOUNT' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.gmpQuerierConfig.startupQueryCount = count
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpStartupQueryCount( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      intfConfig.gmpQuerierConfig.startupQueryCount = \
            intfConfig.startupQueryCountDefault
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp startup-query-interval
#------------------------------------------------------------------------------------

# There is a factor of 10 between the config cli input ( deciseconds )to the value
# stored in the Igmp tac model ( seconds )
def cmdIgmpStartupQueryInterval( mode, args ):
   interval = args[ 'QUERYINTERVAL' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.gmpQuerierConfig.startupQueryInterval = float( interval ) / 10.0
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpStartupQueryInterval( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      intfConfig.gmpQuerierConfig.startupQueryInterval = \
            intfConfig.startupQueryIntervalDefault
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp static-group
#------------------------------------------------------------------------------------
def cmdIgmpStaticGroup( mode, args ):
   groupAddr = args[ 'GROUPADDR' ]
   sourceAddr = args.get( 'SOURCEADDR', '0.0.0.0' )
   err = IpAddrMatcher.validateMulticastIpAddr( groupAddr , False )
   if err:
      mode.addError( err )
      return

   if sourceAddr and not isUnicast( sourceAddr ):
      mode.addError( "Source address is not a valid unicast host" )
      return

   sg = SourceGroup( sourceAddr, groupAddr )
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.staticJoinSourceGroup[ sg ] = True

def cmdNoIgmpStaticGroup( mode, args ):
   groupAddr = args[ 'GROUPADDR' ]
   sourceAddr = args.get( 'SOURCEADDR', '0.0.0.0' )
   sg = SourceGroup( sourceAddr, groupAddr )
   intfConfig = _getOrCreateIgmpIntfConfig( mode, printError=False )
   if intfConfig:
      del intfConfig.staticJoinSourceGroup[ sg ]
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp static-group acl
#------------------------------------------------------------------------------------
def checkAclValidAsClassmap( aclName ):
   acl = aclConfig.config['ip'].acl[ aclName ].currCfg
   countGroupsVal = checkNumberOfGroupsInAcl( aclName )
   if not countGroupsVal[ 0 ]:
      return ( False, "%s" % countGroupsVal[1] )
   for seq, uid in acl.ruleBySequence.iteritems():
      maybeRet = checkSourceGroupRule( acl.ipRuleById[ uid ] )
      if not maybeRet[0]:
         return ( False, "Seq no %d: %s" % ( seq, maybeRet[1] ) )
   return ( True, None )


def cmdIgmpStaticGroupAcl( mode, args ):
   name = args[ 'ACLNAME' ]
   if not name in aclConfig.config['ip'].acl:
      mode.addWarning( "Acl %s not configured. Assigning anyway."
                       % name )
      intfConfig = _getOrCreateIgmpIntfConfig( mode )
      if intfConfig:
         intfConfig.staticJoinAcl[ name ] = True
      return
   valid =  checkAclValidAsClassmap( name )
   if not valid[0]:
      mode.addError( "Acl %s contains rules invalid as static group specifications."
                     "\ntry 'show ip igmp static-group acl %s' for more information."
                     "\nError: %s " % ( name, name, valid[1] ) )
      return
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.staticJoinAcl[ name ] = True

def cmdNoIgmpStaticGroupAcl( mode, args ):
   name = args[ 'ACLNAME' ]
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig :
      del intfConfig.staticJoinAcl[ name ]
      _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# ip igmp static-group range <prefix> [ source <addr> ]
#------------------------------------------------------------------------------------
def _cmdIgmpStaticGroupRange( mode, group, action, source = '0.0.0.0' ):
   if group is None:
      mode.addError( "Group range mask must be contiguous" )
      return
   elif '/' in group:
      if source is None:
         mode.addError( "Source range mask must be contiguous" )
         return
      if '/' in source:
         mode.addError( "Cannot specify source range with group range" )
         return
      if not IraIpRouteCliLib.isValidPrefix( group ):
         mode.addError( "Masked bits of group range must be zero" )
         return
      groupAddr, groupLen = group.split( '/' )
      groupAddr = IpAddress( groupAddr )
      groupLen = int ( groupLen )
      if ( pow( 2, ( 32 - groupLen ) ) > allowedNumberOfStaticGroups() ):
         mode.addError( "Range for static-group too big," +
            " should be less than %d" % allowedNumberOfStaticGroups() )
         return
      m = ( 1 << ( 32 - groupLen ) )  # group mask + 1, name truncated for space
      addrMax = applyMask( groupAddr, ~( m - 1 ), False )
      if not ( isMulticast( groupAddr ) and isMulticast( addrMax ) ):
         mode.addError( "Group range contains invalid multicast addresses" )
         return
      sourceAddr = IpAddress( source )
      if not isUnicast( sourceAddr ):
         mode.addError( "Source address is not a valid unicast host" )
         return
      action( mode, sourceAddr, groupAddr, m, True )
   else : # group address is a single host
      groupAddr =  IpAddress( group )
      if not isMulticast( groupAddr ):
         mode.addError( "Group address is not a valid multicast address" )
         return
      if source is None:
         mode.addError( "Source range mask must be contiguous" )
         return
      if '/' in source:
         if not IraIpRouteCliLib.isValidPrefix( source ):
            mode.addError( "Masked bits of source range must be zero" )
            return
         sourceAddr, sourceLen = source.split( '/' )
         sourceAddr = IpAddress( sourceAddr )
         sourceLen = int ( sourceLen )
         m = ( 1 << ( 32 - sourceLen ) )
         sourceMax = applyMask( sourceAddr, ~( m - 1 ), False )
         if not ( isUnicast( sourceAddr ) and isUnicast( sourceMax ) ):
            mode.addError( "Source range contains invalid unicast addresses" )
            return
         action( mode, sourceAddr, groupAddr, m, False )
      else: #source address is a single host
         sourceAddr = IpAddress ( source )
         if not isUnicast( sourceAddr ):
            mode.addError( "Source address not a valid unicast host" )
            return
         action( mode, sourceAddr, groupAddr, 1 )

def cmdIpIgmpStaticGroupRange( mode, args ):
   group = args.get( 'GROUPADDR' ) or args.get( 'GROUP' )
   source= args.get( 'SOURCEADDR' ) or args.get( 'SOURCE' ) or '0.0.0.0'
   _cmdIgmpStaticGroupRange( mode, group, doAddSgs, source )

def cmdNoIpIgmpStaticGroupRange( mode, args ):
   group = args.get( 'GROUPADDR' ) or args.get( 'GROUP' )
   source= args.get( 'SOURCEADDR' ) or args.get( 'SOURCE' ) or '0.0.0.0'
   _cmdIgmpStaticGroupRange( mode, group, doDeleteSgs, source )

def doAddSgs( mode, saddr, gaddr, rangemax, grange=True ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if not intfConfig:
      return
   if grange:
      addrbase = gaddr.value
      for ga in ( IpAddress( addrbase | x ) for x in xrange( rangemax ) ):
         sg = SourceGroup( str( saddr ), str ( ga ) )
         intfConfig.staticJoinSourceGroup[ sg ] = True

   else:
      addrbase = saddr.value
      for sa in ( IpAddress( addrbase | x ) for x in xrange( rangemax ) ):
         sg = SourceGroup( str( sa ), str( gaddr ) )
         intfConfig.staticJoinSourceGroup[ sg ] = True

# This optimisation is not trivial. without checking the smaller
# input, attempting 'no ip igmp static-group range 224.0.0.0/8'
# takes over 45 minutes.
def doDeleteSgs( mode, saddr, gaddr, rangemax, grange=True ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if not intfConfig:
      return
   if grange: # range of group addresses

      if rangemax  < len( intfConfig.staticJoinSourceGroup ):
         addrbase = gaddr.value
         for ga in ( IpAddress( addrbase | x ) for x in xrange( rangemax ) ):
            sg = SourceGroup( str( saddr ), str( ga ) )
            del intfConfig.staticJoinSourceGroup[ sg ]

      else:
         mask = 0xFFFFFFFF & ~( rangemax - 1 )
         for sg in intfConfig.staticJoinSourceGroup.keys():
            sa, ga = IpAddress( sg.sourceAddr ), IpAddress( sg.groupAddr )
            if ( sa == saddr ) and \
                   ( ( ga.value & mask ) == ( gaddr.value & mask ) ):
               del intfConfig.staticJoinSourceGroup[ sg ]

   else: # range of source addresses
      if rangemax < len( intfConfig.staticJoinSourceGroup ):
         addrbase = saddr.value
         for sa in ( IpAddress( addrbase | x ) for x in xrange( rangemax ) ):
            sg = SourceGroup( str( sa ), str( gaddr ) )
            del intfConfig.staticJoinSourceGroup[ sg ]
      else:
         mask = 0xFFFFFFFF & ~( rangemax - 1 )
         for sg in intfConfig.staticJoinSourceGroup.keys():
            sa, ga = IpAddress( sg.sourceAddr ), IpAddress( sg.groupAddr )
            if ( ga == gaddr ) and \
                   ( ( sa.value & mask ) == ( saddr.value & mask ) ):
               del intfConfig.staticJoinSourceGroup[ sg ]

   _deleteInfoConfigIfAllDefault( mode )

#------------------------------------------------------------------------------------
# (config-if)# ip igmp static-group group <group>
#                                   num-groups <num-groups> [ source <addr> ]
#------------------------------------------------------------------------------------
def cmdIpIgmpStaticNumGroups( mode, args ):
   groupAddr = args[ 'GROUP' ]
   numGroups = args[ 'GROUPNUM' ]
   source = args.get( 'SOURCE', '0.0.0.0' )
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   for group in Arnet.getMcastGroupAddresses( groupAddr, numGroups ):
      sg = SourceGroup( source, group )
      intfConfig.staticJoinSourceGroup[ sg ] = True

def cmdNoIpIgmpStaticNumGroups( mode, args ):
   groupAddr = args[ 'GROUP' ]
   numGroups = args[ 'GROUPNUM' ]
   source = args.get( 'SOURCE', '0.0.0.0' )
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   for group in Arnet.getMcastGroupAddresses( groupAddr, numGroups ):
      sg = SourceGroup( source, group )
      del intfConfig.staticJoinSourceGroup[ sg ]

class IgmpIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      name = self.intf_.name
      del igmpConfig.intfConfig[ name ]

#------------------------------------------------------------------------------------
# (config-if)# [no|default] ip igmp router-alert { mandatory | optional [connected] }
#------------------------------------------------------------------------------------
def cmdIgmpRouterAlertConfigMandatory( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      if 'mandatory' in args:
         intfConfig.routerAlertConfigOption = "routerAlertMandatory"
      elif 'connected' in args:
         intfConfig.routerAlertConfigOption = "routerAlertOptionalConnected"
      else:
         intfConfig.routerAlertConfigOption = "routerAlertOptional"
      _deleteInfoConfigIfAllDefault( mode )

def cmdNoIgmpRouterAlertConfig( mode, args ):
   intfConfig = _getOrCreateIgmpIntfConfig( mode )
   if intfConfig:
      intfConfig.routerAlertConfigOption = intfConfig.routerAlertConfigOptionDefault
      _deleteInfoConfigIfAllDefault( mode )

class RouterIgmpMode( RoutingIgmpMode, BasicCli.ConfigModeBase ):
   name = 'IGMP configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingIgmpMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterIgmpVrfMode( RoutingIgmpVrfMode, BasicCli.ConfigModeBase ):
   name = 'IGMP VRF configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in RESERVED_VRF_NAMES
      self.vrfName = vrfName
      RoutingIgmpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterIgmpSharedModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName

# service acls
def _noServiceAcl( mode, vrfName=DEFAULT_VRF ):
   AclCliLib.noServiceAcl( mode, 'igmp', aclConfig, aclCpConfig,
                           None, 'ip', vrfName )

def _setServiceAcl( mode, aclName, vrfName=DEFAULT_VRF ):
   AclCliLib.setServiceAcl( mode, 'igmp', AclLib.IPPROTO_IGMP,
                            aclConfig, aclCpConfig, aclName, 'ip', vrfName )

def _getServiceAcl( vrfName ):
   serviceAclVrfConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl.get( vrfName )
   if serviceAclVrfConfig:
      serviceConfig = serviceAclVrfConfig.service.get( 'igmp' )
      if serviceConfig:
         return serviceConfig.aclName
   return None

RouterIgmpMode.addModelet( RouterIgmpSharedModelet )
RouterIgmpVrfMode.addModelet( RouterIgmpSharedModelet )

#-------------------------------------------------------------------------------
# "[no|default] router igmp" command, in "config" mode.
#-------------------------------------------------------------------------------
def gotoRouterIgmpMode( mode, args ):
   childMode = mode.childMode( RouterIgmpMode )
   mode.session_.gotoChildMode( childMode )

def deleteRouterIgmpMode( mode, args ):
   # delete non-default config all for all VRFs
   for vrf in aclCpConfig.cpConfig[ 'ip' ].serviceAcl.keys():
      _noServiceAcl( mode, vrf )   
   SsmAwareCmd.noOrDefaultHandler( mode, None ) 

#----------------------------------------------------------------------     
# (config-router-igmp)# [no|default] vrf <vrf>
#----------------------------------------------------------------------      
def gotoRouterIgmpVrfMode( mode, args ):
   vrfName = args[ 'VRF' ]
   if vrfName in RESERVED_VRF_NAMES:
      mode.addError( "vrf name %s is reserved." % vrfName )
      return
   if not vrfExists( vrfName ):
      mode.addWarning( "vrf name %s is not defined." % vrfName )
   IraIpCli.warnIfRoutingDisabled( mode, vrfName )
   childMode = mode.childMode( RouterIgmpVrfMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterIgmpVrfMode( mode, args ):
   vrfName = args[ 'VRF' ]
   assert vrfName not in RESERVED_VRF_NAMES
   _noServiceAcl( mode, vrfName=vrfName )
   
class SsmAwareCmd( CliCommand.CliCommandClass ):
   syntax = 'ssm aware'
   noOrDefaultSyntax = 'ssm aware'
   data = {
         'ssm': CliMatcher.KeywordMatcher( "ssm",
            helpdesc='Source-Specific Multicast configuration' ),
         'aware': CliMatcher.KeywordMatcher( "aware",
            'Ignore *,G reports for groups in SSM range' ),
         }

   @staticmethod
   def handler( mode, args ):
      igmpConfig.ssmAwareEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      igmpConfig.ssmAwareEnabled = igmpConfig.ssmAwareEnabledDefault

if Toggles.IgmpToggleLib.toggleSsmAwareEnabled():
   RouterIgmpMode.addCommandClass( SsmAwareCmd )



#------------------------------------------------------------------------------------
# switch(config-router-igmp)# ip igmp access-group <acl> [in] 
#------------------------------------------------------------------------------------
def setIgmpServiceAcl( mode, args ):
   aclName = args[ 'ACLNAME' ]
   _setServiceAcl( mode, aclName, mode.vrfName )

def noIgmpServiceAcl( mode, args ):
   _noServiceAcl( mode, vrfName=mode.vrfName )

#------------------------------------------------------------------------------------
# show ip igmp access-list [<acl>]
#------------------------------------------------------------------------------------
def showIpAcl( mode, args ):
   name = ( args.get( 'ACLNAME' ),
            'summary' in args, 
            'dynamic' in args,
            'detail' in args )
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 'ip',
                                 name,
                                 serviceName='igmp' )

#-------------------------------------------------------------------------------

#------------------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global igmpConfig, igmpStatus, mfibVrfConfig, ipConfig, ipStatus
   global igmpClearConfigColl, igmpClearStatusColl
   global _entityManager
   global aclConfig, aclCpConfig, aclStatus, aclCheckpoint   

   _entityManager = entityManager
   igmpConfig = ConfigMount.mount( entityManager, 'routing/igmp/config',
                                   'Routing::Igmp::Config', 'w' )
   igmpClearConfigColl = LazyMount.mount( entityManager, 'routing/igmp/clearConfig',
                                   'Routing::Igmp::ClearConfigColl', 'w' )
   igmpStatus = LazyMount.mount( entityManager, 'routing/igmp/status',
                                 'Routing::Igmp::Status', 'r' )
   mfibVrfConfig = LazyMount.mount( entityManager, 'routing/multicast/vrf/config',
                                    'Routing::Multicast::Fib::VrfConfig', 'r' )
   ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )
   ipStatus = LazyMount.mount( entityManager, 'ip/status', 'Ip::Status', 'r' )

   igmpClearStatusColl = LazyMount.mount( entityManager, 
                                          'routing/igmp/clearStatus', 
                                          'Routing::Igmp::ClearStatusColl', 'r' )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint", 
                                    "Acl::CheckpointStatus", "w" )   

   # priority 22 - one higher then pim
   IntfCli.Intf.registerDependentClass( IgmpIntf, priority=22 )
   GmpCli.registerPimEnableIgmpHook( implicitlyEnabledIs )
   GmpCli.registerPimLocalInterfaceIgmpHook( igmpLocalInterfaceIs )

   # Install a hook for igmp interface collection. This
   # way cli keeps track of all multicast routing interfaces engaged.
   def _igmpIntfConfColl( vrfName, af ):
      # Return set of all intf on which Igmp is configured(not from igmp intf status)
      # A count of interfaces per VRF is not enough because
      # same interface may be enabled for pim and for igmp and if they both return
      # collections then it is not counted twice by mcastCollIfHook. If count is
      # returned then it will get counted twice
      allIgmpIntfs = { }

      if af == AddressFamily.ipv6:
         return allIgmpIntfs

      for igmpIntf in igmpConfig.intfConfig.keys():
         if vrfName == VrfCli.getVrfNameFromIntfConfig( ipConfig, igmpIntf ):
            allIgmpIntfs[ igmpIntf ] = igmpConfig.intfConfig[ igmpIntf ]
      return allIgmpIntfs
   McastCommonCliLib.mcastIfCollHook.addExtension( _igmpIntfConfColl )

   def _igmpIntfStatusColl():
      return igmpStatus.intfStatus
   GmpCli.igmpStatusCollHook.addExtension( _igmpIntfStatusColl )
   MrouteCli.routerMcastVrfDefinitionHook.addExtension( _igmpVrfDefinitionHook )
   MrouteCli.routerMcastVrfDeletionHook.addExtension( _igmpVrfDeletionHook )
