# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys, re
import BasicCli
from HttpServiceConstants import ServerConstants
from CliMode.HttpService import HttpServerVrfConfigModeBase
import CliCommand
import CliMatcher
import CliParser
import ConfigMgmtMode
import ConfigMount
from CliPlugin.VrfCli import DEFAULT_VRF, VrfNameExprFactory
import LazyMount
import ShowCommand
import Tac
import HttpServiceModel
from eunuchs.in_h import IPPROTO_TCP
import CliToken
import AclCli
import AclCliLib
import AclCliModel

capiConstants = Tac.Type( "HttpService::Constants" )
sslConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
capiConfig = None
serverStatus = None
sslConfig = None
allOrigin = ServerConstants.ALLOW_ALL_CROSS_ORIGIN_DOMAINS
allOriginStar = ServerConstants.ALLOW_ALL_CROSS_ORIGIN_DOMAINS_STAR
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
allVrfConfig = None
sslStatus = None

def getHttpServices( mode ):
   serviceDict = {}
   for service in capiConfig.service:
      serviceDict[ service ] = "Configure %s service settings" % service
   return serviceDict

protocolHelpdesc = 'Configure server options'
httpHelpdesc = 'Configure HTTP server options'
httpsHelpdesc = 'Configure HTTPs server options'
sslHelpdesc = 'Configure SSL options'
profileHelpdesc = 'Configure SSL profile'
vrfHelpdesc = 'Configure server options on a VRF'
vrfShutdownHelpdesc = 'Disable server access in this VRF'
portHelpdesc = 'Specify the TCP port to serve on'
httpServerShowHelpDesc = 'Show Http Server VRF configuration details'
aclServiceName = ServerConstants.serviceName

# Make sure server-configuring commands are in "api http-commands" mode xor
# 'http-server' mode.
def checkServerConfigNotSplit( mode ):
   errorStr = "Cannot configure server in mode '%s' because there is server "\
              "configuration in mode '%s'. Please use the same mode for server "\
              "configuration."
   if capiConfig.useHttpServiceCli == isinstance( mode, HttpServerConfigMode ):
      return True
   elif capiConfig.useHttpServiceCli:
      mode.addError( errorStr % ( 'api http-commands', 'http-server' ) )
      return False
   elif serverConfigurationExists():
      mode.addError( errorStr % ( 'http-server', 'api http-commands' ) )
      return False
   else:
      return True

# Check protocol, cert/key, SSL profile, cipher, QOS dscp, content frame
# ancestors, cors and log-level to see if there's existing server configuration
def serverConfigurationExists():
   httpsConfig = capiConfig.httpsConfig
   if ( not _checkProtocolsDefault() or
        httpsConfig.sslKey or httpsConfig.sslCertificate or
        httpsConfig.sslProfile or
        httpsConfig.ciphers or httpsConfig.keyExchange or httpsConfig.mac or
        capiConfig.qosDscp or
        capiConfig.contentFrameAncestor or
        not capiConfig.defaultServicesEnabled or
        capiConfig.corsAllowedOrigins or
        capiConfig.syslogLevel != capiConfig.syslogLevelDefault ):
      return True
   return False

# Set useHttpServiceCli flag to true or false. This function assumes we have
# already run checkServerConfigNotSplit to decide if we're in a valid mode.
def setUseHttpServiceCli( mode ):
   if isinstance( mode, HttpServerConfigMode ):
      if serverConfigurationExists():
         capiConfig.useHttpServiceCli = True
      else:
         capiConfig.useHttpServiceCli = False

def server_config_split( function ):
   """
   Checks if the server-configuration would be split if the config got applied,
   apply config if it's okay, then set the useHttpServiceCli flag based on new config
   """
   def wrapper( *args, **kwargs ):
      mode = args[ 0 ]
      if not checkServerConfigNotSplit( mode ):
         return
      function( *args, **kwargs )
      setUseHttpServiceCli( mode )
   return wrapper

#
# Management config mode for http server
#
class HttpServerConfigMode( ConfigMgmtMode.ConfigMgmtMode ):

   name = "HTTP server configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "http-server" )

   def enterCmd( self ):
      return "management http-server"

#-------------------------------------------------------------------------------
# The "[no|default] management http-server" mode command.
#-------------------------------------------------------------------------------
class EnterHttpServerMode( CliCommand.CliCommandClass ):
   syntax = '''management http-server'''
   noOrDefaultSyntax = '''management http-server'''

   data = { 'management': ConfigMgmtMode.managementKwMatcher,
            'http-server': 'Configure the HTTP server' }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( HttpServerConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serverNoVrf( mode, {} )
      if capiConfig.useHttpServiceCli:
         handleDefaultProtocol( "http" )
         handleDefaultProtocol( "https" )
         handleDefaultProtocol( "http localhost")
         handleDefaultProtocol( "unix-socket" )
         capiConfig.defaultServicesEnabled = True
         capiConfig.corsAllowedOrigins.clear()
         capiConfig.httpsConfig.sslProfile = ''
         capiConfig.qosDscp = 0
         capiConfig.contentFrameAncestor = ""
         capiConfig.syslogLevel = capiConfig.syslogLevelDefault
      if not serverConfigurationExists():
         capiConfig.useHttpServiceCli = False

BasicCli.GlobalConfigMode.addCommandClass( EnterHttpServerMode )

#-------------------------------------------------------------------------------
# The "[no | default] shutdown" command,
# in "vrf <vrf name>" mode
# under "management http-server" mode.
#-------------------------------------------------------------------------------

class HttpServerVrfConfigMode( HttpServerVrfConfigModeBase,
                               BasicCli.ConfigModeBase ):
   name = "HTTP server VRF configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      HttpServerVrfConfigModeBase.__init__( self, ( vrfName, capiConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setServiceAcl( self, aclType, aclName, remove=False ):
      if remove:
         AclCliLib.noServiceAcl( self, aclServiceName, aclConfig, aclCpConfig,
                                 aclName, aclType, self.vrfName_ )
      else:
         AclCliLib.setServiceAcl( self, aclServiceName, IPPROTO_TCP,
                                  aclConfig, aclCpConfig, aclName, 
                                  aclType, self.vrfName_,
                                  port=[ capiConfig.httpConfig.port,
                                         capiConfig.httpsConfig.port ] )

   def setIpAcl( self, args ):
      self.setServiceAcl( 'ip', args[ 'ACL_NAME' ] )

   def noIpAcl( self, args=None ):
      aclName = args.get( 'ACL_NAME' ) if args else None
      self.setServiceAcl( 'ip', aclName, remove=True )

   def setIp6Acl( self, args ):
      self.setServiceAcl( 'ipv6', args[ 'ACL_NAME' ] )

   def noIp6Acl( self, args=None ):
      aclName = args.get( 'ACL_NAME' ) if args else None
      self.setServiceAcl( 'ipv6', aclName, remove=True )

class VrfModeShutdown( CliCommand.CliCommandClass ):
   syntax = '''shutdown'''
   noSyntax = '''shutdown ...'''
   defaultSyntax = '''shutdown ...'''
   data = { 'shutdown': vrfShutdownHelpdesc }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.shutdown()

   @staticmethod
   def noHandler( mode, args ):
      mode.noShutdown()

   @staticmethod
   def defaultHandler( mode, args ):
      mode.defaultShutdown()

HttpServerVrfConfigMode.addCommandClass( VrfModeShutdown )

# remove VRF configuration from service
def removeVrfFromService( config, serviceName, vrf=None ):
   def removeVrf( vrfName ):
      if vrfName in config.vrfConfig:
         del config.vrfConfig[ vrfName ].vrfService[ serviceName ]

         if ( len( config.vrfConfig[ vrfName ].vrfService ) == 0 and
               config.vrfConfig[ vrfName ].serverState == 'globalDefault' ):
            del config.vrfConfig[ vrfName ]

   if vrf:
      removeVrf( vrf )
   else:
      for currVrf in config.vrfConfig:
         removeVrf( currVrf )

# {no|default} vrf <vrfName> in manamgent http-server mode
# remove VRF overridden configuration in http-server
def enterHttpServerVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( HttpServerVrfConfigMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def serverNoVrf( mode, args ):
   def setServerNoVrf( vrf ):
      if vrf in capiConfig.vrfConfig:
         capiConfig.vrfConfig[ vrf ].serverState = 'globalDefault'
         if len( capiConfig.vrfConfig[ vrf ].vrfService ) == 0:
            del capiConfig.vrfConfig[ vrf ]

   vrfName = args.get( 'VRF' )
   # Clear http-server VRF configuration
   # This part is not used for now since we hide '[no|default] shutdown' command
   # in http-server VRF submode
   if vrfName:
      setServerNoVrf( vrfName )
   else:
      for currVrf in capiConfig.vrfConfig:
         setServerNoVrf( currVrf )
   
   # Clear http-server service Acl configuration
   for aclType in ( 'ip', 'ipv6' ):
      vrfList = ( [ vrfName ] if vrfName 
                  else aclCpConfig.cpConfig[ aclType ].serviceAcl.keys() )
      for vrf in vrfList:
         childMode = mode.childMode( HttpServerVrfConfigMode, vrfName=vrf )
         childMode.setServiceAcl( aclType, None, remove=True )
   return

class EnterVrfMode( CliCommand.CliCommandClass ):
   syntax = '''vrf VRF'''
   noOrDefaultSyntax = '''vrf [ VRF ]'''
   data = { 'vrf': vrfHelpdesc,
            'VRF': VrfNameExprFactory( inclDefaultVrf=True ) }

   handler = enterHttpServerVrfConfigMode
   noOrDefaultHandler = serverNoVrf

HttpServerConfigMode.addCommandClass( EnterVrfMode )

#-----------------------------------------------------------------------------------
# [ no | default ] ip access-group <name> [ vrf VRF ] in
#-----------------------------------------------------------------------------------
class IpAccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME in'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] in'
   data = {
            'ip': AclCli.ipKwForServiceAclMatcher, 
            'access-group': AclCli.accessGroupKwMatcher, 
            'ACL_NAME': AclCli.ipAclNameMatcher,
            'in': AclCli.inKwMatcher 

          }
   handler = HttpServerVrfConfigMode.setIpAcl
   noOrDefaultHandler = HttpServerVrfConfigMode.noIpAcl

HttpServerVrfConfigMode.addCommandClass( IpAccessGroup )
#-----------------------------------------------------------------------------------
# [ no | default ] ipv6 access-group <name> [ vrf VRF ] in
#-----------------------------------------------------------------------------------
class Ipv6AccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME in'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] in'
   data = {
            'ipv6': AclCli.ipv6KwMatcherForServiceAcl, 
            'access-group': AclCli.accessGroupKwMatcher, 
            'ACL_NAME': AclCli.ip6AclNameMatcher,
            'in': AclCli.inKwMatcher 

          }
   handler = HttpServerVrfConfigMode.setIp6Acl
   noOrDefaultHandler = HttpServerVrfConfigMode.noIp6Acl

HttpServerVrfConfigMode.addCommandClass( Ipv6AccessGroup )

#-------------------------------------------------------------------------------
# The "[no|default] protocol ((http [localhost])|https) [port <number>]" command,
# in "management http-server" mode.
#-------------------------------------------------------------------------------
def _setServerStatus( protocol, config, status, port=None ):
   config.enabled = status
   if port:
      config.port = port
      if protocol in ( 'http', 'https' ):
         portsStr = ','.join( str( i ) for i in [ capiConfig.httpConfig.port,
                                                  capiConfig.httpsConfig.port ] )
         for aclType in ( 'ip', 'ipv6' ):
            for serviceAclVrfConfig in \
                  aclCpConfig.cpConfig[ aclType ].serviceAcl.itervalues():
               serviceAclConfig = serviceAclVrfConfig.service.get( aclServiceName )
               if serviceAclConfig:
                  serviceAclConfig.ports = portsStr

def _checkProtocolsDefault():
   for protocol in ( "https", "http", "http localhost", "unix-socket" ):
      config, port, enabled = _getProtocolDefaults( protocol )
      if ( config.enabled != enabled or
           protocol != 'unix-socket' and config.port != port ):
         return False
   return True

def _getProtocolDefaults( protocol ):
   if protocol == "https":
      return ( capiConfig.httpsConfig, capiConstants.defaultSecurePort,
               capiConstants.defaultSecureEnabled )
   elif protocol == "http":
      return ( capiConfig.httpConfig, capiConstants.defaultInsecurePort,
               capiConstants.defaultInsecureEnabled )
   elif protocol == "http localhost":
      return ( capiConfig.localHttpConfig,
               capiConstants.defaultInsecureLocalPort,
               capiConstants.defaultInsecureLocalEnabled )
   elif protocol == "unix-socket":
      return ( capiConfig.unixConfig, None, capiConstants.defaultUnixEnabled )

def setProtocolStatus( protocol, status, port=None ):
   config, defaultPort, _ = _getProtocolDefaults( protocol )
   _setServerStatus( protocol, config, status,
                     port if port and status else defaultPort )

def handleDefaultProtocol( protocol ):
   config, defaultPort, defaultStatus = _getProtocolDefaults( protocol )
   _setServerStatus( protocol, config, defaultStatus, defaultPort )

def _getProtocolType( args ):
   if "https" in args:
      return "https"
   elif "http" in args:
      if "localhost" in args:
         return "http localhost"
      else:
         return "http"

class HttpProtocolCmd( CliCommand.CliCommandClass ):
   syntax = """protocol ((http[localhost])|https) [port <number>]"""
   noOrDefaultSyntax = """protocol ((http[localhost])|https) [port [<number>]]"""

   data = { "protocol": protocolHelpdesc,
            "http": httpHelpdesc,
            "https": httpsHelpdesc,
            "localhost": "Server bound on localhost",
            "port": portHelpdesc,
            "<number>": CliMatcher.IntegerMatcher( capiConstants.minPort,
                                                   capiConstants.maxPort,
                                                   helpdesc="TCP port" )
          }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      protocol = _getProtocolType( args )
      port = args[ "<number>" ] if "<number>" in args else None
      setProtocolStatus( protocol, True, port )

   @staticmethod
   @server_config_split
   def noOrDefaultHandler( mode, args ):
      protocol = _getProtocolType( args )
      if CliCommand.isDefaultCmd( args ):
         handleDefaultProtocol( protocol )
      else:
         setProtocolStatus( protocol, False, None )

HttpServerConfigMode.addCommandClass( HttpProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol unix-socket" command,
# in "management http-server" mode.
#-------------------------------------------------------------------------------
class UnixProtocolCmd( CliCommand.CliCommandClass ):
   syntax = """protocol unix-socket"""
   noOrDefaultSyntax = """protocol unix-socket"""

   data = { "protocol": protocolHelpdesc,
            "unix-socket": "Configure Unix Domain Socket" }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      setProtocolStatus( "unix-socket", True, None )

   @staticmethod
   @server_config_split
   def noHandler( mode, args ):
      setProtocolStatus( "unix-socket", False, None )

   @staticmethod
   @server_config_split
   def defaultHandler( mode, args ):
      handleDefaultProtocol( "unix-socket" )

HttpServerConfigMode.addCommandClass( UnixProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol https ssl profile" command,
# in "management http-server" mode.
#-------------------------------------------------------------------------------
profileNameRule = CliMatcher.DynamicNameMatcher(
      lambda mode: sslConfig.profileConfig,
      'Profile name' )

class HttpsProfile( CliCommand.CliCommandClass ):
   syntax = """protocol https ssl profile <profileName>"""
   noOrDefaultSyntax = """protocol https ssl profile"""

   data = { "protocol": protocolHelpdesc,
            "https": httpsHelpdesc,
            "ssl": sslHelpdesc,
            "profile": profileHelpdesc,
            "<profileName>": profileNameRule }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      capiConfig.httpsConfig.sslProfile = args[ "<profileName>" ]

   @staticmethod
   @server_config_split
   def noOrDefaultHandler( mode, args ):
      capiConfig.httpsConfig.sslProfile = ''

HttpServerConfigMode.addCommandClass( HttpsProfile )

#-------------------------------------------------------------------------------
# The "[ no | default ] qos dscp <dscpValue>" command
#-------------------------------------------------------------------------------
singleDscpValueMatcher = CliMatcher.IntegerMatcher( 0, 63,
                                            helpdesc='DSCP value between 0 and 63' )

class QosDscp( CliCommand.CliCommandClass ):
   syntax = """qos dscp <dscpValue>"""
   noOrDefaultSyntax = """qos dscp [<dscpValue>]"""

   data = { "qos": "Configure QoS parameters",
            "dscp": "Set the DSCP value for eAPI",
            "<dscpValue>": singleDscpValueMatcher }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      capiConfig.qosDscp = args[ "<dscpValue>" ]

   @staticmethod
   @server_config_split
   def noOrDefaultHandler( mode, args ):
      capiConfig.qosDscp = 0

HttpServerConfigMode.addCommandClass( QosDscp )

#-------------------------------------------------------------------------------
# The "[ no | default ] log-level <severity>" command
#-------------------------------------------------------------------------------
logLevels = Tac.Type( 'HttpService::LogLevel' )
nginxLogLevels = { s: s.upper() + ' log level'
                   for s in logLevels.attributes if s != logLevels.none }

class NginxSyslog( CliCommand.CliCommandClass ):
   syntax = """log-level <severity>"""
   noOrDefaultSyntax = """log-level [<severity>]"""

   data = { "log-level": "Configure nginx logging level",
            "<severity>": CliMatcher.DynamicKeywordMatcher(
                                                      lambda mode: nginxLogLevels ) }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      capiConfig.syslogLevel = args[ "<severity>" ]

   @staticmethod
   @server_config_split
   def noOrDefaultHandler( mode, args ):
      if CliCommand.isDefaultCmd( args ):
         capiConfig.syslogLevel = capiConfig.syslogLevelDefault
      else:
         capiConfig.syslogLevel = logLevels.none

HttpServerConfigMode.addCommandClass( NginxSyslog )

#-------------------------------------------------------------------------------
# The "[ no | default ] default-services" command
#-------------------------------------------------------------------------------
class DefaultServices( CliCommand.CliCommandClass ):
   syntax = """default-services"""
   noSyntax = """default-services"""
   defaultSyntax = """default-services"""

   data = { "default-services": "Enable default services: capi-doc and tapagg" }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      capiConfig.defaultServicesEnabled = True

   @staticmethod
   @server_config_split
   def noHandler( mode, args ):
      capiConfig.defaultServicesEnabled = False

   @staticmethod
   @server_config_split
   def defaultHandler( mode, args ):
      capiConfig.defaultServicesEnabled = True

HttpServerConfigMode.addCommandClass( DefaultServices )

#-------------------------------------------------------------------------------
# The "[ no | default ] content frame ancestors <uri>" command
#-------------------------------------------------------------------------------
URI_REGEX = r"^(([^:/?#]+):)?(//([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?"
tokenUri = CliMatcher.PatternMatcher( URI_REGEX,
                           helpname="URI",
                           helpdesc='URI of allowed host' )
class FrameAncestors( CliCommand.CliCommandClass ):
   syntax = """header csp frame-ancestors <uri>"""
   noOrDefaultSyntax = """header csp frame-ancestors [ <uri> ]"""

   data = { "header": "Additional headers",
            "csp": "Content Security Policy Headers",
            "frame-ancestors": "CSP directive frame-ancestors",
            "<uri>": tokenUri }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      capiConfig.contentFrameAncestor = args[ "<uri>" ]

   @staticmethod
   @server_config_split
   def noOrDefaultHandler( mode, args ):
      capiConfig.contentFrameAncestor = ""

HttpServerConfigMode.addCommandClass( FrameAncestors )

#-------------------------------------------------------------------------------
# The "cors allowed domian [ALL | URL]" command
#-------------------------------------------------------------------------------
corsRegex = r'^(?:http)s?://(?:(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}'\
      r'[a-zA-Z0-9])?\.)+(?:[a-zA-Z]{2,6}\.?)(?::\d+)?$|'\
      r'(localhost)(?::\d+)?$|'\
      r'(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})(?::\d+)?$|'\
      r'(\[?[A-Fa-f0-9]*:[A-Fa-f0-9:]+\]?)(?::\d+)?$)'

corsRegexPartial = '(http)*'

# tokens for CORS ( cross origin request service )
tokenOrigin = CliMatcher.PatternMatcher(
    corsRegex,
    helpname='Origin',
    helpdesc='Origin of form http[s]://<hostname>[:<port>]',
    partialPattern=corsRegexPartial )

corsNameRule = CliMatcher.DynamicNameMatcher(
    lambda mode: capiConfig.corsAllowedOrigins,
    'Origin',
    pattern='.*' )

class CorsOrigins( CliCommand.CliCommandClass ):
   syntax = """cors allowed-origin ( all | * | <originString> )"""
   noOrDefaultSyntax = """cors allowed-origin [ <corsName> ]"""

   data = { "cors": "Configure CORS functionality",
            "allowed-origin": "Enter allowed origins",
            "all": "Allow all Origins",
            "*": "Allow all Origins",
            "<originString>": tokenOrigin,
            "<corsName>": corsNameRule }

   @staticmethod
   @server_config_split
   def handler( mode, args ):
      if allOrigin in args or allOriginStar in args:
         origin = allOrigin
      else:
         origin = args[ "<originString>" ]

      capiConfig.corsAllowedOrigins[ origin ] = True

   @staticmethod
   @server_config_split
   def noOrDefaultHandler( mode, args ):
      if "<corsName>" not in args:
         capiConfig.corsAllowedOrigins.clear()
         return
      originString = args[ "<corsName>" ]
      if ( ( originString == allOrigin or originString == allOriginStar )
            and allOrigin in capiConfig.corsAllowedOrigins ):
         del capiConfig.corsAllowedOrigins[ allOrigin ]
      elif originString in capiConfig.corsAllowedOrigins:
         del capiConfig.corsAllowedOrigins[ originString.lower() ]
      else:
         mode.addError( "'%s' is not in allowed origins" % originString )

HttpServerConfigMode.addCommandClass( CorsOrigins )

#-------------------------------------------------------------------------------
# show management http-server ( ip|ipv6 ) access-list [<acl>] [summary]
#-------------------------------------------------------------------------------
class ShowManagementHttpServerAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management http-server'
              '('
              ' ( ip access-list [ IP_ACL_NAME ] ) | '
              ' ( ipv6 access-list [ IPV6_ACL_NAME ] ) '
              ')' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'http-server': httpServerShowHelpDesc, 
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'IP_ACL_NAME': AclCli.ipAclNameExpression,
            'IPV6_ACL_NAME': AclCli.ip6AclNameExpression
          }

   cliModel = AclCliModel.AllAclList
   
   @staticmethod
   def handler( mode, args ):
      # Note: the name argument is actually a tuple( name, summary )
      aclType = 'ip' if 'ip' in args else 'ipv6'
      return AclCli.showServiceAcl( mode,
                                    aclCpConfig,
                                    aclStatus,
                                    aclCheckpoint,
                                    aclType,
                                    args[ '<aclNameExpr>' ],
                                    serviceName=ServerConstants.serviceName )

BasicCli.addShowCommandClass( ShowManagementHttpServerAcl ) 

#----------------------------------------------------------------
# "clear management http-server counters ( ip | ipv6 ) access-list"
#----------------------------------------------------------------
class ClearHttpServerAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear management http-server counters ( ip | ipv6 ) access-list'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'http-server': httpServerShowHelpDesc,
            'counters': AclCli.countersKwMatcher,
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl }

   @staticmethod
   def handler( mode, args ):
      aclType = 'ip' if 'ip' in args else 'ipv6'
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

BasicCli.EnableMode.addCommandClass( ClearHttpServerAclCounters ) 

#-------------------------------------------------------------------------------
# The "show management http-server" command
#-------------------------------------------------------------------------------
def getCommonServerInfo( ret, config ):
   ret.fipsEnabled = False
   ret.tlsProtocol = ServerConstants.DEFAULT_TLS_PROTOCOLS

   profile = config.httpsConfig.sslProfile
   if profile:
      status = sslStatus.profileStatus.get( profile )
      if status == None:
         ret.sslProfile = HttpServiceModel.SslProfile( name=profile,
                                                       configured=False )
      else:
         profileState = status.state
         ret.sslProfile = HttpServiceModel.SslProfile( name=profile,
                                                       configured=True,
                                                       state=profileState )
         ret.fipsEnabled = status.fipsMode
         tlsVersionsEnabled = []
         tlsVersionMap = [ ( sslConstants.tlsv1, '1.0' ),
                           ( sslConstants.tlsv1_1, '1.1' ),
                           ( sslConstants.tlsv1_2, '1.2' ) ]
         for mask, version in tlsVersionMap:
            if status.tlsVersion & mask:
               tlsVersionsEnabled.append( version )
         if tlsVersionsEnabled:
            ret.tlsProtocol = tlsVersionsEnabled
   ret.dscpValue = config.qosDscp
   ret.logLevel = config.syslogLevel
   if capiConfig.contentFrameAncestor:
      ret.iframeAncestors.append( config.contentFrameAncestor )

def showHttpServer( mode, args ):
   ret = HttpServiceModel.HttpServiceStatus()
   getCommonServerInfo( ret, capiConfig )

   for ( vrf, status ) in serverStatus.vrfStatus.items():
      # check vrf error first
      for error in status.vrfError.values():
         if error:
            mode.addWarning( error )
      # only show enabled VRFs
      if status.enabled:
         currVrf = HttpServiceModel.HttpServiceVrf()
         currVrf.httpsServer = HttpServiceModel.HttpProtocolStatus(
               configured=capiConfig.httpsConfig.enabled,
               running=status.httpsEnabled, port=capiConfig.httpsConfig.port )
         currVrf.httpServer = HttpServiceModel.HttpProtocolStatus(
               configured=capiConfig.httpConfig.enabled,
               running=status.httpEnabled, port=capiConfig.httpConfig.port )
         currVrf.localHttpServer = HttpServiceModel.HttpProtocolStatus(
               configured=capiConfig.localHttpConfig.enabled,
               running=status.httpLocalEnabled, port=capiConfig.localHttpConfig.port)
         currVrf.unixSocketServer = HttpServiceModel.UnixProtocolStatus(
               configured=capiConfig.unixConfig.enabled, running=status.unixEnabled )

         # only show services that enabled in this VRF
         vrfService = []
         for ( service, serviceStatus )in status.vrfService.items():
            if serviceStatus.enabled:
               vrfService.append( capiConfig.service[ service ].serviceName )
         currVrf.services = vrfService
         ret.vrfs[ vrf ] = currVrf

   return ret

class ShowManagementHttpServer( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management http-server'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'http-server': httpServerShowHelpDesc }
   handler = showHttpServer
   cliModel = HttpServiceModel.HttpServiceStatus

BasicCli.addShowCommandClass( ShowManagementHttpServer )
#-------------------------------------------------------------------------------
# The "show management http-server counters" command
#-------------------------------------------------------------------------------
def showHttpServerStats( mode, args ):
   url = ServerConstants.NGINX_STATUS_URL
   cmd = [ 'curl', '-sS', url ]
   ret = HttpServiceModel.HttpServerStats()
   for ( vrf, status ) in serverStatus.vrfStatus.items():
      # only show enabled VRFs
      if status.enabled:
         currVrfStats = HttpServiceModel.VrfStats()
         if vrf == DEFAULT_VRF:
            execCmd = cmd
         else:
            execCmd = [ 'ip', 'netns', 'exec', 'ns-%s' % vrf ]
            execCmd.extend( cmd )
         output = ""
         try:
            output = Tac.run( execCmd, stdout=Tac.CAPTURE,
                              stderr=sys.stderr, asRoot=True )
         except Tac.SystemCommandError, e:
            mode.addWarning( "Fail to get counter info of vrf %s due to %s"
                             % ( vrf, e.output ) )
            continue
         result = [ int( v ) for v in re.findall( r'\d+', output ) ]
         if not result:
            mode.addWarning( "Fail to parse counter info of vrf %s output" % vrf )
            continue
         else:
            assert len( result ) == 7
            currVrfStats.maxConnections = 1024
            currVrfStats.activeConnections = result[ 0 ]
            currVrfStats.acceptConnections = result[ 1 ]
            currVrfStats.handledConnections = result[ 2 ]
            currVrfStats.refusedConnections = result[ 1 ] - result[ 2 ]
            currVrfStats.requests = result[ 3 ]
            currVrfStats.readingConnections = result[ 4 ]
            currVrfStats.writingConnections = result[ 5 ]
            currVrfStats.waitingConnections = result[ 6 ]
            ret.vrfs[ vrf ] = currVrfStats

   return ret

class ShowManagementHttpServerCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management http-server counters'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'http-server': httpServerShowHelpDesc,
            'counters': 'Show http connection stats (per vrf)' }
   handler = showHttpServerStats
   cliModel = HttpServiceModel.HttpServerStats

BasicCli.addShowCommandClass( ShowManagementHttpServerCounters )

# Plug-in definition:
def Plugin( entityManager ):
   global capiConfig
   global serverStatus
   global sslConfig
   global aclConfig
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global allVrfConfig
   global sslStatus
   capiConfig = ConfigMount.mount( entityManager,
                                   "mgmt/capi/config",
                                   "HttpService::Config",
                                   "w" )
   serverStatus = LazyMount.mount( entityManager,
                                   "mgmt/httpserver/status",
                                   "HttpService::Status",
                                   "r" )
   sslConfig = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config",
                                "r" )
   aclConfig = ConfigMount.mount( entityManager,
                                  "acl/config/cli",
                                  "Acl::Input::Config",
                                  "w" )
   aclCpConfig = ConfigMount.mount( entityManager,
                                    "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig",
                                    "w" )
   aclStatus = LazyMount.mount( entityManager, 
                                "acl/status/all",
                                "Acl::Status", 
                                "r" )
   aclCheckpoint = LazyMount.mount( entityManager,
                                    "acl/checkpoint",
                                    "Acl::CheckpointStatus",
                                    "w" )
   allVrfConfig = LazyMount.mount( entityManager,
                                   "ip/vrf/config",
                                   "Ip::AllVrfConfig",
                                   "r" )
   sslStatus = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/status",
                                "Mgmt::Security::Ssl::Status",
                                "r" )
