# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import TableOutput, Ark, Tac
import Ethernet
import CliCommon

from ArnetModel import IpGenericAddress
from ArnetModel import MacAddress
from ArnetModel import IpGenericPrefix
from ArnetModel import IpGenericAddrAndMask
from CliModel import Model, Enum, List, Dict, Str, Int, Float, Bool, Submodel
from VxlanVniLib import VniFormat

from collections import OrderedDict

normFmt = TableOutput.Format( justify='left' )
normFmt.noPadLeftIs( True )
headFmt = TableOutput.Format( isHeading=True, justify='left' )
headFmt.noPadLeftIs( True )

renderInVniInDottedNotation = False

def getFormattedTable( tableCols=None, nCols=0 ):
   if tableCols is None:
      table = TableOutput.TableFormatter()
   else:
      nCols = len( tableCols )
      table = TableOutput.createTable( tableCols )

   fmt = TableOutput.Format( justify='left' )
   fmt.noPadLeftIs( True )
   table.formatColumns( *( [ fmt ] * nCols ) )

   return table

# This attempts to provide a generic way of rendering tables given a Capi model that
# maps "reasonably" to a table. The 'models' arg should be a dict of 'model' types
# that correspond to a 'logical row' on a table. The attribs of the 'model' type
# should corrspond to columns in the table. If a column in a logical row maps to a
# collection the 'logical row' will be expanded to the max length of all column.
#
# Each 'model' attrib can optionally define a 'toString' method (Capi will not allow
# __str__ or variants ) to render the type into a table cell. There is also a hack
# using a splitCell bool to break a single key value attrib into two columns
modelKey = '<key>'
def renderTable( headings, models, modelAttribs ):
   def toString( attrib ):
      if hasattr( attrib, 'toString' ):
         return attrib.toString()
      else:
         return str( attrib )
   table = TableOutput.createTable( [ hscTable[ h ][ 0 ] for h in headings ] )
   table.formatColumns( *[ hscTable[ h ][ 1 ] for h in headings ] )

   logicalRow = []
   maxRows = 0
   for key, model in models.iteritems():
      for modelAttrib in modelAttribs:
         if modelAttrib == modelKey:
            logicalRow.append( [ toString( key ) ] )
            maxRows = max( maxRows, 1 )
            continue

         column = modelAttrib
         splitCells = False
         # This is the hacky bit where a key-value pair of a dict is to be rendered 
         # into two columns instead of toString( k ) + ' ' + toString( v ). If client
         # passes in a tuple, the 2nd arg indicates if the pair should be split
         if hasattr( modelAttrib, '__iter__' ):
            column = modelAttrib[ 0 ]
            splitCells =  modelAttrib[ 1 ]
         attrib = getattr( model, column )
         if hasattr( attrib, '__iter__' ):
            row = None
            if hasattr( attrib, 'iteritems' ):
               # Handle dict
               row = []
               if not splitCells:
                  for k, v in attrib.iteritems():
                     row.append( toString( k ) + ' ' + toString( v ) )
                  maxRows = max( maxRows, len( row ) )
                  # Add a column of key-value pair rendered together
                  logicalRow.append( sorted( row ) )
               else:
                  # We are splitting the key-value pairs into two columns
                  keyRow = []
                  for k in attrib.keys():
                     keyRow.append( toString( k ) )
                  # Add a column of keys
                  logicalRow.append( sorted( keyRow ) )

                  valueRow = []
                  for v in attrib.itervalues():
                     valueRow.append( toString( v ) )
                  # Add a column of values
                  logicalRow.append( sorted( valueRow ) )

                  maxRows = max( maxRows, len( valueRow ) )
            else:
               # Handle list
               row = [ toString( i ) for i in attrib ]
               maxRows = max( maxRows, len( row ) )
               logicalRow.append( sorted( row ) )
         else:
            # Handle single attribute
            maxRows = max( maxRows, 1 )
            logicalRow.append( [ toString( attrib ) ] )

      for i in range( maxRows ):
         row = []
         for column in logicalRow:
            cell = ''
            if i < len( column ):
               cell = column[ i ]
            row.append( cell )
         table.newRow( *row )
      logicalRow = []
      maxRows = 0

   print table.output()

class HscAgentStatusModel( Model ):
   ovsdbConnection = Enum( help='The current agent to OVSDB server connection ' \
         'state.', values=( 'Connected', 'Disconnected', 'Unknown' ) )
   state = Enum( help='The current agent state.',
         values=( 'Stopped', 'Running' ) )

class OvsdbManagerModel( Model ):
   manager = Str( help='The manager connection information.' )
   state = Str( help='The current state of the connection to this manager' )
   uptime = Float( help='UTC Timestamp at time of manager connecting.' )
   connected = Bool( help='Whether OVSDB server is currently connected to this ' \
         'manager.' )

ProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )

class SslProfileModel( Model ):
   name = Str( help="SSL profile name" )
   configured = Bool( help="Whether this profile is configured in SSL Manager" )
   state = Enum( optional=True, help="State of the SSL profile",
                 values=ProfileState.attributes )

class HscStatusModel( Model ):
   __revision__ = 3

   enabled = Bool( help='The configured state of the HSC service.' )
   managers = List( valueType=OvsdbManagerModel,
         help='The currently configured managers and their connection state.',
         optional=True )
   ovsdbState = Enum( help='The state of the local OVSDB server',
         values=( 'Disabled', 'Stopped', 'Started', 'Running', 'Failed', 
                  'WaitingForConvergence' ),
         optional=True )

   agent = Submodel( valueType=HscAgentStatusModel,
         help='The status of the HSC agent.', optional=True)

   vtepFloodMode = Enum( help='The current Vtep flood mode.',
         values=( 'sendToAll', 'sendToAny' ),
         optional=True )

   enableErrorReporting = Bool( help='The error reporting state.',
         optional=True )

   routingMode = Enum( help='The routing mode.',
         values=( 'direct', 'indirect' ),
         optional=True )
   sslProfile = Submodel( optional=True, valueType=SslProfileModel,
                          help="The configuration and status of the SSL profile" )
   fipsEnabled = Bool( optional=True, help="Whether FIPS mode is enabled for "
                                           "HTTPS algorithms" )
   tlsProtocol = List( valueType=str, help="Allowed TLS protocol versions" )

   def degrade( self, dictRepr, revision ):
      def downFrom2to1( dictRepr ):
         for manager in dictRepr.get( 'managers', [] ):
            manager[ 'uptime' ] = int( manager[ 'uptime' ] )

         return dictRepr

      def downFrom3to2( dictRepr ):
         #Version 2 represents agent as a list
         #Version 3 uses a Submodel with valueType: HscAgentStatusModel 
         agent = dictRepr.pop( 'agent', None )
         if agent is not None:
            dictRepr[ 'agents' ] = [ agent ]
         return dictRepr

      funcList = [ downFrom3to2, downFrom2to1 ]
      diffCount = self.__revision__ - revision

      for degradeFunc in funcList[ :diffCount ]:
         dictRepr = degradeFunc( dictRepr )
     
      return dictRepr

   def render( self ):
      if self.enabled:
         state = 'enabled'
      else:
         state = 'disabled'

      print 'HSC is %s.' % ( state, )
      if not self.enabled:
         return

      mode = 'all' if self.vtepFloodMode == 'sendToAll' else 'any'
      print 'Vtep flood mode is %s.' % ( mode, )

      if self.enableErrorReporting:
         print 'Error reporting is enabled.'

      print 'Routing mode is %s.' % ( self.routingMode, )
      print ''

      if self.sslProfile:
         state = 'not configured'
         if self.sslProfile.state:
            state = self.sslProfile.state
         print 'SSL Profile', self.sslProfile.name, 'is', state
         print 'TLS versions:', ', '.join( self.tlsProtocol )
         print 'FIPS:', 'enabled' if self.fipsEnabled else 'disabled'
      else:
         print 'SSL Profile: none'

      if self.ovsdbState != 'Running':
         if self.ovsdbState == 'Stopped':
            print 'OVSDB server has not started.\n'
         elif self.ovsdbState == 'Started':
            print 'OVSDB server is running but not yet ready.\n'
         elif self.ovsdbState == 'Failed':
            print 'Failed to communicate with the OVSDB server.\n'
         elif self.ovsdbState == 'WaitingForConvergence':
            print ( 'Connection to managers is suspended until at least one client '
                    'switch is registered' )
      
      ovsdbTable = getFormattedTable( ( 'Manager', 'Connected', 'State' ) )

      if self.managers:
         for manager in self.managers:
            if manager.uptime:
               connected = '%s (%s)' % ( manager.connected,
                     Ark.utcTimeRelativeToNowStr( manager.uptime ) )
            else:
               connected = manager.connected

            ovsdbTable.newRow( manager.manager, connected, manager.state )

         print ovsdbTable.output()

      if self.agent is not None:
         print 'Hsc agent is %s, OVSDB connection state is %s.' % \
               ( self.agent.state, self.agent.ovsdbConnection )

def dummyFormatter( val ):
   return val

ovsdbSessionStatusFields = [
      [ 'sessionState', 'Session state', dummyFormatter ],
      [ 'lastStateChange', 'Last state change', Ark.timestampToStr ],
      [ 'lastMsgSent', 'Last message sent', dummyFormatter ],
      [ 'timeOfLastMsg', 'Last message time', Ark.timestampToStr ],
      [ 'responseTimeForLastMsg', 'Response time', dummyFormatter ],
      [ 'lastSuccessfulMsg', 'Last successful message', dummyFormatter ],
      [ 'numSuccessfulMsg', 'Successful messages', dummyFormatter ],
      [ 'lastFailedMsg', 'Last failed message', dummyFormatter ],
      [ 'numFailedMsg', 'Failed messages', dummyFormatter ],
      [ 'lastSuccessfulMsgTime', 'Last successful message time', Ark.timestampToStr],
      [ 'lastFailedMsgTime', 'Last failed message time', Ark.timestampToStr],
      [ 'lastErrorCode', 'Last error', dummyFormatter ] ]

class HscOvsdbSessionStatusModel( Model ):
   ovsdbState = Enum( help='The current state of the OVSDB server.',
         values=( 'Stopped', 'Started', 'Running' ) )

   # All the following attributes are optional so that they don't need to be
   # initialized in the case of an error.
   sessionState = Enum( help='The current agent state.',
         values=( 'Connected', 'Disconnected', 'Unknown' ),
         optional=True )
   lastStateChange = Float( help='The last time the state changed.',
         optional=True )
   lastMsgSent = Str( help='The type of the last OVSDB message sent.',
         optional=True )
   lastMsgSentDetail = Str( help='The last OVSDB message sent.',
         optional=True )
   timeOfLastMsg = Float( help='The time the last message was sent.',
         optional=True )
   responseTimeForLastMsg = Float( help='The last message response time.',
         optional=True )
   lastSuccessfulMsg = Str( help='The time of the last successful message.',
         optional=True )
   lastSuccessfulMsgDetail = Str( help='The last successful message sent.',
         optional=True )
   numSuccessfulMsg = Int( help='The number of successfull messages.',
         optional=True )
   lastFailedMsg = Str( help='The last failed message type.',
         optional=True )
   lastFailedMsgDetail = Str( help='The last failed message.',
         optional=True )
   numFailedMsg = Int( help='The number of failed messages.',
         optional=True )
   lastSuccessfulMsgTime = Float( help='The time of the last successful message.',
         optional=True )
   lastFailedMsgTime = Float( help='The time of the last failed message.',
         optional=True )
   lastErrorCode = Str( help='The last error encountered.',
         optional=True )

   _ptcpPort = Int( help='Passive tcp port.', optional=True )

   def render( self ):
      print 'OVSDB Session Status\n'

      table = getFormattedTable( nCols=2 )

      table.newRow( 'OVSDB server state', self.ovsdbState )

      for entry in ovsdbSessionStatusFields:
         table.newRow( entry[1], entry[2]( self.__getattribute__( entry[0] ) ) )

      if self._ptcpPort != Tac.Value( 'Hsc::CliDefaults' ).passiveTcpPort:
         table.newRow( 'Passive tcp port', self._ptcpPort )
      
      print table.output()

      if self.lastMsgSentDetail is not None:
         print 'Last message sent detail:'
         print self.lastMsgSentDetail

         print '\nLast successful message detail:'
         print self.lastSuccessfulMsgDetail

         print '\nLast failed message detail:'
         print self.lastFailedMsgDetail

class HscCertificateModel( Model ):
   certificate = Str( help='SSL certificate used for OVSDB server to manager ' \
         'connections.' )

   def render( self ):
      if self.certificate == '':
         print 'No certificate available.\n'

      print self.certificate

hscTable = {
   'lport': [ 'Logical ports', headFmt ],
   'port': [ 'Port', headFmt ],
   'switch': [ 'Switch', headFmt ],
   'uuid': [ 'UUID', normFmt ],
   'vlan': [ 'VLAN Bindings', normFmt ],
   'vni': [ 'VNI', normFmt ],
   'remoteVtep': [ 'Remote VTEP IP', headFmt ],
   'localVtep': [ 'Local VTEP IP', headFmt ],
   'arpMac': [ 'Source MAC', normFmt ],
   'router': [ 'Router', headFmt ],
   'ipSubnet': [ 'IP/Subnet', normFmt ],
   'logicalSwitch': [ 'Logical Switch', normFmt ],
   'staticRoute': [ 'Static Routes', normFmt ]
}

class HscVlanBindingModel( Model ):
   vlan = Int( help='The vLAN ID.' )
   vni = Int( help='The VXLAN Network Identifier (VNI).' )

class HscPhysicalPortModel( Model ):
   uuid = Str( help='The UUID of the port.' )
   switchName = Str( help='The name of the physical switch.' )
   displayName = Str( help='The port name.' )
   vlanBindings = List( valueType=HscVlanBindingModel,
                        help='The configured VLAN/VNI pairs.' )

   def init( self, ovsdb, port, switchName ):
      self.uuid = str( port.uuid )
      self.displayName = port.displayName
      self.switchName = switchName
      for vlan, lsid in port.vlanBinding.iteritems():
         ls = ovsdb.logicalSwitch.get( lsid )
         if ls:
            self.vlanBindings.append( HscVlanBindingModel(
               vlan=vlan, vni=ls.tunnelKey ) )

class HscPhysicalPortsModel( Model ):
   __revision__ = 2
   ports = Dict( valueType=HscPhysicalPortModel,
                 help='A mapping between uuid and physical ports.' )
   _detail = Bool( help='Whether extra details are shown.', optional=True )
   _vniInDottedNotation = Bool( default=False,
                                help='True if vni is displayed in dotted notation' )

   def degrade( self, dictRepr, revision ):
      if 1 == revision:
         dictRepr[ 'detail' ] = self._detail
         dictRepr[ 'vniInDottedNotation' ] = self._vniInDottedNotation
      return dictRepr

   def put( self, ovsdb, uuid, port, switchName ):
      if switchName:
         self.ports[ str( uuid ) ] = HscPhysicalPortModel()
         self.ports[ str( uuid ) ].init( ovsdb, port, switchName )

   def sortedValues( self ):
      return sorted( self.ports.itervalues(),
                     key=lambda x: x.switchName + x.displayName )

   def renderTable( self, *args ):
      table = TableOutput.createTable( [ hscTable[ arg ][ 0 ] for arg in args ] )
      table.formatColumns( *[ hscTable[ arg ][ 1 ] for arg in args ] )

      def row( **kwargs ):
         table.newRow( *[ kwargs.get( key, '' ) for key in args ] )

      for port in self.sortedValues():
         first = True
         for binding in sorted( port.vlanBindings, key=lambda x: x.vlan ):
            vniStr = VniFormat( binding.vni, self._vniInDottedNotation )
            bstr = 'VLAN %d <-> VNI %s' % ( binding.vlan, vniStr )
            if first:
               row( uuid=str( port.uuid ), switch=port.switchName,
                     port=port.displayName, vlan=bstr )
               first = False
            else:
               row( vlan=bstr )
         if first:
            row( uuid=str( port.uuid ), switch=port.switchName, 
                  port=port.displayName )

      print table.output()

   def render( self ):
      if self._detail is None:
         print ''
         return

      if self._detail:
         self.renderTable( 'switch', 'port', 'vlan', 'uuid' )
      else:
         self.renderTable( 'switch', 'port', 'vlan' )

class HscPhysicalSwitchModel( Model ):
   uuid = Str( help='The UUID of the switch.' )
   displayName = Str( help='The switch name.' )
   mgmtIps = List( valueType=IpGenericAddress,
                   help='The IP addresses of the management ports.' )
   tunnelIps = List( valueType=IpGenericAddress,
                     help='The IP addresses of the tunnel end point.' )

   def init( self, switch ):
      self.uuid = str( switch.uuid )
      self.displayName = switch.displayName
      for ip in switch.mgmtIp:
         self.mgmtIps.append( ip )
      for ip in switch.tunnelIp:
         self.tunnelIps.append( ip )

class HscPhysicalSwitchesModel( Model ):
   __revision__ = 2
   switches = Dict( valueType=HscPhysicalSwitchModel,
                    help='A mapping between uuid and physical switch.' )
   ports = Dict( valueType=HscPhysicalPortsModel,
                 help='A mapping between uuid and physical ports.' )
   _detail = Bool( help='Whether extra details are shown.', optional=True )
   _vniInDottedNotation = Bool( default=False,
                                help='True if vni is displayed in dotted notation' )

   def degrade( self, dictRepr, revision ):
      if 1 == revision: 
         dictRepr[ 'detail' ] = self._detail
         dictRepr[ 'vniInDottedNotation' ] = self._vniInDottedNotation
         # Degrade Ports
         for k in dictRepr[ 'ports' ]:
            dictRepr[ 'ports' ][ k ][ 'detail' ] = self._detail
            dictRepr[ 'ports' ][ k ][ 'vniInDottedNotation' ] = \
                     self._vniInDottedNotation
         # Degrade Switches
         for k in dictRepr[ 'switches' ]:
            if dictRepr[ 'switches' ][ k ].has_key( 'mgmtIps' ):
               tempMgmtIps = dictRepr[ 'switches' ][ k ][ 'mgmtIps' ]
               dictRepr[ 'switches' ][ k ][ 'mgmtIps' ] = \
                     [ str( mgmtIp ) for mgmtIp in tempMgmtIps ] 
            if dictRepr[ 'switches' ][ k ].has_key( 'tunnelIps' ):
               tempTunnelIps = dictRepr[ 'switches' ][ k ][ 'tunnelIps' ]
               dictRepr[ 'switches' ][ k ][ 'tunnelIps' ] = \
                     [ str( tunnelIp ) for tunnelIp in tempTunnelIps ]
      return dictRepr

   def put( self, ovsdb, uuid, switch ):
      uuidString = str( uuid )
      self.switches[ uuidString ] = HscPhysicalSwitchModel()
      self.switches[ uuidString ].init( switch )
      self.ports[ uuidString ] = HscPhysicalPortsModel()
      # pylint: disable-msg=W0212
      self.ports[ uuidString ]._vniInDottedNotation = self._vniInDottedNotation

      for puuid in switch.port:
         pPort = ovsdb.physicalPort.get( puuid )
         if pPort:
            self.ports[ uuidString ].put( ovsdb, puuid, pPort, switch.displayName )

   def renderDetail( self ):
      for switch in sorted( self.switches.itervalues(),
                            key=lambda x: x.displayName ):
         print 'Switch: ' + switch.displayName
         if self._detail:
            print 'UUID: ' + str( switch.uuid )

         mgmtIpsTemp = [ str(mgmtIp) for mgmtIp in switch.mgmtIps ]
         tunnelIpsTemp = [ str(tunnelIp) for tunnelIp in switch.tunnelIps ]
         print 'Management IPs: ' + ', '.join( mgmtIpsTemp )
         print 'Tunnel end-point IPs: ' + ', '.join( tunnelIpsTemp )
         print
         self.ports[ str( switch.uuid ) ].renderTable( 'port', 'vlan' )
         print

   def renderSimple( self ):
      table = TableOutput.createTable( ( 'Switch', 'Management IPs', 'Tunnel IPs',
                                         'Ports' ) )
      table.formatColumns( headFmt, normFmt, normFmt, normFmt )

      for switch in sorted( self.switches.itervalues(),
                            key=lambda x: x.displayName ):
         ports = self.ports[ str( switch.uuid ) ].sortedValues()
         mgmtIps = sorted( switch.mgmtIps )
         tunnelIps = sorted( switch.tunnelIps )
         maxlen = max( len( ports ), len( mgmtIps ), len( tunnelIps ) )
         for i in range( 0, maxlen ):
            table.newRow(
               switch.displayName if i == 0 else '',
               mgmtIps[ i ] if len( mgmtIps ) > i else '',
               tunnelIps[ i ] if len( tunnelIps ) > i else '',
               ports[ i ].displayName if len( ports ) > i else '' )
      print table.output()

   def render( self ):
      if self._detail is None:
         print ''
         return

      if self._detail:
         self.renderDetail()
      else:
         self.renderSimple()

class HscTableRowListFormatter:
   """Class to encapsulate the formatting of a TableFormatter table cell that
      contains a list of items. The TableFormatter makes it easy to list many items
      within a single column, but it is not as easy to compact a list of items in to
      multiple sub-columns. That is what this class attempts to accomplish. The list
      is formatted across multiple sub-columns up to the given maximum column width.
      """
   def __init__( self, itemList, maxColWidth, subColWidth ):
      subColWidth += 1 # One space between cells
      self.strList = [ '{:<{}}'.format( item, subColWidth ) for item in itemList ]
      self.subColsPerCell = int( maxColWidth / subColWidth )

   def __iter__( self ):
      return self

   def next( self ):
      if self.strList:
         cell = ''.join( self.strList[ :self.subColsPerCell ] )
         del self.strList[ :self.subColsPerCell ]
         return cell
      else:
         raise StopIteration

   def itemsLeft( self ):
      return len( self.strList )

   def itemsPerRow( self ):
      return self.subColsPerCell

class HscArpSourceModel( Model ):
   vtepIp = IpGenericAddress( help='The VTEP IP corresponding to a MAC address' )
   uuid = Str( help='The UUID of the ARP entry' )

   def init( self, vtepIp, arp ):
      self.vtepIp = vtepIp
      self.uuid = str( arp.uuid )

class HscArpSourcesModel( Model ):
   __revision__ = 2
   arpSrcs = Dict( keyType=MacAddress, valueType=HscArpSourceModel,
                     help='A mapping between MAC address of HW VTEP and ARP entry.' )
   _detail = Bool( help='Whether extra details are shown.', optional=True )
   _isArpLocal = Bool( default=True, help='Whether ARP entries are local' )

   def degrade( self, dictRepr, revision ):
      raise CliCommon.CliModelNotDegradable( "" )

   def put( self, mac, vtepIp, arp ):
      model = HscArpSourceModel()
      model.init( vtepIp, arp )
      self.arpSrcs[ mac ] = model

   def render( self ):
      vtepCol = 'localVtep' if self._isArpLocal else 'remoteVtep'
      if self._detail:
         self.renderTable( 'arpMac', vtepCol, 'uuid' )
      else:
         self.renderTable( 'arpMac', vtepCol )

   def renderTable( self, *args ):
      table = TableOutput.createTable( [ hscTable[ arg ][ 0 ] for arg in args ] )
      table.formatColumns( *[ hscTable[ arg ][ 1 ] for arg in args ] )
      
      def row( **kwargs ):
         table.newRow( *[ kwargs.get( key, '' ) for key in args ] )

      sortedMac = sorted( self.arpSrcs )
      for srcMac in sortedMac:
         sMac = Ethernet.convertMacAddrToDisplay( srcMac )
         arp = self.arpSrcs[ srcMac ]
         ip = arp.vtepIp
         row( uuid=str( arp.uuid ), remoteVtep='%s' % ip, localVtep='%s' % ip,
               arpMac=sMac )

      print table.output()

class HscPhysicalLocatorsModel( Model ):
   __revision__ = 3
   locators = List( valueType=IpGenericAddress,
                    help='List of VTEP IP.' )
   _detail = Bool( help='Whether extra details are shown.', optional=True )
   _vniInDottedNotation = Bool( default=False,
                                help='True if vni is displayed in dotted notation' )

   def degrade( self, dictRepr, revision ): 
      # Cannot degrade below revision 3 since VNI information is no longer 
      # available in Hsc to have the model compatible with previous versions
      if revision < 3:
         raise CliCommon.CliModelNotDegradable( "" )

   def put( self, locator ):
      self.locators.append( locator.key.dstIp )

   def render( self ):
      # Create string based structure so we just do it once. Also figure out column
      # sizing at the same time.
      headers = [ 'VTEP IP' ]

      # Build the table
      table = TableOutput.createTable( headers )
      table.formatColumns( headFmt, normFmt )
      for ip in sorted( self.locators ):
         table.newRow( ip )
      print table.output()

class HscSimplePhysicalSwitchModel( Model ):
   ports = List( valueType=str,
                 help='The physical ports which comprise the logical switch' )

class HscLogicalSwitchModel( Model ):
   displayName = Str( help='The switch name.' )
   vni = Int( help='The VXLAN Network Identifier (VNI).' )
   pSwitches = Dict( valueType=HscSimplePhysicalSwitchModel,
                     help='A mapping between physical switch name and ports.' )

   def init( self, lSwitch, pSwitches ):
      self.displayName = lSwitch.displayName
      self.vni = lSwitch.tunnelKey
      if pSwitches is not None:
         for psName, ports in pSwitches.iteritems():
            self.pSwitches[ psName ] = \
               HscSimplePhysicalSwitchModel(
                     ports=[ port[ 0 ] for port in ports ] )

class HscLogicalSwitchesModel( Model ):
   __revision__ = 2
   switches = Dict( valueType=HscLogicalSwitchModel,
                    help='A mapping between uuid and configured logical switch.' )
   _detail = Bool( help='Whether extra details are shown.', optional=True )
   _vniInDottedNotation = Bool( default=False,
                                help='True if vni is displayed in dotted notation' )

   def degrade( self, dictRepr, revision ):
      if 1 == revision: 
         dictRepr[ 'detail' ] = self._detail
         dictRepr[ 'vniInDottedNotation' ] = self._vniInDottedNotation
      return dictRepr

   def put( self, ovsdb, uuid, pSwitches ):
      self.switches[ str( uuid ) ] = HscLogicalSwitchModel()
      self.switches[ str( uuid ) ].init( ovsdb.logicalSwitch[ uuid ], pSwitches )

   def renderTable( self, *args ):
      table = TableOutput.createTable( [ hscTable[ arg ][ 0 ] for arg in args ] )
      table.formatColumns( *[ hscTable[ arg ][ 1 ] for arg in args ] )

      def row( **kwargs ):
         table.newRow( *[ kwargs.get( key, '' ) for key in args ] )

      for uuid, switch in sorted( self.switches.iteritems(),
                                  key=lambda (k, v): v.displayName ):
         first = True
         vniStr = VniFormat( switch.vni, self._vniInDottedNotation )
         for pSwitch in sorted( switch.pSwitches ):
            for port in sorted( switch.pSwitches[ pSwitch ].ports ):
               lport = '%s-%s' % ( pSwitch, port )
               if first:
                  row( switch=switch.displayName, vni=vniStr, lport=lport,
                       uuid=uuid )
                  first = False
               else:
                  row( lport=lport )
         if first:
            row( switch=switch.displayName, vni=vniStr, uuid=uuid )

      print table.output()

   def render( self ):
      if self._detail is None:
         print ''
         return

      if self._detail:
         self.renderTable( 'switch', 'vni', 'lport', 'uuid' )
      else:
         self.renderTable( 'switch', 'vni', 'lport' )

class HscSimpleLogicalSwitchModel( Model ):
   displayName = Str( help='The logical switch name.' )
   vni = Int( help='The VXLAN Network Identifier (VNI).', optional=True )

   def toString( self ):
      rep = self.displayName
      if self.vni:
         vniStr = str( VniFormat( self.vni, renderInVniInDottedNotation ) )
         rep += ' (VNI ' + vniStr + ')'
      return rep

class HscStaticRouterNHopsModel( Model ):
   nhops = List( valueType=IpGenericAddress, help='Next hop IP addresses.' )

   def toString( self ):
      return 'via ' + ', '.join( [ str( n ) for n in sorted( self.nhops ) ] )

class HscLogicalRouterModel( Model ):
   switchBindings = Dict( keyType=IpGenericAddrAndMask,
                          valueType=HscSimpleLogicalSwitchModel,
                          help='A mapping between IP/Subnet and logical switch.' )
   staticRoutes = Dict( keyType=IpGenericPrefix,
                        valueType=HscStaticRouterNHopsModel,
                        help='A mapping between an IP prefix and next hop IPs.' )
   uuid = Str( help='The UUID of the logical router.' )

class HscLogicalRoutersModel( Model ):
   routers = Dict( valueType=HscLogicalRouterModel,
                   help='A mapping between logical router name and logical router.' )
   _detail = Bool( help='Whether extra details are shown.', optional=True )

   def degrade( self, dictRepr, revision ):
      return dictRepr

   def put( self, ovsdb, lr ):
      lrModel = HscLogicalRouterModel()
      lrModel.uuid = str( lr.uuid )
      for sb, lsUuid in lr.switchBinding.iteritems():
         lsw = ovsdb.logicalSwitch.get( lsUuid )
         if not lsw:
            continue
         sbLsModel = HscSimpleLogicalSwitchModel()
         sbLsModel.displayName = lsw.displayName
         if lsw.tunnelKey != Tac.Value( 'Vxlan::VniOrNone' ).invalidVni:
            sbLsModel.vni = lsw.tunnelKey
         lrModel.switchBindings[ sb ] = sbLsModel

      for prefix, sr in lr.staticRoute.iteritems():
         srNhopsModel = HscStaticRouterNHopsModel()
         for nhop in sr.nexthop.keys():
            srNhopsModel.nhops.append( nhop )
         lrModel.staticRoutes[ prefix ] = srNhopsModel
      self.routers[ lr.displayName ] = lrModel

   def render( self ):
      if self._detail is None:
         print ''
         return

      headings = [ 'router', 'ipSubnet', 'logicalSwitch', 'staticRoute' ]
      attribs = [ modelKey, ( 'switchBindings', True ), 'staticRoutes' ]
      if self._detail:
         headings.append( 'uuid' )
         attribs.append( 'uuid' )

      sortedLrs = OrderedDict( sorted( self.routers.items(), key=lambda t: t[ 0 ] ) )
      renderTable( headings, sortedLrs, attribs )

class HscDetailModel( Model ):
   __public__ = False
   enabled = Bool( help='Whether OVSDB records exist.' )
   tableOutput = Str( help='The dump of the internal view of the database.' )

   def buildTable( self, ovsdb, noHeadings ):
      def toString( obj ):
         if hasattr( obj, 'iteritems' ):
            if len( obj ) and type( next( obj.itervalues() ) ) is bool:
               # Account for how list are modelled a dict with bool values. Hsc tacc
               # model doesn't model an actual dict where the values are bool types
               return toString( obj.iterkeys() )

            return '{%s}' % ', '.join( [ '%s=%s' % 
                                          ( toString( k ) , toString( obj[ k ] ) )
                                             for k in sorted( obj.keys() ) ] )
         if hasattr( obj, '__iter__' ):
            return '[%s]' % ', '.join( sorted( [ '%s' % toString( k )
                                                   for k in obj ] ) )
         if type( obj ) is Tac.Value( 'Hsc::PhysicalLocatorKey' ).__class__:
            return str( obj.dstIp )

         if type( obj ) is Tac.Value( 'Hsc::StaticRoute' ).__class__:
            # We only need to render the nexthop ip as the key is the prefix
            return toString( obj.nexthop.keys() )

         return str( obj )

      table = {
         'root': [ [ 'name', 'pSwitch', 'manager' ] ],
         'physicalSwitch': [ [ 'name', 'displayName', 'desc', 'port', 'mgmtIp',
                               'tunnelIp' ] ],
         'physicalPort': [ [ 'name', 'displayName', 'desc', 'vlanBinding',
                             'vlanStat' ] ],
         'logicalBindingStats': [ [ 'name', 'inBytes', 'inPackets', 'outBytes',
                                    'outPackets' ] ],
         'logicalSwitch': [ [ 'name', 'displayName', 'desc', 'tunnelKey' ] ],
         'logicalRouterRaw': [ [ 'uuid', 'displayName', 'desc', 'switchBinding',
                                 'staticRoute' ] ],
         'arpSourcesRemote': [ [ 'name', 'srcMac', 'locator' ] ],
         'arpSourcesLocal': [ [ 'name', 'srcMac', 'locator' ] ],
         'physicalLocator': [ [ 'name', 'key' ] ],
         'physicalLocatorSet': [ [ 'name', 'locator' ] ],
      }

      for tableName, attribs in table.iteritems():
         for row in getattr( ovsdb, tableName ).itervalues():
            table[ tableName ].append( [ toString( getattr( 
               row, 'uuid' if attribName=='name' else attribName ) )
               for attribName in attribs[ 0 ] ] )

      # Now actually render the table. While we /could/ do this properly in the
      # render method and have a class for each table, this is only a debug model.
      # Also, don't use the TableOutput suite, we want lines to run on (suitable for
      # redirecting output to a file), not get separated into multiple tables.
      output = []
      for tableName, rows in table.iteritems():
         output.append( tableName )

         # Determine the width of each column (the maximum width of all cells in
         # that column).
         colCount = len( rows[ 0 ] )
         widths = list( ( 0, ) * colCount )
         for row in rows:
            for cell in range( 0, colCount ):
               if len( row[ cell ] ) > widths[ cell ]:
                  width = len( row[ cell ] )
                  # Empty cells in a column will be filled with spaces.  Keep the
                  # column width under 2000 chars to avoid an assert in the vt100
                  # emulator used to gather CLI output (see emulate() in
                  # CliTestClient/__init__.py).  This is only done when the
                  # no-headings keyword is used as that is done in the test context.
                  #
                  # This will not cause any data to be truncated, just cell-padding.
                  if noHeadings and width >= 2000:
                     width = 1999
                  widths[ cell ] = width

         # Add each row to the output.
         first = True
         for row in rows:
            if first and noHeadings:
               first = False
               continue

            output.append( ' '.join( [ row[ cell ].ljust( widths[ cell ] )
                                       for cell in range( 0, colCount ) ] ) )
            if first:
               output.append( ' '.join( [ '-' * widths[ cell ]
                                          for cell in range( 0, colCount ) ] ) )
               first = False

         # End the table with a blank line.
         output.append( '' )
      self.tableOutput = '\n'.join( output )

   def render( self ):
      print self.tableOutput

class HscDetailOvsdbModel( Model ):
   __public__ = False
   database = Str( help='The current contents of the OVSDB database.' )

   def render( self ):
      print self.database

