# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliParser, CliPlugin, BasicCli, CliModel, Tac
import ConfigMount, LazyMount
import os, json, re, functools
import HscAgent
import Arnet
import CliMatcher
import ShowCommand
import CliCommand
import BasicCliModes

from CliPlugin.ControllerCli import addNoCvxCallback, CvxConfigMode
from CliPlugin.ControllerCli import serviceKwMatcher
from CliPlugin.Ssl import profileMatcher, sslMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin import HscModel
from CliMode.Hsc import HscMode
from HscCliLib import ovsdbLogTypes
from VxlanVniLib import VniFormat
from VxlanVniLib import VniMatcher
from CliToken.Reset import resetMatcher


from HscModel import \
      SslProfileModel, \
      HscStatusModel, \
      HscAgentStatusModel, \
      OvsdbManagerModel, \
      HscCertificateModel, \
      ovsdbSessionStatusFields, \
      HscOvsdbSessionStatusModel, \
      HscPhysicalSwitchesModel, \
      HscPhysicalLocatorsModel, \
      HscPhysicalPortsModel, \
      HscLogicalSwitchesModel, \
      HscLogicalRoutersModel, \
      HscDetailModel, \
      HscDetailOvsdbModel, \
      HscArpSourcesModel

# To Box vxlanCntrlConfig for VniMatcher
vxlanCtrlConfigBox = []

matcherArp = CliMatcher.KeywordMatcher( 'arp',
      helpdesc='Show information related to ARP' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Show the contents of the OVSDB database' )
matcherExtraDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Show extra details' )
nodeExtraDetailHidden = CliCommand.Node(
      matcher=matcherExtraDetail, hidden=True )
vniIdMatcher = VniMatcher( "VXLAN Network Identifier", vxlanCtrlConfigBox )
matcherHsc = CliMatcher.KeywordMatcher( 'hsc',
      helpdesc='Show information about the HSC service' )
matcherLocator = CliMatcher.KeywordMatcher( 'locator',
      helpdesc='Show information related to a locator' )
matcherLogical = CliMatcher.KeywordMatcher( 'logical',
      helpdesc='Show information related to logical devices' )
matcherName = CliMatcher.KeywordMatcher( 'name',
      helpdesc='Name of the device to search for' )
matcherNoHeadings = CliMatcher.KeywordMatcher( 'no-headings',
      helpdesc='Do not display column headings' )
matcherPhysical = CliMatcher.KeywordMatcher( 'physical',
      helpdesc='Show information related to physical devices' )
matcherPhysicalSwitch = CliMatcher.KeywordMatcher( 'physical-switch',
      helpdesc='Physical switch name' )
matcherSources = CliMatcher.KeywordMatcher( 'sources',
      helpdesc='Show information related to ARP' )
matcherStatus = CliMatcher.KeywordMatcher( 'status',
      helpdesc='Show status information' )
matcherSwitch = CliMatcher.KeywordMatcher( 'switch',
      helpdesc='Show information related to a switch' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Virtual LAN on a physical port' )
matcherVni = CliMatcher.KeywordMatcher( 'vni',
      helpdesc='VXLAN Network Identifier' )
vlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Identifier for a Virtual LAN' )
matcherManager = CliMatcher.KeywordMatcher( 'manager',
       helpdesc='Set the IP and port of the HSC manager (requires SSL)' )
matcherPassive = CliMatcher.KeywordMatcher( 'passive',
       helpdesc='Reconfigure the passive TCP port of OVSDB server' )
matcherPort = CliMatcher.KeywordMatcher( 'port',
       helpdesc='Reconfigure the passive TCP port of OVSDB server' )
matcherTcp = CliMatcher.KeywordMatcher( 'tcp',
       helpdesc='Reconfigure the passive TCP port of OVSDB server' )

vxlanCntrlConfig = None

def forceMountWithActivityLock( func ):
   @functools.wraps( func )
   def _forceMountWithActivityLock( *margs, **kwargs ):
      # The LazyMount objects must be force mounted before grabbing the activity lock
      # otherwise a deadlock will occur between the blocking mount and the activity
      # thread.
      LazyMount.force( controllerConfig )
      LazyMount.force( config )
      LazyMount.force( status )
      LazyMount.force( vxlanCntrlConfig )

      with Tac.ActivityLockHolder():
         return func( *margs, **kwargs )

   return _forceMountWithActivityLock

# This CLI plugin defines the following modes and configuration commands:
#   cvx
#     service hsc
#       [ no|default ] shutdown
#       [ no|default ] manager <ip> [ <port> ]
#       [ no|default ] ovsdb-shutdown
#       [ no|default ] vtep flood list type [ all | any ]
#       [ no|default ] logical-router mac-address
#       [ no|default ] ssl profile
#
#       ! Below command is hidden until we have stable L3 Indirect support from
#       [ no|default ] routing [ direct | indirect ]
#
#       ! Below command is hidden and used to reconfigure the OVSDB server
#            to start listening for tcp sessions
#       [ no|default ] passive tcp port <port>
#
#       ! Below commands are hidden and used to configurey OVSDB error reporting
#       [ no|default ] error-reporting
#
#       ! Below commands are hidden and used to control OVSDB output
#       [ no|default ] log-console <emergency/error/warning/information/debug>
#       [ no|default ] log-syslog <emergency/error/warning/information/debug>
#       [ no|default ] log-file <emergency/error/warning/information/debug>
#
#       ! Below commands are hidden and used to toggle the location of vtep.db
#       ! between /mnt/flash/openvswitch and /var/run/openvswitch
#       [ no|default ] persist-database
#
# This CLI plugin also defines the following show commands (all CAPI capable unless
# noted):
#   show hsc status [ detail ]
#      |- Show the HSC service status.  If detail is specified, show the status of
#         the agents.
#
#   show hsc status ovsdb
#      |- Show the OVSDB session status for the specified agent.
#
#   show hsc certificate
#      |- Shows the SSL certificate used to connect to HSC.
#
#   show hsc physical switch [ name <name> ] [ ip <ip> ] [ vlan <vlan-id> ]
#               [ vni <vni> ] [ detail ]
#      |- Show filtered information about the physical switches
#
#   show hsc physical port [ name <name> ] [ physical-switch <switch> ]
#               [ vlan <vlan-id> ] [ vni <vni> ] [ detail ]
#      |- Show filtered information about the physical ports
#
#   show hsc physical locator [ ip <ip> ] [ vni <vni> ] [ detail ]
#      |- Show filtered information about the physical locators 
#
#   show hsc logical switch [ name <name> ] [ physical-switch <switch> ]
#               [ physical-port <port> ] [ vlan <vlan-id> ] [ vni <vni> ] [ detail ]
#      |- Show filtered information about the logical switches
#
#   show hsc logical router [ <name> ] [ detail ]
#      |- Show filtered information about the logical routers
#
#   ! The following commands are hidden and used to gather raw debugging data.
#   show hsc detail
#      |- Dump the internal agent state.
#
#   show hsc detail ovsdb [ database ] [ socket ]
#      |- Dump the data in OVSDB.
#
# This CLI plugin also defines the following reset command:
#   reset hsc certificate
#      |- Delete the existing OVSDB certificates and keys and re-generate them.

defaults = Tac.Value( 'Hsc::CliDefaults' )
ovsdbLogLevel = Tac.Type( 'Hsc::OvsdbLogLevel' )
sslConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

#------------------------------------------------------------------------------
# Common CLI Token Definitions
#------------------------------------------------------------------------------

def getSessionStateStr( sessionStatus ):
   if sessionStatus is None:
      return 'Unknown'

   sessionState = Tac.Type( 'Hsc::SessionState' )

   if sessionStatus.sessionState == sessionState.stateConnected:
      return 'Connected'
   elif sessionStatus.sessionState == sessionState.stateDisconnected:
      return 'Disconnected'
   else:
      return 'Unknown'

#-----------------------------------------------------------------------------
# HSC Mode
#------------------------------------------------------------------------------

class HscConfigMode( HscMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'cvx-hsc'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      HscMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setShutdown( self, args ):
      hscServiceConfig = serviceConfigDir.service[ HscAgent.name() ]
      cliConfig.enableAgent = hscServiceConfig.enabled = False

   def noShutdown( self, args ):
      if not controllerConfig.enabled:
         self.addError( "Hsc service cannot be enabled until the CVX server is "
                        "enabled" )
         return

      hscServiceConfig = serviceConfigDir.service[ HscAgent.name() ]
      cliConfig.enableAgent = hscServiceConfig.enabled = True

   def setOvsdbShutdown( self, args ):
      cliConfig.enableOvsdb = False

   def noOvsdbShutdown( self, args ):
      cliConfig.enableOvsdb = True

   def noErrorReporting( self, args ):
      cliConfig.enableErrorReporting = False

   def defErrorReporting( self, args ):
      cliConfig.enableErrorReporting = defaults.enableErrorReporting

   def setErrorReporting( self, args ):
      cliConfig.enableErrorReporting = True

   def setVtepListType( self, args ):
      sendMode = args.get( 'SENDTO' )
      if sendMode == 'all':
         cliConfig.vtepListType = 'sendToAll'
      else:
         cliConfig.vtepListType = 'sendToAny'

   def deleteManager( self, args ):
      ipAddr = args.get( 'IPADDR ' )

      def findManagerIp( mgrIp ):
         for target in cliConfig.managerAddress.keys():
            ip = target.split( ':' )[ 1 ]
            if  ip == mgrIp:
               return target
         return None

      if ipAddr is None:
         # No ip address specified. Delete all managers
         for target in cliConfig.managerAddress.keys():
            del cliConfig.managerAddress[ target ]
      else:
         target = findManagerIp( ipAddr )
         if target is not None:
            del cliConfig.managerAddress[ target ]

   def isValidManagerIp( self, ipAddr ):

      err = ""
      if ipAddr == '0.0.0.0':
         err = "Manager IP cannot be zero. Please enter a valid IP"
         return ( False, err )

      return ( True, err )

   def setManager( self, args ):
      mgrIp = args[ 'IPADDR' ]
      port = args.get( 'PORT' )

      ( valid, err ) = self.isValidManagerIp( mgrIp )
      if not valid:
         self.addError( err )
         return

      if port is None:
         mgrPort = defaults.managerPort
      else:
         mgrPort = int( port )

      def findManagerIp( mgrIp ):
         for target in cliConfig.managerAddress.keys():
            ip = target.split( ':' )[ 1 ]
            port = target.split( ':' )[ 2 ]
            if  ip == mgrIp:
               return ( ip, int( port ) )
         return None

      oldTarget = findManagerIp( mgrIp )
      target = "ssl:%s:%s" % ( mgrIp, mgrPort )

      if oldTarget is None:
         # IP does not exist. Add the new manager
         cliConfig.managerAddress[ target ] = True
      else: 
         ( oldIp, oldPort ) = oldTarget

         # Check to see if port has changed. If so, add the
         # new ip/port
         if oldPort != mgrPort:
            toDelete = "ssl:%s:%s" % ( oldIp, oldPort )
            del cliConfig.managerAddress[ toDelete ]
            cliConfig.managerAddress[ target ] = True

   def setPassiveTcpPort( self, args ):
      port = args.get( 'PORT' )

      if port is None:
         cliConfig.passiveTcpPort = defaults.passiveTcpPort
      else:
         cliConfig.passiveTcpPort = port

   def setRoutingMode( self, args ):
      routingMode = args.get( 'ROUTING' )
      if routingMode is not None:
         cliConfig.routingModeType = routingMode
      else:
         cliConfig.routingModeType = 'direct'

   def setLogLevel( self, args ):
      logDest = args[ 'LOGDEST' ]
      spec = ovsdbLogTypes[ logDest ]

      levelVal = args.get( 'LOGLEVEL', spec.default )

      spec.getSet( cliConfig, levelVal )

   def setPersistentDatabase( self, args ):
      cliConfig.persistDatabase =  True
   
   def noPersistentDatabase( self, args ):
      cliConfig.persistDatabase =  False

def gotoHscMode( mode, args ):
   childMode = mode.childMode( HscConfigMode )
   mode.session_.gotoChildMode( childMode )

def noHsc( mode, args=None ):
   cliConfig.reset()

   hscServiceConfig = serviceConfigDir.service[ HscAgent.name() ]
   hscServiceConfig.enabled = False

addNoCvxCallback( noHsc )

#--------------------------------------------------------------------------------
# [ no | default ] service hsc
#--------------------------------------------------------------------------------
class ServiceHscCmd( CliCommand.CliCommandClass ):
   syntax = 'service hsc'
   noOrDefaultSyntax = syntax
   data = {
      'service': serviceKwMatcher,
      'hsc': matcherHsc,
   }
   handler = gotoHscMode
   noOrDefaultHandler = noHsc

CvxConfigMode.addCommandClass( ServiceHscCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown the HSC service',
   }

   handler = HscConfigMode.setShutdown
   noHandler = HscConfigMode.noShutdown
   defaultHandler = handler

HscConfigMode.addCommandClass( ShutdownCmd )

#------------------------------------------------------------------------------
# (config-hsc)# [ no|default ] ovsdb-shutdown
#------------------------------------------------------------------------------

class OvsdbShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'ovsdb-shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'ovsdb-shutdown': 'Shutdown the OVSDB server',
   }

   handler = HscConfigMode.setOvsdbShutdown
   noHandler = HscConfigMode.noOvsdbShutdown
   defaultHandler = handler

HscConfigMode.addCommandClass( OvsdbShutdownCmd )


#--------------------------------------------------------------------------------
# [ no | default ] vtep flood list type ( all | any )
#--------------------------------------------------------------------------------
class VtepFloodListTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'vtep flood list type SENDTO'
   noOrDefaultSyntax = 'vtep flood list type ...'
   data = {
      'vtep': 'Vtep property',
      'flood': 'VTEP flood list property',
      'list': 'VTEP flood list property',
      'type': 'VTEP flood list property',
      'SENDTO': CliMatcher.EnumMatcher( {
         'all': 'Send to all the VTEPs',
         'any': 'Send to any one of the VTEPs'
      } )
   }
   handler = HscConfigMode.setVtepListType
   noOrDefaultHandler = HscConfigMode.setVtepListType

HscConfigMode.addCommandClass( VtepFloodListTypeCmd )

#------------------------------------------------------------------------------
# (config-hsc)# [no|default] manager <ip> [ <port> ]
#------------------------------------------------------------------------------

class ManagerCmd( CliCommand.CliCommandClass ):
   syntax = 'manager IPADDR [ PORT ]'
   noOrDefaultSyntax = 'manager [ IPADDR ] ...'
   data = {
      'manager': matcherManager,
      'IPADDR': IpAddrMatcher( helpdesc='IP address' ),
      'PORT': CliMatcher.IntegerMatcher( 1, 65535, 
         helpdesc='Port (default is 6632)' ),
   }
   noOrDefaultHandler = HscConfigMode.deleteManager
   handler = HscConfigMode.setManager

HscConfigMode.addCommandClass( ManagerCmd )

#------------------------------------------------------------------------------
# (config-hsc)# [no|default] passive tcp port <port>
#------------------------------------------------------------------------------
class PassiveTcpPortCmd( CliCommand.CliCommandClass ):
   syntax = 'passive tcp port PORT'
   noOrDefaultSyntax = 'passive tcp port ...'
   data = {
      'passive': matcherPassive,
      'tcp': matcherTcp,
      'port': matcherPort,
      'PORT': CliMatcher.IntegerMatcher( 1, 65535, 
         helpdesc='Port (default is 0)' ), 
   }
   handler = HscConfigMode.setPassiveTcpPort
   noOrDefaultHandler = handler

HscConfigMode.addCommandClass( PassiveTcpPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] routing ( direct | indirect )
#--------------------------------------------------------------------------------
class RoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'routing ROUTING'
   noOrDefaultSyntax = 'routing ...'
   data = {
      'routing': 'Routing mode',
      'ROUTING': CliMatcher.EnumMatcher( {
         'direct': 'Direct routing mode',
         'indirect': 'Indirect routing mode'
      } )
   }
   handler = HscConfigMode.setRoutingMode
   noOrDefaultHandler = HscConfigMode.setRoutingMode

HscConfigMode.addCommandClass( RoutingCmd )


#------------------------------------------------------------------------------
# (config-hsc)# [ no | default ] error-reporting
#------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
class ErrorReportingCmd( CliCommand.CliCommandClass ):
   syntax = 'error-reporting'
   noOrDefaultSyntax = syntax
   data = {
      'error-reporting': 'Enable error reporting',
   }

   handler = HscConfigMode.setErrorReporting
   noHandler = HscConfigMode.noErrorReporting
   defaultHandler = HscConfigMode.defErrorReporting

HscConfigMode.addCommandClass( ErrorReportingCmd )

#------------------------------------------------------------------------------
#------------------------------------------------------------------------------

def getLogDestination( mode ):
   a = {}
   for key, val in ovsdbLogTypes.iteritems():
      a[ key ]  = val.helpPrefix + ' logging level '

   return a

#--------------------------------------------------------------------------------
# [ no | default ] log-[console|syslog|file] 
#     ( disabled | information | warning | emergency | error | debug )
#--------------------------------------------------------------------------------
class LogConsoleCmd( CliCommand.CliCommandClass ):
   syntax = 'LOGDEST LOGLEVEL'
   noOrDefaultSyntax = 'LOGDEST ...'
   data = {
      'LOGDEST': CliMatcher.DynamicKeywordMatcher( getLogDestination ),
      'LOGLEVEL': CliMatcher.EnumMatcher( {
         'disabled': 'Disabled',
         'information': 'Information level or higher messages',
         'warning': 'Warning level or higher messages',
         'emergency': 'Emergency level or higher messages',
         'error': 'Error level or higher messages',
         'debug': 'Debug level or higher messages',
      } )
   }
   handler = HscConfigMode.setLogLevel
   noOrDefaultHandler = HscConfigMode.setLogLevel

HscConfigMode.addCommandClass( LogConsoleCmd )

#------------------------------------------------------------------------------
# (config-hsc)# [ no|default ] persist-database
#--------------------------------------------------------------------------------
class PersistDatabaseCmd( CliCommand.CliCommandClass ):
   syntax = 'persist-database'
   noOrDefaultSyntax = syntax
   data = {
      'persist-database': 'Save database to persistent storage',
   }

   handler = HscConfigMode.setPersistentDatabase
   noHandler = HscConfigMode.noPersistentDatabase
   defaultHandler = handler

HscConfigMode.addCommandClass( PersistDatabaseCmd )

def hscSetSslProfile( mode, args ):
   cliConfig.sslProfile = args.get( 'PROFILENAME', '' )

#--------------------------------------------------------------------------------
# ( no | default ) ssl profile ...
#--------------------------------------------------------------------------------
class SslProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': sslMatcher,
      'profile': profileMatcher,
      'PROFILENAME': CliMatcher.DynamicNameMatcher( 
         lambda mode: sslConfig.profileConfig, 'Profile name' )
   }
   handler = hscSetSslProfile
   noOrDefaultHandler = handler

HscConfigMode.addCommandClass( SslProfileCmd )

#------------------------------------------------------------------------------
# show hsc status
#------------------------------------------------------------------------------

def isRunning( pidFilename ):
   if os.path.exists( pidFilename ):
      with open( pidFilename, 'r' ) as pidFile:
         pid = pidFile.readline()

      if pid != '' and os.path.exists( '/proc/%s' % ( pid[ :-1 ], ) ):
         return True

   return False

def addHscStatusDetail( mode, model ):
   def hscAgentStatus( mode ):
      agentFileName = '/var/run/agents/%s.%s' % ( mode.sysname, HscAgent.name() )

      if isRunning( agentFileName ):
         return 'Running'
      else:
         return 'Stopped'

   model.agent = HscAgentStatusModel( state=hscAgentStatus( mode ), 
         ovsdbConnection=getSessionStateStr( status.ovsdbSessionStatus ) )

def getOvsdbState( stoppedDesc='Stopped' ):
   if not isRunning( config.ovsdbPidFile ):
      return stoppedDesc

   if os.path.exists( config.ovsdbSocketFile ):
      return 'Running'
   else:
      return 'Started'

def showHscStatus( mode, args ):
   detail = 'detail' in args

   model = HscStatusModel( enabled=cliConfig.enableAgent,
                           vtepFloodMode=status.vtepListType,
                           enableErrorReporting=cliConfig.enableErrorReporting,
                           routingMode=status.routingModeType )

   if not model.enabled:
      return model

   if detail:
      addHscStatusDetail( mode, model )

   query = '[ "hardware_vtep", ' \
           '{ "op": "select", "table": "Manager", ' \
             '"columns": [ "is_connected", "status", "target" ], "where": [] } ]'

   argv = [ 'ovsdb-client', 'transact', 'unix:%s' % ( config.ovsdbSocketFile, ),
            '%s' % ( query, ) ]

   if not cliConfig.enableOvsdb:
      model.ovsdbState = 'Disabled'
      return model

   model.ovsdbState = getOvsdbState()
   if model.ovsdbState != 'Running':
      return model

   dataStr = ''

   try:
      dataStr = Tac.run( argv, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
   except ( Tac.Timeout, Tac.SystemCommandError ):
      model.ovsdbState = 'Failed'
      return model

   data = json.loads( dataStr )
   rows = data[0][ 'rows' ]

   # check if rows are empty because HscAgent is waiting for topology or
   # VCS to converge
   if not rows and getSessionStateStr( status.ovsdbSessionStatus ) == 'Connected':
      model.ovsdbState = 'WaitingForConvergence'
      return model

   for row in rows:
      mmodel = OvsdbManagerModel()

      mmodel.manager = row[ 'target' ].encode( 'utf-8' )
      mmodel.state = 'unknown'
      mmodel.uptime = 0.

      for data in row[ 'status' ][1]:
         if data[0] == 'state':
            mmodel.state = data[1]
         if data[0] == 'sec_since_connect':
            mmodel.uptime = Tac.utcNow() - float( data[1] )   
         if data[0] == 'locks_held':
            # If HSC is connected to multiple managers a '*' in manager name 
            # indicates that this manager has acquired the lock
            mmodel.manager += '*'

      if row[ 'is_connected' ] == True:
         mmodel.connected = True
      else:
         mmodel.connected = False

      model.managers.append( mmodel )

   model.fipsEnabled = False
   model.tlsProtocol = [ '1.0', '1.1', '1.2' ]

   profile = cliConfig.sslProfile
   if profile:
      profileStatus = sslStatus.profileStatus.get( profile )
      if profileStatus is None:
         model.sslProfile = SslProfileModel( name=profile,
                                             configured=False )
      else:
         profileState = profileStatus.state
         model.sslProfile = SslProfileModel( name=profile,
                                             configured=True,
                                             state=profileState )
         model.fipsEnabled = profileStatus.fipsMode
         tlsVersionsEnabled = []
         tlsVersionMap = [ ( sslConstants.tlsv1, '1.0' ),
                           ( sslConstants.tlsv1_1, '1.1' ),
                           ( sslConstants.tlsv1_2, '1.2' ) ]
         for mask, version in tlsVersionMap:
            if profileStatus.tlsVersion & mask:
               tlsVersionsEnabled.append( version )
         if tlsVersionsEnabled:
            model.tlsProtocol = tlsVersionsEnabled

   return model

class HscStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc status [ detail ]'
   data = {
      'hsc': matcherHsc,
      'status': matcherStatus,
      'detail': matcherExtraDetail,
   }
   handler = showHscStatus
   cliModel = HscStatusModel

BasicCli.addShowCommandClass( HscStatusCmd )

#------------------------------------------------------------------------------
# show hsc status ovsdb
#------------------------------------------------------------------------------

def showHscStatusOvsdb( mode, args ):
   detail = 'detail' in args

   model = HscOvsdbSessionStatusModel()
   source = status.ovsdbSessionStatus

   model.ovsdbState = getOvsdbState()
   # pylint: disable-msg=W0212
   model._ptcpPort = cliConfig.passiveTcpPort

   if source is None:
      return model

   for entry in ovsdbSessionStatusFields:
      field = entry[0]

      if field == 'sessionState':
         model.sessionState = getSessionStateStr( source )
         continue

      data = source.__getattribute__( field )
      attr = model.__attributes__.get( field )
      if isinstance( attr, CliModel.Int ):
         data = int( data )
      elif isinstance( attr, CliModel.Float ):
         data = float( data )

      model.__setattr__( field, data )

   if detail:
      model.lastMsgSentDetail = source.lastMsgSentDetail
      model.lastSuccessfulMsgDetail = source.lastSuccessfulMsgDetail
      model.lastFailedMsgDetail = source.lastFailedMsgDetail

   return model

class HscStatusOvsdbCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc status ovsdb [ detail ]'
   data = {
      'hsc': matcherHsc,
      'status': matcherStatus,
      'ovsdb': 'OVSDB status information',
      'detail': nodeExtraDetailHidden,
   }
   handler = showHscStatusOvsdb
   cliModel = HscOvsdbSessionStatusModel

BasicCli.addShowCommandClass( HscStatusOvsdbCmd )

#------------------------------------------------------------------------------
# show hsc certificate
#------------------------------------------------------------------------------

def showHscCert( mode, args ):
   model = HscCertificateModel( certificate='' )

   profile = cliConfig.sslProfile
   profileStatus = sslStatus.profileStatus.get( profile )

   if ( profileStatus and
        profileStatus.state == "valid" and
        profileStatus.certKeyPath ):
      with open( profileStatus.certKeyPath, 'r' ) as f:
         cert = f.read().rpartition( 'PRIVATE KEY-----\n' )[ -1 ]
         model.certificate = cert
   elif os.path.exists( config.ovsdbCertFile ):
      with open( config.ovsdbCertFile, 'r' ) as f:
         model.certificate = f.read()
   return model

class HscCertificateCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc certificate'
   data = {
      'hsc': matcherHsc,
      'certificate': 'Show the HSC SSL certificate',
   }
   handler = showHscCert
   cliModel = HscCertificateModel

BasicCli.addShowCommandClass( HscCertificateCmd )

#------------------------------------------------------------------------------
# show hsc arp sources [remote|local] [detail]
#------------------------------------------------------------------------------

@forceMountWithActivityLock
def showHscArpSources( mode, args ):
   detail = 'detail' in args
   arpType = args.get( 'SOURCE' )

   model = HscArpSourcesModel()
   if not status.ovsdb:
      mode.addError( 'OVSDB is not connected.' )
      return model

   if not status.running:
      return model

   # pylint: disable-msg=W0212
   model._isArpLocal = arpType == 'local'
   model._detail = detail
   arpSrcs = status.ovsdb.arpSourcesRemoteRaw
   if arpType == 'local':
      arpSrcs = status.ovsdb.arpSourcesLocalRaw

   phyLocators = status.ovsdb.physicalLocatorRaw
   for arp in arpSrcs.itervalues():
      if arp.locator in phyLocators:
         ip = phyLocators[ arp.locator ].key.dstIp
         srcMac = arp.srcMac
         model.put( srcMac, ip, arp )

   return model

#--------------------------------------------------------------------------------
# show hsc arp sources ( local | remote ) [ detail ]
#--------------------------------------------------------------------------------
class HscArpSourcesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc arp sources SOURCE [ detail ]'
   data = {
      'hsc': matcherHsc,
      'arp': matcherArp,
      'sources': matcherSources,
      'SOURCE': CliMatcher.EnumMatcher( {
            'local': 'Show information related to ARP entries of local VTEPs',
            'remote': 'Show information related to ARP entries of remote VTEPs',
         } ),
      'detail': nodeExtraDetailHidden,
   }
   handler = showHscArpSources
   cliModel = HscArpSourcesModel

BasicCli.addShowCommandClass( HscArpSourcesCmd )

#------------------------------------------------------------------------------
# show hsc physical switch [name <name>] [ip <ip>] [vlan <vlan-id>] [vni <vni>]
#   [detail]
#------------------------------------------------------------------------------

@forceMountWithActivityLock
def showHscPhySwitch( mode, args ):
   name = args.get( 'SWITCH' )
   ip = args.get( 'IPADDR' )
   vlan = args.get( 'VLANRANGE' )
   vniDotted = args.get( 'VNI' )
   detail = 'detail' in args

   model = HscPhysicalSwitchesModel()
   if not status.ovsdb:
      mode.addError( 'OVSDB is not connected.' )
      return model

   if not status.running:
      return model

   # pylint: disable-msg=W0212
   model._detail = detail
   model._vniInDottedNotation = vxlanCntrlConfig.vniInDottedNotation
   vni = VniFormat( vniDotted ).toNum() if vniDotted else None

   switches = status.ovsdb.physicalSwitch
   for sn, switch in switches.iteritems():
      # Apply filters
      if name and sn != name:
         continue
      if ip and Arnet.IpGenAddr( ip ) not in switch.mgmtIp:
         continue
      ports = [ status.ovsdb.physicalPort[ puuid ] for puuid in switch.port
                if puuid in status.ovsdb.physicalPort ]
      if vlan and vlan not in [ v for p in ports for v in p.vlanBinding ]:
         continue
      if vni and vni not in [ status.ovsdb.logicalSwitch[ lsid ].tunnelKey
                              for p in ports
                              for lsid in p.vlanBinding.values()
                              if lsid in status.ovsdb.logicalSwitch ]:
         continue

      # Passed all filters
      model.put( status.ovsdb, switch.uuid, switch )

   return model

class HscPhysicalSwitchCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc physical switch [ name SWITCH ] [ ip IPADDR ] '\
         '[ vlan VLANRANGE ] [ vni VNI ] [ detail ]'
   data = {
      'hsc': matcherHsc,
      'physical': matcherPhysical,
      'switch': matcherSwitch,
      'name': matcherName,
      'SWITCH': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z:\\.\\-\\_]+',
         helpdesc='Name of the physical switch', helpname='WORD' ),
      'ip': 'IP address of a management port',
      'IPADDR': IpAddrMatcher( helpdesc='IP address' ),
      'vlan': matcherVlan,
      'VLANRANGE': vlanIdMatcher,
      'vni': matcherVni,
      'VNI': vniIdMatcher,
      'detail': nodeExtraDetailHidden,
   }
   handler = showHscPhySwitch
   cliModel = HscPhysicalSwitchesModel

BasicCli.addShowCommandClass( HscPhysicalSwitchCmd )

#------------------------------------------------------------------------------
# show hsc physical locator vni <vni> [ detail ]
#------------------------------------------------------------------------------

@forceMountWithActivityLock
def deprecateShowHscPhyLocatorVni( mode, args ):
   model = HscPhysicalLocatorsModel()

   mode.addError( 'The vni filter for show hsc physical locator has been '
                  'deprecated.' )

   return model

class HscPhysicalLocatorVniCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc physical locator vni VNI [ detail ]'
   data = {
      'hsc': matcherHsc,
      'physical': matcherPhysical,
      'locator': matcherLocator,
      'vni': matcherVni,
      'VNI': vniIdMatcher,
      'detail': nodeExtraDetailHidden,
   }
   handler = deprecateShowHscPhyLocatorVni
   cliModel = HscPhysicalLocatorsModel
   hidden = True

BasicCli.addShowCommandClass( HscPhysicalLocatorVniCmd )

#------------------------------------------------------------------------------
# show hsc physical locator [ ip <ip> ] [ detail ]
#------------------------------------------------------------------------------
@forceMountWithActivityLock
def showHscPhyLocator( mode, args ):
   ip = args.get( 'IPADDR' )
   detail = 'detail' in args

   model = HscPhysicalLocatorsModel()
   if not status.ovsdb:
      mode.addError( 'OVSDB is not connected.' )
      return model

   if not status.running:
      return model

   # pylint: disable-msg=W0212
   model._detail = detail is not None
   model._vniInDottedNotation = vxlanCntrlConfig.vniInDottedNotation

   locators = status.ovsdb.physicalLocator
   for locator in locators.values():
      # Apply filters
      if ip and ip != str( locator.key.dstIp ):
         continue

      # Passed all filters
      model.put( locator )

   return model

class HscPhysicalLocatorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc physical locator [ ip IPADDR ] [ detail ]'
   data = {
      'hsc': matcherHsc,
      'physical': matcherPhysical,
      'locator': matcherLocator,
      'ip': 'IP address of a physical locator',
      'IPADDR': IpAddrMatcher( helpdesc='IP address' ),
      'detail': nodeExtraDetailHidden,
   }
   handler = showHscPhyLocator
   cliModel = HscPhysicalLocatorsModel

BasicCli.addShowCommandClass( HscPhysicalLocatorCmd )

#------------------------------------------------------------------------------
# show hsc physical port [name <name>] [physical-switch <switch>]
#   [vlan <vlan-id>] [vni <vni>] [detail]
#------------------------------------------------------------------------------

# Match contractions of port names, such as po1 = Port-Channel1 and eth2 = Ethernet2.
# Do this by splitting off the letters.
def portMatch( fullName, shortName ):
   if fullName == shortName:
      return True

   # Aside: if shortName ends with a letter, split ends with an empty string. This is
   # quite nice, it means 'po' can be used to mean 'all Port-Channel interfaces'.
   split = re.split( r'([A-Za-z\-]+)', shortName )
   # On the other hand, if shortName begins with a letter (normal), we don't want to
   # match anything before it.
   if split[ 0 ] == '':
      del split[ 0 ]
   # Special case: eth1 should match Ethernet1/2, but not Ethernet13. To achieve
   # this, append (/|$) to the regular expression (match / or EOL).
   regex = r'[A-Za-z\-]*'.join( split ) + '(/|$)'
   return re.match( regex, fullName, re.IGNORECASE ) is not None

@forceMountWithActivityLock
def showHscPhyPort( mode, args ):
   name = args.get( 'PORT' )
   switch = args.get( 'SWITCH' )
   vlan = args.get( 'VLANRANGE' )
   vniDotted = args.get( 'VNI' )
   detail = 'detail' in args

   model = HscPhysicalPortsModel()
   if not status.ovsdb:
      mode.addError( 'OVSDB is not connected.' )
      return model

   if not status.running:
      return model

   # pylint: disable-msg=W0212
   model._detail = detail
   model._vniInDottedNotation = vxlanCntrlConfig.vniInDottedNotation
   vni = VniFormat( vniDotted ).toNum() if vniDotted else None

   # Generate mapping of physical port to switch name now to save time later
   pPortSwitchNames = {} # Indexed by port UUID
   # Loop through all physical switches
   for pSwitch in status.ovsdb.physicalSwitch.itervalues():
      # Loop through all ports for the physical switch
      for puuid in pSwitch.port:
         pPortSwitchNames[ puuid ] = pSwitch.displayName

   ports = status.ovsdb.physicalPort
   for uuid, port in ports.iteritems():
      # Apply filters
      if name and not portMatch( port.displayName, name ):
         continue
      if switch and ( switch not in status.ovsdb.physicalSwitch or
                      uuid not in status.ovsdb.physicalSwitch[ switch ].port ):
         continue
      if vlan and vlan not in port.vlanBinding:
         continue
      if vni and vni not in [ status.ovsdb.logicalSwitch[ lsid ].tunnelKey
                              for lsid in port.vlanBinding.values() ]:
         continue

      # Passed all filters
      model.put( status.ovsdb, uuid, port, pPortSwitchNames.get( uuid ) )

   return model

class HscPhysicalPortCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc physical port [ name PORT ] [ physical-switch SWITCH ] '\
      '[ vlan VLANRANGE ] [ vni VNI ] [ detail ]'
   data = {
      'hsc': matcherHsc,
      'physical': matcherPhysical,
      'port': 'Show information related to an interface port',
      'name': matcherName,
      'PORT': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z\\/\\-]+',
         helpdesc='Physical port name', helpname='WORD' ),
      'physical-switch': matcherPhysicalSwitch,
      'SWITCH': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z:\\.\\-\\_]+',
         helpdesc='Name of the physical switch', helpname='WORD' ),
      'vlan': matcherVlan,
      'VLANRANGE': vlanIdMatcher,
      'vni': matcherVni,
      'VNI': vniIdMatcher,
      'detail': nodeExtraDetailHidden,
   }
   handler = showHscPhyPort
   cliModel = HscPhysicalPortsModel

BasicCli.addShowCommandClass( HscPhysicalPortCmd )

#------------------------------------------------------------------------------
# show hsc logical switch [name <name>] [physical-switch <switch>]
#   [physical-port <port>] [vlan <vlan>] [vni <vni>] [detail]
#------------------------------------------------------------------------------
@forceMountWithActivityLock
def showHscLogSwitch( mode, args ):
   name = args.get( 'NAME' )
   pswitch = args.get( 'SWITCH' )
   pport = args.get( 'PORT' )
   vlan = args.get( 'VLANRANGE' )
   vniDotted = args.get( 'VNI' )
   detail = 'detail' in args

   model = HscLogicalSwitchesModel()
   if not status.ovsdb:
      mode.addError( 'OVSDB is not connected.' )
      return model

   if not status.running:
      return model

   # pylint: disable-msg=W0212
   model._detail = detail
   model._vniInDottedNotation = vxlanCntrlConfig.vniInDottedNotation
   vni = VniFormat( vniDotted ).toNum() if vniDotted else None

   # Generate internal data structure to speed up processing
   lSwitches = {} # Indexed by UUID
   # Loop through all physical switches
   for psName, pSwitch in status.ovsdb.physicalSwitch.iteritems():
      # Loop through all ports for the physical switch
      for puuid in pSwitch.port:
         port = status.ovsdb.physicalPort[ puuid ]
         # Loop through the vlan bindings for the port
         for vlTag, luuid in port.vlanBinding.iteritems():
            if luuid not in lSwitches:
               lSwitches[ luuid ] = {}
            if psName not in lSwitches[ luuid ]:
               lSwitches[ luuid ][ psName ] = []
            # Store port display name and vlan tag (needed for filtering) in a tuple
            lSwitches[ luuid ][ psName ].append( ( port.displayName, vlTag ) )

   # Apply filters
   for luuid, lSwitch in status.ovsdb.logicalSwitch.iteritems():
      if name and lSwitch.displayName != name:
         continue
      if pswitch and pswitch not in lSwitches[ luuid ]:
         continue
      if pport and len( [
            p[ 0 ] for ps in lSwitches[ luuid ].itervalues()
            for p in ps
            if portMatch( p[ 0 ], pport ) ] ) == 0:
         continue
      if vlan and vlan not in [
            p[ 1 ] for ps in lSwitches[ luuid ].itervalues()
            for p in ps ]:
         continue
      if vni and lSwitch.tunnelKey != vni:
         continue

      # Passed all filters
      model.put( status.ovsdb, luuid, lSwitches.get( luuid ) )

   return model

class HscLogicalSwitchCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc logical switch [ name NAME ] [ physical-switch SWITCH ] '\
         '[ physical-port PORT ] [ vlan VLANRANGE ] [ vni VNI ] [ detail ]'
   data = {
      'hsc': matcherHsc,
      'logical': matcherLogical,
      'switch': matcherSwitch,
      'name': matcherName,
      'NAME': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z:\\.\\-\\_]+',
         helpdesc='Name of the physical switch', helpname='WORD' ),
      'physical-switch': matcherPhysicalSwitch,
      'SWITCH': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z:\\.\\-\\_]+',
         helpdesc='Name of the physical switch', helpname='WORD' ),
      'physical-port': 'Physical port name',
      'PORT': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z\\/\\-]+',
         helpdesc='Physical port name', helpname='WORD' ),
      'vlan': matcherVlan,
      'VLANRANGE': vlanIdMatcher,
      'vni': matcherVni,
      'VNI': vniIdMatcher,
      'detail': nodeExtraDetailHidden,
   }
   handler = showHscLogSwitch
   cliModel = HscLogicalSwitchesModel

BasicCli.addShowCommandClass( HscLogicalSwitchCmd )

#------------------------------------------------------------------------------
# show hsc logical router [<name>] [detail]
#------------------------------------------------------------------------------
@forceMountWithActivityLock
def showHscLogRouter( mode, args ):
   name = args.get( 'NAME' )
   detail = 'detail' in args

   HscModel.renderInVniInDottedNotation = vxlanCntrlConfig.vniInDottedNotation

   model = HscLogicalRoutersModel()
   if not status.ovsdb:
      mode.addError( 'OVSDB is not connected.' )
      return model

   if not status.running:
      return model

   # pylint: disable-msg=W0212
   model._detail = detail

   for lr in status.ovsdb.logicalRouterRaw.itervalues():
      if name and lr.displayName != name:
         continue

      model.put( status.ovsdb, lr )

   return model

class HscLogicalRouterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc logical router [ NAME ] [ detail ]'
   data = {
      'hsc': matcherHsc,
      'logical': matcherLogical,
      'router': 'Show information related to a router',
      'NAME': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z\\-\\_]+',
         helpdesc='Name of the logical router', helpname='WORD' ),
      'detail': nodeExtraDetailHidden,
   }
   handler = showHscLogRouter
   cliModel = HscLogicalRoutersModel

BasicCli.addShowCommandClass( HscLogicalRouterCmd )

#------------------------------------------------------------------------------
# show hsc detail [ no-headings ]
#--------------------------------------------------------------------------------

@forceMountWithActivityLock
def showHscDetail( mode, args ):
   noHeadings = 'no-headings' in args

   model = HscDetailModel( tableOutput='' )
   model.enabled = bool( status.ovsdb )

   if not model.enabled:
      mode.addError( 'OVSDB is not connected' )
      return model

   model.buildTable( status.ovsdb, noHeadings )
   return model

class HscDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc detail [ no-headings ]'
   data = {
      'hsc': matcherHsc,
      'detail': matcherDetail,
      'no-headings': matcherNoHeadings,
   }
   handler = showHscDetail
   cliModel = HscDetailModel
   hidden = True

BasicCli.addShowCommandClass( HscDetailCmd )

#--------------------------------------------------------------------------------
# show hsc detail ovsdb [ DATABASE ] [ SOCKET ] [ no-headings ]
#--------------------------------------------------------------------------------

def showHscDetailOvsdb( mode, args ):
   socket = args.get( 'SOCKET' )
   database = args.get( 'DATABASE' )
   noHeadings = 'no-headings' in args

   model = HscDetailOvsdbModel( database='' )

   if not cliConfig.enableAgent or not cliConfig.enableOvsdb:
      mode.addError( 'OVSDB not enabled.' )
      return model

   if socket is None:
      if not config.ovsdbSocketFile:
         mode.addError( 'Could not find OVSDB socket.' )
         return model

      socket = 'unix:' + config.ovsdbSocketFile

   if database is None:
      database = 'hardware_vtep'

   try:
      cmd = [ 'ovsdb-client', 'dump' ]
      if noHeadings:
         cmd.append( '--no-headings' )

      cmd += [ socket, database ]
      output = Tac.run( cmd, stdout=Tac.CAPTURE, asRoot=True )

      if noHeadings:
         # Empty cells in a column will be filled with spaces.  Keep the number of
         # repeated spaces under 2000 to avoid an assert in the vt100 emulator used
         # to gather CLI output (see emulate() in CliTestClient/__init__.py).  This
         # is only done when the no-headings keyword is used as that is done in the
         # test context.
         #
         # This will not cause any data to be truncated, just cell-padding.
         model.database = re.sub( ' {2000,}', ' ' * 1999, output )
      else:
         model.database = output
   except ( Tac.Timeout, Tac.SystemCommandError ):
      mode.addError( 'Failed to dump OVSDB database.' )

   return model

# Note that you cannot enter a database name after specifying a socket name in
# the below command because the database rule will have been skipped when
# matching the socket name.  To enter a database and socket name, type the database
# name first.
class HscDetailOvsdbCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hsc detail ovsdb [ DATABASE ] [ SOCKET ] [ no-headings ]'
   data = {
      'hsc': matcherHsc,
      'detail': CliCommand.Node( matcher=matcherDetail, hidden=True ),
      'ovsdb': 'Dump OVSDB data',
      'DATABASE': CliMatcher.PatternMatcher( pattern='[0-9A-Za-z_]+',
         helpdesc='Database name', helpname='database' ),
      'SOCKET': CliMatcher.PatternMatcher( pattern='(tcp|udp|unix):[^:]+',
         helpdesc='Socket name', helpname='socket' ),
      'no-headings': matcherNoHeadings,
   }
   handler = showHscDetailOvsdb
   cliModel = HscDetailOvsdbModel

BasicCli.addShowCommandClass( HscDetailOvsdbCmd )

#------------------------------------------------------------------------------
# reset hsc certificate
#------------------------------------------------------------------------------
def resetHscCert( mode, args ):
   if config.ovsdbCertDir is None or config.ovsdbCertDir == '':
      # ovsdbCertDir is set when the OvsdbServer plugin is first initialized at
      # SuperServer startup.  That means that this path should always be set
      # unless SuperServer has never run which should only be the case on
      # namespace DUTs.
      mode.addError( 'The certificate path has not been set.' )
      return

   restartOvsdb = getOvsdbState() != 'Stopped'
   oldOvsdbState = cliConfig.enableOvsdb

   try:
      cliConfig.enableOvsdb = False

      try:
         Tac.waitFor( lambda: getOvsdbState() == 'Stopped',
               description='the OVSDB server to stop', sleep=True )
      except Tac.Timeout:
         mode.addError(
               'Timed out waiting for the OVSDB server to stop.  Disable HSC and ' \
               'then retry this command.' )
         return

      try:
         Tac.run( [ 'rm', '-rf', config.ovsdbCertDir ], stdout=Tac.DISCARD,
               stderr=Tac.DISCARD, asRoot=True )
      except Tac.SystemCommandError:
         mode.addError( 'Failed to remove existing certificate and keys.' )
         return

      cliConfig.enableOvsdb = oldOvsdbState

      if restartOvsdb:
         try:
            Tac.waitFor( lambda: getOvsdbState() != 'Stopped',
                  description='the OVSDB server to start', sleep=True )
         except Tac.Timeout:
            mode.addError( 'Timed out waiting for the OVSDB server to start.' )
            return

         print 'The HSC certificate and keys have been successfully re-generated.'
      else:
         print 'Successfully cleared the HSC certificate and keys.  They will ' \
               'be re-generated the next time the OVSDB server starts.'

   except KeyboardInterrupt:
      mode.addError( 'Interrupted by user.' )
   finally:
      cliConfig.enableOvsdb = oldOvsdbState

#--------------------------------------------------------------------------------
# reset hsc certificate
#--------------------------------------------------------------------------------
class ResetHscCertificateCmd( CliCommand.CliCommandClass ):
   syntax = 'reset hsc certificate'
   data = {
      'reset': resetMatcher,
      'hsc': 'HSC service',
      'certificate': 'Re-generate the HSC certificate and keys',
   }
   handler = resetHscCert

BasicCliModes.EnableMode.addCommandClass( ResetHscCertificateCmd )

#------------------------------------------------------------------------------
# show tech-support commands
#------------------------------------------------------------------------------
def _showTechHscCmds():
   cmds = []
   if cliConfig and cliConfig.enableAgent and status and status.running:
      vtepDb = config.ovsdbDataDir + "/vtep.db"
      cmds += [ "show hsc status",
                "show hsc detail ovsdb",
                "bash ovsdb-tool -mm show-log %s" % ( vtepDb ) ]
   return cmds

timestamp = '2018-09-11 14:31:35'
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      timestamp, _showTechHscCmds )

#------------------------------------------------------------------------------
# Plugin Setup
#------------------------------------------------------------------------------

config = None
cliConfig = None
status = None
sslConfig = None
sslStatus = None
controllerConfig = None
serviceConfigDir = None

def Plugin( entityManager ):
   # If mounts are added here that use LazyMount then forceMountWithActivityLock
   # at the top of this file must be updated with the new mount.
   #
   # Technically this only needs to be done for LazyMount objects that may become
   # mounted from within a show command that uses forceMountWithActivityLock;
   # however, to avoid surprises in the future and keep it simple, it is best if all
   # LazyMounts are added.
   global cliConfig
   global config
   global status
   global sslConfig
   global sslStatus
   global controllerConfig
   global serviceConfigDir
   global vxlanCntrlConfig

   controllerConfig = LazyMount.mount( entityManager,
         'controller/config',
         'Controllerdb::Config', 'r' )

   cliConfig = ConfigMount.mount( entityManager,
         'hsc/cliconfig', 
         'Hsc::CliConfig', 'w' )

   config = LazyMount.mount( entityManager,
         'hsc/config', 
         'Hsc::Config', 'r' )

   status = LazyMount.mount( entityManager,
         'hsc/status',
         'Hsc::Status', 'r' )

   sslStatus = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/status",
                                "Mgmt::Security::Ssl::Status",
                                "r" )
   sslConfig = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config",
                                "r" )

   # To let the CVX infrastructure to know that the Hsc service is
   # enabled/disabled
   serviceConfigDir = ConfigMount.mount( entityManager,
                                         'controller/service/config',
                                         'Controller::ServiceConfigDir', 'w' )

   # To get to vniDottedNotation setting
   vxlanCntrlConfig = LazyMount.mount( entityManager,
                                       "vxlancontroller/config",
                                       "VxlanController::Config", "r" )
   # Boxing vxlanCtrlConfig since VniMatcher needs it at mod-load
   vxlanCtrlConfigBox.append( vxlanCntrlConfig )
