# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel
import Tracing
import TableOutput
import ArnetModel
from IntfModel import Interface

__defaultTraceHandle__ = Tracing.Handle( 'HostInjectCliModels' )

#-----------------------------------------------------------------------------------
#  Models used for "show ip attached-hosts route export [ vlan < vlanId > ]"
#-----------------------------------------------------------------------------------

class AttachedHostEntry( CliModel.Model ):
   address = ArnetModel.IpGenericAddress( help="IP Address" )
   interface = Interface( help="Interface on which host is attached to" )
   prefix = ArnetModel.IpGenericPrefix( help="Injected prefix" )

class AttachedHostList( CliModel.Model ):
   hosts = CliModel.List( help="List of all the injected hosts",
                          valueType=AttachedHostEntry )
   def setAttrFromData( self, data ):
      ahe = AttachedHostEntry()
      setattr( ahe, 'address', data[ 'ipAddr' ] )
      setattr( ahe, 'interface', data[ 'interface' ] )
      setattr( ahe, 'prefix', data[ 'prefix' ] )
      # pylint: disable-msg=E1101
      self.hosts.append( ahe )
      # pylint: enable-msg=E1101


class AttachedHostVlan( CliModel.Model ):
   vlans = CliModel.Dict( valueType=AttachedHostList, keyType=str,
                          help="A mapping of vlan to hosts information" )
   
class AttachedHostModel( CliModel.Model ):
   __revision__ = 2
   vrfs = CliModel.Dict( valueType=AttachedHostVlan, keyType=str,
                         help="A mapping of VRF to vlan information" )

   def setAttachedHosts( self, data ):
      for item in data:
         vrf = item[ 'vrf' ]
         ahVrf = self.vrfs[ vrf ] if vrf in self.vrfs else AttachedHostVlan()
         self.vrfs[ vrf ] = ahVrf

         vId = item[ 'vlan' ]
         ahVlan = ahVrf.vlans[ vId ] if vId in ahVrf.vlans else AttachedHostList()
         ahVlan.setAttrFromData( item )
         ahVrf.vlans[ vId ] = ahVlan

   def render( self ):
      header = [ 'IP Address', 'VLAN', "Interface", "Prefix", "VRF" ]
      table = TableOutput.createTable( ( h, 'l' ) for h in header )

      # pylint: disable-msg=E1101
      for vrf, aheVrf in self.vrfs.iteritems():
         for vlan, aheList in aheVrf.vlans.iteritems():
            for ahe in aheList.hosts:
               table.newRow( ahe.address, vlan, ahe.interface.stringValue,
                             ahe.prefix.stringValue, vrf )
      # pylint: enable-msg=E1101
      print table.output()

   def degrade( self, dictRepr, revision ):
      # The new revision is introduced to accomodate the vrf changes.
      if revision == 1:
         # this newDict will be populated from dictRepr to represent revision 1.
         newDict = {}
         newDict[ "attachedHostEntries" ] = {} #default entry
         allVlans = {}
         for vrf in dictRepr[ "vrfs" ]:
            if vrf != "default":
               continue
            for vlan in dictRepr[ "vrfs" ][ vrf ][ "vlans" ]:
               allVlans[ vlan ] = {}
               hostList = dictRepr[ "vrfs" ][ vrf ][ "vlans" ][ vlan ][ "hosts" ]
               allVlans[ vlan ][ "attachedHosts" ] = hostList
         newDict[ "attachedHostEntries" ] = allVlans
         dictRepr = newDict
      return dictRepr

