# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Enum, Int, Float, Str, Bool, Model, Submodel
import TableOutput
from ArnetModel import IpGenericAddress

HmAdminState = ( "enabled", "shutdown" )
HmOperState = ( "down", "up" )

class HealthMonitorStatusEntry( Model ):
   adminState = Enum( values=HmAdminState, help="Administrative state" )
   operState = Enum( values=HmOperState, help="Operational state" )
   weight = Int( help="Server weight" )

class HealthMonitorProbeStatus( Model ):
   serverStatus = Dict( keyType=IpGenericAddress, valueType=HealthMonitorStatusEntry,
         help="A mapping of Server IP to its health-monitor status" )

class HealthMonitorStatus( Model ):
   __revision__ = 2
   probeStatus = Dict( keyType=str, valueType=HealthMonitorProbeStatus,
         help="A mapping of nexthop-group name to its probe status" )

   def render( self ):
      headings = \
            ( "Nexthop Group", "Server IP", "AdminState", "OperState", "Weight" )
      table = TableOutput.createTable( headings )

      fmt = TableOutput.Format( justify="left", minWidth=3 )
      fmt.noPadLeftIs( True )
      fmt.padLimitIs( True )
      table.formatColumns( fmt, fmt, fmt, fmt, fmt )

      for nhgName, probeStatus in sorted( self.probeStatus.iteritems() ):
         for serverIp, hmStatus in sorted( probeStatus.serverStatus.iteritems() ):
            table.newRow( nhgName, serverIp, hmStatus.adminState, hmStatus.operState,
                          hmStatus.weight )

      print table.output()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ "entries" ] = {}
      return dictRepr

class HealthMonitorLbSummary( Model ):
   probeInterval = Int( help="Probe interval in seconds" )
   sslProfile = Str( help="SSL profile configured" )
   sslProfileValid = Bool( help="SSL profile is valid" )
   caFile = Str( help="CA file configured" )

class HealthMonitorSummary( Model ):
   monitoring = Bool( help="Monitoring is enabled" )
   configProcessed = Bool( help="Config processing is completed" )
   masterConfigFile = Str( help="Master config file" )
   hostConfigFile = Str( help="Host config file" )
   lbSummary = Submodel( valueType=HealthMonitorLbSummary, help="LB summary" )

class HmServerCounterEntry( Model ):
   numChecks = Int( help="Total number of health-checks for this server" )
   numResponses = Int(
         help="Total number of health-check responses for this server" )
   numConnectFailures = Int(
         help="Total number of health-check connect failures for this server" )
   numFailures = Int( help="Total number of health-check failures for this server" )
   numTimeouts = Int( help="Total number of health-check timeouts for this server" )

class HmNhgCounterEntry( Model ):
   servers = Dict( keyType=IpGenericAddress, valueType=HmServerCounterEntry,
                    help="A mapping of server IP to counters" )

class HealthMonitorCounters( Model ):
   groups = Dict( keyType=str, valueType=HmNhgCounterEntry,
                    help="A mapping of nexthop-group to server counters" )

class HmServerTimestampEntry( Model ):
   lastResponseTime = Int(
         help="Time taken by the last health-check to connect in ms" )
   lastCheckTime = Int( help="Last time (UTC) this health-check was conducted" )
   nextCheckTime = Int(
         help="Estimated next time (UTC) this health-check will be conducted" )

class HmNhgTimestampEntry( Model ):
   servers = Dict( keyType=IpGenericAddress,
         valueType=HmServerTimestampEntry,
         help="A mapping of server IP to timestamps" )

class HealthMonitorTimestamps( Model ):
   groups = Dict( keyType=str, valueType=HmNhgTimestampEntry,
                    help="A mapping of nexthop-group to server timestamps" )

class HmPbEntry( Model ):
   numProbes = Int( help="Number of probes in this interval" )
   percentProbes = Float( help="Percentage of probes in this interval" )

class HealthMonitorProbeDelay( Model ):
   intervals = Dict( keyType=str, valueType=HmPbEntry,
         help="A mapping of interval in seconds to probes with delay within \
               that interval" )
