# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model
from CliModel import Enum
from CliModel import Int
from CliModel import Dict
from CliModel import Bool
from CliModel import List 
import TableOutput

 # A dictionary of the prefixes to add before each line of the subtrees.
prefixes = {
   # If a node has multiple children, firstChild, middleChildren and 
   # lastChild are all used. If a node has a single child only, it is represented by 
   # lastChild.
   # firstChild is used for the first child in the set, aka children[ 0 ]
   # middleChildren is used for all of the subset children[ 1:-1 ]
   # lastChild is used for the last child in the set, aka children[ -1 ]
   'firstChild' : {
      'firstLine' : '|-- ',
      'tailingLines' : '|   '
   },
   'middleChildren' : {
      'firstLine' : '|-- ',
      'tailingLines' : '|   '
   },
   'lastChild' : {
      'firstLine' : '`-- ',
      'tailingLines' : '    '
   }
}

headerSuffix = ' -' # Added to the header if it has any children

# Added as a workaround for the fact that there's a circular dependancy with having
# a class C with a member of type CliModel.List(valueTye=C) because C doesn't exist
# in the namespace yet because the object hasn't been created.
class _HealthMgrTree( Model ):
   pass

class HealthMgrTree( _HealthMgrTree ):
   __public__ = False
   healthScore = Int( help="Describes the health index score of this node. Usually,"\
                           " this is simply the sum of all of the health scores of"\
                           " the children of this node, unless the node itself"\
                           " presents additional health concerns.\n"\
                           "A health score of 0 represents no health issues." )
   nodeType = Enum( help="Describes the classification of the node based on what"\
                         " kind of attribute it has been sorted by.",
                    values=[ 'metric', 'category', 'component' ] )
   children = Dict( help="The collection of subtrees that have all been sorted in"\
                         " a similar way",
                    keyType=str,
                    valueType=_HealthMgrTree )

   # Named toStr because you can't define _repr__ or __str__ in a subclass of Model
   def toStr( self, name ):
      # Recursive function used to generate a string displaying the tree structure
      # Basic idea is to generate the tree-string for each node, then
      # join them all together with the given prefixes

      # Generate information about the node at this level.
      header = "{0} ({1})".format( name, self.healthScore )

      # Check to see if we have children that we need to iterate over. If not, return
      # just the header. If so, recurse on the remaining children.
      if self.children == {}:
         # Base case:
         return header
      else:
         # Helper function that adds prefixes before each line in a subtree
         def prependSpace( subtrees, prefixType, addLine=True ):
            lines = subtrees.split( '\n' )
            # For the first line, add a different prefix than the rest
            firstPrefix = prefixes[ prefixType ][ 'firstLine' ]
            tailPrefix = prefixes[ prefixType ][ 'tailingLines' ]
            lines[ 0 ] = firstPrefix + lines[ 0 ]
            # For the rest of the lines, add the space and then the tailPrefix
            lines[ 1: ] = [ tailPrefix + line for line in lines[ 1: ] ]
            subtreeWithSpace = '\n'.join( lines )
            return subtreeWithSpace

         # Use the map function to get each bad child's subtree string
         # pylint: disable-msg=no-member
         # (because children.iteritems() is added at runtime)
         subtrees = [ child.toStr( name ) for ( name, child )
                      in self.children.iteritems() ]

         if len( subtrees ) == 1: # In other words, if we only have 1 child
            subtrees[ 0 ] = prependSpace( subtrees[ 0 ], 'lastChild' )
         else:
            subtrees[ 0 ] = prependSpace( subtrees[ 0 ], 'firstChild' )
            subtrees[ 1:-1 ] = [ prependSpace( subtree, 'middleChildren' )
                                 for subtree in subtrees[ 1:-1 ] ]
            subtrees[ -1 ] = prependSpace( subtrees[ -1 ], 'lastChild' )

         finalString = '\n'.join( subtrees ) # Put all the subtrees together

         # Replace the first headerLength characters section with the header
         finalString = header + '\n' + finalString
         return finalString

   def pruneHealthyNodes( self ):
      # Recursively deletes all of the healthy nodes that are children of this node
      # Note: We have to make a copy of nodesDict so that the iteritems() doesn't
      # get confused about the dictionary changing size during iteration
      for childName, childNode in dict( self.children ).iteritems():
         if childNode.healthScore == 0:
            del self.children[ childName ]
         else:
            childNode.pruneHealthyNodes()

   def recountScore( self ):
      # Recursively update the health score for all of the children, then update
      # ourself
      # Note: This function might need to be changed when we implement weighting
      if not self.isLeaf():
         score = sum( [ child.recountScore() for child in self.children.values() ] )
         self.healthScore = score
      elif self.nodeType != 'component':
         self.healthScore = 0
      return self.healthScore

   def isLeaf(self):
      return self.children == {}

   def render( self ):
      print( self.toStr( "Score" ) )

class HealthMetricModel( Model ):
   handlers = List( help="A mapping between handler and handler information.",
                    valueType=str )

class HealthCategoryModel( Model ):
   metrics = Dict( help="A mapping between metric and metric information.",
                   valueType=HealthMetricModel )

# Model for "show monitor device-health" command
class DeviceHealthModel( Model ):
   healthMonitoringEnabled = Bool( help="Device health monitoring enabled." )
   categories = Dict( help="A mapping between category and category information.",
                   valueType=HealthCategoryModel )
   
   def render( self ):
      if self.healthMonitoringEnabled:
         print "Device health monitoring is enabled." 
         table = TableOutput.createTable( ( 'Category', 'Metric', 'Handler' ) )
         fmt = TableOutput.Format( justify="left" )
         table.formatColumns( fmt, fmt, fmt )
         tableRows = []
         for ( catName, category ) in self.categories.iteritems():
            categoryStr = catName 
            if not category.metrics:
               tableRows.append( 
                     [ categoryStr, 'None configured', 'None configured' ] )
               continue
            for ( metName, metric ) in category.metrics.iteritems() :
               metricStr = metName
               if not metric.handlers:
                  tableRows.append( [ categoryStr, metricStr, 'None enabled' ] )
                  categoryStr = ''
                  continue
               for handlerName in metric.handlers:
                  tableRows.append( [ categoryStr, metricStr, handlerName ] )
                  categoryStr = ''
                  metricStr = ''
         for row in tableRows:
            table.newRow( *row )
         
         out = table.output()
         width = len( out.splitlines()[0] )
         print out
         print '-' * width
      else:
         print "Device health monitoring is disabled." 

