# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import IntfCli
from IntfCli import IntfConfigMode
import ConfigMount
import LazyMount

healthIndicatorConfig = None
xcvrConfig = None

#---------------------------------------------------------------
# The "[no | default] health indicator <indicator>" command, in "config-if" mode.
#---------------------------------------------------------------
class HealthIndicatorCmd( CliCommand.CliCommandClass ):
   syntax = 'health indicator ( good | bad )'
   noOrDefaultSyntax = 'health indicator'
   data = {
      'health': 'Set health parameters of an interface',
      'indicator': 'Set health indicator of an interface',
      'good': 'Set health indicator to "good"',
      'bad': 'Set health indicator to "bad"',
   }

   @staticmethod
   def handler( mode, args ):
      # Add for all lanes
      intfs = getAllLanes( mode.intf.name )
      for intf in intfs:
         cfg = healthIndicatorConfig.newIntfConfig( intf )
         if 'good' in args:
            cfg.indicator = "healthIndicatorGood"
         elif 'bad' in args:
            cfg.indicator = "healthIndicatorBad"
         else:
            mode.addError( "Invalid health indicator value" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      deleteHealthIndicator( mode.intf.name )

IntfConfigMode.addCommandClass( HealthIndicatorCmd )

def getAllLanes( intf ):
   baseIntf = intf.rsplit( "/", 1 )[ 0 ]
   if baseIntf not in xcvrConfig.xcvrConfig.keys():
      return [ intf ]
   return xcvrConfig.xcvrConfig[ baseIntf ].intfName.values()

def deleteHealthIndicator( inputIntf ):
   # remove from list
   intfs = getAllLanes( inputIntf )
   for intf in intfs:
      if intf in healthIndicatorConfig.intfConfig:
         del healthIndicatorConfig.intfConfig[ intf ]

class HealthIndicatorIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      deleteHealthIndicator( self.intf_.name )

def Plugin( entityManager ):
   global healthIndicatorConfig, xcvrConfig
   healthIndicatorConfig = ConfigMount.mount( entityManager,
      "openconfig/config/healthIndicator", 
      "Hercules::HealthIndicatorConfigDir", "w" )
   xcvrConfig = LazyMount.mount( entityManager, 
      "hardware/xcvr/config/all",
      "Xcvr::AllConfigDir", "r" )

   IntfCli.Intf.registerDependentClass( HealthIndicatorIntfCleaner )
