# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Monitor
import ConfigMount
import LazyMount
import ShowCommand
import Tac
import TechSupportCli
import Tracing
from CliMode.DeviceHealth import CategoryMode, DeviceHealthMode
from HealthMgrCliModel import ( DeviceHealthModel,
                                HealthCategoryModel,
                                HealthMetricModel,
                                HealthMgrTree )

# Also gotta do TRACEFILE=/some/file
traceHandle = Tracing.Handle( 'HealthCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

hwConfig = None
cliConfig = None
status = None
scoreTree = None
rootEventDir = None

#
# Management config mode for health monitor.
#
class DeviceHealthConfigMode( DeviceHealthMode, BasicCli.ConfigModeBase ):
   name = "mon-device-health"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      DeviceHealthMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def eventHandlerDataFn( metricName ):
   return []
#-------------------------------------------------------------------------------
# The "[no|default] monitor device-health" mode command.
#-------------------------------------------------------------------------------
class MonitorDeviceHealth( CliCommand.CliCommandClass ):
   syntax = 'monitor device-health'
   noOrDefaultSyntax = syntax
   data = {
         'monitor': CliToken.Monitor.monitorMatcher,
         'device-health': 'Health monitor configuration'
   }

   @staticmethod
   def handler( mode, args ):
      t0( "'monitor device-health handler" )
      childMode = mode.childMode( DeviceHealthConfigMode )
      mode.session_.gotoChildMode( childMode )
      # this sets cliConfig.enabled and cliConfig.configExists to 'True'.
      cliConfig.configExists = True
      cliConfig.enabled = True

   @staticmethod
   def noHandler( mode, args ):
      t0( "'no monitor device-health handler'" )
      # this sets cliConfig.configExists to 'True' and cliConfig.enabled to 'False'.
      cliConfig.configExists = True
      cliConfig.enabled = False

   @staticmethod
   def defaultHandler( mode, args ):
      t0( "'default monitor device-health handler'" )
      # this sets cliConfig.enabled and cliConfig.configExists to 'False'.
      cliConfig.configExists = False
      cliConfig.enabled = False

BasicCli.GlobalConfigMode.addCommandClass( MonitorDeviceHealth )

#-------------------------------------------------------------------------------
# The "show monitor device-health" show command.
#-------------------------------------------------------------------------------
class DeviceHealthShow( ShowCommand.ShowCliCommandClass ):
   syntax = '''show monitor device-health'''
   data = {
      'monitor' : CliToken.Monitor.monitorMatcherForShow,
      'device-health' : 'Show metrics about system health'
   }
   cliModel = DeviceHealthModel

   @staticmethod
   def handler( mode, args ):
      t0( "'show monitor device-health' handler" )
      deviceHealthMonitor = DeviceHealthModel()
      deviceHealthMonitor.healthMonitoringEnabled = healthMgrEnabled()

      if healthMgrEnabled():
         for ( categoryName, categoryInfo ) in status.category.items():
            if not categoryInfo.builtin or \
                  ( categoryInfo.builtin and not categoryInfo.disabled ):
               categoryObj = HealthCategoryModel()
               deviceHealthMonitor.categories[ categoryName ] = categoryObj 

         for ( metricName, metricInfo ) in status.metric.items():
            if not metricInfo.builtin or \
                  ( metricInfo.builtin and not metricInfo.disabled ):
               metricObj = HealthMetricModel()
               handlerList = eventHandlerDataFn( metricName )
               for handler in handlerList:
                  metricObj.handlers.append( handler )
               categoryModel = deviceHealthMonitor.categories[ metricInfo.category ]
               categoryModel.metrics[ metricName ] = metricObj

      return deviceHealthMonitor

def healthMgrEnabled():
   return ( hwConfig.supported and (
               ( not cliConfig.configExists and hwConfig.enabledByDefault ) or
               ( cliConfig.configExists and cliConfig.enabled ) )
            or status.enabled ) 

BasicCli.addShowCommandClass( DeviceHealthShow )

#-------------------------------------------------------------------------------
# The "[no] category CATEGORY" mode command.
#-------------------------------------------------------------------------------
class AddCategory( CliCommand.CliCommandClass):
   syntax = 'category CATEGORY'
   noOrDefaultSyntax = syntax
   data = {
         'category': 'Configure category',
         'CATEGORY': CliMatcher.DynamicNameMatcher( lambda m: cliConfig.category,
                                                    helpname='WORD',
                                                    helpdesc='Name of category' )
   }
   _categoryName = None

   @staticmethod
   def handler( mode, args ):
      AddCategory._categoryName = args[ 'CATEGORY' ]
      t0( "'category %s'" % AddCategory._categoryName )
      assert cliConfig != None

      childMode = mode.childMode( DeviceHealthCategoryConfigMode,
                                  categoryName=AddCategory._categoryName )
      mode.session_.gotoChildMode( childMode )
      childMode.action = 'add'

   @staticmethod
   def noHandler( mode, args ):
      AddCategory._categoryName = args[ 'CATEGORY' ]
      t0( "'no category %s'" % AddCategory._categoryName )
      assert cliConfig != None

      # now remove category-name from cliConfig.category collection
      if AddCategory._categoryName in cliConfig.category:
         removeCategory( AddCategory._categoryName )
      else:
         mode.addWarning( "Category '%s' not registered." %
                          AddCategory._categoryName )

   @staticmethod
   def defaultHandler( mode, args ):
      AddCategory._categoryName = args[ 'CATEGORY' ]
      t0( "'default category %s'" % AddCategory._categoryName )
      assert cliConfig != None
      # enable category if builtin.
      #         enable builtin metrics under this category.
      #         remove user added metris under this category.
      # remove if not builtin
      if AddCategory._categoryName in cliConfig.category:
         catConfig = cliConfig.category [ AddCategory._categoryName ]
         if catConfig.builtin:
            childMode = mode.childMode( DeviceHealthCategoryConfigMode,
                                        categoryName=AddCategory._categoryName )
            mode.session_.gotoChildMode( childMode )
            childMode.action = 'default'
         elif not catConfig.builtin:
            removeCategory( AddCategory._categoryName )
      else:
         mode.addWarning( "No default configuration for category: %s" %
                          AddCategory._categoryName )

DeviceHealthConfigMode.addCommandClass( AddCategory )

#-------------------------------------------------------------------------------
# The "[no] metric METRIC" command.
#-------------------------------------------------------------------------------
class AddMetric( CliCommand.CliCommandClass):
   syntax = 'metric METRIC'
   noOrDefaultSyntax = syntax
   data = {
         'metric': 'Configure metric',
         'METRIC': CliMatcher.DynamicNameMatcher( lambda m: cliConfig.metric,
                                                  helpname='WORD',
                                                  helpdesc='Name of metric' ),
   }

   @staticmethod
   def handler( mode, args ):
      metricName = args[ 'METRIC' ]
      t0( "'metric %s'" % metricName )
      assert cliConfig != None
      if metricName in cliConfig.metric:
         metConfig = cliConfig.metric[ metricName ]
         if metConfig.category != mode.categoryName:
            mode.addError( 'Metric %s is already configured under category %s' %
                           ( metricName, metConfig.category ) )
            return
      mode.metricsInfo[ metricName ] = 'add'

   @staticmethod
   def noHandler( mode, args ):
      metricName = args[ 'METRIC' ]
      t0( "'no metric %s'" % metricName )
      assert cliConfig != None
      if metricName in cliConfig.metric:
         metConfig = cliConfig.metric[ metricName ]
         if metConfig.category != mode.categoryName:
            mode.addWarning( "Metric %s is already configured under category %s. "
                             "Ignoring this command." %
                           ( metricName, metConfig.category ) )
            return
      mode.metricsInfo[ metricName ] = 'no'

   @staticmethod
   def defaultHandler( mode, args ):
      metricName = args[ 'METRIC' ]
      t0( "'default metric %s'" % metricName )
      assert cliConfig != None
      if metricName in cliConfig.metric:
         metConfig = cliConfig.metric[ metricName ]
         if metConfig.category != mode.categoryName:
            mode.addWarning( "Metric %s is already configured under category %s. "\
                             "Ignoring this command." %
                           ( metricName, metConfig.category ) )
            return
      mode.metricsInfo[ metricName ] = 'default'

#
# Management config mode for health monitor category.
#
class DeviceHealthCategoryConfigMode( CategoryMode, BasicCli.ConfigModeBase ): 
   name = "device-health category configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, categoryName ):
      self.categoryName = categoryName
      self.action = None
      # metricsInfo = { 'metricName' : "add|no|default" }
      self.metricsInfo = {}
      CategoryMode.__init__( self, categoryName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def addMetric( self, metricName ):
      # now add metric-name to cliConfig.metric collection
      if metricName not in cliConfig.metric: 
         cliConfig.metric.newMember( metricName, self.categoryName, False )
      else:
         metConfig = cliConfig.metric[ metricName ]
         if metConfig.builtin and metConfig.disabled:
            metConfig.disabled = False
         else:
            self.addWarning( "Metric '%s' already registered." % metricName )

   def noMetric( self, metricName ):
      # now remove metric-name from cliConfig.metric.collection
      if metricName in cliConfig.metric: 
         removeMetric( metricName )
      else:
         self.addWarning( "metric '%s' not registered." % metricName )
   
   def defaultMetric( self, metricName ):
      # enable if builtin and disabled
      # do nothing in builtin and enabled
      # remove if not builtin
      if metricName in cliConfig.metric:
         metConfig = cliConfig.metric[ metricName ]
         if metConfig.builtin and metConfig.disabled:
            metConfig.disabled = False
         elif not metConfig.builtin:
            removeMetric( metricName )
      else: 
         self.addWarning( "No default configuration for metric: %s" % metricName )

   def handleMetrics( self ):
      for ( metric, action ) in self.metricsInfo.iteritems():
         if action == "add":
            self.addMetric( metric )
         elif action == "no":
            self.noMetric( metric )
         elif action == "default":
            self.defaultMetric( metric )
         else:
            assert False, "unrecognised metric action."

   def addCategory( self ):
      # now add category-name to cliConfig.category collection
      if self.categoryName not in cliConfig.category:
         cliConfig.category.newMember( self.categoryName, False )
      else:
         catConfig = cliConfig.category[ self.categoryName ]
         if catConfig.builtin and catConfig.disabled:
            enableBuiltinCategory( self.categoryName )
         else:
            self.addWarning( "Category '%s' already registered." %
                             self.categoryName )
         
   def onExit( self ):
      t0( "starting commit" )
      # update Metrics
      self.handleMetrics()
      # update Category 
      if self.action == 'add':
         self.addCategory()
      elif self.action == 'default':
         defaultBuiltinCategory( self.categoryName )
      else:
         assert False, "unrecognised category command."
      t0( "finished commiting, trying  to exit" )
      BasicCli.ConfigModeBase.onExit( self )

def removeMetric( metricName ):
   if cliConfig.metric[ metricName ].builtin == True:
      cliConfig.metric[ metricName ].disabled = True
   else:
      del cliConfig.metric[ metricName ]

def enableBuiltinCategory( catName ):
   catConfig = cliConfig.category[ catName ]
   catConfig.disabled = False
   # enable builtin metrics under this category here.
   for metric in cliConfig.metric.values():
      # remove user metrics.
      if metric.category == catName and metric.builtin:
         metric.disabled = False

def defaultBuiltinCategory( catName ):
   enableBuiltinCategory( catName )
   for ( metricName, metric ) in cliConfig.metric.items():
      if metric.category == catName and not metric.builtin:
         del cliConfig.metric[ metricName ]

def removeCategory( categoryName ):
   # delete metrics under this category here.
   for ( name, metric ) in cliConfig.metric.items():
      if metric.category == categoryName:
         removeMetric( name ) 
   if cliConfig.category[ categoryName ].builtin == True:
      cliConfig.category[ categoryName ].disabled = True
   else:
      del cliConfig.category[ categoryName ]

DeviceHealthCategoryConfigMode.addCommandClass( AddMetric )

#-------------------------------------------------------------------------------
# The "show monitor device-health component" show command.
#-------------------------------------------------------------------------------
class DeviceHealthComponent( ShowCommand.ShowCliCommandClass ):
   syntax = '''show monitor device-health component'''
   data = {
      'monitor' : CliToken.Monitor.monitorMatcherForShow,
      'device-health' : 'Show metrics about system health',
      'component' : 'Display a tree of bad health events sorted by components first'
   }
   cliModel = HealthMgrTree

   @staticmethod
   def handler( mode, args ):
      t0( "'show monitor device-health component' handler" )
      componentTree = makeComponentTree( includeHealthyChildren=False )
      # Note: After creating the tree, we want to recompute the score to ensure
      # that it's valid, and then possibly prune off any healthy branches that
      # slipped in due to inconsistent/temporary Sysdb state
      componentTree.recountScore()
      # in the future, add 'if not includeHealthyChildren:' here when we have a
      # command for it
      componentTree.pruneHealthyNodes()
      return componentTree

#-------------------------------------------------------------------------------
# The "show monitor device-health category" show command.
#-------------------------------------------------------------------------------
class DeviceHealthCategory( ShowCommand.ShowCliCommandClass ):
   syntax = '''show monitor device-health category'''
   data = {
      'monitor' : CliToken.Monitor.monitorMatcherForShow,
      'device-health' : 'Show metrics about system health',
      'category' : 'Display a tree of bad health events sorted by category first'
   }
   cliModel = HealthMgrTree

   @staticmethod
   def handler( mode, args ):
      t0( "'show monitor device-health category' handler" )
      return makeCategoryTree( includeEmptySubtrees=False )

BasicCli.addShowCommandClass( DeviceHealthComponent )
BasicCli.addShowCommandClass( DeviceHealthCategory )

def makeComponentTree( includeHealthyChildren=False ):
   # Converts the TAC scoreTree object into a CAPI HealthMgrTree tree structure
   assert scoreTree != None
   assert rootEventDir != None
   if scoreTree.rootName == '':
      return HealthMgrTree( healthScore=0, nodeType='component', children={} )

   # Helper function to remove all the healthy nodes from a dictionary
   # We take a two-layer approach to filtering out unhealthy children:
   #    If a node is totally healthy
   #         Skip computing child branches and add a blank HealthMgrTree node.
   #    Else:
   #         Compute all the child branches and then get rid of the healthy ones
   # Because of the first step, we will frequently skip lots of unnecessary
   # computation of child branches, thus speeding up calculation and reducing errors
   # from data-racing
   def filterOutHealthyNodes( nodesDict ):
      # Note: We have to make a copy of nodesDict so that the iteritems() doesn't
      # get confused about the dictionary changing size during iteration
      for nodeName, node in dict( nodesDict ).iteritems():
         if not includeHealthyChildren and node.healthScore == 0:
            del nodesDict[ nodeName ]

   # The structure in TAC is as follows:
   # We have Source, Category, and Metric nodes that inherit from ScoreNode.
   #  - Source nodes contain both Category nodes and additional Source nodes.
   #  - Category nodes contain Metric nodes.
   #  - Metric nodes contain no children directly in TAC, but in this model we want
   #    Component Nodes to be a child of Metric Nodes. We have no structure for this
   #    in TAC directly, so we have to fish around for a bit in the rootEventDir as
   #    evidenced in traverseMetric()
   def traverseComponent( componentName ):
      # Right now, the score is simply set to 1 for all leaf events. This will be
      # amended later when weighting is added.
      return HealthMgrTree( healthScore=1,
                            nodeType='component',
                            children={} )

   def traverseMetric( metricNode ):
      t1( 'Traversing metricNode: {0}'.format( metricNode ) )
      metricChildren = {}
      if ( rootEventDir.metric.get( metricNode.name ) and
            ( metricNode.score != 0 or includeHealthyChildren ) ):
         metricDir = rootEventDir.metric[ metricNode.name ]
         for handler in metricDir.handler.values():
            if metricNode.source in handler.componentList:
               for name in handler.componentList[ metricNode.source ].component:
                  for sev in [ 'info', 'warning', 'error', 'fatal' ]:
                     key = Tac.Value( 'Health::EventKey', name, sev )
                     if key in handler.event:
                        if name not in metricChildren:
                           metricChildren[ name ] = traverseComponent( name )
                           metricChildren[ name ].healthScore = 1 
                        else:
                           metricChildren[ name ].healthScore += 1
         filterOutHealthyNodes( metricChildren )
         # healthScore = len( children) here is a check that the score is still
         # accurate for the state that we're seeing
         # Change this later when weighting is added
      return HealthMgrTree( healthScore=len( metricChildren ),
                            nodeType='metric',
                            children=metricChildren )
      # add a corner case test for when the state is in an intermediate state
      # and the health is not the sum of the number of children

   def traverseCategory( categoryNode ):
      t1( 'Traversing categoryNode: {0}'.format( categoryNode ) )
      categoryChildren = {}
      # If this node is healthy, just skip computation of all the child nodes unless
      # we're specifically interested in seeing them.
      if categoryNode.score != 0 or includeHealthyChildren:
         categoryChildren.update( { node.name : traverseMetric( node )
                                    for node in categoryNode.metric.values() } )
         filterOutHealthyNodes( categoryChildren )
      return HealthMgrTree( healthScore=categoryNode.score,
                            nodeType='category',
                            children=categoryChildren )

   def traverseSource( sourceNode ):
      # Note that source nodes can contain both category nodes as children as
      # well as additional source nodes.
      t1( 'Traversing sourceNode: {0}'.format( sourceNode ) )
      sourceChildren = {}
      # If this node is healthy, just skip computation of all the child nodes unless
      # we're specifically interested in seeing them.
      if sourceNode.score != 0 or includeHealthyChildren:
         sourceChildren.update( { name : traverseSource( scoreTree.source[ name ] )
                                  for name in sourceNode.childName } )
         sourceChildren.update( { node.name : traverseCategory( node )
                                  for node in sourceNode.category.values() } )
         filterOutHealthyNodes( sourceChildren )
      return HealthMgrTree( healthScore=sourceNode.score,
                            nodeType='component',
                            children=sourceChildren )

   rootScoreNode = scoreTree.source[ scoreTree.rootName ]
   return traverseSource( rootScoreNode )


def makeCategoryTree( includeEmptySubtrees=False ):
   # Sorting by category is a bit different because we don't maintain a category
   # tree structure in Sysdb.
   # We maintain a list of all metrics. Each metric can tell us what category
   # it's been defined under. Additionally, each metric has a handler that helps
   # us find all the components that reported that kind of metric. Then we have a
   # mapping from components -> source, so we can find out which source a
   # component is registered to.

   def mergePathToTree( parentNode, pathTuples ):
      # Merges a HealthMgrTree path into a larger tree
      # Idempotent if path already exists
      # "treeRoot" is a HealthMgrTree node
      # "path" is a list of strings representing the names of sources
      # Returns the leaf node of the path
      #
      # For example behavior, imagine we have the following tree structure:
      #     [ Root-+-mammal--cat--meow ]
      #     [      \-bird--owl--hoot   ]
      # and we have the path below that we want to merge into the tree.
      #     [ mammal--dog--bark ]
      # Then the correct output should look like the following tree:
      #     [ Root-+-mammal-+-cat--meow ]
      #     [      |        \-dog--bark ]
      #     [      \-bird--owl--hoot    ]
      for (childName, childType) in pathTuples:
         if childName not in parentNode.children:
            newChildNode = HealthMgrTree( healthScore=1,
                                          nodeType=childType,
                                          children={} )
            parentNode.children[ childName ] = newChildNode
         parentNode = parentNode.children[ childName ]
      return parentNode

   def addComponentSourcePath( parentName, componentName, metricNode ):
      # Components only store their immediatly containing source's name. To get
      # around this, we have to iterate all the way up to the Root node, and then
      # add all of the sources we saw along the way.
      # Essentially, follow the parent path until we hit the top
      path = [ ( componentName, 'component' ) ]
      while parentName != 'Root' and parentName != '':
         if parentName in path:
            t0( "Error: The parent {0} is already in the source path {1}! This will "
                "cause an infinite loop.".format( parentName, path ) )
            return None
         elif parentName not in scoreTree.source:
            t0( "Error: {0} not found in scoreTree.source".format( parentName ) )
            return None
         else:
            t2( "Adding parent {0} to path {1}".format( parentName, path ) )
            path.append( ( parentName, 'component' ) )
            parentName = scoreTree.source[ parentName ].parentName
      path.reverse()
      return mergePathToTree( metricNode, path )

   # For each component node, we generate a trail of source nodes from Root to the
   # component, convert each node to a HealthMgrTree node, and merge it in with
   # the rest of the children
   def populateMetricChildren( metricName, metricNode ):
      t1( "Creating source subtree for metric {0}".format( metricName ) )
      metricDir = rootEventDir.metric[ metricName ]
      componentsVisited = set()
      for handlerName in metricDir.handler:
         t1( "Exploring handler {0}".format( handlerName ) )
         eventDir = metricDir.handler[ handlerName ]
         for componentList in  eventDir.componentList.values():
            for componentName in componentList.component:
               for sev in [ 'info', 'warning', 'error', 'fatal' ]:
                  key = Tac.Value( 'Health::EventKey', componentName, sev )
                  t1( "Creating new component leaf node: "
                      "{0}".format( componentName ) )
                  if( key not in eventDir.event or
                      componentName not in eventDir.source ):
                     t0( "Component {0} does not have any registered "
                         "events/sources".format( componentName ) )
                     continue
                  parentName = eventDir.source[ componentName ].source
                  componentNode = addComponentSourcePath( parentName, componentName,
                                                          metricNode )
                  if componentNode is None:
                     continue
                  assert componentNode.nodeType == 'component'
                  componentNode.healthScore = 1 \
                     if componentName not in componentsVisited \
                     else componentNode.healthScore + 1 
                  componentsVisited.update( [ componentName ] )

   def populateCategoryAndMetricNodes( parentNode ):
      metrics = rootEventDir.metric
      for metricName in metrics:
         t1( "Creating subtree for metric {0}".format( metricName ) )
         categoryName = metrics[ metricName ].category
         metricPath = [ ( categoryName, 'category' ), ( metricName, 'metric' ) ]
         metricNode = mergePathToTree( parentNode, metricPath )
         populateMetricChildren( metricName, metricNode )

   t1( "Creating root node" )
   rootNode = HealthMgrTree( healthScore=0,
                             nodeType='category',
                             children={} )
   populateCategoryAndMetricNodes( rootNode )
   # !IMPORTANT! We do not keep track of the healthScores for the category tree in
   # the TAC structure, so we have to recalculate them using the node.recountScore()
   # function.
   # Until we implement weighted-scores, essentially each leaf node is assumed to
   # have a healthScore of 1, and each internal node has
   # node.healthScore = sum( [child health scores] )
   rootNode.recountScore()
   rootNode.pruneHealthyNodes()
   return rootNode

def _showTechCmds():
   return [ 'show monitor device-health',
            'show monitor device-health component',
            'show monitor device-health category', ]

TechSupportCli.registerShowTechSupportCmdCallback( '2018-01-10 00:00:00',
                                                   _showTechCmds )

def Plugin( entityManager ):
   global hwConfig, cliConfig, status, scoreTree, rootEventDir

   cliConfig = ConfigMount.mount( entityManager, "health/cliConfig",
                                  "Health::CliConfig", "w" )
   status = LazyMount.mount( entityManager, "health/status",
                             "Health::Status", "r" )
   scoreTree = LazyMount.mount( entityManager, "health/scoreTree",
                                "Health::ScoreDir", "r" )
   rootEventDir = LazyMount.mount( entityManager, "health/eventData",
                                   "Health::AllMetricDir", "r" )
   hwConfig = LazyMount.mount( entityManager, "health/hwConfig",
                                  "Health::HwConfig", "r" )
