# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import HardwareLedMode as LedMode
import LazyMount
import Tac
import LedCommon
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Model
from CliPlugin import FruCli
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import Intf.IntfRange
import ShowCommand

#------------------------------------------------------------------------------------
# Constants declaration
#------------------------------------------------------------------------------------
SET_NAME = 'HardwareLedCliSet'
RAMP_RATE = 0.0104
MAX_BRIGHTNESS = 100

entityMib = None
hwLedConfig = None
ledConfig = None
plutoStatus = Tac.Type( 'Led::PlutoStatus' )

#------------------------------------------------------------------------------------
# Model classes for show command
#------------------------------------------------------------------------------------
class Led( Model ):
   color = Enum( values=( 'green', 'yellow', 'red', 'blue', 'off' ),
                 help='Color of LED' )
   flash = Bool( help='True if this LED flashes on and off' )

class LedModel( Model ):
   leds = Dict( keyType=str, valueType=Led,
                help='All LEDs; key is the display name of the LED' )
   def render( self ):
      print 'User configured LEDs:'
      print '---------------------'
      for led in sorted( self.leds ):
         print '%s (color:%s%s)' % ( led, self.leds[ led ].color,
                                     ' flashing' if self.leds[ led ].flash else '' )
      print ''

class ChassisStatusMatcher( CliMatcher.Matcher ):
   '''
   Matches all linecards/fabrics/supervisors on modular systems.
   Matches the status on fixed.
   '''
   def __init__( self, **kwargs ):
      super( ChassisStatusMatcher, self ).__init__( helpdesc='', **kwargs )

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         return CliMatcher.noMatch
      if root.tacType.fullTypeName == 'EntityMib::FixedSystem':
         if token == 'Status1':
            return CliMatcher.MatchResult( 'Status1', 'Status1' )
      elif root.tacType.fullTypeName == 'EntityMib::Chassis':
         for slot in root.cardSlot.values():
            if '%s%s' % ( slot.tag, slot.label ) == token:
               result = '%s%s' % ( slot.tag, slot.label )
               return CliMatcher.MatchResult( result, result )
      return CliMatcher.noMatch

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         return []
      if root.tacType.fullTypeName == 'EntityMib::FixedSystem':
         return [ CliParser.Completion( 'Status1', 'Chassis Status LED' ) ]
      elif root.tacType.fullTypeName == "EntityMib::Chassis":
         completions = []
         for slot in root.cardSlot.values():
            completions.append( CliParser.Completion( '%s%s' % ( slot.tag,
                                                                 slot.label ), '' ) )
         return completions
      return []

#------------------------------------------------------------------------------------
# Helper functions
#------------------------------------------------------------------------------------
def setLed( mode, led=None, color=None, disable=None, flashing=None ):
   '''
   Setting the configuration for one LED
   '''
   ledSet = LedCommon.getLedSet( SET_NAME, ledConfig )

   # In case of Supervisors the LEDs are called 'Status1' and 'Status2'
   led = led.replace( 'Supervisor', 'Status' )

   if disable:
      if led in ledSet.led:
         del ledSet.led[ led ]
         return
      print 'LED already not user configured!'
      return

   if not color:
      print 'Please enter a color!'
      return

   ledSetting = Tac.Value( 'Led::LightSetting' )
   ledSetting.rampRate = RAMP_RATE
   ledSetting.maxBright = MAX_BRIGHTNESS
   ledSetting.flashRate = 1 if flashing else 0
   if flashing:
      ledSetting.plutoStatus = plutoStatus.plutoStatusBeacon
      if color == 'off':
         print 'Cannot be flashing and be set to off!'
         return
   elif color == 'green':
      ledSetting.plutoStatus = plutoStatus.plutoStatusGood
   elif color == 'yellow':
      ledSetting.plutoStatus = plutoStatus.plutoStatusInactive
   elif color == 'red':
      ledSetting.plutoStatus = plutoStatus.plutoStatusBad
   elif color == 'off':
      ledSetting.plutoStatus = plutoStatus.plutoStatusOff
   else:
      ledSetting.plutoStatus = plutoStatus.plutoStatusUnknown
   if color != 'off' and color != 'blue':
      setattr( ledSetting, color, True )
   if color == 'blue':
      setattr( ledSetting, color, True )
      if not getattr( hwLedConfig.leds.get( led ), 'ledAttribute', False ) \
         or not getattr( hwLedConfig.leds.get( led ).ledAttribute, 'blue', False ):
         print 'WARNING: This LED does not support blue!'

   ledSet.led[ led ] = ledSetting
   return

#--------------------------------------------------------------------------------
# [ no | default ] ( ( chassis ( CHASSIS | LED ) ) |
#                    ( fan-tray ( FAN_TRAY | LED ) ) |
#                    ( power ( POWER_SUPPLY | LED ) ) |
#                    ( interface ( INTF | LED ) ) ) color COLOR [ flashing ]
#--------------------------------------------------------------------------------
class LedCmd( CliCommand.CliCommandClass ):
   syntax = ( '( ( chassis ( CHASSIS | LED ) ) | '
                '( fan-tray ( FAN_TRAY | LED ) ) | '
                '( power ( POWER_SUPPLY | LED ) ) | '
                '( interface ( INTF | LED ) ) ) color COLOR [ flashing ]' )
   noOrDefaultSyntax = ( '( chassis CHASSIS ) |'
                         '( fan-tray FAN_TRAY ) |'
                         '( power POWER_SUPPLY ) |'
                         '( interface INTF ) |'
                         '( [ interface | chassis | fan-tray | power ] LED )' )
   data = {
      'chassis' : 'Chassis LEDs',
      'CHASSIS' : ChassisStatusMatcher(),
      'fan-tray' : 'Fan-tray LEDs',
      'FAN_TRAY' : FruCli.FanTrayMatcher(),
      'power' : 'Power supply LEDs',
      'POWER_SUPPLY' : FruCli.PowerSupplyMatcher(),
      'interface' : 'Interface LEDs',
      'INTF' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, ) ),
      'LED' : CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern='.+',
            helpdesc='LED target', helpname='WORD',
            priority=CliParser.PRIO_LOW ),
         hidden=True # NOTE: See BUG254320 for more info
      ),
      'color' : 'LED color',
      'COLOR' : LedCommon.colorEnumMatcher,
      'flashing' : 'Flash this LED',
   }

   @staticmethod
   def handler( mode, args ):
      color = args.get( 'COLOR' )
      flashing = 'flashing' in args
      disable = CliCommand.isNoOrDefaultCmd( args )

      if 'CHASSIS' in args:
         led = args[ 'CHASSIS' ]
         setLed( mode, led=led, color=color, disable=disable, flashing=flashing )
      elif 'FAN_TRAY' in args:
         # In modular systems the fan number is declared as 1/1, 1/2, 2/1, 2/2, etc.
         # By sorting a list of all LEDs that start with 'Fan', we not only include
         # Fan as well as FanTray naming schemes, but we can also catch the modular
         # scheme.
         led = args[ 'FAN_TRAY' ]
         fanLeds = [ fan for fan in hwLedConfig.leds if fan.startswith( 'Fan' ) ]
         fanLeds = sorted( fanLeds )
         if len( fanLeds ) >= int( led.label ):
            setLed( mode, fanLeds[ int( led.label ) - 1 ], color, disable, flashing )
         else:
            setLed( mode, led.label, color, disable, flashing )
      elif 'POWER_SUPPLY' in args:
         # As with the fan-tray LEDs, need to get a sorted list for the
         # powerSupplyRule to act correctly.
         led = args[ 'POWER_SUPPLY' ]
         powerLeds = [ power for power in hwLedConfig.leds
                        if power.startswith( 'Power' ) ]
         powerLeds = sorted( powerLeds )
         if len( powerLeds ) >= int( led.label ):
            setLed( mode, powerLeds[ int( led.label ) - 1 ], color, disable,
                  flashing )
         else:
            setLed( mode, led.label, color, disable, flashing )
      elif 'INTF' in args:
         led = args[ 'INTF' ]
         for interface in led.intfNames():
            setLed( mode, interface, color, disable, flashing )
      elif 'LED' in args:
         led = args[ 'LED' ]
         setLed( mode, led=led, color=color, disable=disable, flashing=flashing )
      else:
         assert False, 'Unknown type'

   noOrDefaultHandler = handler

LedMode.LedMode.addCommandClass( LedCmd )

#--------------------------------------------------------------------------------
# show led
#--------------------------------------------------------------------------------
def ledToShowColor( led, ledSet ):
   for color in [ 'green', 'yellow', 'red', 'blue' ]:
      if getattr( ledSet.led[ led ], color ):
         return color
   return 'off'

class ShowLedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led'
   data = {
      'led' : 'User configured LED',
   }
   cliModel = LedModel

   @staticmethod
   def handler( mode, args ):
      ledSet = LedCommon.getLedSet( SET_NAME, ledConfig )
      ret = LedModel()
      for led in ledSet.led:
         ledC = Led()
         ledC.color = ledToShowColor( led, ledSet )
         ledC.flash = True if ledSet.led[ led ].flashRate == 1 else False
         ret.leds[ led ] = ledC
      return ret

LedMode.LedMode.addShowCommandClass( ShowLedCmd )

#------------------------------------------------------------------------------------
# Plugin initialization
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global entityMib, hwLedConfig, ledConfig
   
   entityMib = LazyMount.mount( entityManager, "hardware/entmib",
                                "EntityMib::Status", "r" )
   hwLedConfig = LazyMount.mount( entityManager, 'hardware/led/configInit',
                                  'Hardware::Led::LedSystemConfigDir', 'r' )
   ledConfig = ConfigMount.mount( entityManager, 'led/config', 'Tac::Dir', 'w' )
