# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time
from datetime import datetime

from ArnetModel import Ip4Address
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from IntfModels import Interface

import TableOutput

fmt1 = "  %-25s: %s"
fmt2 = "  %-28s: %s"
left = TableOutput.Format( justify="left" )
f1 = TableOutput.Format( justify="left", wrap=True, maxWidth=15 )
f2 = TableOutput.Format( justify="left", wrap=True, maxWidth=10 )

def _createTable( headings ):
   table = TableOutput.createTable( headings )
   f = TableOutput.Format( justify="left" )
   f.noPadLeftIs( True )
   table.formatColumns( *( [ f ] * len( headings ) ) )
   return table

def _shortJobName( name ):
   return name[ :32 ]

def _shortClusterName( name ):
   return name[ :8 ]

def _shortTtName( ttName ):
   return ttName.split( "." )[ 0 ]

def _intfName( intf ):
   return intf.stringValue

def humanReadableBytes( givenBytes ):
   """Converts a number of bytes into a human-readable size (e.g. 3.42MB)."""
   if givenBytes >= 1099511627776:
      teraBytes = givenBytes / 1099511627776.0
      size = '%.2fTB' % teraBytes
   elif givenBytes >= 1073741824:
      gigaBytes = givenBytes / 1073741824.0
      size = '%.2fGB' % gigaBytes
   elif givenBytes >= 1048576:
      megaBytes = givenBytes / 1048576.0
      size = '%.2fMB' % megaBytes
   elif givenBytes >= 1024:
      kiloBytes = givenBytes / 1024.0
      size = '%.2fKB' % kiloBytes
   else:
      size = '%d' % givenBytes
   return size

def _printCountersWarning():
   print ( "Note: these counters are derived from Hadoop counters and represent "
           "approximate network bandwidth utilization" )

def _percentStringValue( value ):
   return "%.2f" % ( value*100 ) + "%"

def _getTimeStr( timeVal ):
   updated = time.localtime( timeVal )
   timeStr = time.strftime( "%Y-%m-%d %H:%M:%S", updated )
   return timeStr


class HasUpdateTime( Model ):
   lastUpdated = Float( help="UNIX timestamp in seconds of the last update"
                        " from JobTracker(MR1)/ResourceManager(MR2)", optional=True )

   def _printLastUpdated( self ):
      if self.lastUpdated:
         updateStr = _getTimeStr( self.lastUpdated )
      else:
         updateStr = "Never"
      print "Last updated: %s" % updateStr


class ClusterError( Model ):
   errorSeenAt = Float( help="UNIX timestamp in seconds when the error was seen"
                        " during an update from the JobTracker(MR1)/"
                        "ResourceManager(MR2)" )
   error = Str( help="Error message" )


# Model that stores cluster summary
class HadoopStatus( HasUpdateTime ):
   adminStatus = Bool( help="MapReduce feature is administratively enabled or not" )
   operStatus = Bool( help="Operational status of MapReduce feature" )
   numClusters = Int( help="Number of clusters configured" )
   numTaskTrackers = Int( help="Number of local Nodes" )
   numRunningJobs = Int( help="Number of jobs running locally" )
   clusterErrors = Dict( keyType=str, valueType=ClusterError,
                         help="Error messages for clusters if there are any. Key"
                         " is cluster name" )
   def render( self ):
      fmt3 = "  %-40s: %s"
      self._printLastUpdated()
      print "Mapreduce Tracer status:"
      print fmt3 % ( "Admin status", "Enabled" if self.adminStatus else "Disabled" )
      print fmt3 % ( "Operational status", "Enabled" if self.operStatus
                     else "Disabled" )
      print fmt3 % ( "Number of clusters configured", self.numClusters )
      print fmt3 % ( "Number of local Nodes", self.numTaskTrackers )
      print fmt3 % ( "Number of jobs running locally", self.numRunningJobs )

      # print cluster errors if any but only if operational status is enabled
      if self.operStatus:
         print fmt3 % ( "Cluster Errors", len( self.clusterErrors ) if
                        self.clusterErrors else "None" )
         if self.clusterErrors:
            table = TableOutput.TableFormatter()
            format1 = TableOutput.Format( justify="left", maxWidth=14 )
            format2 = TableOutput.Format( justify="right", maxWidth=2 )
            format3 = TableOutput.Format( justify="left", wrap=True, maxWidth=60 )
            table.formatColumns( format2, format1, format2, format3 )
            for cluster, clusterError in self.clusterErrors.iteritems():
               table.newRow( "  ", "Cluster", ":", cluster )
               table.newRow( "  ", "  Error Time", ":",
                             _getTimeStr( clusterError.errorSeenAt ) )
               table.newRow( "  ", "  Error", ":", clusterError.error )
               table.newRow( "  ", " ", " ", " " ) # one blank line between clusters
            print table.output()


# Model for counters that we care about
class Counters( Model ):
   hdfsBytesRead = Int( help="HDFS bytes read so far for this job" )
   hdfsBytesWritten = Int( help="HDFS bytes written so far for this job" )
   reduceShuffleBytes = Int( help="Bytes read during reduce shuffle phase" )


# Per interface counters and list of all the TTs on that interface. There could
# be more than one TT per interface when we support VMs
class CountersPerInterface( Counters ):
   interface = Interface( help="Interface to which the counters belong" )
   taskTrackers = List( valueType=str, help="List of Nodes on the "
                        "interface on which counters were collected" )


# Basic information for a job. This model is mostly used when a collection of jobs
# is returned.
class BasicJobInfo( HasUpdateTime ):
   name = Str( help="Job name" )
   jobId = Int( help="Job Id" )
   jtId = Str( help="JobTracker Id" )
   user = Str( help="User for the job" )
   cluster = Str( help="Cluster on which the job is running" )
   maps = Int( help="Number of map tasks for this jobs" )
   reduces = Int( help="Number of reduce tasks for this jobs" )
   startTime = Float( help="Time when job was started" )
   mapProgress = Float( help="Progress of all map tasks so far" )
   reduceProgress = Float( help="Progress of all reduce tasks so far" )
   countersPerInterface = Dict( valueType=CountersPerInterface,
                                keyType=Interface, help="Counters per interface" )
   taskTrackers = List( valueType=str,
                        help="List of Nodes the job is running on" )


# For all the details of a given job this model is used
class JobInfo( BasicJobInfo ):
   cleanupProgress = Float( help="Cleanup progress of the job" )
   setupProgress = Float( help="Setup progress of the job" )
   priority = Enum( values=( "veryHigh", "high", "normal", "low", "veryLow",
                             "none" ), help="Priority of this job" )
   runState = Enum( values=( "unknown", "running", "succeeded", "failed", "prep",
                             "killed" ), help="Running state of the job" )
   failureInfo = Str( help="Failure information for the job, if any" )
   schedulingInfo = Str( help="Scheduling information for the job" )
   queueName = Str( help="Name of the queue for the job" )
   url = Str( help="Web URL for the job" )

   def render( self ):
      self._printLastUpdated()
      print "Information for job:", self.name, "running on cluster:", self.cluster
      table = TableOutput.TableFormatter()
      format1 = TableOutput.Format( justify="left", maxWidth=22 )
      format2 = TableOutput.Format( justify="right", maxWidth=2 )
      format3 = TableOutput.Format( justify="left", wrap=True, maxWidth=50 )
      format2.noPadLeftIs( True )
      format3.noPadLeftIs( True )
      table.formatColumns( format1, format2, format3 )

      table.newRow( "Cluster", ":", self.cluster )
      table.newRow( "Id", ":", self.jobId )
      table.newRow( "Name", ":", self.name )
      table.newRow( "User", ":", self.user )
      if self.priority != "none":
         table.newRow( "Priority", ":", self.priority )
      table.newRow( "Running state", ":", self.runState )
      table.newRow( "Number of map tasks", ":", self.maps )
      table.newRow( "Number of reduce tasks", ":", self.reduces )
      td = datetime.fromtimestamp( self.startTime )
      table.newRow( "Start time", ":", td.strftime( "%Y-%m-%d %H:%M:%S" ) )
      table.newRow( "Bytes In", ":", humanReadableBytes(
            sum( c.hdfsBytesRead for
                 c in self.countersPerInterface.itervalues() ) +
            sum( c.reduceShuffleBytes for
                 c in self.countersPerInterface.itervalues() ) ) )
      table.newRow( "Bytes Out", ":", humanReadableBytes(
            sum( c.hdfsBytesWritten for
                 c in self.countersPerInterface.itervalues() ) ) )
      table.newRow( "Map Progress", ":", _percentStringValue( self.mapProgress ) )
      table.newRow( "Reduce Progress", ":",
                    _percentStringValue( self.reduceProgress ) )
      if not self.cleanupProgress < 0: #negative values are ignored
         table.newRow( "Cleanup Progress", ":",
                       _percentStringValue( self.cleanupProgress ) )
      if not self.setupProgress < 0: #negative values are ignored
         table.newRow( "Setup Progress", ":",
                       _percentStringValue( self.setupProgress ) )
      table.newRow( "Nodes", ":",
                    ", ".join( _shortTtName( ttName )
                               for ttName in sorted( self.taskTrackers) ) )

      print table.output()


# Represents all running jobs on a cluster or node
class RunningJobs( HasUpdateTime ):
   jobs = Dict( keyType=str, valueType=BasicJobInfo,
                help="Dictonary of all running jobs on "
                "directly attached Nodes.  Keyed by composite JobId as "
                "shown on Hadoop web interface plus cluster name" )

   def printFormatted( self ):
      table = _createTable( [ "JobId", "Job Name", "Cluster", "Maps(#/%)",
                              "Reduces(#/%)", "Start Time" ] )
      table.formatColumns( left, f1, left, left, left, left )
      for jobKey, job in sorted( self.jobs.iteritems() ):
         td = datetime.fromtimestamp( job.startTime )
         maps = "%d/%s" % ( job.maps, _percentStringValue( job.mapProgress ) )
         reduces = "%d/%s" % ( job.reduces,
                                   _percentStringValue( job.reduceProgress ) )
         table.newRow( jobKey.split("-")[ 2 ], _shortJobName( job.name ),
                       _shortClusterName( job.cluster ), maps, reduces,
                       td.strftime( "%Y-%m-%d %H:%M:%S" ) )
      print table.output()

   def render( self ):
      if self.jobs and len( self.jobs ) :
         self._printLastUpdated()
         print "Currently running jobs: %d" % len( self.jobs )
         self.printFormatted()
      else:
         print "No running jobs found"


class TaskTrackerRunningJobs( HasUpdateTime ):
   runningJobs = Dict( valueType=RunningJobs,
                       help="Dictionay of running jobs per Node"
                       " keyed by Node name" )

   def render( self ):
      # we are looping twice to figure out if there are any running tasks in any of
      # the Nodes. If not then print the message and return
      for runningJobs in self.runningJobs.itervalues():
         if runningJobs.jobs:
            break
      else:
         print "No running jobs found"
         return
      self._printLastUpdated()
      for ttName, runningJobs in self.runningJobs.iteritems():
         print "Running job for Node: %s" % _shortTtName( ttName )
         runningJobs.printFormatted()
         print


# Derives from RunningJobs. We need this separately as it displays counters in the
# render method as oposed to task progress in RunningJobs
class HadoopCounters( RunningJobs ):
   def printFormatted( self ):
      table = _createTable( [ "JobId", "Job Name", "Cluster", "Bytes In",
                              "Bytes Out", "Start Time" ] )
      table.formatColumns( left, f1, left, left, left, left )
      for job in self.jobs.itervalues():
         bytesIn = ( sum( c.hdfsBytesRead for
                          c in job.countersPerInterface.itervalues() ) +
                     sum( c.reduceShuffleBytes for
                          c in job.countersPerInterface.itervalues() ) )
         bytesOut = sum( c.hdfsBytesWritten for
                         c in job.countersPerInterface.itervalues() )
         td = datetime.fromtimestamp( job.startTime )
         table.newRow( job.jobId, _shortJobName( job.name ),
                       _shortClusterName( job.cluster ),
                       humanReadableBytes( bytesIn ),
                       humanReadableBytes( bytesOut ),
                       td.strftime( "%Y-%m-%d %H:%M:%S" ) )
      print table.output()
      _printCountersWarning()

   def render( self ):
      if not self.jobs:
         print "No running jobs found"
         return
      self._printLastUpdated()
      print "Counters for running jobs:"
      self.printFormatted()


class TaskTrackerPerJobCounters( HasUpdateTime ):
   runningJobs = Dict( valueType=HadoopCounters,
                       help="Dictionay of HadoopCounters per Node"
                       " keyed by Node name" )

   def render( self ):
      # we are looping twice to figure out if there are any running tasks in any of
      # the Node. If not then print the message and return
      for runningJobs in self.runningJobs.itervalues():
         if runningJobs.jobs:
            break
      else:
         print "No running jobs found"
         return
      self._printLastUpdated()
      for ttName, runningJobs in self.runningJobs.iteritems():
         print "Running job for Node: %s" % _shortTtName( ttName )
         runningJobs.printFormatted()


# Counters per job for each interface. Basically a list of jobs as BasicInfo
# that has counters in it
class PerJobPerInterfaceCounters( HasUpdateTime ):
   jobInfo = Submodel( valueType=BasicJobInfo,
                       help="Job information along with the interface counters" )

   def render( self ):
      if not self.jobInfo:
         print "Job not found"
         return
      self._printLastUpdated()
      print fmt1 % ( "Cluster", self.jobInfo.cluster )
      print fmt1 % ( "Job Name", self.jobInfo.name )
      print fmt1 % ( "Job Id", self.jobInfo.jobId )

      table = _createTable( [ "Interface", "HDFS Bytes Read", "HDFS Bytes Written",
                              "Reduce Shuffle Bytes" ] )
      for counter in self.jobInfo.countersPerInterface.itervalues():
         table.newRow( _intfName( counter.interface ) if counter.interface else
                       "Unknown",
                       humanReadableBytes( counter.hdfsBytesRead ),
                       humanReadableBytes( counter.hdfsBytesWritten ),
                       humanReadableBytes( counter.reduceShuffleBytes ) )
      print table.output()


# Data for one historical job
class JobHistoryData( HasUpdateTime ):
   jobId = Int( help="Job Id of  the job" )
   jtId = Str( help="JT Id of  the JobTracker" )
   jobName = Str( help="Name of the job" )
   cluster = Str( help="Cluster on which the job ran" )
   user = Str( help="User associated with the job" )
   startTime = Float( help="Start time of the job" )
   endTime = Float( help="End time of the job" )
   countersPerInterface = Dict( valueType=CountersPerInterface, keyType=Interface,
                                help="Counters per interface" )

   def render( self ):
      print "Job history data for job: %s" % self.jobName
      print fmt1 % ( "Cluster", self.cluster )
      print fmt1 % ( "Job Id", self.jobId )
      print fmt1 % ( "JT Id", self.jtId )
      print fmt1 % ( "User", self.user )
      start = datetime.fromtimestamp( self.startTime )
      end = datetime.fromtimestamp( self.endTime )
      print fmt1 % ( "Job start time", start.strftime( "%Y-%m-%d %H:%M:%S" ) )
      print fmt1 % ( "Job end time", end.strftime( "%Y-%m-%d %H:%M:%S" ) )
      print "Per Interface job counters:"
      table = _createTable( [ "Interface", "Node", "Bytes In", "Bytes Out" ] )
      for counter in self.countersPerInterface.itervalues():
         table.newRow( _intfName( counter.interface ) if counter.interface else
                       "Unknown", counter.taskTrackers[ 0 ],
                       humanReadableBytes( counter.hdfsBytesRead  +
                                           counter.reduceShuffleBytes ),
                       humanReadableBytes( counter.hdfsBytesWritten ) )
      print table.output()


# Job history for a cluster.
class JobHistory( HasUpdateTime ):
   jobHistory = Dict( keyType=str, valueType=JobHistoryData,
                      help="List of historical jobs. Keyed by cluster-jobId" )

   def render( self ):
      if not self.jobHistory:
         print "No job history found"
         return
      print "Job history for all clusters:"
      table = _createTable( [ "JobId", "Job Name", "Cluster", "Start Time",
                              "End Time", "Bytes In", "Bytes Out" ] )
      table.formatColumns( left, f1, left, f2, f2, left, left )
      for jobKey, job in sorted( self.jobHistory.iteritems(),
                                 key=lambda j: j[ 1 ].endTime, reverse=True ):
         start = datetime.fromtimestamp( job.startTime )
         end = datetime.fromtimestamp( job.endTime )

         inBytes = outBytes = 0
         for c in job.countersPerInterface.itervalues():
            inBytes += c.hdfsBytesRead + c.reduceShuffleBytes
            outBytes += c.hdfsBytesWritten
         jobKey = jobKey.split( "-" )
         displayKey = jobKey[ 2 ]
         table.newRow( displayKey, job.jobName[ : 40 ],
                       _shortClusterName( job.cluster ),
                       start.strftime( "%Y-%m-%d %H:%M:%S" ),
                       end.strftime( "%Y-%m-%d %H:%M:%S" ),
                       humanReadableBytes( inBytes ),
                       humanReadableBytes( outBytes ) )
      print table.output()


# A counter that represents a burst.
class BurstCounter( Model ):
   value = Int( help="Burst value" )
   jobId = Int( help="Job Id" )
   jobName = Str( help="Job name" )
   timestamp = Float( help="Time of the burst" )


# Model that stores input and output bursts per interface. It will have up to
# 5 bursts in the list
class InterfaceBurst( HasUpdateTime ):
   inBursts = List( valueType=BurstCounter,
                    help="List of top input bursts for a Node during last "
                    "update interval" )
   outBursts = List( valueType=BurstCounter,
                    help="List of top output bursts for a Node during last "
                    "update interval" )
   interface = Interface( help="Interface on which the traffic was transmitted" )
   cluster = Str( help="Cluster on which the burst happened" )

   def render( self ):
      # if there are no bursts then say so and return
      if not self.inBursts and not self.outBursts:
         print "No bursts data found"
         return
      print "Bursts on Interface: %s in cluster: %s" % ( self.interface,
                                                         self.cluster )
      if self.inBursts:
         print "Top %d input bursts:" % len( self.inBursts )
         table = _createTable( [ "JobId", "Job Name", "Burst", "Time" ] )
         table.formatColumns( left, f1, left, left )
         for burst in self.inBursts:
            td = datetime.fromtimestamp( burst.timestamp )
            table.newRow( burst.jobId, _shortJobName( burst.jobName ),
                          humanReadableBytes( burst.value ),
                          td.strftime( "%Y-%m-%d %H:%M:%S" ) )
         print table.output()
      else:
         print "Input bursts: No input bursts available"
         print
      if self.outBursts:
         print "Top %d output bursts:" % len( self.outBursts )
         table = _createTable( [ "JobId", "Job Name", "Burst", "Time" ] )
         table.formatColumns( left, f1, left, left )
         for burst in self.outBursts:
            td = datetime.fromtimestamp( burst.timestamp )
            table.newRow( burst.jobId, _shortJobName( burst.jobName ),
                          humanReadableBytes( burst.value ),
                          td.strftime( "%Y-%m-%d %H:%M:%S" ) )
         print table.output()
      else:
         print "Output bursts: No output bursts available"
         print

# List of per interface bursts. We can have a list if multiple interfaces
# are given in the command or if we are showing for all interfaces
class InterfaceBursts( HasUpdateTime ):
   bursts = List( valueType=InterfaceBurst,
                  help="List of bursts per interface" )
   def render( self ):
      if not self.bursts:
         print "No bursts found"
         return
      self._printLastUpdated()
      for intfBurst in sorted( self.bursts,
                               key=lambda burst: burst.interface ):
         intfBurst.render()

# A task report with information as returned by JobTracker RPC
class TaskReport( HasUpdateTime ):
   taskTrackerName = Str( help="Name of the Node where tasks are running" )
   interface = Interface( help="Interface on which the traffic was transmitted" )
   jobId = Int( help="Job id  of a job this task belongs to" )
   taskId = Int( help="Task id of the task" )
   cluster = Str( help="Cluster where the task is running" )
   attemptId = Int( help="Attempt id of the task" )
   isMap = Bool( help="Whether task is Map or not" )
   status = Enum( help="Status of the task",
                  values=( "pending", "running", "complete", "killed", "failed" ) )
   state = Str( help="State of the task" )
   startTime = Float( help="Start time of the task" )
   endTime = Float( help="End time of the task" )
   progress = Float( help="Progress of the task" )
   hdfsBytesRead = Int( help="HDFS bytes read so far for this task" )
   hdfsBytesWritten = Int( help="HDFS bytes written so far for this task" )
   reduceShuffleBytes = Int( help="Bytes read during reduce shuffle phase" )

   def render( self ):
      self._printLastUpdated()
      print "Task details for one task as given below:"
      print fmt1 % ( "Node name", self.taskTrackerName )
      print fmt1 % ( "Interface", self.interface if self.interface else "Unknown" )
      print fmt1 % ( "Cluster", self.cluster )
      print fmt1 % ( "Job Id", self.jobId )
      print fmt1 % ( "Task Id", self.taskId )
      print fmt1 % ( "Attempt Id", self.attemptId )
      print fmt1 % ( "Task type", "Map" if self.isMap else "Reduce" )
      print fmt1 % ( "Status", self.status )
      print fmt1 % ( "State", self.state )
      td = datetime.fromtimestamp( self.startTime )
      print fmt1 % ( "Start time", td.strftime( "%Y-%m-%d %H:%M:%S" ) )
      print fmt1 % ( "Progress", _percentStringValue( self.progress ) )
      print fmt1 % ( "HDFS bytes read", humanReadableBytes( self.hdfsBytesRead ) )
      print fmt1 % ( "HDFS bytes written",
                     humanReadableBytes( self.hdfsBytesWritten ) )
      print fmt1 % ( "Reduce shuffle bytes",
                     humanReadableBytes( self.reduceShuffleBytes ) )

# List of task reports. This signifies all running tasks on a Node
class RunningTasks( Model ):
   taskTrackerName = Str( help="Name of the Node where tasks are running" )
   interface = Interface( help="Interface to which the Node is attached" )
   taskReports = List( valueType=TaskReport,
                       help="List of running tasks on a Node" )

   def render( self ):
      if not self.taskReports:
         return
      print ( "Running tasks for Node: %s on interface %s"
              % ( _shortTtName( self.taskTrackerName ),
                  _intfName( self.interface ) ) )
      table = _createTable( [ "JobId", "TaskId", "Cluster", "Type", "Progress",
                              "Status", "HDFS Read", "HDFS Write", "Shuffle" ] )
      for tr in sorted( self.taskReports, key=lambda tr: tr.hdfsBytesRead +
                        tr.hdfsBytesWritten + tr.reduceShuffleBytes,
                        reverse=True ):
         table.newRow( tr.jobId, tr.taskId, _shortClusterName( tr.cluster ),
                       "Map" if tr.isMap else "Reduce",
                       _percentStringValue( tr.progress ), tr.status,
                       humanReadableBytes( tr.hdfsBytesRead ),
                       humanReadableBytes( tr.hdfsBytesWritten ),
                       humanReadableBytes( tr.reduceShuffleBytes ) )
      print table.output()

class TaskTrackerRunningTasks( HasUpdateTime ):
   runningTasksList = List( valueType=RunningTasks,
                            help="List of running tasks per Node" )

   def render( self ):
      # we are looping twice to figure out if there are any runing tasks in any of
      # the Nodes. If not then print the message and return
      for runningTasks in self.runningTasksList:
         if runningTasks.taskReports:
            break
      else:
         print "No running tasks found"
         return
      self._printLastUpdated()
      for runningTasks in self.runningTasksList:
         runningTasks.render()


# Maps the state of a Node to a human readable description.
STATE_DETAILS = {
      "resolved": "ARP and DNS are resolved",
      "unresolvedDnsName": "Unresolved DNS name",
      "noArpEntry": "There is no ARP entry for this IP",
      "unresolvedArpEntry": "ARP entry cannot be resolved to an interface",
      "peerArpEntry": "Node is directly connected to peer switch only",
}

# Model to represent a Node. We keep lists of running jobs and running tasks
# as they are used in many other places for various CLI commands. With this object
# user gets a very good picture about a Node
class TaskTracker( HasUpdateTime ):
   name = Str( help="Node hostname" )
   cluster = Str( help="Name of the cluster to which the Node belongs" )
   ipAddress = Ip4Address( help="Ip Address of the Node" )
   interface = Interface( help="Interface to which the Node is attached" )
   state = Enum( help="Node state of active or inactive",
                 values=( "active", "inactive" ) )
   stateDetail = Enum( help="Node state detail",
                       values=( "resolved", "unresolvedDnsName", "noArpEntry",
                                "unresolvedArpEntry", "peerArpEntry" ) )
   runningJobs = Submodel( valueType=RunningJobs,
                           help="List of currently running jobs on the Node" )
   runningTasks = Submodel( valueType=RunningTasks,
                            help="List of currently running tasks on the Node" )

   def getBytesReadWritten( self ):
      bytesRead = bytesWritten = 0
      for _, jobInfo in self.runningJobs.jobs.iteritems():
         bytesRead += sum( c.hdfsBytesRead + c.reduceShuffleBytes for
                           c in jobInfo.countersPerInterface.itervalues() if
                           c.interface == self.interface )
         bytesWritten += sum( c.hdfsBytesWritten for
                              c in jobInfo.countersPerInterface.itervalues() if
                              c.interface == self.interface )
      return bytesRead, bytesWritten

   def printFormatted( self ):
      print fmt1 % ( "Node", self.name )
      print fmt1 % ( "IP Address", self.ipAddress )
      print fmt1 % ( "Interface", _intfName( self.interface ) )
      print fmt1 % ( "State", self.state )
      print fmt1 % ( "State detail", STATE_DETAILS[ self.stateDetail ] )
      print fmt1 % ( "Running jobs", len( self.runningJobs.jobs ) )
      print fmt1 % ( "Running tasks", sum( 1 for tr in self.runningTasks.taskReports
                                           if tr.state == "running" ) )
      print fmt1 % ( "Map Tasks", sum(1 for tr in self.runningTasks.taskReports
                                      if tr.isMap) )
      print fmt1 % ( "Reduce Tasks", sum(1 for tr in self.runningTasks.taskReports
                                         if not tr.isMap) )
      read, written = self.getBytesReadWritten()
      print fmt1 % ( "Total bytes read", humanReadableBytes( read ) )
      print fmt1 % ( "Total bytes written", humanReadableBytes( written ) )

   def render( self ):
      self._printLastUpdated()
      self.printFormatted()
      # print extra blank line so that we are consistent with printFormatted of
      # TaskTrackers and InterfaceTaskTrackers as they need this extra line between
      # two TTs
      print

# A dictionary of all Nodes attached to the switch
class TaskTrackers( HasUpdateTime ):
   taskTrackers = Dict( valueType=TaskTracker, help="Dictionary of locally attached "
                        "Nodes with running task information" )
   _printDetail = Bool( help="Flag that tells whether to print Node details "
                        "or not", optional=True )

   def printFormatted( self ):
      if self._printDetail:
         for _, tt in self.taskTrackers.iteritems():
            tt.printFormatted()
            print
      else:
         table = _createTable( [ "Node", "Cluster", "IP Address", "Interface",
                                 "State", "Maps", "Reduces" ] )
         table.formatColumns( f1, left, left, left, left, left, left )
         for ttName, tt in sorted( self.taskTrackers.iteritems(),
                                   key=lambda t: t[ 1 ].interface ):
            maps = sum( 1 for tr in tt.runningTasks.taskReports if tr.isMap )
            reduces = sum( 1 for tr in tt.runningTasks.taskReports if not tr.isMap )
            table.newRow( _shortTtName( ttName ), _shortClusterName( tt.cluster ),
                          tt.ipAddress, _intfName( tt.interface ), tt.state,
                          maps, reduces )
         print table.output()

   def render( self ):
      if not self.taskTrackers:
         print "No Nodes found"
         return
      self._printLastUpdated()
      print "All local Nodes:"
      self.printFormatted()

# Model that represents Nodes on a set of interfaces. When user gives multiple
# interfaces on the CLI command he/she gets this model object. If user specifies
# detail option then _printDetail is set
class InterfaceTaskTrackers( TaskTrackers ):
   def render( self ):
      if not self.taskTrackers:
         print "No Nodes found"
         return

      self._printLastUpdated()
      if self._printDetail:
         for _, tt in sorted( self.taskTrackers.iteritems() ):
            tt.printFormatted()
            print
      else:
         table = _createTable( [ "Interface", "IP Address", "Node", "State",
                                 "Map Tasks", "Reduce Tasks" ] )
         for ttName, tt in sorted( self.taskTrackers.iteritems(),
                                   key=lambda t: t[ 1 ].interface ):
            maps = sum( 1 for tr in tt.runningTasks if tr.isMap )
            reduces = sum( 1 for tr in tt.runningTasks if not tr.isMap )
            table.newRow( _intfName( tt.interface ), tt.ipAddress,
                          _shortTtName( ttName ), tt.state, maps, reduces )
         print table.output()


# Model that represents counters for a Node. We generate this object
# separately for Nodes to just show Node counters
class TaskTrackerCounter( Model ):
   name = Str( help="Node hostname" )
   ipAddress = Ip4Address( help="Ip Address of the Node" )
   interface = Interface( help="Interface to which the Node is attached" )
   bytesIn = Int( help="Total bytes read in by all the jobs running on this "
                  "Node" )
   bytesOut = Int( help="Total bytes written in by all the jobs running on this "
                   "Node" )


# List of above counter
class TaskTrackerCounters( HasUpdateTime ):
   counters = Dict( keyType=str, valueType=TaskTrackerCounter,
                    help="Node counters keyed by Node host name" )

   def render( self ):
      if not self.counters:
         print "No Nodes found"
         return
      self._printLastUpdated()
      print "Counters for all Nodes:"
      table = _createTable( [ "Node", "IP Address", "Interface", "Bytes Read",
                              "Bytes Written" ] )
      table.formatColumns( f1, left, left, left, left )
      for name, counter in sorted( self.counters.iteritems() ):
         table.newRow( _shortTtName( name ), counter.ipAddress,
                       _intfName( counter.interface ),
                       humanReadableBytes( counter.bytesIn ),
                       humanReadableBytes( counter.bytesOut ) )
      print table.output()

# Model that represents configuration for a cluster
class ClusterConfig( Model ):
   host = Str( help="JobTracker hostname" )
   rpcPort = Int( help="RPC port number for the JobTracker" )
   user = Str( help="Username used to connect with the JobTracker" )
   interval = Int( help="Polling interval to poll from the JobTracker using RPC" )
   adminStatus = Bool( help="Is cluster administratively enabled or not" )
   httpPort = Int( help="HTTP port number for TaskTrackers" )
   mapReduceVersion = Str( help="MapReduce Version", optional=True )
   resourceManagerHost = Str( help="ResourceManager hostname", optional=True )
   resourceManagerPort = Int( help="Port number for the ResourceManager", 
                              optional=True )      
   jobHistoryHost = Str( help="JobHistoryServer hostname", optional=True )
   jobHistoryPort = Int( help="Port number for the JobHistoryServer", 
                         optional=True )      

   def printFormatted( self ):
      if self.mapReduceVersion is not None:
         print fmt2 % ( "MapReduce Version", self.mapReduceVersion )
      print fmt2 % ( "Admin status", "Enabled" if self.adminStatus
                     else "Disabled" )
      print fmt2 % ( "Polling interval", str( self.interval ) + " seconds" )
      if self.mapReduceVersion == "MR1":
         print fmt2 % ( "JobTracker host", self.host )
         print fmt2 % ( "JobTracker RPC port", self.rpcPort )
         print fmt2 % ( "JobTracker user", self.user )
         print fmt2 % ( "TaskTracker HTTP port", self.httpPort )
      elif self.mapReduceVersion == "MR2":
         if self.resourceManagerHost is not None:
            print fmt2 % ( "ResourceManager host", self.resourceManagerHost )
         if self.resourceManagerPort is not None:
            print fmt2 % ( "ResourceManager port", self.resourceManagerPort )
         if self.jobHistoryHost is not None:
            print fmt2 % ( "JobHistoryServer host", self.jobHistoryHost )
         if self.jobHistoryPort is not None:
            print fmt2 % ( "JobHistoryServer port", self.jobHistoryPort )

# Model that represents basic summary for a cluster as gotten from the JobTracker
# or Resource Manager.
class ClusterInfo( Model ):
   clusterConfig = Submodel( valueType=ClusterConfig,
                             help="Configuration for the cluster" )
   name = Str( help="Name of the cluster" )
   operState = Bool( help="Is cluster data being actively polled from "
                     "the JobTracker(MR1)/ResourceManager(MR2)", optional=True )
   status = Enum( values=( "initializing", "running" ),
                  help="Current status of the JobTracker(MR1)/ResourceManager(MR2)",
                  optional=True )
   numActiveTrackers = Int( help="Number of Nodes currently serving "
         "in the cluster", optional=True )
   numBlacklistedTrackers = Int( help="Number of Nodes currently in the "
         "blacklist of the JobTracker(MR1)/ResourceManager(MR2)", optional=True )
   numDecommissionedNodes = Int( help="Number of Nodes that were "
         "decommissioned from the cluster", optional=True )
   trackerExpiryInterval = Float( help="Amount of time (in seconds) the JobTracker"
                                  "(MR1)/ResourceManager(MR2) "
                                  "will wait before considering a node dead and its"
                                  " tasks lost", optional=True )
   mapSlots = Int( help="Number of map slots available in the cluster",
                   optional=True )
   reduceSlots = Int( help="Number of reduce slots available in the cluster",
                      optional=True )
   numMapTasksRunning = Int( help="Number of map tasks currently running",
                             optional=True )
   numReduceTasksRunning  = Int( help="Number of reduce tasks currently running",
                                 optional=True )
   jobTrackerHeapSize  = Int( help="Current size of the JobTracker heap in bytes",
                              optional=True )
   jobTrackerMaxHeapSize  = Int( help="Maximum JobTracker heap size in bytes",
                                 optional=True )
   allocatedMemory = Int( help="Total memory currently allocated across all nodes"
                          " in MegaBytes(MapReduce2 only)", optional=True )
   totalMemory = Int( help="Total memory available across all Nodes in MegaBytes"
                        "(MapReduce2 only)", 
                        optional=True )
   
   def printFormatted( self ):
      version = self.clusterConfig.mapReduceVersion 
      print fmt2 % ( "Operational status", "Enabled"
                     if self.operState else "Disabled" )
      # if oper status is not enabled we don't have info to print
      if not self.operState:
         return
      print fmt2 % ( "Active Nodes", self.numActiveTrackers )
      print fmt2 % ( "Decommissioned Nodes", self.numDecommissionedNodes )
      if version == "MR1":
         print fmt2 % ( "Blacklisted Nodes", self.numBlacklistedTrackers )
         print fmt2 % ( "Tracker expiry interval", self.trackerExpiryInterval )
         print fmt2 % ( "Map slots (used/total)",
                        "%d/%d" % ( self.numMapTasksRunning, self.mapSlots ) )
         print fmt2 % ( "Reduce slots (used/total)",
                        "%d/%d" % ( self.numReduceTasksRunning, self.reduceSlots ) )
         print fmt2 % ( "JobTracker heap size", "%s (max: %s)"
                     % ( humanReadableBytes( self.jobTrackerHeapSize ),
                         humanReadableBytes( self.jobTrackerMaxHeapSize ) ) )
      elif version == "MR2":
         print fmt2 % ( "Map Tasks", self.numMapTasksRunning )
         print fmt2 % ( "Reduce Tasks", self.numReduceTasksRunning )
         if ( self.allocatedMemory and self.totalMemory ) is not None:
            print fmt2 % ( "Allocated Memory", "%s (total: %s)" %
                           ( humanReadableBytes( self.allocatedMemory 
                                                 * 1024 * 1024 ),
                             humanReadableBytes( self.totalMemory * 1024 * 1024) ) )

   def render( self ):
      print "Cluster information for the cluster: %s" % self.name
      self.printFormatted()

# Has cluster info plus Nodes on the cluster and all running jobs on those
# Nodes and that cluster
class ClusterStatus( HasUpdateTime ):
   clusterError = Submodel( valueType=ClusterError,
                            help="Error that was seen during last cluster update,"
                            " if any", optional=True )
   clusterInfo = Submodel( valueType=ClusterInfo,
                           help="Information about the cluster" )
   localTaskTrackers = Submodel( valueType=TaskTrackers,
                                 help="List of Nodes directly attached "
                                 "to the switch" )
   runningJobs = Submodel( valueType=RunningJobs,
                           help="List of currently running jobs on the cluster" )

   def render( self ):
      # print last updated only if operationally enabled
      if self.clusterInfo.operState:
         self._printLastUpdated()
      print "Cluster status for cluster: ", self.clusterInfo.name
      self.clusterInfo.clusterConfig.printFormatted()
      self.clusterInfo.printFormatted()
      # if operationally active then print info from the JobTracker
      if self.clusterInfo.operState:
         if self.clusterError:
            print fmt2 % ( "Cluster error", "" )
            table = TableOutput.TableFormatter()
            format1 = TableOutput.Format( justify="left", maxWidth=14 )
            format2 = TableOutput.Format( justify="right", maxWidth=2 )
            format3 = TableOutput.Format( justify="left", wrap=True, maxWidth=60 )
            table.formatColumns( format1, format2, format3 )

            table.newRow( " Error Time", ":",
                          _getTimeStr( self.clusterError.errorSeenAt ) )
            table.newRow( " Error", ":", self.clusterError.error )
            print table.output()
            # print fmt4 % ( "  Error Time",
            #                _getTimeStr( self.clusterError.errorSeenAt ) )
            # print fmt4 % ( "  Error", self.clusterError.error )

# A list of all the clusters in the system
class Clusters( HasUpdateTime ):
   clusters = Dict( valueType=ClusterInfo, help="List of all the clusters "
                    "configured" )

   def render( self ):
      if not self.clusters:
         print "No clusters found"
         return
      print "Total number of clusters configured: %d" % len( self.clusters )
      for cluster, clusterInfo in sorted( self.clusters.iteritems() ):
         print fmt2 % ( "Cluster", cluster )
         clusterInfo.clusterConfig.printFormatted()
         clusterInfo.printFormatted()
         print

# List of all running jobs. Derives from RunningJobs mainly due to different render()
# method
class ClusterRunningJobs( RunningJobs ):
   name = Str( help="Name of the cluster the jobs are running on" )

   def render( self ):
      if not self.jobs:
         print "No running jobs found"
         return
      self._printLastUpdated()
      print "Currently running jobs on cluster: %s" % self.name
      table = _createTable( [ "JobId", "Job Name", "User", "Maps",
                              "Reduces", "Start Time" ] )
      table.formatColumns( left, f1, left, left, left, left )
      for jobKey, job in self.jobs.iteritems():
         td = datetime.fromtimestamp( job.startTime )
         table.newRow( jobKey.split("-")[ 2 ], _shortJobName( job.name ),
                       job.user,job.maps, job.reduces,
                       td.strftime( "%Y-%m-%d %H:%M:%S" ) )
      print table.output()

# List of all running jobs. Derives from RunningJobs mainly due to different render()
# method
class ClusterHadoopCounters( HadoopCounters ):
   name = Str( help="Name of the cluster the jobs are running on" )

   def render( self ):
      if not self.jobs:
         print "No running jobs found"
         return
      self._printLastUpdated()
      print "Counters for currently running jobs on cluster: %s" % self.name
      table = _createTable( [ "JobId", "Job Name", "User", "Bytes In",
                              "Bytes Out", "Start Time" ] )
      table.formatColumns( left, f1, f2, left, left, left )
      for _, job in self.jobs.iteritems():
         bytesIn = ( sum( c.hdfsBytesRead for
                          c in job.countersPerInterface.itervalues() ) +
                     sum( c.reduceShuffleBytes for
                          c in job.countersPerInterface.itervalues() ) )
         bytesOut = sum( c.hdfsBytesWritten for
                         c in job.countersPerInterface.itervalues() )
         td = datetime.fromtimestamp( job.startTime )
         table.newRow( job.jobId, _shortJobName( job.name ), job.user,
                       humanReadableBytes( bytesIn ),
                       humanReadableBytes( bytesOut ),
                       td.strftime( "%Y-%m-%d %H:%M:%S" ) )
      print table.output()
      _printCountersWarning()

# Job history for a single cluster
class ClusterJobHistory( JobHistory ):
   name = Str( help="Name of the cluster the jobs are running on" )
   _printDetail = Bool( help="Used for internal purpose of displaying details",
                        optional=True )
   def render( self ):
      # _printDetail is set means user asked detail of one or more tasks with
      # a given jobId. Handle that separately.
      if self._printDetail:
         if not self.jobHistory:
            print "Job history for given JobId not found on the cluster"
            return
         for job in self.jobHistory.values():
            job.render()
         return

      if not self.jobHistory:
         print "No history found"
         return
      print "Jobs history on cluster: %s" % self.name
      table = _createTable( [ "JobId", "Job Name", "Start Time",
                              "End Time", "Bytes In", "Bytes Out" ] )

      table.formatColumns( f1, f1, f2, f2, left, left )
      for jobKey, job in sorted( self.jobHistory.iteritems(),
                                 key=lambda j: j[ 1 ].endTime, reverse=True ):
         start = datetime.fromtimestamp( job.startTime )
         end = datetime.fromtimestamp( job.endTime )

         inBytes = outBytes = 0
         for c in job.countersPerInterface.itervalues():
            inBytes += c.hdfsBytesRead + c.reduceShuffleBytes
            outBytes += c.hdfsBytesWritten
         jobKey = jobKey.split( "-" )
         displayKey = jobKey[ 2 ]
         table.newRow( displayKey, job.jobName[ : 42 ],
                       start.strftime( "%Y-%m-%d %H:%M:%S" ),
                       end.strftime( "%Y-%m-%d %H:%M:%S" ),
                       humanReadableBytes( inBytes ),
                       humanReadableBytes( outBytes ) )
      print table.output()

class ClusterTaskTrackers( TaskTrackers ):
   name = Str( help="Name of the cluster Nodes belong to" )
   def printFormatted( self ):
      if self._printDetail:
         for _, tt in self.taskTrackers.iteritems():
            tt.printFormatted()
            print
      else:
         table = _createTable( [ "Node", "IP Address", "Interface", "State",
                                 "Maps", "Reduces" ] )
         table.formatColumns( f1, left, left, left, left )
         for ttName, tt in sorted( self.taskTrackers.iteritems(),
                                   key=lambda t: t[ 1 ].interface ):
            maps = sum( 1 for tr in tt.runningTasks.taskReports if tr.isMap )
            reduces = sum( 1 for tr in tt.runningTasks.taskReports if not tr.isMap )
            table.newRow( _shortTtName( ttName ), tt.ipAddress,
                          _intfName( tt.interface ) if tt.interface else "Unknown",
                          tt.state, maps, reduces )
         print table.output()

   def render( self ):
      if not self.taskTrackers:
         print "No Nodes found"
         return
      self._printLastUpdated()
      print "Total %d Nodes on cluster %s:" % ( len( self.taskTrackers ),
                                                  self.name )
      self.printFormatted()

