#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from operator import attrgetter

from CliPlugin.GribiModel import (
   GribiMplsAftType,
   GribiMplsAftDetail,
   GribiFecDetail,
   GribiFecVia,
   GribiMplsAftTable,
   GribiNhgNhAftType,
   GribiNhgAftType,
   GribiNhgAftTable,
   GribiNhAftType,
   GribiNhAftDetail,
   GribiNhAftTable,
)
from CliPlugin.SrTePolicyLibCli import getSegmentListVias
from CliPlugin.TunnelCli import (
   TunnelTableIdentifier,
   readMountTunnelTable,
)
from SrTePolicyLib import MplsLabel
from Toggles import RoutingLibToggleLib
from TypeFuture import TacLazyType

import BasicCli
import CliMatcher
import LazyMount
import SharedMem
import ShowCommand
import Smash

DynTunnelIntfId = TacLazyType( "Arnet::DynamicTunnelIntfId" )
TunnelId = TacLazyType( "Tunnel::TunnelTable::TunnelId" )

gribiAftEntries = None
gribiStatus = None
gribiNhToTunnelMap = None
slTunnelTable = None
srteForwardingStatus = None

matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
      helpdesc='Traffic Engineering related information' )
matcherGribi = CliMatcher.KeywordMatcher( 'gribi',
      helpdesc='Show gRIBI AFT entries' )
matcherNexthopGroupAft = CliMatcher.KeywordMatcher( 'nexthop-group-aft',
      helpdesc='Show gRIBI Nexthop Group AFT entries' )
matcherNhgId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Nexthop Group AFT ID' )
matcherMplsAft = CliMatcher.KeywordMatcher( 'mpls-aft',
      helpdesc='Show gRIBI MPLS AFT entries' )
matcherNexthopAft = CliMatcher.KeywordMatcher( 'nexthop-aft',
      helpdesc='Show gRIBI Nexthop AFT entries' )
matcherLabel = CliMatcher.KeywordMatcher( 'label',
      helpdesc='Show gRIBI MPLS AFT entry for a label' )
matcherNhId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Nexthop AFT ID' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Show AFT detail' )

def populateNhgAft( nhgId, detail=False ):
   # get keys
   keys = []
   if nhgId is not None:
      keys.append( nhgId )
   else:
      keys = gribiAftEntries.nhgAft.entry.keys()

   # populate nexthop-group AFT entries
   nhgAftTable = GribiNhgAftTable()
   nhgAftTable.nhgAft = {}
   nhgAftTable.details_ = detail
   for key in sorted( keys ):
      nhgAftEntry = gribiAftEntries.nhgAft.entry.get( key )
      if nhgAftEntry is None:
         continue
      aft = GribiNhgAftType()
      aft.nhgId = nhgAftEntry.nhgAftEntryId
      if detail:
         aft.fecId = gribiStatus.nhgToFecEntry.get( aft.nhgId )

      for index in sorted( nhgAftEntry.nhIdColl.keys() ):
         nhgNhIdCollEntry = nhgAftEntry.nhIdColl[ index ]
         nhId = nhgNhIdCollEntry.nhAftEntryId
         weight = nhgNhIdCollEntry.weight

         nhgNhEntry = GribiNhgNhAftType()
         nhgNhEntry.nhId = nhId
         nhgNhEntry.weight = weight
         # default value of a 'List' is "[]" json output will print "[]"
         # which could cause confusion in non-detailed version of the
         # command so explicitly set it to None value so that it will
         # not be printed in json output.
         nhgNhEntry.mplsVias = None

         if detail:
            nhEntry = gribiAftEntries.nhAft.entry.get( nhId )
            if nhEntry is not None and \
               RoutingLibToggleLib.toggleGribiTeVrfPiEnabled():
               if nhEntry.ipAddr:
                  nhgNhEntry.ipAddress = nhEntry.ipAddr
               if nhEntry.intf:
                  nhgNhEntry.interface = nhEntry.intf
            tunnelId = gribiNhToTunnelMap.nhToTunnelMap.get( nhId )
            if tunnelId:
               tunnelEntry = slTunnelTable.entry.get( tunnelId, None )
               if tunnelEntry:
                  nhgNhEntry.mplsVias, _ = getSegmentListVias( tunnelEntry )

         aft.nhgNhs.append( nhgNhEntry )

      nhgAftTable.nhgAft[ long( nhgAftEntry.nhgAftEntryId ) ] = aft
   return nhgAftTable

def populateMplsAft( label, detail=False ):
   # get keys
   keys = []
   if label is not None:
      keys.append( label )
   else:
      keys = gribiAftEntries.mplsAft.entry.keys()

   # populate mpls-aft
   mplsAftTable = GribiMplsAftTable()
   mplsAftTable.mplsAft = {}
   for key in sorted( keys ):
      mplsAft = gribiAftEntries.mplsAft.entry.get( key )
      if mplsAft is None:
         continue
      aft = GribiMplsAftType()
      aft.label = mplsAft.label
      aft.nhgId = mplsAft.nhgAftEntryId

      if detail:
         # populate mpls-aft detail
         aft.mplsAftDetail = GribiMplsAftDetail()
         fecId = gribiStatus.nhgToFecEntry.get( aft.nhgId )
         if fecId is None:
            mplsAftTable.mplsAft[ long( mplsAft.label ) ] = aft
            continue
         fecDetail = GribiFecDetail()
         fecDetail.fecId = fecId
         fec = srteForwardingStatus.fec.get( fecId )
         if fec:
            for via in fec.via.itervalues():
               if not DynTunnelIntfId.isDynamicTunnelIntfId( via.intfId ):
                  continue
               fecVia = GribiFecVia()
               fecVia.tunnelIdx = \
                     TunnelId( DynTunnelIntfId.tunnelId( via.intfId )
                             ).tunnelIndex()
               fecVia.weight = via.weight
               # get tunnel info
               tunnelId = TunnelId( DynTunnelIntfId.tunnelId( via.intfId ) )
               tunnelEntry = slTunnelTable.entry.get( tunnelId, None )
               if tunnelEntry:
                  fecVia.mplsVias, _ = getSegmentListVias( tunnelEntry )
               fecDetail.fecVias.append( fecVia )
            fecDetail.fecVias = sorted( fecDetail.fecVias,
                                         key=attrgetter( 'tunnelIdx' ) )
         aft.mplsAftDetail.fecDetail = fecDetail
      mplsAftTable.mplsAft[ long( mplsAft.label ) ] = aft
   return mplsAftTable

def populateNhAft( nhId, detail=False ):
   # get keys
   keys = []
   if nhId is not None:
      keys.append( nhId )
   else:
      keys = gribiAftEntries.nhAft.entry.keys()

   # populate nexthop-aft
   nhAftTable = GribiNhAftTable()
   nhAftTable.nhAft = {}
   for key in sorted( keys ):
      nhAft = gribiAftEntries.nhAft.entry.get( key )
      if nhAft is None:
         continue
      aft = GribiNhAftType()
      aft.nhId = nhAft.nhAftEntryId
      aft.pushMplsLabels = []
      for i in range( nhAft.pushMplsLabelStack.stackSize ):
         aft.pushMplsLabels.append( nhAft.pushMplsLabelStack.labelStack( i ) )
      if RoutingLibToggleLib.toggleGribiTeVrfPiEnabled():
         if nhAft.ipAddr:
            aft.ipAddress = nhAft.ipAddr
         if nhAft.intf:
            aft.interface = nhAft.intf

      if detail:
         aft.nhAftDetail = GribiNhAftDetail()
         tunnelIdx = gribiNhToTunnelMap.nhToTunnelMap.get( aft.nhId )
         if tunnelIdx is None:
            nhAftTable.nhAft[ long( nhAft.nhAftEntryId ) ] = aft
            continue
         # populate nexthop-aft detail
         fecVia = GribiFecVia()
         aft.nhAftDetail.tunnelDetail = fecVia
         tunnelId = TunnelId( tunnelIdx ).tunnelIndex()
         fecVia.tunnelIdx = tunnelId
         tunnelEntry = slTunnelTable.entry.get( tunnelIdx, None )
         if tunnelEntry:
            fecVia.mplsVias, _ = getSegmentListVias( tunnelEntry )
      nhAftTable.nhAft[ long( nhAft.nhAftEntryId ) ] = aft
   return nhAftTable

class ShowGribiNhgAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi nexthop-group-aft [ id ID ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'nexthop-group-aft': matcherNexthopGroupAft,
      'id': matcherNhgId,
      'ID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFFFFFFFFFF,
                                       helpdesc='Nexthop AFT ID' ),
      'detail': matcherDetail,
   }
   cliModel = GribiNhgAftTable

   @staticmethod
   def handler( mode, args ):
      detail = ( 'detail' in args )
      return populateNhgAft( args.get( 'ID' ), detail )

class ShowGribiMplsAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi mpls-aft [ label LABEL ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'mpls-aft': matcherMplsAft,
      'label': matcherLabel,
      'LABEL': CliMatcher.IntegerMatcher( MplsLabel.min,
                                          MplsLabel.max,
                                          helpdesc='MPLS label' ),
      'detail': matcherDetail,
   }
   cliModel = GribiMplsAftTable

   @staticmethod
   def handler( mode, args ):
      detail = ( 'detail' in args )
      return populateMplsAft( args.get( 'LABEL' ), detail )

class ShowGribiNhAftCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering gribi nexthop-aft [ id ID ] [ detail ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'gribi': matcherGribi,
      'nexthop-aft': matcherNexthopAft,
      'id': matcherNhId,
      'ID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFFFFFFFFFF,
                                       helpdesc='Nexthop AFT ID' ),
      'detail': matcherDetail,
   }
   cliModel = GribiNhAftTable

   @staticmethod
   def handler( mode, args ):
      detail = ( 'detail' in args )
      return populateNhAft( args.get( 'ID' ), detail )

BasicCli.addShowCommandClass( ShowGribiNhgAftCmd )
BasicCli.addShowCommandClass( ShowGribiMplsAftCmd )
BasicCli.addShowCommandClass( ShowGribiNhAftCmd )

def Plugin( entityManager ):
   global gribiAftEntries
   global gribiStatus
   global gribiNhToTunnelMap
   global slTunnelTable
   global srteForwardingStatus

   gribiAftEntries = LazyMount.mount( entityManager,
                                      "routing/gribi/afts",
                                      "Gribi::GribiAfts", "r" )
   gribiStatus = LazyMount.mount( entityManager,
                                  "routing/gribi/status",
                                  "Gribi::GribiStatus", "r" )
   gribiNhToTunnelMap = LazyMount.mount( entityManager,
                                         "routing/gribi/nhtotunnmap",
                                         "Gribi::NhToTunnelMap", "r" )

   slTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTeSegmentListTunnelTable, entityManager )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   # pkgdeps: library FibLib
   srteForwardingStatus = smashEm.doMount( "forwarding/srte/status",
                                           "Smash::Fib::ForwardingStatus",
                                           readerInfo )
