# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import IpGenericAddress
from CliModel import ( Bool,
                       Dict,
                       Int,
                       List,
                       Model,
                       Str,
                       Submodel )
from IntfModels import Interface
from CliPlugin.SrTePolicyLibModel import SrTeSegmentListVia
from operator import attrgetter
from Toggles import RoutingLibToggleLib

class GribiEndpointStatus( Model ):
   enabled = Bool( help="The endpoint is enabled" )
   port = Int( help="The port this protocol's server is listening on" )
   sslProfile = Str( help="SSL profile name", optional=True )
   error = Str( help="Error which occurred while starting the endpoint",
                optional=True )
   vrfName = Str( help="The VRF this protocol's server is listening in",
                  default="default" )

   def render( self ):
      print( "Enabled:\t\t%s" % ( "Yes" if self.enabled else "No" ) )
      print( "Server:\t\t\t%s" % ( ( "running on port %d, in %s VRF" %
                                    ( self.port, self.vrfName )
                                    if self.enabled else "Not yet running" ) ) )
      if self.error:
         print( "Error:\t\t%s" % self.error )
      print( "SSL Profile:\t\t%s" %
               ( self.sslProfile if self.sslProfile else "none" ) )

class GribiNhgNhAftType( Model ):
   nhId = Int( help='Nexthop ID' )
   weight = Int( help='Weight of the nexthop' )
   if RoutingLibToggleLib.toggleGribiTeVrfPiEnabled():
      ipAddress = IpGenericAddress( optional=True, help="Next hop address" )
      interface = Interface( optional=True, help="Next hop interface" )
   mplsVias = List( valueType=SrTeSegmentListVia,
                    optional=True,
                    help='Segment Routing Traffic Engineering'
                         ' policy information' )

   def render( self, detail ): # pylint: disable=arguments-differ
      print( "\tnexthop ID: %u, weight: %u" % ( self.nhId, self.weight ) )
      if detail:
         ipAddrIntf = ""
         if RoutingLibToggleLib.toggleGribiTeVrfPiEnabled():
            if self.ipAddress:
               ipAddrIntf += "\t\tIP address: {}".format( self.ipAddress )
            if self.interface:
               if self.ipAddress:
                  ipAddrIntf += ", "
               else:
                  ipAddrIntf += "\t\t"
               # pylint: disable=no-member
               ipAddrIntf += "Interface: {}".format( self.interface.stringValue )
               # pylint: enable=no-member
            if ipAddrIntf:
               print( ipAddrIntf )
         if self.mplsVias:
            for mplsVia in self.mplsVias:
               print( "\t\tresolved nexthop: %s, interface: %s" % (
                  mplsVia.nexthop, mplsVia.interface.stringValue ) )
               labelStackStr = "\t\t\tresolved label stack: ["
               labelStackStr += ' '.join( '%u' % label
                                          for label in mplsVia.mplsLabels )
               labelStackStr += "]"
               print( labelStackStr )
         elif not ipAddrIntf:
            print( "\t\tunresolved" )

class GribiNhgAftType( Model ):
   nhgId = Int( help='Nexthop Group ID' )
   nhgNhs = List( valueType=GribiNhgNhAftType,
                  help='List of nexthops in a Nexthop Group AFT entry' )
   fecId = Int( help='FEC ID', optional=True )

class GribiNhgAftTable( Model ):
   nhgAft = Dict( keyType=long, valueType=GribiNhgAftType,
                  help='Map of Nexthop Group AFT entries, keyed by'
                       ' Nexthop Group ID' )
   details_ = Bool( default=False, help='gRIBI Nexthop Group AFT Detail' )

   def render( self ):
      for key in sorted( self.nhgAft.keys() ):
         aft = self.nhgAft[ key ]
         print( "Nexthop Group ID: %u" % aft.nhgId )
         if self.details_ and aft.fecId:
            print( "\tgRIBI FEC ID: %u" % aft.fecId )
         for nh in sorted( aft.nhgNhs, key=attrgetter( 'nhId' ) ):
            nh.render( self.details_ )

class GribiFecVia( Model ):
   tunnelIdx = Int( help="Tunnel index " )
   weight = Int( "Weight of the nexthop", optional=True )
   mplsVias = List( valueType=SrTeSegmentListVia,
                    optional=True,
                    help="Segment Routing Traffic Engineering"
                         " policy information" )

class GribiFecDetail( Model ):
   fecId = Int( help='FEC ID' )
   fecVias = List( valueType=GribiFecVia,
                   optional=True,
                   help="List of FEC vias" )

class GribiMplsAftDetail( Model ):
   fecDetail = Submodel( valueType=GribiFecDetail, optional=True,
                         help='gRIBI FEC detail' )

   def render( self ):
      if self.fecDetail:
         print( "\tgRIBI FEC ID: %u" % ( self.fecDetail.fecId ) )
         for fecVia in self.fecDetail.fecVias:
            print( "\t\ttunnel ID: gRIBI tunnel index %u, weight: %u" % (
                  fecVia.tunnelIdx, fecVia.weight ) )
            for mplsVia in fecVia.mplsVias:
               viaStr = "\t\t\tresolved nexthop: %s, interface: %s\n" % (
                     mplsVia.nexthop, mplsVia.interface.stringValue )
               viaStr += "\t\t\t\tresolved label stack: ["
               viaStr += ' '.join( '%u' % label for label in mplsVia.mplsLabels )
               viaStr += "]"
               print( viaStr )
      else:
         print( "\tunresolved" )

class GribiNhAftDetail( Model ):
   tunnelDetail = Submodel( valueType=GribiFecVia, optional=True,
                            help='Tunnel detail' )

   # pylint: disable=W0221
   def render( self, labelStackString ):
      if self.tunnelDetail and labelStackString:
         print( "%s, tunnel ID: gRIBI tunnel index %u" % ( labelStackString,
                                                   self.tunnelDetail.tunnelIdx ) )
         for mplsVia in self.tunnelDetail.mplsVias:
            viaStr = "\t\tresolved nexthop: %s, interface: %s\n" % (
                  mplsVia.nexthop, mplsVia.interface.stringValue )
            viaStr += "\t\t\tresolved label stack: ["
            viaStr += ' '.join( '%u' % label for label in mplsVia.mplsLabels )
            viaStr += "]"
            print( viaStr )
      else:
         if labelStackString:
            print( labelStackString )
            print( "\t\tunresolved" )

class GribiMplsAftType( Model ):
   label = Int( help='Label' )
   nhgId = Int( help='Nexthop Group ID' )
   mplsAftDetail = Submodel( valueType=GribiMplsAftDetail, optional=True,
                              help='gRIBI MPLS AFT detail' )

class GribiMplsAftTable( Model ):
   mplsAft = Dict( keyType=long, valueType=GribiMplsAftType,
                   help='Map of MPLS AFT entries, keyed by Label' )

   def render( self ):
      for key in sorted( self.mplsAft.keys() ):
         aft = self.mplsAft[ key ]
         print( "Label: %u Nexthop Group ID: %u" % ( aft.label, aft.nhgId ) )
         if aft.mplsAftDetail:
            aft.mplsAftDetail.render()

class GribiNhAftType( Model ):
   nhId = Int( help='Nexthop ID' )
   pushMplsLabels = List( valueType=int,
                          help='Push MPLS Label Stack Data' )
   if RoutingLibToggleLib.toggleGribiTeVrfPiEnabled():
      ipAddress = IpGenericAddress( optional=True, help="Next hop address" )
      interface = Interface( optional=True, help="Next hop interface" )
   nhAftDetail = Submodel( valueType=GribiNhAftDetail, optional=True,
                           help='gRIBI Nexthop AFT detail' )

class GribiNhAftTable( Model ):
   nhAft = Dict( keyType=long, valueType=GribiNhAftType,
                 help='Map of Nexthop AFT entries, keyed by Nexthop ID' )

   def render( self ):
      for _, aftEntry in sorted( self.nhAft.items() ):
         print( "Nexthop ID: %u" % aftEntry.nhId )
         ipAddrIntf = ""
         if RoutingLibToggleLib.toggleGribiTeVrfPiEnabled():
            if aftEntry.ipAddress:
               ipAddrIntf += "\tIP address: {}".format( aftEntry.ipAddress )
            if aftEntry.interface:
               if aftEntry.ipAddress:
                  ipAddrIntf += ", "
               else:
                  ipAddrIntf += "\t"
               ipAddrIntf += "Interface: {}".format( aftEntry.interface.stringValue )
         if ipAddrIntf:
            print( ipAddrIntf )
         labelStackString = ""
         if aftEntry.pushMplsLabels:
            labelStackString += "\tlabel stack: [%s]" % \
                                ' '.join( '%u' % l for l in aftEntry.pushMplsLabels )
         if aftEntry.nhAftDetail:
            aftEntry.nhAftDetail.render( labelStackString )
         else:
            if labelStackString:
               print( labelStackString )
