# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
from CliMode.GribiMode import MgmtGribiMode, GribiTransportMode
import CliParser
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
import CliCommand
import CliMatcher
import ConfigMount
import LazyMount
import ShowCommand
from GribiModel import (
   GribiEndpointStatus,
)

gribiConfig = None
sslConfig = None
gribiStatus = None

matcherApiForShow = CliMatcher.KeywordMatcher( 'api',
                                               helpdesc='Show management APIs' )
matcherManagement = ConfigMgmtMode.managementShowKwMatcher

# ------------------------------------------------------
# gRIBI config commands
# ------------------------------------------------------
class MgmtGribiConfigMode( MgmtGribiMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'management api gribi'"""

   name = "gRIBI configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.config_ = gribiConfig
      MgmtGribiMode.__init__( self, "api-gribi" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMgmtGribiConfigMode( mode, args ):
   childMode = mode.childMode( MgmtGribiConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMgmtGribiConfigMode( mode, args ):
   """Resets gRIBI agent configuration to default."""
   gribiConfig.enabled = False
   gribiConfig.endpoint = None

# ------------------------------------------------------
# [ no | default ] management api gribi
# ------------------------------------------------------
class GribiMgmtApiConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'management api gribi'
   noOrDefaultSyntax = syntax
   data = {
         'management': 'Configure management services',
         'api': 'Configure management APIs for the switch',
         'gribi': 'Configure gRIBI',
   }

   handler = gotoMgmtGribiConfigMode
   noOrDefaultHandler = noMgmtGribiConfigMode

BasicCliModes.GlobalConfigMode.addCommandClass( GribiMgmtApiConfigCmd )

class GribiTransportConfigMode( GribiTransportMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'transport grpc <name>'."""

   name = 'Transport for gRIBI'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name ):
      self.config_ = gribiConfig
      self.name = name

      GribiTransportMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoGribiTransportConfigMode( mode, args ):
   transportName = args.get( 'TRANSPORT_NAME' )
   endpoint = gribiConfig.endpoint
   if endpoint is None:
      gribiConfig.endpoint = ( transportName, )
      endpoint = gribiConfig.endpoint
   else:
      if endpoint.name != transportName:
         mode.addError( "transport '%s' of type '%s' already enabled; "
                        "can not enable another" % ( endpoint.name,
                                                     endpoint.transport ) )
         return
   endpoint.transport = 'grpc'
   endpoint.port = endpoint.portDefault
   endpoint.enabled = True
   gribiConfig.enabled = True

   childMode = mode.childMode( GribiTransportConfigMode, name=transportName )
   mode.session_.gotoChildMode( childMode )

def noGribiTransportConfigMode( mode, args ):
   transportName = args.get( 'TRANSPORT_NAME' )
   endpoint = gribiConfig.endpoint
   if endpoint is not None and endpoint.name == transportName:
      gribiConfig.endpoint = None
      gribiConfig.enabled = False

# ------------------------------------------------------
# [ no | default ] transport grpc <name>
# ------------------------------------------------------
class GribiTransportConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'transport grpc TRANSPORT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'transport': 'Configure a transport',
         'grpc': 'Configure grpc transport for gRIBI',
         'TRANSPORT_NAME': CliMatcher.StringMatcher( helpdesc='Transport name',
                                                      helpname='WORD' ),
   }

   handler = gotoGribiTransportConfigMode
   noOrDefaultHandler = noGribiTransportConfigMode

MgmtGribiConfigMode.addCommandClass( GribiTransportConfigCmd )

def shutdown( mode, args ):
   endpoint = mode.config_.endpoint
   endpoint.enabled = False
   mode.config_.enabled = False

def noShutdown( mode, args ):
   endpoint = mode.config_.endpoint
   endpoint.enabled = True
   mode.config_.enabled = True

# ------------------------------------------------------
# [ no | default ] shutdown
# ------------------------------------------------------
class GribiShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
         'shutdown': 'Disable protocol',
   }

   handler = shutdown
   noOrDefaultHandler = noShutdown

def populateGribiEndpointStatus():
   endpointStatus = GribiEndpointStatus()
   endpointStatus.enabled = gribiStatus.enabled
   endpointStatus.port = gribiStatus.port
   endpointStatus.error = gribiStatus.error
   endpointStatus.sslProfile = gribiStatus.sslProfile
   return endpointStatus

#--------------------------------------------------------------------------------
# show management api gribi
#--------------------------------------------------------------------------------
class ManagementApiGribiCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api gribi'
   data = {
            'management': matcherManagement,
            'api': matcherApiForShow,
            'gribi': 'Show status of gRIBI agent',
         }
   cliModel = GribiEndpointStatus

   @staticmethod
   def handler( mode, args ):
      return populateGribiEndpointStatus()

BasicCli.addShowCommandClass( ManagementApiGribiCmd )

GribiTransportConfigMode.addCommandClass( GribiShutdownCmd )

def setPort( mode, args ):
   endpoint = mode.config_.endpoint
   port = args.get( 'PORT', endpoint.portDefault )
   endpoint.port = port

# ------------------------------------------------------
# [ no | default ] port <port number>
# ------------------------------------------------------
class GribiPortCmd( CliCommand.CliCommandClass ):
   syntax = 'port PORT'
   noOrDefaultSyntax = 'port ...'
   data = {
      'port': 'The port number to listen on',
      'PORT': CliMatcher.IntegerMatcher( 1, 65535,
                                         helpdesc='Port number to use' ),
   }
   handler = setPort
   noOrDefaultHandler = handler

GribiTransportConfigMode.addCommandClass( GribiPortCmd )

def setSslProfile( mode, args ):
   endpoint = mode.config_.endpoint
   profileName = args.get( 'PROFILENAME', '' )
   endpoint.sslProfile = profileName

# ------------------------------------------------------
# [ no | default ] ssl profile <profile name>
# ------------------------------------------------------
class GribiSslProfileNameCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': 'Configure SSL related options',
      'profile': 'Configure SSL profile',
      'PROFILENAME': CliMatcher.DynamicNameMatcher(
                        lambda mode: sslConfig.profileConfig,
                        'Profile name' ),
   }
   handler = setSslProfile
   noOrDefaultHandler = handler

GribiTransportConfigMode.addCommandClass( GribiSslProfileNameCmd )

def Plugin( entityManager ):
   global gribiConfig, gribiStatus, sslConfig

   gribiConfig = ConfigMount.mount( entityManager, "mgmt/gribi/config",
                                    "Gribi::Config", "w" )
   gribiStatus = LazyMount.mount( entityManager, "mgmt/gribi/status",
                                    "Gribi::Status", "r" )
   sslConfig = LazyMount.mount( entityManager, "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config", "r" )
