# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclCliLib
import BasicCli
from CliMode.Gnmi import MgmtGnmiMode, GnmiTransportMode
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli
import ConfigMount
import DscpCliLib
from IpLibConsts import DEFAULT_VRF
import LazyMount
import OpenConfigCliLib

sslConfig = None
gnmiConfig = None
gnmiStatus = None
gnmiCheckpoint = None
aclConfig = None
aclCpConfig = None
octaConfig = None

# ------------------------------------------------------
# GNMI config commands
# ------------------------------------------------------

class MgmtGnmiConfigMode( MgmtGnmiMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'management api gnmi'."""

   name = "GNMI configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.config_ = gnmiConfig
      MgmtGnmiMode.__init__( self, "api-gnmi" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMgmtGnmiConfigMode( mode, args ):
   childMode = mode.childMode( MgmtGnmiConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMgmtGnmiConfigMode( mode, args ):
   """Resets OpenConfig agent configuration to default."""
   gnmiConfig.enabled = False
   for name in gnmiConfig.endpoints:
      noGnmiTransportConfigMode( mode, { 'TRANSPORT_NAME': name } )

def setEOSNative( mode, args ):
   octaConfig.enabled = True

def noEOSNative( mode, args ):
   octaConfig.enabled = False

# ------------------------------------------------------
# switch(config)# management api openconfig
#
# made obsolete by "management api gnmi" but still
# available as hidden command for ease of transitions
# ------------------------------------------------------
def gotoMgmtOpenConfigConfigMode( mode, args ):
   mode.addWarning(
      'This command is deprecated and will be saved as \'management api gnmi\'.' )
   gotoMgmtGnmiConfigMode( mode, args )

class GnmiTransportConfigMode( GnmiTransportMode,
                                     BasicCli.ConfigModeBase ):
   """CLI configuration submode 'transport grpc <name>'."""

   name = 'Transport for GNMI'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name ):
      self.config_ = gnmiConfig
      self.name = name

      GnmiTransportMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoGnmiTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT_NAME' ]
   if name not in gnmiConfig.endpoints:
      if OpenConfigCliLib.otherEnabledTransportExists( mode, name ):
         return

      endpoint = gnmiConfig.newEndpoints( name )
      endpoint.transport = 'grpc'
      endpoint.vrfName = DEFAULT_VRF
      # since 'initially' attributes don't get attrlogged, be explicit for now
      endpoint.port = endpoint.portDefault
      endpoint.enabled = True

      OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )

   childMode = mode.childMode( GnmiTransportConfigMode, name=name )
   mode.session_.gotoChildMode( childMode )

def noGnmiTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT_NAME' ]
   endpoint = gnmiConfig.endpoints.get( name )
   if endpoint is not None:
      if endpoint.enabled:
         AclCliLib.noServiceAclTypeVrfMap( mode,
                                           gnmiConfig.serviceAclTypeVrfMap,
                                           None, "ip", endpoint.vrfName )
         endpoint.enabled = False
      endpoint.port = endpoint.portDefault
      try:
         del gnmiConfig.endpoints[ name ]
      except KeyError:
         pass

   OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )

def shutdown( mode, args ):
   endpoint = mode.config_.endpoints[ mode.name ]
   if endpoint.enabled:
      AclCliLib.noServiceAclTypeVrfMap( mode,
                                        mode.config_.serviceAclTypeVrfMap,
                                        None, "ip", endpoint.vrfName )
      endpoint.enabled = False
      OpenConfigCliLib.updateLevelEnabledFlag( mode.config_ )

def noShutdown( mode, args ):
   # This stanza can only be enabled if there is no other
   # stanza with same 'transport' type already enabled
   endpoint = mode.config_.endpoints.get( mode.name )
   for e in mode.config_.endpoints.itervalues():
      if e.name != endpoint.name and e.enabled and e.transport == endpoint.transport:
         mode.addError( "transport '%s' of type '%s' already "
               "enabled; can not enable another" % ( e.name, e.transport ) )
         return
   endpoint.enabled = True
   mode.config_.enabled = True
   AclCliLib.setServiceAclTypeVrfMap( mode, mode.config_.serviceAclTypeVrfMap,
                  endpoint.serviceAcl, "ip", endpoint.vrfName )

def setSslProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   mode.config_.endpoints[ mode.name ].sslProfile = profileName

def noSslProfile( mode, args ):
   mode.config_.endpoints[ mode.name ].sslProfile = ''

profileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: sslConfig.profileConfig,
      'Profile name')

def setVrfName( mode, args ):
   vrfName = args.get( 'VRFNAME', DEFAULT_VRF )
   mode.config_.endpoints[ mode.name ].vrfName = vrfName

def setPort( mode, args ):
   port = args.get( 'PORT', mode.config_.endpoints[ mode.name ].portDefault )
   mode.config_.endpoints[ mode.name ].port = port

# ---------------------------------------------------------------------
# switch(config-mgmt-api-gnmi-transport-<name>)# authorization requests
# ---------------------------------------------------------------------
def setAuthorization( mode, args ):
   mode.config_.endpoints[ mode.name ].authorization = True

def noAuthorization( mode, args ):
   mode.config_.endpoints[ mode.name ].authorization = False

# ---------------------------------------------------------------------
# switch(config-mgmt-api-gnmi-transport-<name>)# qos dscp <dscpValue>
# ---------------------------------------------------------------------
def setDscp( mode, args ):
   mode.config_.endpoints[ mode.name ].qosDscp = args[ 'DSCP' ]

def noDscp( mode, args ):
   mode.config_.endpoints[ mode.name ].qosDscp = 0

DscpCliLib.addQosDscpCommandClass( GnmiTransportConfigMode, setDscp, noDscp )

def setOpenConfigAcl( mode, args ):
   aclName = args[ 'ACLNAME' ]
   AclCliLib.checkServiceAcl( mode, aclConfig, aclName )
   endpoint = mode.config_.endpoints.get( mode.name )
   endpoint.serviceAcl = aclName
   if endpoint.enabled:
      AclCliLib.setServiceAclTypeVrfMap( mode,
                  mode.config_.serviceAclTypeVrfMap,
                  endpoint.serviceAcl, "ip", endpoint.vrfName )

def noOpenConfigAcl( mode, args ):
   endpoint = mode.config_.endpoints.get( mode.name )
   if endpoint.enabled:
      AclCliLib.noServiceAclTypeVrfMap( mode,
                        mode.config_.serviceAclTypeVrfMap,
                        None, "ip", endpoint.vrfName )
   endpoint.serviceAcl = ""

#-------------------------------------------------------------------------------
# The "show management api gnmi access-list" command
#-------------------------------------------------------------------------------
def showOpenConfigAcl( mode, args ):
   params = [ args.get( 'ACL' ), 'summary' in args ]
   return AclCli.showServiceAcl( mode,
                                 gnmiConfig.serviceAclTypeVrfMap,
                                 gnmiStatus.aclStatusService,
                                 gnmiCheckpoint,
                                 'ip', params, supressVrf=True )

def clearOpenConfigAclCounters( mode, args ):
   AclCli.clearServiceAclCounters( mode,
                                   gnmiStatus.aclStatusService,
                                   gnmiCheckpoint, 'ip' )

def Plugin( entityManager ):
   global gnmiConfig, sslConfig, aclConfig, aclCpConfig
   global gnmiStatus, gnmiCheckpoint, octaConfig

   gnmiConfig = ConfigMount.mount( entityManager, "mgmt/gnmi/config",
                                   "Gnmi::Config", "w" )
   gnmiStatus = LazyMount.mount( entityManager, "mgmt/gnmi/status",
                                       "Gnmi::Status", "r" )
   gnmiCheckpoint = LazyMount.mount( entityManager,
                                           "mgmt/gnmi/checkpoint",
                                           "Acl::CheckpointStatus", "w" )
   sslConfig = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config",
                                "r" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   octaConfig = ConfigMount.mount( entityManager, "mgmt/octa/config",
                                   "Octa::Config", "w" )
