# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import AclCliLib
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
from CliPlugin.TcpMssCli import CfgTcpMssCeilingCmdBase
import CliPlugin.VrfCli as VrfCli
import CliToken.Ip
import CliToken.Ipv6
import CliToken.System
import ConfigMount
import Plugins

cliCpConfig = None
paramConfig = None

tcpMssConfig4 = None
tcpMssConfig6 = None

#--------------------------------------------------------------------------------
# system control-plane
#--------------------------------------------------------------------------------
class SystemControlPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'system control-plane'
   data = {
      'system': CliToken.System.systemMatcherForConfig,
      'control-plane': 'Configure control-plane features',
   }

   @staticmethod
   def handler( mode, args ):
      mode.session_.gotoChildMode( mode.childMode( AclCli.CpConfigMode ) )

BasicCliModes.GlobalConfigMode.addCommandClass( SystemControlPlaneCmd )

#--------------------------------------------------------------------------------
# control-plane (deprecated)
#--------------------------------------------------------------------------------
class ControlPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'control-plane'
   data = {
      'control-plane': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'control-plane',
            helpdesc='Configure control-plane features' ),
         deprecatedByCmd='system control-plane' )
   }

   handler = SystemControlPlaneCmd.handler

BasicCliModes.GlobalConfigMode.addCommandClass( ControlPlaneCmd )

def configureControlPlaneAcl( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   assert aclType in args
   aclName = args[ 'ACL_NAME' ]
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   if aclName == paramConfig.cpAclNameDefault:
      del cliCpConfig.cpConfig[ aclType ].globalCpAcl[ vrfName ]
   else:
      cliCpConfig.cpConfig[ aclType ].globalCpAcl[ vrfName ] = aclName
   AclCliLib.tryWaitForWarmup( mode )

def configureNoControlPlaneAclWithName( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   assert aclType in args
   aclName = args.get( 'ACL_NAME' )
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   # Get the effective cpAcl name
   aclConfig = cliCpConfig.cpConfig[ aclType ].globalCpAcl
   currentAclName = aclConfig.get( vrfName, paramConfig.cpAclNameDefault )

   if aclName is None or aclName == currentAclName:
      del aclConfig[ vrfName ]
      AclCliLib.tryWaitForWarmup( mode )
   else:
      AclCli.printNotConfiguredError( mode, aclType, aclName,
                                      'control-plane(%s VRF)' % vrfName )

class CpIpAclCmd( CliCommand.CliCommandClass ):
   syntax =  'ip access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ip': CliToken.Ip.ipMatcherForConfigCp,
            'access-group': AclCli.accessGroupKwMatcher,
            'ACL_NAME': AclCli.ipAclNameMatcher,
            'VRF': AclCli.vrfKwAndNameExprFactory,
            'in': AclCli.inKwMatcher
          }
   handler = configureControlPlaneAcl
   noOrDefaultHandler = configureNoControlPlaneAclWithName
 
AclCli.CpConfigMode.addCommandClass( CpIpAclCmd )

class CpIp6AclCmd( CliCommand.CliCommandClass ):
   syntax =  'ipv6 access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForConfigCp,
            'access-group': AclCli.accessGroupKwMatcher,
            'ACL_NAME': AclCli.ip6AclNameMatcher,
            'VRF': AclCli.vrfKwAndNameExprFactory,
            'in': AclCli.inKwMatcher
          }
   handler = configureControlPlaneAcl
   noOrDefaultHandler = configureNoControlPlaneAclWithName
 
AclCli.CpConfigMode.addCommandClass( CpIp6AclCmd )

class CfgTcpMssCeilingCmd( CfgTcpMssCeilingCmdBase ):
   # The control-plane version of this command does not support directionality
   # (ingress / egress).
   syntax = '''tcp mss ceiling { ( ipv4 MSS4 ) | ( ipv6 MSS6 ) }'''
   data = {
      'tcp': 'TCP',
   }
   data.update( CfgTcpMssCeilingCmdBase._baseData )

   @staticmethod
   def _getConfig( mode ):
      cpConfig = cliCpConfig.cpConfig
      ipv4Config = cpConfig[ 'ip' ].tcpMssConfig
      ipv6Config = cpConfig[ 'ipv6' ].tcpMssConfig
      assert ipv4Config and ipv6Config
      return ipv4Config, ipv6Config

   @staticmethod
   def handler( mode, args ):
      ipv4Config, ipv6Config = CfgTcpMssCeilingCmd._getConfig( mode )
      CfgTcpMssCeilingCmdBase._commonHandler( mode, args, ipv4Config, ipv6Config )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4Config, ipv6Config = CfgTcpMssCeilingCmd._getConfig( mode )
      CfgTcpMssCeilingCmdBase._commonNoOrDefaultHandler( mode, ipv4Config,
                                                         ipv6Config )

AclCli.CpConfigMode.addCommandClass( CfgTcpMssCeilingCmd )

@Plugins.plugin()
def Plugin( entityManager ):
   global cliCpConfig
   global paramConfig
   cliCpConfig = ConfigMount.mount( entityManager, 'acl/cpconfig/cli',
                                    'Acl::Input::CpConfig', 'w' )
   paramConfig = ConfigMount.mount( entityManager, 'acl/paramconfig',
                             'Acl::ParamConfig', 'w' )
