#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from ArnetModel import Ip4Address
from ArnetModel import IpGenericAddress
import Tac
from CliModel import Bool
from CliModel import Enum
from CliModel import GeneratorList
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from BgpCliModels import BgpPeerListBase
from BgpCliModels import _degradeToIpGenAddress
from BgpCliModels import peerKeyFromData
from BgpCliModels import trimIntfName
from BgpLib import PeerConfigKey
from HumanReadable import formatTimeInterval
from DeviceNameLib import kernelIntfFilter
import GatedConstants

#-------------------------------------------------------------------------------
# "show ip[v6] bgp neighbors [<ip>] [bfd] [vrf <vrfName>]"
#-------------------------------------------------------------------------------

PEER_TCP_STATE_LIST  = ( 'NOTUSED', 'ESTABLISHED', 'SYN-SENT',
                         'SYN-RECEIVED', 'FIN-WAIT-1', 'FIN-WAIT-2',
                         'TIME-WAIT', 'CLOSED', 'CLOSE-WAIT',
                         'LAST-ACK', 'LISTEN', 'CLOSING' )
class BgpPeerTcpInfo( Model ):
   __revision__ = 2

   def getMioAttrId( self ):
      return GatedConstants.MioDgetTcpInfo.MIO_DGET_BGP_PEER_TCPINFO

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BUG279484 for details.
         renamedAttributes = {
               'inqLen': 'inputQueueLength',
               'inqMaxLen': 'inputMaxQueueLength',
               'outqLen': 'outputQueueLength',
               'outqMaxLen': 'outputMaxQueueLength',
               'slowStartThr': 'slowStartThreshold'
               }
         for before, after in renamedAttributes.iteritems():
            dictRepr[ before ] = dictRepr.pop( after )
      return dictRepr

   state = Enum( values=PEER_TCP_STATE_LIST, help='TCP Socket current state' )
   options = Int( help='TCP socket options' )
   sendWindowScale = Int( help='TCP socket send window scale' )
   rcvWindowScale = Int( help='TCP socket receive window scale' )
   retransTimeout = Int( help='TCP socket retransmission timeout (microseconds)' )
   delayedAckTimeout = Int( help='TCP socket delayed ack timeout (microseconds)' )
   maxSegmentSize = Int( help='TCP outgoing maximum segment size' )
   sndRtt = Int( help='TCP round-trip time (microseconds)' )
   sndRttVariance = Int( help='TCP round-trip time variance (microseconds)' )
   slowStartThreshold = Int( help='TCP send slow start size threshold' )
   congestionWindow = Int( help='TCP send congestion window' )
   rcvRtt = Int( help='TCP receive round-trip time (microseconds)' )
   rcvWindow = Int( help='TCP advertised receive window' )
   totalRetrans = Int( help='Total number of TCP retransmissions' )
   outputQueueLength = Int( help='TCP output queue length' )
   outputMaxQueueLength = Int( help='TCP output queue max length' )
   inputQueueLength = Int( help='TCP input queue length' )
   inputMaxQueueLength = Int( help='TCP input queue max length' )

   def processData( self, data ):
      if 'tcpiState' in data:
         self.state = data.pop( 'tcpiState' )
      if 'tcpiOptions' in data:
         self.options = data.pop( 'tcpiOptions' )
      if 'tcpiSndWscale' in data:
         self.sendWindowScale = data.pop( 'tcpiSndWscale' )
      if 'tcpiRcvWscale' in data:
         self.rcvWindowScale = data.pop( 'tcpiRcvWscale' )
      if 'tcpiRto' in data:
         self.retransTimeout = data.pop( 'tcpiRto' )
      if 'tcpiAto' in data:
         self.delayedAckTimeout = data.pop( 'tcpiAto' )
      if 'tcpiSndMss' in data:
         self.maxSegmentSize = data.pop( 'tcpiSndMss' )
      if 'tcpiRtt' in data:
         self.sndRtt = data.pop( 'tcpiRtt' )
      if 'tcpiRttvar' in data:
         self.sndRttVariance = data.pop( 'tcpiRttvar' )
      if 'tcpiSndSsthresh' in data:
         self.slowStartThreshold = data.pop( 'tcpiSndSsthresh' )
      if 'tcpiSndCwnd' in data:
         self.congestionWindow = data.pop( 'tcpiSndCwnd' )
      if 'tcpiRcvRtt' in data:
         self.rcvRtt = data.pop( 'tcpiRcvRtt' )
      if 'tcpiRcvSpace' in data:
         self.rcvWindow = data.pop( 'tcpiRcvSpace' )
      if 'tcpiTotalRetrans' in data:
         self.totalRetrans = data.pop( 'tcpiTotalRetrans' )
      if 'tcpiOutqLen' in data:
         self.outputQueueLength = data.pop( 'tcpiOutqLen' )
      if 'tcpiOutqMaxLen' in data:
         self.outputMaxQueueLength = data.pop( 'tcpiOutqMaxLen' )
      if 'tcpiInqLen' in data:
         self.inputQueueLength = data.pop( 'tcpiInqLen' )
      if 'tcpiInqMaxLen' in data:
         self.inputMaxQueueLength = data.pop( 'tcpiInqMaxLen' )

   def render( self ):
      if self.state is None:
         return

      inputQueueLength = self.inputQueueLength if self.inputQueueLength is not None \
               else "not available"
      inqMax = self.inputMaxQueueLength if self.inputMaxQueueLength is not None \
               else "not available"
      outputQueueLength = self.outputQueueLength if self.outputQueueLength \
                          is not None \
                else "not available"
      outqMax = self.outputMaxQueueLength if self.outputMaxQueueLength is not None \
                else "not available"

      print "TCP Socket Information:"
      print "  TCP state is %s" % ( self.state )
      print "  Recv-Q: %s/%s" % ( inputQueueLength, inqMax )
      print "  Send-Q: %s/%s" % ( outputQueueLength, outqMax )
      print "  Outgoing Maximum Segment Size (MSS): %s" % ( self.maxSegmentSize )
      print "  Total Number of TCP retransmissions: %s" % ( self.totalRetrans )

      print "  Options:"
      print "    Timestamps enabled: %s" % \
            ( "yes" if self.options & \
             GatedConstants.DabitBgpPeerTcpInfo.\
             DABIT_BGP_PEER_TCPINFO_OPT_TIMESTAMPS \
             else "no" )
      print "    Selective Acknowledgments enabled: %s" % \
            ( "yes" if self.options & \
              GatedConstants.DabitBgpPeerTcpInfo.DABIT_BGP_PEER_TCPINFO_OPT_SACK \
              else "no" )
      print "    Window Scale enabled: %s" % \
            ( "yes" if self.options & \
              GatedConstants.DabitBgpPeerTcpInfo.DABIT_BGP_PEER_TCPINFO_OPT_WSCALE \
              else "no" )
      print "    Explicit Congestion Notification (ECN) enabled: %s" % \
            ( "yes" if self.options & \
              GatedConstants.DabitBgpPeerTcpInfo.DABIT_BGP_PEER_TCPINFO_OPT_ECN \
              else "no" )
      print "  Socket Statistics:"
      print "    Window Scale (wscale): %s,%s" % \
            ( self.sendWindowScale, self.rcvWindowScale )
      print "    Retransmission Timeout (rto): %.1fms" % \
            ( float( self.retransTimeout )/1000 )
      print "    Round-trip Time (rtt/rtvar): %.1fms/%.1fms" % \
            ( float( self.sndRtt )/1000, float( self.sndRttVariance )/1000 )
      print "    Delayed Ack Timeout (ato): %.1fms" % \
            ( float( self.delayedAckTimeout )/1000 )
      print "    Congestion Window (cwnd): %s" % ( self.congestionWindow )

      if self.slowStartThreshold < 65535:
         print "    Slow-start Threshold (ssthresh): %s" % \
            ( self.slowStartThreshold )
      if self.sndRtt > 0 and self.maxSegmentSize and self.congestionWindow:
         print "    TCP Throughput: %.2f Mbps" % \
               ( float( self.congestionWindow ) * \
                 ( float( self.maxSegmentSize ) * 8. / float( self.sndRtt ) ) )

      if self.rcvRtt:
         print "    Recv Round-trip Time (rcv_rtt): %.1fms" % \
               ( float( self.rcvRtt )/1000 )
      if self.rcvWindow:
         print "    Advertised Recv Window (rcv_space): %s" % ( self.rcvWindow )

      print " "

BITS_PER_PEERFLAG = 32

def bfdStateName( state ):
   if state == GatedConstants.BfdStateFlag.BFD_ADMIN_DOWN:
      return "AdminDown"
   elif state == GatedConstants.BfdStateFlag.BFD_INIT:
      return "Init"
   elif state == GatedConstants.BfdStateFlag.BFD_UP:
      return "Up"
   elif state == GatedConstants.BfdStateFlag.BFD_DOWN:
      return "Down"
   else:
      return "Unknown"

class ErrorTimeInfo( Model ):
   time = Int( optional=True,
         help='Time at which last error occured' )
   firstTime = Int( optional=True,
         help='Time at which last error occured for the first time' )
   repeats = Int( optional=True,
         help='Number of times last error repeated' )

# used for input and output message statistics
class MessageStatInfo( Model ):
   updates = Int( optional=True, help='Updates' )
   keepalives = Int( optional=True, help='Keepalives' )
   opens = Int( optional=True, help='Opens' )
   rtRefreshes = Int( optional=True, help='Route Refreshes' )
   notifications = Int( optional=True, help='Notifications' )
   queueDepth = Int( optional=True, help='Queue Depth' )

# used to collapse routemap and prefixlist variables
# with simlar names into a single class
class RouteFilterInfo( Model ):
   filterType = Enum( help='Type of route filter',
                      values=( 'RouteMap', 'PrefixList' ) )
   inboundIpv4Uni = Str( optional=True, help='Inbound Ipv4 Unicast' )
   outboundIpv4Uni = Str( optional=True, help='Outbound Ipv4 Unicast' )
   inboundIpv6Uni = Str( optional=True, help='Inbound Ipv6 Unicast' )
   outboundIpv6Uni = Str( optional=True, help='Outbound Ipv6 Unicast' )
   inboundIpv4SrTe = Str( optional=True, help='Inbound Ipv4 SR-TE' )
   inboundIpv6SrTe = Str( optional=True, help='Inbound Ipv6 SR-TE' )

class PeerDropStats( Model ):
   inDropAsloop = Int( optional=True, help='In packets dropped due to Asloop' )
   inDropEnforceFirstAs = Int( optional=True,
         help='In packets dropped due to Enforce First As' )
   inDropMalformedMpbgp = Int( optional=True,
         help='In packets dropped due to Malformed MpBgp' )
   inDropOrigId = Int( optional=True,
         help='In packets dropped due to Orig Id' )
   inDropNhLocal = Int( optional=True,
         help='In packets dropped due to Nexthop Local' )
   inDropNhInvalid = Int( optional=True,
         help='In packets dropped due to Nexthop Invalid' )
   inDropNhAfV6 = Int( optional=True,
         help=' In packets dropped due to Nexthop AF V6' )
   outDropV4LocalAddr = Int( optional=True,
         help='Out packets dropped due to V4 Local Address' )
   outDropV6LocalAddr = Int( optional=True,
         help='Out packets dropped due to V6 Local Address' )
   prefixLuDroppedV4 = Int( optional=True,
         help='Packets dropped due to V4 Prefix LU ' )
   prefixLuDroppedV6 = Int( optional=True,
         help='Packets dropped due to V6 Prefix LU ' )
   prefixVpnIpv4DroppedImportMatchFailure = Int( optional=True,
         help='VPN-IPv4 NLRIs dropped due to route import match failure' )
   prefixVpnIpv6DroppedImportMatchFailure = Int( optional=True,
         help='VPN-IPv6 NLRIs dropped due to route import match failure' )
   prefixEvpnDroppedImportMatchFailure = Int( optional=True,
         help='L2VPN EVPN NLRIs dropped due to route import match failure' )

class BgpPeerInUpdateErrors( Model ):

   def getMioAttrId( self ):
      # We cannot depend on gated. So we must hardcode the value here to match
      # the one in gated/gated-ctk/src/mioagt/bgp_api.h. In fact having "ribd"
      # specific element in BGP Cli Model is violation of modularity. Make sure
      # that the value here matches MIO_DGET_BGP_PEER_IN_UPDATE_ATTR_ERRORS
      return 3

   inUpdErrWithdraw = Int( optional=True, help='in updates treated as withdraw' )
   inUpdErrIgnore = Int( optional=True, help='in updates ignored attributes' )
   inUpdErrDisableAfiSafi = Int( optional=True,
         help='in updates disabling AFI/SAFI' )
   disabledAfiSafi = Str( optional=True, help='disabled AFI/SAFI' )
   withdrawAttr = Str( optional=True, help='last attribute resulting in withdraw' )
   if withdrawAttr is None:
      withdrawAttr = ''
   ignoreAttr = Str( optional=True, help='last attribute resulting in ignore' )
   if ignoreAttr is None:
      ignoreAttr = ''
   disableAfiSafiAttr = Str( optional=True,
         help='last attribute resulting in AFI/SAFI disable' )
   if disableAfiSafiAttr is None:
      disableAfiSafiAttr = ''
   lastUpdErrTime = Int( optional=True, help='last time of Update inbound error' )

   def render( self ):
      print "  Inbound updates with attribute errors:"
      print "    Resulting in removal of all paths in update " \
         "(treat-as-withdraw): %d" % ( self.inUpdErrWithdraw )
      print "    Resulting in AFI/SAFI disable: %d" \
         % ( self.inUpdErrDisableAfiSafi )
      print "    Resulting in attribute ignore: %d" % ( self.inUpdErrIgnore )
      if self.disabledAfiSafi is not None:
         print "    Disabled AFI/SAFI: %s" % ( self.disabledAfiSafi )
      if self.withdrawAttr is not None:
         print "    Last treat-as-withdraw attribute error: %s" \
               % ( self.withdrawAttr )
      if self.ignoreAttr is not None:
         print "    Last ignored attribute error: %s" % ( self.ignoreAttr )
      if self.disableAfiSafiAttr is not None:
         print "    Last disable AFI/SAFI attribute error: %s" \
            % ( self.disableAfiSafiAttr )
      lastRcvdErrStr = "    Last update with error received at: "
      if self.lastUpdErrTime:
         lastRcvdErrStr += "%8s" % formatTimeInterval( self.lastUpdErrTime )
         print lastRcvdErrStr

class BgpPeer( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         _degradeToIpGenAddress( dictRepr[ 'peerAddress' ] )
      return dictRepr

   peerAddress = Str( help='Peer address' )
   peerTcpInfo = Submodel( valueType=BgpPeerTcpInfo, optional=True,
                    help='Peer TCP information' )
   peerInUpdateErrors = Submodel( valueType=BgpPeerInUpdateErrors, optional=True,
                    help='Peer inbound update errors' )
   asn = Str( help='Autonomous system number' )
   routerId = Ip4Address( help='BGP router identity' )
   updateSource = IpGenericAddress( optional=True, help='Local TCP address' )
   localAddressV4 = IpGenericAddress( optional=True, help='Local IPv4 address' )
   localAddressV6 = IpGenericAddress( optional=True, help='Local IPv6 address' )
   localAddressPort = Int( optional=True, help='Local address port' )
   autoLocalAddress = Str( optional=True, help='Auto local address' )
   localAsn = Str( help='Local autonomous system number' )
   localRouterId = Ip4Address( help='Local router ID' )
   linkType = Str( help='Link type' )

   version = Int( help='BGP version' )
   negotiatedVersion = Int( help='Negotiated BGP version' )
   updateGroupIndex = Int( help='Update group index' )
   ttl = Int( help='TTL value for peer connections' )
   maxTtlHops = Int( help='Maximum hops allowed for BGP neighbor' )
   holdTime = Int( optional=True, help='Hold time' )
   confHoldTime = Int( optional=True, help='Configured hold time' )
   connectFailed = Int( optional=True, help='Failed connection attempts' )
   connectInterval = Int( optional=True, help='Connection interval' )
   connectTimeLeft = Int( optional=True, help='Connection time left' )
   advertisedRestartTime = Int( optional=True, help='Advertised restart time' )
   restart = Int( optional=True, help='Received restart time' )
   restartTimeLeft = Int( optional=True, help='Restart time left' )
   endOfRibTimeLeft = Int( optional=True, help='End of rib time left' )
   configIdleHoldTime = Int( optional=True, help='Configured idle hold time' )
   idleHoldTimeLeft = Int( optional=True, help='Idle hold time left' )
   state = Str( optional=True, help='BGP state' )
   idleReason = Str( optional=True, help='Idle reason if session state is idle' )
   lastState = Str( optional=True, help='Last BGP state' )
   lastEvent = Str( optional=True, help='Last BGP event' )
   lastErrorCode = Str( optional=True, help='Last send error code' )
   lastErrorSubcode = Str( optional=True, help='Last send error subcode' )
   lastErrorTimeInfo = Submodel( optional=True, valueType=ErrorTimeInfo,
         help='Last send error time information' )
   lastErrorData = Str( optional=True, help='Last send error data' )
   lastErrorRcvdCode = Str( optional=True, help='Last receive error code' )
   lastErrorRcvdSubcode = Str( optional=True, help='Last receive error subcode' )
   lastErrorRcvdTimeInfo = Submodel( optional=True,
         valueType=ErrorTimeInfo,
         help='Last receive error time information' )
   lastErrorRcvdData = Str( optional=True, help='Last receive error data' )
   lastSocketOutErrorMsg = Str( optional=True,
         help='Last socket out error message' )
   lastSocketOutErrorTimeInfo = Submodel( optional=True,
         valueType=ErrorTimeInfo,
         help='Last socket out error time information' )
   lastSocketInErrorMsg = Str( optional=True,
         help='Last socket in error message' )
   lastSocketInErrorTimeInfo = Submodel( optional=True,
         valueType=ErrorTimeInfo,
         help='Last socket in error time information' )
   bgpPeerHiscaps = Int( optional=True, help='BGP capabilities received from peer' )
   bgpPeerCaps = Int( optional=True, help='BGP capabilities sent by us' )
   bgpPeerOptions = Int( optional=True, help='BGP peer options' )
   bgpPeerOptions2 = Int( optional=True, help='BGP peer options 2' )
   bgpPeerOptions3 = Int( optional=True, help='BGP peer options 3' )
   bgpPeerFlags = List( valueType=long, optional=True, help='BGP peer flag list' )
   miscFlags = Int( optional=True, help='BGP miscellaneous flags' )
   v4gateway = IpGenericAddress( optional=True, help='IPv4 gateway address' )
   v6gateway = IpGenericAddress( optional=True, help='IPv6 gateway address' )
   remoteAddressV4 = IpGenericAddress( optional=True, help='Remote IPv4 address' )
   remoteAddressV6 = IpGenericAddress( optional=True, help='Remote IPv6 address' )
   remoteAddrPort = Int( optional=True, help='Remote address port' )
   sentUpdates = Int( optional=True, help='Sent updates' )
   receivedUpdates = Int( optional=True, help='Received updates' )
   sentMessages = Int( optional=True, help='Sent messages' )
   receivedMessages = Int( optional=True, help='Received messages' )
   establishedTransitions = Int( optional=True, help='Established transitions' )
   establishedTime = Int( optional=True, help='Established time' )
   lastSent = Int( optional=True, help='Last sent' )
   lastRcvd = Int( optional=True, help='Last received' )
   keepaliveTime = Int( optional=True, help='Keepalive time' )
   configuredKeepaliveTime = Int( optional=True, help='Configured keepalive time' )
   inMessageStats = Submodel( optional=True,
         valueType=MessageStatInfo,
         help='Input message statistics' )
   outMessageStats = Submodel( optional=True,
         valueType=MessageStatInfo,
         help='Output message statistics' )
   localPort = Int( optional=True, help='Local port' )
   remotePort = Int( optional=True, help='Remote port' )
   confRemotePort = Int( optional=True, help='Configured remote port' )
   description = Str( optional=True, help='Description' )
   routeMapInbound = Str( optional=True, help='Route map inbound' )
   routeMapOutbound = Str( optional=True, help='Route map outbound' )
   routeMapInfo = Submodel( optional=True,
         valueType=RouteFilterInfo,
         help='Neighbor route map information' )
   prefixListInfo = Submodel( optional=True,
         valueType=RouteFilterInfo,
         help='Neighbor prefix list information' )
   bfdState = Int( optional=True, help='BFD state' )
   vrf = Str( help='Vrf' )
   peerGroup = Str( optional=True, help='Peer group name' )
   prefixesReceived = Int( optional=True, help='IPv4 Prefixes received' )
   v6PrefixesReceived = Int( optional=True, help='IPv6 prefixes received' )
   v6SixPePrefixesReceived = Int( optional=True,
         help='IPv6 provider edge (6PE) prefixes received' )
   v6SixPePrefixesSent = Int( optional=True,
         help='IPv6 provider edge (6PE) prefixes sent' )
   prefixesSent = Int( optional=True, help='Prefixes sent' )
   v6PrefixesSent = Int( optional=True, help='IPv6 prefixes sent' )
   v4SrTePrefixesReceived = Int( optional=True,
         help='IPv4 segment routing traffic engineering policies received' )
   v6SrTePrefixesReceived = Int( optional=True,
         help='IPv6 segment routing traffic engineering policies received' )
   v4SrTePrefixesSent = Int( optional=True,
         help='IPv4 segment routing traffic engineering policies sent' )
   v6SrTePrefixesSent = Int( optional=True,
         help='IPv6 segment routing traffic engineering policies sent' )
   linkStateSent = Int( optional=True,
         help="Link State NLRIs sent" )
   linkStateReceived = Int( optional=True,
         help="Link State NLRIs received" )
   evpnPrefixesReceived = Int( optional=True,
         help='Ethernet VPN prefixes received' )
   evpnPrefixesSent = Int( optional=True,
         help='Ethernet VPN prefixes sent' )
   v4FlowspecPrefixesReceived = Int( optional=True,
         help='IPv4 flow specification rules received' )
   v6FlowspecPrefixesReceived = Int( optional=True,
         help='IPv6 flow specification rules received' )
   v4FlowspecPrefixesSent = Int( optional=True,
         help='IPv4 flow specification rules sent' )
   v6FlowspecPrefixesSent = Int( optional=True,
         help='IPv6 flow specification rules sent' )
   v4BestPaths = Int( optional=True,
         help='IPv4 best paths received' )
   v6BestPaths = Int( optional=True,
         help='IPv6 best paths received' )
   v4SrTeBestPaths = Int( optional=True,
         help='IPv4 segment routing traffic engineering best paths received' )
   v6SrTeBestPaths = Int( optional=True,
         help='IPv6 segment routing traffic engineering best paths received' )
   v4BestEcmpPaths = Int( optional=True,
         help='IPv4 best ecmp paths received' )
   v6BestEcmpPaths = Int( optional=True,
         help='IPv6 best ecmp paths received' )
   v4SrTeBestEcmpPaths = Int( optional=True,
         help='IPv4 segment routing traffic engineering best ecmp paths received' )
   v6SrTeBestEcmpPaths = Int( optional=True,
         help='IPv6 segment routing traffic engineering best ecmp paths received' )
   linkStateBestPaths = Int( optional=True,
         help="Link State best paths" )
   linkStateBestEcmpPaths = Int( optional=True,
         help="Link State best ECMP paths" )
   rtMembershipPrefixesSent = Int( optional=True,
                                   help="RT membership prefixes sent" )
   rtMembershipPrefixesReceived = Int( optional=True,
                                       help="RT membership prefixes received" )
   rtMembershipBestPaths = Int( optional=True,
                                help="RT membership best paths" )
   rtMembershipBestEcmpPaths = Int( optional=True,
                                    help="RT membership best ECMP paths" )
   v4MplsVpnPrefixesSent = Int( optional=True,
                                help="IPv4 MPLS VPN prefixes sent" )
   v4MplsVpnPrefixesReceived = Int( optional=True,
                                    help="IPv4 MPLS VPN prefixes received" )
   v4MplsVpnBestPaths = Int( optional=True,
                             help="IPv4 MPLS VPN best paths" )
   v4MplsVpnBestEcmpPaths = Int( optional=True,
                                 help="IPv4 MPLS VPN best ECMP paths" )
   v6MplsVpnPrefixesSent = Int( optional=True,
                                help="IPv6 MPLS VPN prefixes sent" )
   v6MplsVpnPrefixesReceived = Int( optional=True,
                                    help="IPv6 MPLS VPN prefixes received" )
   v6MplsVpnBestPaths = Int( optional=True,
                             help="IPv6 MPLS VPN best paths" )
   v6MplsVpnBestEcmpPaths = Int( optional=True,
                                 help="IPv6 MPLS VPN best ECMP paths" )
   updownTime = Int( optional=True, help='Up down time' )
   ifName = Str( optional=True, help='Interface name' )
   maintenance = Bool( optional=True, help='Peering under maintenance' )
   maintenanceRm = Str( optional=True, help='Maintenance mode route map' )
   maintenanceRmIn = Str( optional=True, help='Maintenance mode inbound route map' )
   maintenanceRmOut = Str( optional=True,
                           help='Maintenance mode outbound route map' )
   noComms = Bool( optional=True, help='Send no communities ' )
   as4 = Int( optional=True, help='AS' )
   prependOwnDisabled = Bool( optional=True, help='Do not prepend own AS number' )
   establishFailHint = Str( optional=True,
         help='Establish failure hint' )
   dropStats = Submodel( optional=True,
         valueType=PeerDropStats,
         help='Prefix drop statistics' )
   bgpSoftReconfigInbound = Enum(
         help='Which inbound routes that are not accepted to \
               retain for soft reconfig',
         values=( 'Default', 'All', 'None' ) )
   rpkiOriginValidationMethod = Enum( optional=True,
         help='Method used for validating prefix origin AS',
         values=( 'disabled', 'local', 'community', 'preferCommunity' ) )
   rpkiOriginValidationSendExtComm = Bool( optional=True,
         help='Attach an extended community carrying the prefix origin validity' )
   rpkiOriginValidationRouteMap = Str( optional=True,
         help='Route map to filter which prefixes should be validated' )
   aigpSessionIpv4Uni = Bool( optional=True, help='AIGP session for IPv4 Unicast' )
   aigpSessionIpv6Uni = Bool( optional=True, help='AIGP session for IPv6 Unicast' )
   nexthopLuOriginateIpv4Uni = Bool( optional=True,
         help='Next hop labeled unicast origination for IPv4 Unicast' )
   nexthopLuOriginateIpv6Uni = Bool( optional=True,
         help='Next hop labeled unicast origination for IPv6 Unicast' )

   # Check if the peerFlag is set in self.bgpPeerFlags list
   def peerFlagTest( self, peerFlag ):
      flagNum = peerFlag / BITS_PER_PEERFLAG
      flagMask = ( 1 << ( peerFlag % BITS_PER_PEERFLAG ) )
      return self.bgpPeerFlags[ flagNum ] & flagMask

   def render( self ):
      print "BGP neighbor is %s, remote AS %s, %s link" % \
            ( self.peerAddress, self.asn, self.linkType )
      if self.description:
         print "  Description: %s" % self.description

      print "  BGP version %s, remote router ID %s, VRF %s" % \
            ( self.version, self.routerId, self.vrf )

      if self.peerGroup:
         print "  Inherits configuration from and member of peer-group %s" % \
               ( self.peerGroup )
      print "  Negotiated BGP version %s" % ( self.negotiatedVersion )

      if self.updateGroupIndex:
         print "  Member of update group %s" % ( self.updateGroupIndex )

      lastReadWriteStr = "  Last read "
      if self.lastRcvd:
         lastReadWriteStr += "%8s, " % formatTimeInterval( self.lastRcvd )
      else:

         lastReadWriteStr += "never, "

      lastReadWriteStr += "last write "
      if self.lastSent:
         lastReadWriteStr += "%8s" % formatTimeInterval( self.lastSent )
      else:
         lastReadWriteStr += "never"

      print lastReadWriteStr

      print "  Hold time is %s, keepalive interval is %s seconds" % \
            ( self.holdTime, self.keepaliveTime )

      print "  Configured hold time is %s, keepalive interval is %s seconds" % \
            ( self.confHoldTime, self.configuredKeepaliveTime )

      connectTimerStr = "  Connect timer is %s" % \
            ( 'active' if self.miscFlags & \
              GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_CONNECT_TIMER_STATE \
               else 'inactive' )
      if self.connectTimeLeft is not None:
         connectTimerStr += ", time left: %s" % \
               formatTimeInterval( self.connectTimeLeft )
      print connectTimerStr

      if self.connectInterval:
         print "  Connection interval is %s seconds" % \
                ( self.connectInterval )
      if self.connectFailed:
         print "  Failed connection attempts is %s" % \
                ( self.connectFailed )

      if self.configIdleHoldTime:
         print "  Configured idle-restart time is %s seconds" % \
               ( self.configIdleHoldTime )

      idleRestartStr = "  Idle-restart timer is %s" % \
            ( 'active' if self.miscFlags & \
              GatedConstants.DabitBnpEntryMF.\
              DABIT_BNP_ENTRY_MF_IDLEHOLD_TIMER_STATE \
              else 'inactive' )

      if self.idleHoldTimeLeft:
         idleRestartStr += ", time left: %s" % \
               formatTimeInterval( self.idleHoldTimeLeft )
      print idleRestartStr

      if self.maintenance:
         print "  Session is under maintenance"

      bgpStateStr = "  BGP state is %s" % self.state
      if self.establishedTime is not None and long( self.establishedTime ) >= 0:
         bgpStateStr += ", up for %8s" % \
               ( formatTimeInterval( self.establishedTime ) )
      elif self.idleReason:
         bgpStateStr += ", %s" % self.idleReason
      print bgpStateStr

      if self.establishedTime is None and self.establishFailHint:
         print "  Peering failure hint: %s" % ( self.establishFailHint )

      print "  Number of transitions to established: %s" % \
            ( self.establishedTransitions )
      print "  Last state was %s" % ( self.lastState )
      print "  Last event was %s" % ( self.lastEvent )

      if self.lastErrorCode is not None:
         lastErrorStr = "  Last sent notification:%s/%s, Last time %8s" % \
               ( self.lastErrorCode, self.lastErrorSubcode,
                  formatTimeInterval( self.lastErrorTimeInfo.time ) )
         if self.lastErrorTimeInfo.repeats is not None:
            lastErrorStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastErrorTimeInfo.firstTime ),
                  self.lastErrorTimeInfo.repeats )
         print lastErrorStr
      if self.lastErrorData is not None:
         print "    Sent data: %s" % ( self.lastErrorData )
      if self.lastErrorRcvdCode is not None:
         lastErrorRcvdStr = "  Last rcvd notification:%s/%s, Last time %8s" % \
               ( self.lastErrorRcvdCode, self.lastErrorRcvdSubcode,
                  formatTimeInterval( self.lastErrorRcvdTimeInfo.time ) )
         if self.lastErrorRcvdTimeInfo.repeats is not None:
            lastErrorRcvdStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastErrorRcvdTimeInfo.firstTime ),
                  self.lastErrorRcvdTimeInfo.repeats )
         print lastErrorRcvdStr
      if self.lastErrorRcvdData is not None:
         print "    Rcvd data: %s" % ( self.lastErrorRcvdData )
      if self.lastSocketOutErrorMsg is not None:
         lastSocketOutStr = "  Last sent socket-error:%s, Last time %8s" % \
               ( self.lastSocketOutErrorMsg,
                  formatTimeInterval( self.lastSocketOutErrorTimeInfo.time ) )
         if self.lastSocketOutErrorTimeInfo.repeats is not None:
            lastSocketOutStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastSocketOutErrorTimeInfo.firstTime ),
                  self.lastSocketOutErrorTimeInfo.repeats )
         print lastSocketOutStr
      if self.lastSocketInErrorMsg is not None:
         lastSocketInStr = "  Last rcvd socket-error:%s, Last time %8s" % \
               ( self.lastSocketInErrorMsg,
                  formatTimeInterval( self.lastSocketInErrorTimeInfo.time ) )
         if self.lastSocketInErrorTimeInfo.repeats is not None:
            lastSocketInStr += ", First time %8s, Repeats %s" % \
               ( formatTimeInterval( self.lastSocketInErrorTimeInfo.firstTime ),
                  self.lastSocketInErrorTimeInfo.repeats )
         print lastSocketInStr

      if self.miscFlags & \
         GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_RR_CLIENT:
         if self.miscFlags \
            & GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_NO_CLIENT_REFLECT:
            print "  Neighbor is a route reflector client (client-to-client" \
                  " reflection disabled)"
         else:
            print "  Neighbor is a route reflector client"
      elif self.miscFlags & \
           GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_RR_MESHED:
         print "  Neighbor is a route reflector client (meshed)"

      print "  Neighbor Capabilities:"

      peerOptionsNoCaps = self.bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_NOCAPS
      hisCapsNoCaps =  self.bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_NO_CAPS


      def renderCapabilities():
         def printAddPathCapabilityHelper( labelString, capFlag,
                                           hisCapFlag, peerFlag ):
            sentCaps = self.bgpPeerCaps & capFlag
            rcvdCaps = self.bgpPeerHiscaps & hisCapFlag
            negotiatedCaps = self.peerFlagTest( peerFlag )
            printAddPathCapability( labelString, sentCaps, rcvdCaps, negotiatedCaps )

         def printCapabilityHelper( labelString, capFlag, peerFlag ):
            sentCaps = self.bgpPeerCaps & capFlag
            rcvdCaps = self.bgpPeerHiscaps & capFlag
            negotiatedCaps = self.peerFlagTest( peerFlag )
            printCapability( labelString, sentCaps, rcvdCaps, negotiatedCaps )

         def printAddPathCapability( labelString, sentCaps,
                                     rcvdCaps, negotiatedCaps ):
            if sentCaps or rcvdCaps or negotiatedCaps:
               capabilitiesStr = "    %s: " % labelString
               if negotiatedCaps:
                  capabilitiesStr += "negotiated"
               elif sentCaps:
                  capabilitiesStr += "advertised"
               elif rcvdCaps:
                  capabilitiesStr += "received"
               print capabilitiesStr

         def printCapability( labelString, sentCaps, rcvdCaps, negotiatedCaps ):
            capabilitiesStr = "    %s: " % labelString

            if sentCaps or rcvdCaps:
               if sentCaps:
                  capabilitiesStr += "advertised"
                  if rcvdCaps:
                     capabilitiesStr += " and received"
               else:
                  capabilitiesStr += "received"
               if negotiatedCaps:
                  capabilitiesStr += " and negotiated"
               print capabilitiesStr

         printCapabilityHelper( "Multiprotocol IPv4 Unicast",
                                GatedConstants.BgpcFlag.BGPC_MP_V4_UNI,
                                GatedConstants.BgpPeerFlag.BGPPF_V4_UNI )
         printCapabilityHelper( "Multiprotocol IPv4 Labeled Unicast",
                                GatedConstants.BgpcFlag.BGPC_MP_V4_LABELS,
                                GatedConstants.BgpPeerFlag.BGPPF_V4_LABELS )
         printCapabilityHelper( "Multiprotocol IPv4 SR-TE",
                                GatedConstants.BgpcFlag.BGPC_MP_V4_SR_TE,
                                GatedConstants.BgpPeerFlag.BGPPF_V4_SR_TE )
         printCapabilityHelper( "Multiprotocol IPv6 Unicast",
                                GatedConstants.BgpcFlag.BGPC_MP_V6_UNI,
                                GatedConstants.BgpPeerFlag.BGPPF_V6_UNI )
         printCapabilityHelper( "Multiprotocol IPv6 Labeled Unicast",
                                GatedConstants.BgpcFlag.BGPC_MP_V6_LABELS,
                                GatedConstants.BgpPeerFlag.BGPPF_V6_LABELS )
         printCapabilityHelper( "Multiprotocol IPv6 SR-TE",
                                GatedConstants.BgpcFlag.BGPC_MP_V6_SR_TE,
                                GatedConstants.BgpPeerFlag.BGPPF_V6_SR_TE )
         printCapabilityHelper( "Multiprotocol IPv4 VPN",
                                GatedConstants.BgpcFlag.BGPC_MP_V4_VPNIPV4,
                                GatedConstants.BgpPeerFlag.BGPPF_V4_VPNIPV4 )
         printCapabilityHelper( "Graceful Restart IPv4 VPN",
                                GatedConstants.BgpcFlag.BGPC_V4_VPN_RESTART,
                                GatedConstants.BgpPeerFlag.BGPPF_V4_VPN_RESTART )
         printCapabilityHelper( "Extended Next-Hop Capability:\n      IPv4 Unicast",
                                GatedConstants.BgpcFlag.BGPC_EXT_NEXTHOP,
                                GatedConstants.BgpPeerFlag.BGPPF_EXT_NEXTHOP )
         printCapabilityHelper( "Four Octet ASN",
                                GatedConstants.BgpcFlag.BGPC_AS4,
                                GatedConstants.BgpPeerFlag.BGPPF_AS4 )

         printCapabilityHelper( "Route Refresh",
                                GatedConstants.BgpcFlag.BGPC_RT_REFRESH,
                                GatedConstants.BgpPeerFlag.BGPPF_RT_REFRESH )

         printCapabilityHelper( "Send End-of-RIB messages",
                                GatedConstants.BgpcFlag.BGPC_RESTART_ANY,
                                GatedConstants.BgpPeerFlag.BGPPF_RESTART )

         printCapabilityHelper( "Dynamic Capabilities",
                                GatedConstants.BgpcFlag.BGPC_DYN_CAPS,
                                GatedConstants.BgpPeerFlag.BGPPF_DYN_CAPS )

         sendMy = self.bgpPeerCaps & GatedConstants.BgpcApFlags.BGPC_AP_SEND_ANY
         recvMy = self.bgpPeerCaps & GatedConstants.BgpcApFlags.BGPC_AP_RECV_ANY
         sendHis = self.bgpPeerHiscaps & GatedConstants.BgpcApFlags.BGPC_AP_SEND_ANY
         recvHis = self.bgpPeerHiscaps & GatedConstants.BgpcApFlags.BGPC_AP_RECV_ANY

         if recvMy or sendHis:
            print "    Additional-paths recv capability:"
            printAddPathCapabilityHelper( "  IPv4 Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_RECV_IPV4_UNI )
            printAddPathCapabilityHelper( "  IPv6 Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_RECV_IPV6_UNI )
            printAddPathCapabilityHelper( "  IPv4 Labeled Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_LABELED_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_LABELED_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_RECV_IPV4_LABELED_UNI
            )
            printAddPathCapabilityHelper( "  IPv6 Labeled Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_LABELED_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_LABELED_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_RECV_IPV6_LABELED_UNI
            )
         if sendMy or recvHis:
            print "    Additional-paths send capability:"
            printAddPathCapabilityHelper( "  IPv4 Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_SEND_IPV4_UNI )
            printAddPathCapabilityHelper( "  IPv6 Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_RECV_IPV6_UNI )
            printAddPathCapabilityHelper( "  IPv4 Labeled Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_LABELED_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V4_LABELED_UNI_AP_RECV,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_SEND_IPV4_LABELED_UNI
            )
            printAddPathCapabilityHelper( "  IPv6 Labeled Unicast",
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_LABELED_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpcFlag.BGPC_V6_LABELED_UNI_AP_SEND,
                                          GatedConstants.\
                                          BgpPeerFlag.BGPPF_AP_SEND_IPV6_LABELED_UNI)


         if ( self.bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART or
                 self.bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART) :
            print "    Graceful Restart advertised:"
            if self.advertisedRestartTime:
               print "       Restart-time is %s" % self.advertisedRestartTime
               print "       Restart-State bit: %s" % \
                     ( "yes" if self.miscFlags & \
                       GatedConstants.DabitBnpEntryMF.\
                       DABIT_BNP_ENTRY_MF_LOCAL_RESTART_STATE \
                       else "no" )
               print "       Graceful notification: %s" % \
                     ( "yes" if self.miscFlags &
                       GatedConstants.DabitBnpEntryMF.
                       DABIT_BNP_ENTRY_MF_LOCAL_GR_NOTIFICATION
                       else "no" )
            if ( self.bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART ):
               print "       IPv4 Unicast is enabled, Forwarding State is " \
                     "%s preserved" % \
                    ( ( "" if self.miscFlags & \
                        GatedConstants.DabitBnpEntryMF.\
                        DABIT_BNP_ENTRY_MF_LOCAL_GR_V4_FWD_PRESERVE \
                        else "not" ) )

            if ( self.bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART and
                 self.bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_MP_V6_UNI ) :
               print "       IPv6 Unicast is enabled, Forwarding State is " \
                     "%s preserved" % \
                    ( ( "" if self.miscFlags & \
                        GatedConstants.DabitBnpEntryMF.\
                        DABIT_BNP_ENTRY_MF_LOCAL_GR_V6_FWD_PRESERVE \
                        else "not" ) )

         if ( self.bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART or
                 self.bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART):
            print "    Graceful Restart received:"
            if self.restart:
               print "       Restart-time is %s" % self.restart
               print "       Restart-State bit: %s" % \
                     ( "yes" if self.miscFlags & \
                       GatedConstants.DabitBnpEntryMF.\
                       DABIT_BNP_ENTRY_MF_RESTART_STATE else "no" )
               print "       Graceful notification: %s" % \
                     ( "yes" if self.miscFlags &
                       GatedConstants.DabitBnpEntryMF.
                       DABIT_BNP_ENTRY_MF_GR_NOTIFICATION else "no" )
            if ( self.bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_V4_UNI_RESTART ):
               print "       IPv4 Unicast is enabled, Forwarding State is " \
                     "%s preserved" % \
                     ( ( "" if self.miscFlags & \
                         GatedConstants.DabitBnpEntryMF.\
                         DABIT_BNP_ENTRY_MF_REMOTE_GR_V4_FWD_PRESERVE else "not" ) )

            if ( self.bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_V6_UNI_RESTART
                 and
                 self.bgpPeerHiscaps & GatedConstants.BgpcFlag.BGPC_MP_V6_UNI ):
               print "       IPv6 Unicast is enabled, Forwarding State is "\
                     "%s preserved" % \
                     ( ( "" if self.miscFlags & \
                               GatedConstants.DabitBnpEntryMF.\
                         DABIT_BNP_ENTRY_MF_REMOTE_GR_V6_FWD_PRESERVE else "not" ) )

      if( peerOptionsNoCaps or hisCapsNoCaps ):
         capabilityStr = "    Capability Negotiation: "
         if peerOptionsNoCaps:
            capabilityStr += "disabled"
         if hisCapsNoCaps:
            if peerOptionsNoCaps:
               capabilityStr += " and "

            capabilityStr += "not capable"
         print capabilityStr
      else:
         renderCapabilities()

      restartTimerStr = "  Restart timer is %s" % \
            ( 'active' if self.miscFlags & \
              GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_RESTART_TIMER_STATE \
              else 'inactive' )
      if self.restartTimeLeft:
         restartTimerStr += ", time left: %s" % \
               formatTimeInterval( self.restartTimeLeft )
      print restartTimerStr

      eorTimerStr = "  End of rib timer is %s" % \
            ( 'active' if self.miscFlags & \
              GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_EOR_TIMER_STATE \
              else 'inactive' )
      if self.endOfRibTimeLeft:
         eorTimerStr += ", time left: %s" % \
               formatTimeInterval( self.endOfRibTimeLeft )
      print eorTimerStr

      msgStatFormat = "    %-15s%10s%10s"
      msgStatFormat2 = "    %-15s%10s%10s%15s%20s"
      print "  Message Statistics:"
      print "    InQ depth is %s" % ( self.inMessageStats.queueDepth )
      print "    OutQ depth is %s" % ( self.outMessageStats.queueDepth )
      print msgStatFormat % ( "", "Sent", "Rcvd" )
      print msgStatFormat % ( "Opens:", self.outMessageStats.opens ,
                              self.inMessageStats.opens )
      print msgStatFormat % ( "Notifications:",
                              self.outMessageStats.notifications,
                              self.inMessageStats.notifications )
      print msgStatFormat % ( "Updates:", self.outMessageStats.updates,
                              self.inMessageStats.updates )
      print msgStatFormat % ( "Keepalives:",
                              self.outMessageStats.keepalives,
                              self.inMessageStats.keepalives )
      print msgStatFormat % ( "Route-Refresh:",
                              self.outMessageStats.rtRefreshes,
                              self.inMessageStats.rtRefreshes )
      print msgStatFormat % ( "Total messages:",
                                    self.sentMessages, self.receivedMessages )
      print "  Prefix Statistics:"
      print msgStatFormat2 % ( "", "Sent", "Rcvd", "Best Paths", "Best ECMP Paths" )
      print msgStatFormat2 % ( "IPv4 Unicast:",
                                    self.prefixesSent, self.prefixesReceived,
                                    self.v4BestPaths, self.v4BestEcmpPaths )
      if self.v6SixPePrefixesReceived is not None or \
         self.v6SixPePrefixesSent is not None:
         print msgStatFormat2 % ( "IPv6 6PE(MPLS Label):",
                                 self.v6SixPePrefixesSent,
                                 self.v6SixPePrefixesReceived, '-', '-' )
      elif self.v6PrefixesReceived is not None:
         print msgStatFormat2 % ( "IPv6 Unicast:",
                                 self.v6PrefixesSent, self.v6PrefixesReceived,
                                 self.v6BestPaths, self.v6BestEcmpPaths )
      if self.v4SrTePrefixesSent is not None:
         print msgStatFormat2 % ( "IPv4 SR-TE:  ",
                              self.v4SrTePrefixesSent, self.v4SrTePrefixesReceived,
                              self.v4SrTeBestPaths, self.v4SrTeBestEcmpPaths )
      if self.v6SrTePrefixesSent is not None:
         print msgStatFormat2 % ( "IPv6 SR-TE:  ",
                              self.v6SrTePrefixesSent, self.v6SrTePrefixesReceived,
                              self.v6SrTeBestPaths, self.v6SrTeBestEcmpPaths )
      print "  Inbound updates dropped by reason:"
      print "    AS path loop detection: %d" % ( self.dropStats.inDropAsloop )
      print "    Enforced First AS: %d" % ( self.dropStats.inDropEnforceFirstAs )
      print "    Originator ID matches local router ID: %d" % \
            ( self.dropStats.inDropOrigId )
      print "    Nexthop matches local IP address: %d" % \
            ( self.dropStats.inDropNhLocal )
      print "    Unexpected IPv6 nexthop for IPv4 routes: %d" % \
            ( self.dropStats.inDropNhAfV6 )

      if self.asn != self.localAsn and \
         not self.bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_CONFED \
            and not self.bgpPeerOptions2 & \
            GatedConstants.Bgpo2Flag.BGPO2_EBGP_MULTIHOP:
         print "    Nexthop invalid for single hop eBGP: %d" % \
               ( self.dropStats.inDropNhInvalid )

      if self.peerInUpdateErrors is not None:
         self.peerInUpdateErrors.render()

      print "  Inbound paths dropped by reason:"
      print "    IPv4 labeled-unicast NLRIs dropped due to excessive labels: %d" % \
            ( self.dropStats.prefixLuDroppedV4 )
      print "    IPv6 labeled-unicast NLRIs dropped due to excessive labels: %d" % \
            ( self.dropStats.prefixLuDroppedV6 )
      print "  Outbound paths dropped by reason:"
      print "    IPv4 local address not available: %d" % \
            ( self.dropStats.outDropV4LocalAddr )
      print "    IPv6 local address not available: %d" % \
            ( self.dropStats.outDropV6LocalAddr )

      if self.miscFlags & \
         GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_IN_DEFAULT_DENY_POLICY :
         print "  Missing policy/default deny import action is active"
      if self.miscFlags & \
         GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_OUT_DEFAULT_DENY_POLICY :
         print "  Missing policy/default deny export action is active"
      if self.routeMapInbound:
         print "  Inbound route map is %s" % ( self.routeMapInbound )
      if self.routeMapOutbound:
         print "  Outbound route map is %s" % ( self.routeMapOutbound )
      if self.routeMapInfo.inboundIpv4Uni:
         print "  Inbound route map for IPv4 unicast is %s" % \
               ( self.routeMapInfo.inboundIpv4Uni )
      if self.routeMapInfo.inboundIpv6Uni:
         print "  Inbound route map for IPv6 unicast is %s" % \
               ( self.routeMapInfo.inboundIpv6Uni )
      if self.routeMapInfo.inboundIpv4SrTe:
         print "  Inbound route map for IPv4 SR-TE is %s" % \
               ( self.routeMapInfo.inboundIpv4SrTe )
      if self.routeMapInfo.inboundIpv6SrTe:
         print "  Inbound route map for IPv6 SR-TE is %s" % \
               ( self.routeMapInfo.inboundIpv6SrTe )
      if self.routeMapInfo.outboundIpv4Uni:
         print "  Outbound route map for IPv4 unicast is %s" % \
               ( self.routeMapInfo.outboundIpv4Uni )
      if self.routeMapInfo.outboundIpv6Uni:
         print "  Outbound route map for IPv6 unicast is %s" % \
               ( self.routeMapInfo.outboundIpv6Uni )
      if self.prefixListInfo.inboundIpv4Uni:
         print "  Inbound prefix list for IPv4 unicast is %s" % \
               ( self.prefixListInfo.inboundIpv4Uni )
      if self.prefixListInfo.inboundIpv6Uni:
         print "  Inbound prefix list for IPv6 unicast is %s" % \
               ( self.prefixListInfo.inboundIpv6Uni )
      if self.prefixListInfo.outboundIpv4Uni:
         print "  Outbound prefix list for IPv4 unicast is %s" % \
               ( self.prefixListInfo.outboundIpv4Uni )
      if self.prefixListInfo.outboundIpv6Uni:
         print "  Outbound prefix list for IPv6 unicast is %s" % \
               ( self.prefixListInfo.outboundIpv6Uni )
      if self.bgpPeerCaps & GatedConstants.BgpcFlag.BGPC_EXT_NEXTHOP_ORIGINATE:
         print "  Local IPv4 NLRIs are advertised with IPv6 next-hops"

      if self.maintenance:
         print "  Maintenance-mode:"
         if self.noComms:
            print "    send-community not enabled"

      rmStr = "      Route map is "
      print "    Inbound policy"
      if self.maintenanceRmIn:
         print "%s%s" % ( rmStr, self.maintenanceRmIn )
      print "    Outbound policy"
      if self.maintenanceRmOut:
         print "%s%s" % ( rmStr, self.maintenanceRmOut )

      if self.bgpSoftReconfigInbound != 'Default':
         print '  Soft reconfiguration inbound is "%s"' % self.bgpSoftReconfigInbound

      print "Local AS is %s, local router ID %s" % \
            ( self.localAsn, self.localRouterId )
      ttlStr = "TTL is %s" % self.ttl

      if self.bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_TTLSEC_MAXHOP:
         ttlStr += ", BGP neighbor may be up to %s hops away" % \
                       ( self.maxTtlHops )
      elif self.bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_EBGP_MULTIHOP:
         ttlStr += ", external peer can be %s hops away" % ( self.ttl )
      print ttlStr

      if self.updateSource:
         localTcpStr =  "Local TCP address is %s" % ( self.updateSource )
         if self.localPort:
            localTcpStr += ", local port is %s" % ( self.localPort )
         print localTcpStr

      if self.peerAddress:
         remoteTcpStr = "Remote TCP address is %s" % ( self.peerAddress )
         if self.remotePort:
            remoteTcpStr += ", remote port is %s" % ( self.remotePort )
         print remoteTcpStr

      if self.confRemotePort:
         print "Configured remote port is %s" % ( self.confRemotePort )

      if self.localAddressV4:
         localAddrStr = "Local IPv4 address is %s" % ( self.localAddressV4 )
         if self.localAddressPort:
            localAddrStr += ", local port is %s" % ( self.localAddressPort )
         print localAddrStr

      if self.localAddressV6:
         localAddrStr = "Local IPv6 address is %s" % ( self.localAddressV6 )
         if self.localAddressPort:
            localAddrStr += ", local port is %s" % ( self.localAddressPort )
         print localAddrStr

      if self.remoteAddressV4:
         remoteAddrStr = "Remote IPv4 address is %s" % ( self.remoteAddressV4 )
         if self.remoteAddrPort:
            remoteAddrStr += ", remote port is %s" % ( self.remoteAddrPort )
         print remoteAddrStr

      if self.remoteAddressV6:
         remoteAddrStr = "Remote IPv6 address is %s" % ( self.remoteAddressV6 )
         if self.remoteAddrPort:
            remoteAddrStr += ", remote port is %s" % ( self.remoteAddrPort )
         print remoteAddrStr

      if self.autoLocalAddress:
         print "Auto-Local-Addr is %s" % ( self.autoLocalAddress )

      if self.bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_BFD_ENABLE:
         print "Bfd is enabled and state is %s" % bfdStateName( self.bfdState )

      if self.bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_PASSIVE:
         print "Passive TCP connection-mode is enabled"

      if self.miscFlags & GatedConstants.DabitBnpEntryMF.DABIT_BNP_ENTRY_MF_MD5:
         print "MD5 authentication is enabled"

      if self.bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_REPLACE_AS:
         print "Private AS number from outbound updates replaced with " \
               "local AS to this neighbor"
      elif self.bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_REPLACE_AS:
         print "Private AS number always removed from " \
               "outbound updates to this neighbor"
      elif self.bgpPeerOptions & GatedConstants.BgpoFlag.BGPO_REMOVE_PRIVATE:
         print "Private AS number removed from outbound updates to this neighbor"

      if self.prependOwnDisabled:
         print "Not prepending own AS Number in outbound updates to this neighbor"

      if self.peerTcpInfo is not None:
         self.peerTcpInfo.render()
      # Add newline after printing output for each neighbor
      print ""

   def renderBfd( self ):
      bfdFlags = 'N'
      if self.bgpPeerOptions2 & GatedConstants.Bgpo2Flag.BGPO2_BFD_ENABLE:
         if self.bfdState == GatedConstants.BfdStateFlag.BFD_UP:
            bfdFlags = 'U'
         elif self.bfdState == GatedConstants.BfdStateFlag.BFD_DOWN:
            bfdFlags = 'D'
         elif self.bfdState == GatedConstants.BfdStateFlag.BFD_INIT:
            bfdFlags = 'I'
         else:
            bfdFlags = 'N'

      print "%-18s %-18s %-10s %-11s %-5c" % \
            ( self.peerAddress,
              kernelIntfFilter( self.ifName ) \
              if self.ifName is not None else "none",
              formatTimeInterval( self.updownTime ) \
                 if self.updownTime is not None else "--:--",
              self.state, bfdFlags )

   def processData( self, data ):
      self.peerAddress = peerKeyFromData( data )

      if 'updateSource6' in data or 'updateSource4' in data:
         if 'updateSource6' in data and data[ 'updateSource6' ]:
            ipAddr = data[ 'updateSource6' ]
            del data[ 'updateSource6' ]
         elif 'updateSource4' in data and data[ 'updateSource4' ]:
            ipAddr = data[ 'updateSource4' ]
            del data[ 'updateSource4' ]
         self.updateSource = Arnet.IpGenAddr( trimIntfName( ipAddr ) )

      flagNum = 0
      peerFlagName = 'bgp_peer_flags%d' % flagNum
      while peerFlagName in data:
         self.bgpPeerFlags.append ( data[ peerFlagName ] )
         del data[ peerFlagName ]
         flagNum += 1
         peerFlagName = 'bgp_peer_flags%d' % flagNum

      if( data[ 'maintenance' ] ):
         self.maintenance = ord( data[ 'maintenance' ] ) != 0
         del data[ 'maintenance' ]
      if( data[ 'noComms' ] ):
         self.noComms = ord( data[ 'noComms' ] ) != 0
         del data[ 'noComms' ]

      # for backward compatability when single 'inout' route-map was supported
      if 'maintenanceRmIn' in data and 'maintenanceRmOut' in data and \
         ( data[ 'maintenanceRmIn' ] == data[ 'maintenanceRmOut' ] ):
         self.maintenanceRm = data[ 'maintenanceRmOut' ]

      disabled = data.pop( 'prependOwnDisabled', None )
      if disabled is not None:
         self.prependOwnDisabled = ord( disabled ) != 0

      if data[ 'bgpPeerOptions' ] & GatedConstants.BgpoFlag.BGPO_KEEPALL:
         self.bgpSoftReconfigInbound = "All"
      elif data[ 'bgpPeerOptions' ] & GatedConstants.BgpoFlag.BGPO_KEEPNONE:
         self.bgpSoftReconfigInbound = "None"
      else:
         self.bgpSoftReconfigInbound = "Default"

      # the DGET has a large number of attributes with common prefixes
      # processFields is used to map those names to BgpPeer submodels
      # to reduce clutter
      def processFields( data, model, prefixStr ):

         fields = model.__attributes__.keys()

         # input field suffixes have camel case words
         if prefixStr:
            inputFields = [ prefixStr + i[0].upper() + i[1:] for i in fields ]
         else:
            inputFields = fields

         for field, inField in zip( fields, inputFields ):
            if inField in data:
               setattr( model, field, data[ inField ] )
               del data[ inField ]

         return model

      if 'lastErrorCode' in data:
         self.lastErrorTimeInfo = processFields( data, ErrorTimeInfo(),
                                                 'lastError' )
      if 'lastErrorRcvdCode' in data:
         self.lastErrorRcvdTimeInfo = processFields( data, ErrorTimeInfo(),
                                                     'lastErrorRcvd' )
      if 'lastSocketOutErrorMsg' in data:
         self.lastSocketOutErrorTimeInfo = processFields( data, ErrorTimeInfo(),
                                                          'lastSocketOutError' )
      if 'lastSocketInErrorMsg' in data:
         self.lastSocketInErrorTimeInfo = processFields( data, ErrorTimeInfo(),
                                                         'lastSocketInError' )

      self.inMessageStats = processFields( data, MessageStatInfo(), 'in' )
      self.outMessageStats = processFields( data, MessageStatInfo(), 'out' )

      self.routeMapInfo = processFields( data, RouteFilterInfo(), 'routeMap' )
      self.routeMapInfo.filterType = 'RouteMap'

      self.prefixListInfo = processFields( data, RouteFilterInfo(), 'prefixList' )
      self.prefixListInfo.filterType = 'PrefixList'

      self.dropStats = processFields( data, PeerDropStats(), '' )

      return data

class BgpPeerList( BgpPeerListBase ):
   peerList = GeneratorList( valueType=BgpPeer,
                             help='Bgp Peer List ' )
   _showBfd = Bool( help='Bfd output is requested')

   def renderBfdHeader( self ):
      print "BGP BFD Neighbor Table"
      print "Flags: U - BFD is enabled for BGP neighbor "\
         "and BFD session state is UP"
      print "       I - BFD is enabled for BGP neighbor "\
         "and BFD session state is INIT"
      print "       D - BFD is enabled for BGP neighbor "\
         "and BFD session state is DOWN"
      print "       N - BFD is not enabled for BGP neighbor"
      print "%-18s %-18s %-10s %-11s %-5s" % ( "Neighbor", "Interface",
                                               "Up/Down", "State", "Flags" )

   def render( self ):
      if self._showBfd:
         self.renderBfdHeader()

      peers = sorted( [ peer for peer in self.peerList ],
                      key=lambda p: PeerConfigKey( p.peerAddress ) )
      for peer in peers:
         if self._showBfd:
            peer.renderBfd()
         else:
            peer.render()
