#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Int, Float, Bool, Dict, Str, List, Model
import ArnetModel
import Ark
import Tac
import TableOutput
import IntfModels

def _createFsfTable( headings ):
   table = TableOutput.createTable( headings )
   f = TableOutput.Format( justify="left" )
   f.noPadLeftIs( True )
   table.formatColumns( *( [ f ] * len( headings ) ) )
   return table

def _intfName( intf ):
   if not isinstance( intf, str ):
      intf = intf.stringValue
   return intf.replace( "'", "" )
   
class MonitorServerFailure( Model ):
   """Base class for Fsf 'show monitor server-failure' command """

   enabled = Bool( help="Server-failure feature is enabled or not" )
   proxyService = Bool( help="Is proxy service enabled or not" )
   proxyLifetimeDefault = Int( help="Default proxy lifetime" )
   proxyLifetime = Int( help="Current lifetime",
                   optional=True )
   networkServers = Dict( help="Number of servers in each monitored network" + \
                             "Indexed by network", 
                          keyType=str, valueType=int )

   def render( self ):
      print "Server-failure monitor is", \
               "enabled" if self.enabled else "not enabled"
      print "Proxy service:", "enabled" if self.proxyService else "disabled"
      if self.proxyService:
         default = ( self.proxyLifetime == self.proxyLifetimeDefault )
         print "Proxy lifetime: %d minutes %s" % ( self.proxyLifetime,
                                                   "(default)" if default else "" )
      print "Networks being monitored:", len( self.networkServers )
      for network in self.networkServers.keys():
         print "   %-18s: %d servers" % ( network, 
                                          self.networkServers[ network ] )


class Server( Model ):
   """ Class to hold server information"""
   isValid = Bool( help="Is the server info valid" )
   ipAddress = ArnetModel.Ip4Address( help="Server IP Address" )
   ethAddress = ArnetModel.MacAddress( help="Server Ethernet Address" )
   state = Str( help="Server state" )
   intf = IntfModels.Interface( help="Interface the server is on" )
   subnet = Str( help="Subnet to which the server belongs" )
   discoveryTime = Float( help="Time at which the server was discovered" )
   lastFailureTime = Float( help="Time at which the server last failed" )
   lastProxyTime = Float( help="Time at which the server was last proxied" )
   lastInactiveTime = Float( help="Time at which the server was last made inactive" )
   numFailed = Int( help="Number of times the server failed" )
   numProxied = Int( help="Number of times the server was proxied" )
   numInactive = Int( help="Number of times the server was made inactive" )
      
   def toUtc( self, timestamp ):
      utcTime = timestamp + Tac.utcNow() - Tac.now() if timestamp else timestamp
      return utcTime

   def utcToStr( self, utcThen, relative=True ):
      timeDiff = Tac.utcNow() - utcThen
      tacTimeThen = Tac.now() - timeDiff if utcThen else 0
      return Ark.timestampToStr( tacTimeThen, relative )

   def render( self ):
      if not self.isValid:
         print "Server with IP Address %s does not exist." % str( self.ipAddress )
         return
      print "Server information:"
      print "  %-25s: %s" % ( "Server Ip Address", self.ipAddress )
      print "  %-25s: %s" % ( "MAC Address", self.ethAddress )
      print "  %-25s: %s" % ( "Subnet", self.subnet )
      print "  %-25s: %s" % ( "Current state", self.state )
      print "  %-25s: %s" % ( "Interface", _intfName( self.intf ) )
      print "  %-25s: %s" % ( "Last Discovered", self.utcToStr( self.discoveryTime,
                                                                False ) )
      print "  %-25s: %s" % ( "Last Failed", self.utcToStr( self.lastFailureTime,
                                                            False ) )
      print "  %-25s: %s" % ( "Last Proxied", self.utcToStr( self.lastProxyTime,
                                                             False ) )
      print "  %-25s: %s" % ( "Last Inactive", self.utcToStr( self.lastInactiveTime,
                                                              False ) )
      print "  %-25s: %d" % ( "Number of times failed", self.numFailed )
      print "  %-25s: %d" % ( "Number of times proxied", self.numProxied )
      print "  %-25s: %d" % ( "Number of times inactive", self.numInactive )
      
class Servers( Model ):
   """Class that holds collection of servers"""
   summaryStatement = Str( help="Summary statement for type of servers in the list" )
   servers = List( valueType=Server,
                   help="List of servers monitored by Server-failure" )
   fsfEnabled = Bool( help="Is Server-failure feature is enabled or not" )
   
   def render( self ):
      if not self.fsfEnabled:
         print "Server-failure monitor is not enabled"
         return
      headings = ( "Server IP", "Server MAC", "Interface",
                   "State", "Last Failed" )
      table = _createFsfTable( headings )
      numServers = len( self.servers )
      for server in self.servers:
         table.newRow( server.ipAddress, server.ethAddress, _intfName( server.intf ),
                       server.state, server.utcToStr( server.lastFailureTime ) )
      print self.summaryStatement, numServers, "\n"
      print table.output()

class ServerHistory( Model ):
   """Class that holds history for Server-failure"""
   servers = List( valueType=Server,
                   help="List of servers monitored by Server-failure" )
   fsfEnabled = Bool( help="Is Server-failure feature is enabled or not" )

   def render( self ):
      if not self.fsfEnabled:
         print "Server-failure monitor is not enabled"
         return
      headings = ( "Server IP", "Server MAC", "Interface", "Last Failed" )
      table = _createFsfTable( headings )
      numFailed = 0
      for server in self.servers:
         table.newRow( server.ipAddress, server.ethAddress, _intfName( server.intf) ,
                       server.utcToStr( server.lastFailureTime ) )
         numFailed += 1
      print "Total server failures:", numFailed, "\n"
      print table.output()

