#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
import re
import HumanReadable 
from CliModel import Submodel, Dict
from CliModel import Model, Str, Int, Enum
from TableOutput import Format
from TableOutput import createTable

# Helper Functions

def printDashes( formatStr ):
   # This prints out a series of dashes to match a format string that
   # looks like "%-16s %-12s %-16s %-20s %-10s".

   out = ""
   while formatStr:
      c = formatStr[ 0 ]
      if c == ' ':
         out += c
         formatStr = formatStr[ 1: ]
         continue
      if c == '%':
         spec = formatStr.split()[ 0 ]
         formatStr = formatStr[ len( spec ): ]
         # Match optional '-', number, type specifier.
         match = re.search( r'-?(\d+).*', spec )
         if not match:
            # Not good, but what else can we do?
            continue
         count = int( match.group( 1 ) )
         out += '-' * count
         continue
      out += c
      formatStr = formatStr[ 1: ]
   print out

# Eapi Models for the Fru package 

class SystemInformation( Model ):
   name = Str( help="Model name", default="Unknown" )
   description = Str( help="Model Description", default="" )
   hardwareRev = Str( help="Hardware Revision", default="" )
   serialNum = Str( help="Serial Number", default="" )
   mfgDate = Str( help="Manufacture Date ( YYYY-MM-DD )", default="" )
   hwEpoch = Str( help="Hardware Epoch", default="" )

storageTypes = {
   'UNKNOWN' : 'unknown',
   'EMMC'    : 'eMMC',
   'SSD'     : 'SSD',
   'EUSB'    : 'eUSB',
   'NVME'    : 'NVME',
}

class StorageDevice( Model ):
   model = Str( help="Model Name of the Storage Device" )
   mount = Str( help="Mount Point of the Storage Device" )
   storageType = Enum( values=storageTypes.values(),
                       help="Storage Type of the Storage Device" )
   serialNum = Str( help="Serial Number of the Storage Device" )
   firmwareRev = Str( help="Firmware Revision of the Storage Device" )
   storageSize = Int( help="Size in GB of the Storage Device" )

# For CAPI backward compatibility, this model is retained. The elements
# are logically just different storage devices.
class EmmcFlashDevice( Model ):
   model = Str( help="Model Name of the Flash Device" )
   mount = Str( help="Mount point of the Flash Device" )
   serialNum = Str( help="Serial Number of the Flash Device" )
   firmwareRev = Str( help="Firmware Revision of the Flash Device" )
   storageSize = Int( help="Size in GB of the Flash Device" )

class PowerSupplySlot( Model ):
   name = Str( help="Model name of the power supply slot", default="Not Inserted" )
   serialNum = Str( help="Serial number for the power supply slot", default="" )

class FanTraySlot( Model ):
   numFans = Int( help="Number of fans in the fan tray slot.", default=0 )
   name = Str( help="Model name of the fan tray slot", default="" )
   serialNum = Str( help="Serial number for the fan tray slot",
                    default="" )

class XcvrSlot( Model ):
   _pos = Int( help="Relative position of the xcvr slot" )
   mfgName = Str( help="Transceiver Manufacturer name", default="Not Present" )
   modelName = Str( help="Transceiver model name", default="" )
   serialNum = Str( help="Transceiver serial number", default="" )
   hardwareRev = Str( help="Transceiver hardware revision number", default="" )

class CardSlot( Model ):
   _pos = Int( help="Relative position of the card slot" )
   modelName = Str( help="Model name of the card slot", default="Not Inserted" )
   hardwareRev = Str( help="Hardware version of the card slot", default="" )
   serialNum = Str( help="Serial Number of the card slot", default="" )
   mfgDate = Str( help="Manufacture Date of the card slot (YYYY-MM-DD)", default="" )
   hwEpoch = Str( help="Hardware Epoch", default="" )

class Fpga( Model ):
   fpgaRev = Str( help="Revision of the FPGA if present", default="Not Programmed" )

class Clock( Model ):
   pass

class Inventory( Model ):
      
   systemInformation = Submodel( valueType=SystemInformation, 
                                 help="System Information" )
   # Additional system information, may not be present for namespace duts
   powerSupplySlots = Dict( keyType=int, valueType=PowerSupplySlot,
                            help="Dictionary of power supply slots" )
   fanTraySlots = Dict( keyType=int, valueType=FanTraySlot, 
                        help="Dictionary of fan tray slots" )
   portCount = Int( help="Total count of ports", default=0 )
   internalPortCount = Int( help="Total count of Internal ports", default=0 )
   dataLinkPortCount = Int( # not used, here for backwards compatibility
      help="Total count of data link ports", default=0 )
   switchedPortCount = Int( help="Total count of Switched ports", default=0 )
   switchedBootstrapPortCount = Int( help="Total count of Switched/Bootstrap ports",
                                     default=0 )
   unconnectedPortCount = Int( help="Total count of unconnected ports", default=0,
                               optional=True )
   switchedFortyGOnlyPortCount = Int( # not used, here for backwards compatibility
      help="Total count of Switched 40G Only ports", default=0 )
   managementPortCount = Int( help="Total count of Management ports", default=0 )
   xcvrSlots = Dict( valueType=XcvrSlot, 
                     help="Dictionary of Transceiver slots indexed by name" )
   # This attribute is only populated for the modular systems
   cardSlots = Dict( valueType=CardSlot, help=
                     "Dictionary of Card slots, only present for modular systems" )

   # Additional inventory, which could be populated by registered
   # callback functions. 
   storageDevices = Dict( valueType=StorageDevice, help=
                          "Dictionary of Storage Devices indexed by model name" )

   emmcFlashDevices = Dict( valueType=EmmcFlashDevice, help=
                            "Dictionary of Flash Devices indexed by model name" )

   # Some systems have subcomponents with their own serial numbers.
   subcompSerNums = Dict( keyType=str, valueType=str, optional=True,
                          help="Dictionary of subcomponent serial numbers" )

   # The 'fpgas' and 'precisionClock' attributes are not and should not be used.
   # The associated "show inventory" callback functions lived within the Stratix and
   # Ruby packages (respectively), which have now been deprecated. They're kept
   # because we cannot remove attributes from CLI models once they have shipped.
   fpgas = Dict( valueType=Fpga, help="Dictionary of fpgas indexed by fpga name" )
   precisionClock = Submodel( valueType=Clock, help="System Precision clock", 
                              optional=True )

   def render( self ):
      # Split system inventory onto two lines for longer SKU/desc
      def makeTable( header, minWidths=None, padLimit=True ):
         if minWidths is None:
            minWidths = [ None ] * len( header )
         assert len( header ) == len( minWidths )
         t = createTable( header, indent=2 )
         formats = []
         for minWidth in minWidths:
            f = Format( justify="left", minWidth=minWidth )
            f.noPadLeftIs( True )
            f.padLimitIs( padLimit )
            formats.append( f )
         t.formatColumns( *formats )
         return t
      print "System information"
      sysTitleL1 = ( "Model", "Description" )
      sysTitleL2 = ( "HW Version", "Serial Number", "Mfg Date", "Epoch" )
      whatL1 = ( self.systemInformation.name, self.systemInformation.description )
      whatL2 = ( self.systemInformation.hardwareRev,
                 self.systemInformation.serialNum, self.systemInformation.mfgDate,
                 self.systemInformation.hwEpoch )
      t = makeTable( sysTitleL1, minWidths=[ 24, 52 ] )
      t.newRow( *whatL1 )
      print t.output()
      t = makeTable( sysTitleL2, minWidths=[ 11, 14, 10, 5 ] )
      t.newRow( *whatL2 )
      print t.output()

      if self.subcompSerNums:
         print "Subcomponent serial numbers"
         t = makeTable( ( "Component", "Serial Number" ), padLimit=False )
         for ( c, sn ) in self.subcompSerNums.iteritems():
            t.newRow( c, sn )
         print t.output()

      if self.cardSlots:
         cs = len( self.cardSlots )
         print "System has %s card slots" % cs
         t = makeTable( ( "Module", "Model", "HW Version", "Serial Number",
                          "Mfg Date", "Epoch" ), padLimit=False )
         # pylint: disable-msg=W0212
         for ( name, card ) in sorted( self.cardSlots.iteritems(),
                                       key=lambda x : x[1]._pos ):
            t.newRow( name, card.modelName, card.hardwareRev,
                      card.serialNum, card.mfgDate, card.hwEpoch )
         print t.output()

      if self.powerSupplySlots:
         ps = len( self.powerSupplySlots )
         print "System has %s" % HumanReadable.plural( ps, "power supply slot" )
         psHeader = ( "Slot", "Model", "Serial Number" )
         t = makeTable( psHeader, minWidths=[ 4, 16, 16 ] )
         for ( i, ps ) in self.powerSupplySlots.iteritems():
            t.newRow( i, ps.name, ps.serialNum )
         print t.output()

      if self.fanTraySlots:
         num = len( self.fanTraySlots )
         print "System has %s" % HumanReadable.plural( num, "fan module" )
         fanHeader = ( "Module", "Number of Fans", "Model", "Serial Number" )
         t = makeTable( fanHeader, minWidths=[ 7, 15, 16, 16 ] )
         for ( i, ft ) in self.fanTraySlots.iteritems():
            if ft.numFans > 0:
               t.newRow( i, ft.numFans, ft.name, ft.serialNum )
            else:
               t.newRow( i, "Not Inserted", ft.name, ft.serialNum )
         print t.output()

      if self.portCount:
         print "System has %s" % HumanReadable.plural( self.portCount, "port" )
         portTypes = ( "management", "switched", "internal", "unconnected",
                       "switchedBootstrap" )
         # We can have ports that have not been allocated any roles
         # (ie. switched, management etc), in which case pc will be 0 but
         # self.portCount will not be 0. Hence we need to make an
         # additional check here.
         pc = 0
         for portType in portTypes:
            pc += getattr( self, "%sPortCount" % portType )
         if pc:
            # Convert only the 1st alphabet of the word into upper case
            firstCharToUpper = lambda s : s[ :1 ].upper() + s[ 1: ]
            portFormat = "  %-18s %-4s"
            print "  %-18s %-4s" % ( "Type", "Count" )
            printDashes( portFormat )
            for portType in portTypes:
               portCount = getattr( self, "%sPortCount" % portType )
               if portCount:
                  print portFormat % ( firstCharToUpper( portType ), portCount )
         print

      if self.xcvrSlots:
         xcvrs = len( self.xcvrSlots )
         print "System has %s" % HumanReadable.plural( xcvrs, "transceiver slot" )
         xcvrFormat = "  %-4s %-16s %-16s %-16s %-4s"
         print xcvrFormat % \
             ( "Port", "Manufacturer", "Model", "Serial Number", "Rev" )
         printDashes( xcvrFormat )
         # pylint: disable-msg=W0212         
         for ( xcvrName, xcvr ) in sorted( self.xcvrSlots.iteritems(), 
                                           key=lambda x : x[1]._pos ):
            print xcvrFormat % ( xcvrName, xcvr.mfgName, xcvr.modelName, 
                                 xcvr.serialNum, xcvr.hardwareRev )
         print

      if self.fpgas:
         raise ValueError( "self.fpgas should be empty; see model definition" )

      # To maintain CAPI backward compatability, storageDevices and emmcFlashDevices
      # are modeled separately, but are combined in a single output here. They are
      # sorted by mount point, where /mnt/flash is displayed first, followed by
      # /mnt/drive. For future compatibility, any other mounts are displayed in
      # alphabetical order after that.
      deviceIter = itertools.chain( self.storageDevices.values(),
                                    self.emmcFlashDevices.values() )
      emmcType = storageTypes[ 'EMMC' ]
      storageDevices = [ ( device.mount, getattr( device, 'storageType', emmcType ),
                           device.model, device.serialNum, device.firmwareRev,
                           device.storageSize ) for device in deviceIter ]

      if storageDevices:
         print "System has %s" % HumanReadable.plural( len( storageDevices ),
                                                       "storage device" )
         t = makeTable( ( "Mount", "Type", "Model", "Serial Number", "Rev",
                          "Size (GB)" ) )
         def mountKey( elem ):
            if elem[ 0 ] == '/mnt/flash':
               return '.0'
            if elem[ 0 ] == '/mnt/drive':
               return '.1'
            return elem[ 0 ]
         for dev in sorted( storageDevices, key=mountKey ):
            t.newRow( *dev )
         print t.output()

      if self.precisionClock:
         raise ValueError(
            "self.precisionClock should be empty; see model definition" )
