# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import ShowCommand
import Tac, CliParser
import CliParserCommon
import BasicCliModes
import BasicCliSession
import CliCommand
import CliMatcher
import CliGlobal
import CommonGuards
import re, collections
import Tracing
import Fru
import CliPlugin.FruModel as FruModel
from CliPlugin.FruModel import Inventory
from CliPlugin.MacAddr import macAddrMatcher, compareMacs
from CliToken.System import systemMatcherForConfig
import ConfigMount
import LazyMount
from Toggles.FruToggleLib import toggleOverrideSystemMacEnabled

# ------------------------------------------------------
# Registration system for fixed vs modular system rules.
# CliPlugins register callbacks with the SystemTypeReactor
# to be called once the system type has been determined
# (ie the entity mib root has been set).
# ------------------------------------------------------


__defaultTraceHandle__ = Tracing.Handle( "FruCli" )
t0 = Tracing.trace0

# -----------------------------------------------------
# Maximum number of cards of each type
# -----------------------------------------------------
maxNumOfSups = 2
maxNumOfLinecards = 16
maxNumOfFabricCards = 8
maxNumOfSwitchcards = 1

systemTypeReactorInitialized_ = False
modularSystemCallbacksRegisteredWithReadline_ = False
fixedSystemCallbacksRegisteredWithReadline_ = False

gv = CliGlobal.CliGlobal( dict( powerFuseConfig=None, redundancyConfig=None,
                                entmib=None, epochStatus=None ) )
   
EthAddr = Tac.Type( 'Arnet::EthAddr' )
macAddrConfig = None
hwEntMib = None

class SystemTypeReactor( Tac.Notifiee ):

   notifierTypeName = "EntityMib::Status"

   def __init__( self, entmib ):
      Tac.Notifiee.__init__( self, entmib )
      self.systemType_ = None
      self.handleSystemType()

   @Tac.handler( 'root' )
   def handleSystemType( self ):
      """ React to the entity mib root being set by registering the
      appropriate callbacks to be invoked by the readline thread. We
      cannot simply invoke the callbacks ourselves, since we're
      in the activities thread, and the readline thread may be in
      the process of iterating through rules lists, etc """

      systemType = (
         self.notifier_.root.tacType.fullTypeName if self.notifier_.root else None )
      if systemType is None:
         # System type not yet set
         assert self.systemType_ is None
      elif systemType == "EntityMib::Chassis":
         if self.systemType_ is None:
            while modularSystemCallbacks_:
               callback = modularSystemCallbacks_.pop( 0 )
               BasicCliSession.registerReadlineThreadWork( callback )
            self.systemType_ = systemType
            global modularSystemCallbacksRegisteredWithReadline_
            modularSystemCallbacksRegisteredWithReadline_ = True
         else:
            assert self.systemType_ == systemType
      elif systemType == "EntityMib::FixedSystem":
         if self.systemType_ is None:
            while fixedSystemCallbacks_:
               callback = fixedSystemCallbacks_.pop( 0 )
               BasicCliSession.registerReadlineThreadWork( callback )
            self.systemType_ = systemType
            global fixedSystemCallbacksRegisteredWithReadline_
            fixedSystemCallbacksRegisteredWithReadline_ = True
         else:
            assert self.systemType_ == systemType

fixedSystemCallbacks_ = []
modularSystemCallbacks_ = []

# calls to this should be replaced with fixedSystemGuard
def registerFixedSystemCallback( callback ):
   """ Register a callback to be invoke by the readline thread
   at the point that we determine that we're in a fixed system."""

   assert not systemTypeReactorInitialized_
   fixedSystemCallbacks_.append( callback )

# Will be replaced by registration using modularSystemGuard. ref BUG 23010.
def registerModularSystemCallback( callback ):
   """ Register a callback to be invoke by the readline thread
   at the point that we determine that we're in a module system."""
   t0( 'Registering Modular Callback', callback )

   assert not systemTypeReactorInitialized_
   modularSystemCallbacks_.append( callback )

def modularSystemCallbacksRegisteredWithReadline():
   return modularSystemCallbacksRegisteredWithReadline_

def fixedSystemCallbacksRegisteredWithReadline():
   return fixedSystemCallbacksRegisteredWithReadline_

def _defaultTags( root ):
   if root is None:
      return {}
   cardSlot = root.defaultChildTags.get( 'CardSlot' )
   if cardSlot:
      return cardSlot.tag.values()
   return []

def _tagLabels( root, prefix='', validTags=None ):
   # return a tag: label-list dictionary
   labels = {}
   if not root:
      if prefix == 'Fabric':
         labels[ prefix ] = []
         labels[ prefix ] = [ str( n ) for n in range( 1, maxNumOfFabricCards + 1 ) ]
      elif prefix == 'Switchcard':
         labels[ prefix ] = []
         labels[ prefix ] = [ str( n ) for n in range( 1, maxNumOfSwitchcards + 1 ) ]
      else:
         if 'Supervisor'.startswith( prefix ):
            labels[ 'Supervisor' ] = [ str( n ) for n in range( 1,
               maxNumOfSups + 1 ) ]
         if 'Linecard'.startswith( prefix ):
            # Start linecard labels at 2 because of modulars with only 1 supervisor.
            labels[ 'Linecard' ] = [ str( n ) for n in range( maxNumOfSups,
                     maxNumOfSups + maxNumOfLinecards + 1 ) ]

   elif root.tacType.fullTypeName == "EntityMib::Chassis":
      prefix = prefix.lower()
      for slot in root.cardSlot.values():
         if validTags is None or slot.tag in validTags:
            if slot.tag.lower( ).startswith( prefix ):
               if slot.tag in labels:
                  labels[ slot.tag ].append( slot.label )
               else:
                  labels[ slot.tag ] = [ slot.label ]
   return labels

# Named Tuple that is written if slot specified has not been discovered yet.
SlotDescription = collections.namedtuple( "SlotDescription",
      "slot, tag, label" )

class SlotMatcher( CliMatcher.Matcher ):
   """Matches a token either in the form of tag+number or just number.
   If forceTags is specified, only matches the number corresponding to
   the specified tags (no tag part).
   """
   # regexs for matching
   labelRegEx = re.compile( '^([0-9]+)$' )
   tagLabelRegEx = re.compile( '^([a-zA-Z]+)([0-9]+)$' )

   def __init__( self, forceTags=None, **kargs ):
      super( SlotMatcher, self ).__init__( **kargs )
      self.forceTags_ = forceTags

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root

      t0( "guardsEnabled is ", mode.session.guardsEnabled() )
      # try to match tag+number
      m = self.tagLabelRegEx.match( token )

      if m:
         ( tag, label ) = m.groups()

      if root is None:
         # Hardware has not yet been discovered.
         # Any well formed token will be accepted in Cli when Guards are disabled,
         # one such case being startup.
         if not mode.session.guardsEnabled() and m:
            result = SlotDescription( None, tag, label )
            return CliParserCommon.MatchResult( result, token )
         else:
            return CliParserCommon.noMatch

      if self.forceTags_ is None:
         # All possible tags are to be considered
         if m:
            taglabels = _tagLabels( root, tag )
            # return the dictionary of tags with 'tag' as the prefix
            # it has to contain exactly one item
            if len( taglabels ) == 1:
               for tag in taglabels:
                  for slot in root.cardSlot.values():
                     if slot.tag == tag and slot.label == label:
                        result = SlotDescription( slot, tag, label )
                        return CliParserCommon.MatchResult( result, token )
            return CliParserCommon.noMatch

      m = self.labelRegEx.match( token )
      if m:
         # number only
         label = m.group( 1 )
         validTags = self.forceTags_ if self.forceTags_ \
                     else _defaultTags( root )

         for slot in root.cardSlot.values():
            if slot.label == label and slot.tag in validTags:
               result = SlotDescription( slot, slot.tag, slot.label )
               return CliParserCommon.MatchResult( result, token )
         return CliParserCommon.noMatch
      return CliParserCommon.noMatch

   def _slotNumberCompletions( self, root, number, taglabels ):
      # Return completion for a partially input number. If the number is None or
      # falls into the valid range, add the range a non-literal completion.
      completions = []

      for ( tag, label ) in taglabels.iteritems( ):
         intLabel = [ int( d ) for d in label ]
         minLabel = min( intLabel )
         maxLabel = max( intLabel )

         if number is None or \
                CliParser.isPrefixOfValueInRange( number, minLabel, maxLabel ):
            completions.append( CliParser.Completion( '<%d-%d>'
                                                      %( minLabel, maxLabel ),
                                                      "%s Number" % tag,
                                                      literal=False ) )
      return completions

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None or root.tacType.fullTypeName != "EntityMib::Chassis":
         # Hardware has not yet been discovered
         return []

      if token and not self.forceTags_:
         # try to match tag+number
         m = self.tagLabelRegEx.match( token )
         if m:
            taglabels = _tagLabels( root, m.group( 1 ) )
            if len( taglabels ) == 1:
               return self._slotNumberCompletions( root, int( m.group( 2 ) ),
                                                   taglabels )
            return []

      # just return the number range
      m = self.labelRegEx.match( token )
      if m or not token:
         validTags = self.forceTags_ if self.forceTags_ else \
                     _defaultTags( root )
         taglabels = _tagLabels( root, '', validTags=validTags )
         number = int( m.group( 1 ) ) if m else None
         return self._slotNumberCompletions( root, number, taglabels )
      # nothing matches
      return []
         
   def __str__( self ):
      return "<slot>"

class Card( object ):
   def __init__( self, mode, name, entmibRoot ):
      self.mode = mode
      self.name = name
      self.entmibRoot = entmibRoot
      self.tag = None
      self.label = None
      match = SlotMatcher.tagLabelRegEx.match( self.name )
      if match:
         self.tag, self.label = match.groups()
      else:
         assert SlotMatcher.labelRegEx.match( self.name )
         tagLabels = _tagLabels( self.entmibRoot,
                                 validTags=[ 'Linecard', 'Supervisor' ] )
         for tag in tagLabels :
            if self.name in tagLabels[ tag ]:
               self.tag = tag
               self.label = self.name
               break
         self.name = '%s%s' % ( self.tag, self.label )
      assert self.tag
      assert self.name

   def powerOff( self ):
      if self.name == ( "Supervisor%d" % Fru.slotId() ):
         self.mode.addError( "Powering off the current supervisor is not supported. "
                             "Use 'reload power' instead." )
         return
      t0( 'powering off', self.name )
      gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ self.name ] = True
      gv.powerFuseConfig.powerOffRequested[ self.name ] = "user configuration"

      # Power off uplink, is a noop on supervisors without an uplink
      if self.tag == 'Supervisor':
         uplinkName = 'Linecard{}'.format( self.label )
         gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ uplinkName ] = True
         gv.powerFuseConfig.powerOffRequested[ uplinkName ] = "user configuration"

      # See BUG8484 for why this sleep is required
      if not self.mode.session.inConfigSession():
         import time
         time.sleep( 0.5 )

   def powerOn( self ):
      t0( 'powering on', self.name )
      del gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ self.name ]
      del gv.powerFuseConfig.powerOffRequested[ self.name ]

      # Power on uplink, is a noop on supervisors without an uplink
      if self.tag == 'Supervisor':
         uplinkName = 'Linecard{}'.format( self.label )
         del gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ uplinkName ]
         del gv.powerFuseConfig.powerOffRequested[ uplinkName ]


def cardTypeGuard( mode, token ):
   # Dont show a card type if it's not present on the System
   # Test only for card labels and not slot ranges

   if not token or token.title() not in [ 'Linecard', 'Supervisor',
                                          'Fabric', 'Switchcard' ]:
      return None

   if not gv.entmib or not gv.entmib.root:
      return CliParser.guardNotThisPlatform
   else:
      slotLabels = _tagLabels( gv.entmib.root, '' )
      if token.title() not in slotLabels:
         return CliParser.guardNotThisPlatform

   return None

def modularSystemGuard( mode, token ):
   # etba dut has no entmib.root, considered as fixed system
   if ( not gv.entmib.root or
        gv.entmib.root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
      return CliParser.guardNotThisPlatform
   return None

def fixedSystemGuard( mode, token ):
   if ( gv.entmib.root and
        gv.entmib.root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
      return None
   return CliParser.guardNotThisPlatform

def powerSupplyGuard( mode, token ):
   if gv.entmib and gv.entmib.root and gv.entmib.root.powerSupplySlot:
      return None
   return CliParser.guardNotThisPlatform

"""The SlotRule allows the following input:
1. a tag followed by a space and a number, such as 'Linecard 1'
2. a tag followed immediately by a number, such as 'Linecard1'
3. a number without a tag for default tags, such as 1

1 is handled by a series of ConcatRule, and 2/3 are handled by a TokenRule
with SlotMatcher.

Partial tokens are allowed as long as it's not ambiguous, such as 'lin1'
"""
# new parser
class SlotExpressionFactory( CliCommand.CliExpressionFactory ):
   slotMatcher_ = SlotMatcher()
   moduleTypes_ = ( 'Supervisor', 'Linecard', 'Fabric', 'Switchcard' )

   @staticmethod
   def modularAndCardTypeGuard( mode, token ):
      return modularSystemGuard( mode, token ) or cardTypeGuard( mode, token )

   def generate( self, name ):
      expression_ = name
      data_ = { name : SlotExpressionFactory.slotMatcher_ }
      names = []
      for k in ( 'Supervisor', 'Linecard', 'Fabric', 'Switchcard' ):
         kwName = name + '_' + k.lower()
         slotName = name + '_' + k.upper()
         expression_ += '| ( %s %s )' % ( kwName, slotName )
         data_[ kwName ] = CliCommand.guardedKeyword(
            k, "%s modules" % k, SlotExpressionFactory.modularAndCardTypeGuard )
         data_[ slotName ] = SlotMatcher( forceTags=[ k ] )
         names.append( slotName )

      class _SlotExpression( CliCommand.CliExpression ):
         expression = expression_
         data = data_
         slotNames_ = names

         @staticmethod
         def adapter( mode, args, argsList ):
            for n in _SlotExpression.slotNames_:
               val = args.pop( n, None )
               if val is not None:
                  if isinstance( val, list ):
                     # the expression is used in a list
                     args.setdefault( name, [] )
                     args[ name ].extend( val )
                  else:
                     args[ name ] = val
                     # only one arg is possible, just stop
                     break
      return _SlotExpression

def getCardsFromIdList( mode, idList, tagLong ):
   '''
   idList is a list of card Ids ( labels ) such as [ 3, 4, 5, 7 ]
   tagLong could be 'Supervisor', 'Linecard', 'Fabric' or '' ( as in the case
   of commands without tags such as 'power enable module 3' )
   '''
   return [ Card( mode, '%s%s' % ( tagLong, cardId ), gv.entmib.root ) \
            for cardId in idList ]

def modularRprActiveSupeGuard( mode, token ):
   if gv.redundancyConfig:
      if gv.redundancyConfig.protocol == "rpr":
         return CommonGuards.standbyGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def cardTypeOrModularRprActiveSupeGuard( model, token ):
   return cardTypeGuard( model, token ) or modularRprActiveSupeGuard( model, token )

def rangeFn( tagLong ):
   labels = _tagLabels( gv.entmib.root, tagLong )
   matchingLabels = []
   if tagLong == '':
      if "Linecard" in labels:
         matchingLabels += labels[ "Linecard" ]
      if "Supervisor" in labels:
         matchingLabels += labels[ "Supervisor" ]
   else:
      if tagLong in labels:
         matchingLabels += labels[ tagLong ]
   if matchingLabels:
      matchingLabels = [ int( label ) for label in matchingLabels ]
      return ( min( matchingLabels ), max( matchingLabels ) )
   else:
      return ( 0, 0 )

class PowerSupplyMatcher( CliMatcher.Matcher ):
   """Matches a power supply token in the form of a number.
   If fixedConfig is specified, the command works on both modular
   and fixed config switches, otherwise only on modular.
   """
   # regexs for matching
   labelRegEx = re.compile( '^([0-9]+)$' )

   def __init__( self, fixedConfig=True, **kargs ):
      super( PowerSupplyMatcher, self ).__init__( **kargs )
      self.fixedConfig_ = fixedConfig

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return CliParserCommon.noMatch

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
         m = self.labelRegEx.match( token )
         if m:
            # ok, the token is a number
            label = m.group( 1 )
            for powersupply in root.powerSupplySlot.values():
               if powersupply.label == label:
                  return CliParserCommon.MatchResult( powersupply, token )
            return CliParserCommon.noMatch
         else:
            return CliParserCommon.noMatch
      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )
      return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return []

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):

         # just return the number range
         m = self.labelRegEx.match( token )
         if m or not token:
            # Return completion for a partially input number. If the number is
            # None or falls into the valid range, add the range a non-literal
            # completion.
            completions = []

            values = root.powerSupplySlot.keys()
            minLabel = min( values )
            maxLabel = max( values )

            number = int( m.group( 1 ) ) if m else None

            if number is None or \
                   CliParser.isPrefixOfValueInRange( number, minLabel, maxLabel ):
               completions.append( CliParser.Completion( '<%d-%d>'
                                                         %( minLabel, maxLabel ),
                                                         "Power Supply Number",
                                                         literal=False ) )
            return completions

         # nothing matches
         return []
      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )

class FanTrayMatcher( CliMatcher.Matcher ):
   """Matches a fan tray token in the form of a number.
   If fixedConfig is specified, the command works on both modular
   and fixed config switches, otherwise only on modular.
   """
   # regexs for matching
   labelRegEx = re.compile( '^([0-9]+)$' )

   def __init__( self, fixedConfig=True, **kargs ):
      super( FanTrayMatcher, self ).__init__( **kargs )
      self.fixedConfig_ = fixedConfig

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return CliParserCommon.noMatch

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
         m = self.labelRegEx.match( token )
         if m:
            # ok, the token is a number
            label = m.group( 1 )
            for fantray in root.fanTraySlot.values():
               if fantray.label == label:
                  return CliParserCommon.MatchResult( fantray, token )
            return CliParserCommon.noMatch
      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )
      return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return []

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
         # just return the number range
         m = self.labelRegEx.match( token )
         if m or not token:
            # Return completion for a partially input number. If the number is
            # None or falls into the valid range, add the range a non-literal
            # completion.
            completions = []

            values = root.fanTraySlot.keys()
            minLabel = min( values )
            maxLabel = max( values )

            number = int( m.group( 1 ) ) if m else None

            if number is None or \
                   CliParser.isPrefixOfValueInRange( number, minLabel, maxLabel ):
               completions.append( CliParser.Completion( '<%d-%d>'
                                                         %( minLabel, maxLabel ),
                                                         "Fan Tray Number",
                                                         literal=False ) )
            return completions

         # nothing matches
         return []
      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )

class ModuleExpression( CliCommand.CliExpression ):
   expression = 'MODULE | ( PowerSupply POWER_SUPPLY )'
   data = {
      'MODULE' : SlotExpressionFactory(),
      'PowerSupply' : CliCommand.guardedKeyword( 'PowerSupply',
                                                 'Power Supply modules',
                                                 guard=powerSupplyGuard ),
      'POWER_SUPPLY' : PowerSupplyMatcher(),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args.pop( 'PowerSupply', None )
      ps = args.pop( 'POWER_SUPPLY', None )
      if ps is not None:
         args[ 'MODULE' ] = ps
         return

class SwitchOrModuleExpression( CliCommand.CliExpression ):
   expression = 'Switch | Chassis | MODULE'
   data = {
      'Switch' : CliCommand.guardedKeyword( 'Switch', 'The switch',
                                            guard=fixedSystemGuard ),
      'Chassis' : CliCommand.guardedKeyword( 'Chassis', 'The chassis',
                                             guard=modularSystemGuard ),
      'MODULE' : ModuleExpression,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      for key in ( 'Switch', 'Chassis' ):
         module = args.pop( key, None )
         if module is not None:
            args[ 'MODULE' ] = gv.entmib.root
            break

def _strip( s ):
   if not s:
      return ""
   return s.strip().encode( 'string_escape' )

def _safe( s ):
   # escape any weird non-printable characters that may have come out
   # of a corrupted serial eeprom on some device
   return s.encode( 'string_escape' )

showInvCallbackList = []
def registerShowInventoryCallback( fn ):
   showInvCallbackList.append( fn )

def invokeShowInventoryCallbacks( mode, model ):
   entMib = mode.entityManager.lookup( 'hardware/entmib' )
   for fn in showInvCallbackList:
      fn( entMib, model )

def _populatePowerSupplyInfo( powerSupplySlots, model ):
   for ( i, slot ) in sorted( powerSupplySlots.iteritems() ):
      ps = slot.powerSupply
      if ps:
         serialNum = _safe( ps.serialNum ) or "Unknown"
         name = _safe( ps.modelName ) or "Unknown"
         model.powerSupplySlots[ i ] = FruModel.PowerSupplySlot( 
               name=name, serialNum=serialNum )
      else:
         model.powerSupplySlots[ i ] = FruModel.PowerSupplySlot()
   
def _populateFanTrayInfo( fanTraySlots, model ):
   for ( i, fanTraySlot ) in sorted( fanTraySlots.iteritems() ):
      fanTray = fanTraySlot.fanTray
      if fanTray:
         serialNum = _safe( fanTray.serialNum ) or "Unknown"
         name = _safe( fanTray.modelName ) or "Unknown"
         fans = len( fanTray.fan )
         model.fanTraySlots[ i ] = FruModel.FanTraySlot( 
            numFans=fans, name=name, serialNum=serialNum )
      else:
         model.fanTraySlots[ i ] = FruModel.FanTraySlot()

def _populatePortInfo( portSlotDir, model ):
   roleCounts = {}
   model.portCount = sum( len( port ) for port in portSlotDir.itervalues() )
   for cardName in portSlotDir:
      port = portSlotDir[ cardName ]
      for p in port.values():
         if p.role:
            if p.role == "Internal":
               # BUG5354 workaround - do not display internal interfaces
               model.portCount -= 1
            else:
               if p.tag.startswith( "UnconnectedEthernet" ):
                  assert p.role == 'Switched' or p.role == 'SwitchedBootstrap'
                  roleCounts[ 'Unconnected' ] = \
                        roleCounts.setdefault( 'Unconnected', 0 ) + 1
               else:
                  roleCounts[ p.role ] = roleCounts.setdefault( p.role, 0 ) + 1
   for ( portType, portCount ) in roleCounts.iteritems():
      # Convert only the 1st alphabet of the word into lower case
      firstCharToLower = lambda s : s[ :1 ].lower() + s[ 1: ]
      portName = "%sPortCount" % firstCharToLower( portType )
      setattr( model, portName, portCount )

def _populateXcvrInfo( xcvrSlotDir, model ):
   pos = 0
   for cardName in sorted( xcvrSlotDir ):
      xcvrSlots = xcvrSlotDir[ cardName ]
      for i in sorted( xcvrSlots ):
         pos += 1
         xcvr = xcvrSlots[ i ].xcvr
         # For modular, prepend card number and "/"
         xcvrName = "%s/%s" % ( cardName, i ) if cardName else str( i )
         if xcvr:
            model.xcvrSlots[ xcvrName ] = FruModel.XcvrSlot( 
                  mfgName=_strip( xcvr.mfgName ), 
                  modelName=_strip( xcvr.modelName ),
                  serialNum=_strip( xcvr.serialNum ), 
                  hardwareRev=_strip( xcvr.hardwareRev ) )
         else:
            model.xcvrSlots[ xcvrName ] = FruModel.XcvrSlot()
         model.xcvrSlots[ xcvrName ]._pos = pos
   
def _populateCardInfo( cardSlots, model ):
   for i in cardSlots:
      slot = cardSlots[ i ]
      name = "%s%s" % ( slot.tag, slot.label )
      card = slot.card
      if card:
         # Until we have RedSup SSO the standby supervisor is shown as 'Inserted'
         # with the rest of the fields blank
         if slot.tag == 'Supervisor' and _safe( card.modelName ) == 'Unknown':
            model.cardSlots[ name ] = FruModel.CardSlot( modelName="Inserted" )
         else:
            hwEpoch = gv.epochStatus.card.get( name ).hwEpoch if \
                      gv.epochStatus.card.get( name ) else _safe( card.hwEpoch )
            model.cardSlots[ name ] = FruModel.CardSlot( 
                  modelName=_safe( card.modelName ), 
                  hardwareRev=_safe( card.hardwareRev ),
                  serialNum=_safe( card.serialNum ), 
                  mfgDate=_safe( card.mfgDate ),
                  hwEpoch=_safe( hwEpoch ) )
      else:
         model.cardSlots[ name ] = FruModel.CardSlot()
      model.cardSlots[ name ]._pos = i

def doShowModularInventory( chassis, model ):
   _populateCardInfo( chassis.cardSlot, model )
   # BUG5354: as-is, this does not count the management ports on the
   #          redundant supervisor, but does count (and displays) the
   #          "Internal" ports.
   # We currently work around the "Internal" ports. We cannot fix the
   # redundant supervisor part until BUG9772 is fixed.

   portSlotDir = {}
   for cardSlotId in chassis.cardSlot:
      if chassis.cardSlot[ cardSlotId ].card:
         portSlotDir[ cardSlotId ] = chassis.cardSlot[ cardSlotId ].card.port
   _populatePortInfo( portSlotDir, model )

   xcvrSlotDir = {}
   for cardSlotId in chassis.cardSlot:
      # This assert is necessary, because the code that prints out the
      # xcvrs prefixes xcvrIds with "cardId/" when this is non-false.
      assert cardSlotId
      if chassis.cardSlot[ cardSlotId ].card:
         xcvrSlotDir[ cardSlotId ] = chassis.cardSlot[ cardSlotId ].card.xcvrSlot
   _populateXcvrInfo( xcvrSlotDir, model )

def doShowFixedInventory( fs, model ):
   _populatePortInfo( { '' : fs.port }, model )
   _populateXcvrInfo( { '' : fs.xcvrSlot }, model )

def doShowPrecisionClock( fs, model ):
   if fs.modelName:
      if re.search( '-?CL-?', fs.modelName ):
         model.precisionClock = FruModel.Clock()

def doShowInventory( mode, args ):
   model = FruModel.Inventory()
   sys = mode.entityManager.lookup( 'hardware/entmib' ).fixedSystem or \
       mode.entityManager.lookup( 'hardware/entmib' ).chassis
   cardStatus = gv.epochStatus.card.get( 'FixedSystem' ) or \
       gv.epochStatus.card.get( 'Chassis' )
   if sys:
      model.systemInformation = FruModel.SystemInformation( 
         name=_safe( sys.modelName ), description=_safe( sys.description ),
         serialNum=_safe( sys.serialNum ), hardwareRev=_safe( sys.hardwareRev ),
         mfgDate=_safe( sys.mfgDate ),
         hwEpoch=_safe( cardStatus.hwEpoch if cardStatus else '' ) )
      _populatePowerSupplyInfo( sys.powerSupplySlot, model )
      _populateFanTrayInfo( sys.fanTraySlot, model )
      if mode.entityManager.lookup( 'hardware/entmib' ).fixedSystem:
         doShowFixedInventory( sys, model )
      elif mode.entityManager.lookup( 'hardware/entmib' ).chassis:
         doShowModularInventory( sys, model )
      # Invoke Registered Callback functions
      invokeShowInventoryCallbacks( mode, model )
      doShowPrecisionClock( sys, model )
   else:
      model.systemInformation = FruModel.SystemInformation()
   return model

def prepareSlotRule( entityManager ):
   mg = entityManager.mountGroup()
   gv.entmib = mg.mount( "hardware/entmib", "EntityMib::Status", "r" )
   gv.epochStatus = mg.mount( 'hwEpoch/status', 'HwEpoch::Status', 'r' )
   def _mountsComplete():
      global systemTypeReactorInitialized_
      systemTypeReactorInitialized_ = True
      entityManager.systemTypeReactor = SystemTypeReactor( gv.entmib )
   mg.close( _mountsComplete )

#--------------------------------------------------------------------------------
# show inventory
#--------------------------------------------------------------------------------
class InventoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show inventory'
   data = {
      'inventory': 'Hardware components of the system',
   }
   handler = doShowInventory
   cliModel = Inventory
   privileged = False

BasicCli.addShowCommandClass( InventoryCmd )

#------------------------------------------------------------------------------------
# [no|default] system mac-address <mac_addr>
#------------------------------------------------------------------------------------
def setMacAddress( mode, args ):
   macAddr = args.get( 'MACADDR', EthAddr.ethAddrZero )
   if not compareMacs( macAddr, hwEntMib.systemMacAddr ):
      mode.addWarning( 
              "Change will take effect only after switch reboot." )
   macAddrConfig.macAddr = macAddr

class SystemMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'system mac-address MACADDR'
   noOrDefaultSyntax = 'system mac-address ...'
   data = {
      'system': systemMatcherForConfig,
      'mac-address': CliCommand.guardedKeyword( 'mac-address', 
                                                'Configure MAC address',
                                                guard=fixedSystemGuard ),
      'MACADDR': macAddrMatcher,
   }
   handler = setMacAddress
   noOrDefaultHandler = handler

if toggleOverrideSystemMacEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( SystemMacAddressCmd )

#--------------------------------------------------
# Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global macAddrConfig, hwEntMib
   gv.powerFuseConfig = ConfigMount.mount( entityManager,
                                           "power/fuse/config/admin",
                                           "Power::SoftwareFuse", "w" )
   gv.redundancyConfig = LazyMount.mount( entityManager, "redundancy/config",
                                          "Redundancy::RedundancyConfig", "r" )
   macAddrConfig = ConfigMount.mount( entityManager,
                                      "sys/macAddress/config",
                                      "Fru::MacAddressConfig", "w" )
   hwEntMib = LazyMount.mount( entityManager,
                               "hardware/entmib", 
                               "EntityMib::Status", "r" )

   # Call and set global variable _slotId once
   Fru.slotId()
   # Call and set global variable fruUtil_ once
   Fru.fruUtil()

   prepareSlotRule( entityManager )
