# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliMatcher

import CliPlugin.ForwardingDestinationHelper as ForwardingDestinationHelper
import CliPlugin.IpAddrMatcher
import CliPlugin.Ip6AddrMatcher
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.PhysicalIntfRule import PhysicalIntfMatcher
from CliPlugin.PromptTree import PromptTreeBase, PromptTreeField, PromptTreeChoice

matchers = {
   '<ingressIntf>': PhysicalIntfMatcher( 'Ethernet' ),
   '<rawPacket>': CliMatcher.StringMatcher( helpdesc='Hexadecimal String',
                                            pattern=r'[a-fA-F0-9]+' ),
   '<srcMac>': MacAddrMatcher( helpdesc='MAC address of the source' ),
   '<dstMac>': MacAddrMatcher( helpdesc='MAC address of the destination' ),
   '<etherType>': CliMatcher.IntegerMatcher( 0, 0xffff, helpdesc='Ethertype' ),
   '<vlan>': CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Identifier for a Virtual LAN' ),
   '<innerVlan>': CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Identifier for a Virtual LAN' ),
   '<srcIpv4>': CliPlugin.IpAddrMatcher.IpAddrMatcher( 'Source IPv4 Address' ),
   '<dstIpv4>': CliPlugin.IpAddrMatcher.IpAddrMatcher( 'Destination IPv4 Address' ),
   '<ipTtl>': CliMatcher.IntegerMatcher( 1, 0xff, helpdesc='IP TTL' ),
   '<ipProto>': CliMatcher.IntegerMatcher( 0, 0xff, helpdesc='IP Protocol' ),
   '<srcL4Port>': CliMatcher.IntegerMatcher( 0, 0xffff,
                                             helpdesc='Source TCP/UDP Port' ),
   '<dstL4Port>': CliMatcher.IntegerMatcher( 0, 0xffff,
                                             helpdesc='Destination TCP/UDP Port' ),
   '<srcIpv6>': CliPlugin.Ip6AddrMatcher.Ip6AddrMatcher( 'Source IPv6 Address' ),
   '<dstIpv6>': CliPlugin.Ip6AddrMatcher.Ip6AddrMatcher(
         'Destination IPv6 Address' ),
   '<hopLimit>': CliMatcher.IntegerMatcher( 1, 0xff, helpdesc='IPv6 Hop Limit' ),
   '<nextHeader>': CliMatcher.IntegerMatcher( 0, 0xff,
                                              helpdesc='IPv6 Next Header' ),
   '<flowLabel>': CliMatcher.IntegerMatcher( 0, 0xfffff,
                                             helpdesc='IPv6 Flow Label' ),
}

class L4PromptTree( PromptTreeBase ):
   def __init__( self ):
      super( L4PromptTree, self ).__init__()
      for field in ForwardingDestinationHelper.L4Fields:
         ptField = PromptTreeField( field, matchers[ field ],
                                    ForwardingDestinationHelper.ArgToLabel[ field ] )
         self.fields.append( ptField )

class Ipv4PromptTree( PromptTreeBase ):
   def __init__( self ):
      super( Ipv4PromptTree, self ).__init__()
      for field in ForwardingDestinationHelper.IPv4Fields:
         ptField = PromptTreeField( field, matchers[ field ],
                                    ForwardingDestinationHelper.ArgToLabel[ field ] )
         # <ipProto> is the only exception, as we don't prompt for it when the L4
         # type is UDP or TCP
         if field == '<ipProto>':
            ptField.promptLambda = \
                  lambda treeDict: treeDict[ '<l4Type>' ] not in [ 'tcp', 'udp' ]
         self.fields.append( ptField )
      self.furtherPromptTrees = [
         ( L4PromptTree(),
           lambda treeDict: treeDict[ '<l4Type>' ] in [ 'tcp', 'udp' ] ),
      ]

class Ipv6PromptTree( PromptTreeBase ):
   def __init__( self ):
      super( Ipv6PromptTree, self ).__init__()
      for field in ForwardingDestinationHelper.IPv6Fields:
         ptField = PromptTreeField( field, matchers[ field ],
                                    ForwardingDestinationHelper.ArgToLabel[ field ] )
         # <nextHeader> is the only exception, as we don't prompt for it when the
         # L4 type is UDP or TCP
         if field == '<nextHeader>':
            ptField.promptLambda = \
                  lambda treeDict: treeDict[ '<l4Type>' ] not in [ 'tcp', 'udp' ]
         self.fields.append( ptField )
      self.furtherPromptTrees = [
         ( L4PromptTree(),
           lambda treeDict: treeDict[ '<l4Type>' ] in [ 'tcp', 'udp' ] ),
      ]

class RawPacketPromptTree( PromptTreeBase ):
   def __init__( self ):
      super( RawPacketPromptTree, self ).__init__()
      self.fields = [
         PromptTreeField(
            '<rawPacket>', matchers[ '<rawPacket>' ],
            ForwardingDestinationHelper.ArgToLabel[ '<rawPacket>' ] ),
      ]

class PacketPromptTree( PromptTreeBase ):
   def __init__( self, rawPacketSupported ):
      super( PacketPromptTree, self ).__init__()
      packetTypes = [ 'ethernet', 'ipv4', 'ipv6' ]
      if rawPacketSupported:
         packetTypes.append( 'raw' )

      def isEthernet( treeDict ):
         return treeDict[ '<packetType>' ] == 'ethernet'

      def isRaw( treeDict ):
         return treeDict[ '<packetType>' ] == 'raw'

      def isIp( treeDict ):
         return treeDict[ '<packetType>' ] in [ 'ipv4', 'ipv6' ]

      self.fields = [
         PromptTreeField(
            '<ingressIntf>', matchers[ '<ingressIntf>' ],
            ForwardingDestinationHelper.ArgToLabel[ '<ingressIntf>' ] ),
         PromptTreeChoice( '<packetType>', 'Packet type', packetTypes ),
         PromptTreeChoice(
            '<l4Type>', 'L4 header', [ 'none', 'tcp', 'udp' ],
            promptLambda=isIp ),
         PromptTreeField(
            '<srcMac>', matchers[ '<srcMac>' ],
            ForwardingDestinationHelper.ArgToLabel[ '<srcMac>' ],
            promptLambda=lambda treeDict: not isRaw( treeDict ) ),
         PromptTreeField(
            '<dstMac>', matchers[ '<dstMac>' ],
            ForwardingDestinationHelper.ArgToLabel[ '<dstMac>' ],
            promptLambda=lambda treeDict: not isRaw( treeDict ) ),
         PromptTreeField(
            '<etherType>', matchers[ '<etherType>' ],
            ForwardingDestinationHelper.ArgToLabel[ '<etherType>' ],
            promptLambda=isEthernet ),
         PromptTreeField(
            '<vlan>', matchers[ '<vlan>' ],
            ForwardingDestinationHelper.ArgToLabel[ '<vlan>' ],
            required=False,
            promptLambda=lambda treeDict: not isRaw( treeDict ) ),
         PromptTreeField(
            '<innerVlan>', matchers[ '<innerVlan>' ],
            ForwardingDestinationHelper.ArgToLabel[ '<innerVlan>' ],
            required=False,
            promptLambda=lambda treeDict : treeDict.get( '<vlan>' ) ),
      ]
      self.furtherPromptTrees = [
         ( Ipv4PromptTree(),
           lambda treeDict: treeDict[ '<packetType>' ] == 'ipv4' ),
         ( Ipv6PromptTree(),
           lambda treeDict: treeDict[ '<packetType>' ] == 'ipv6' ),
         ( RawPacketPromptTree(), isRaw ),
      ]
