#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import threading

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.ForwardingDestinationHelper as ForwardingDestinationHelper
import CliPlugin.ForwardingDestinationPromptTree as ForwardingDestinationPromptTree
from CliPlugin.PromptTree import FieldException
import CliToken
import ConfigMount
import LazyMount
import ShowCommand
import Tac

from Ark import synchronized
from Arnet import intfNameKey
from CliModel import Enum, List, Model
from IntfModels import Interface

# Client name for packettracer config/status in sysdb
CLI_CLIENT_NAME = 'TracerCli'

RequestId = Tac.Type( 'PacketTracer::RequestId' )
State = Tac.Type( 'PacketTracer::State' )

# Keeps track of the most recently used ID for Sysdb request entries
currentId = None
idLock = threading.Lock()

clientConfig = None
clientStatus = None
packetTracerHwStatus = None
packetTracerSwStatus = None

# -----------------------------------------------------------------------------------
# CLI models
# -----------------------------------------------------------------------------------
class EgressInterface( Model ):
   logicalEgressInterface = Interface( help="Logical egress interface" )
   physicalEgressInterface = Interface( help="Physical egress interface" )

class PacketDestination( Model ):
   status = Enum( values=[ 'success', 'timeout' ],
                  help="Status of the request" )
   egressInterfaces = List( help="List of egress interfaces",
                            valueType=EgressInterface )

# -----------------------------------------------------------------------------------
# Helper functions, CLI guards, etc.
# -----------------------------------------------------------------------------------
@synchronized( idLock )
def getNewRequestId():
   global currentId
   if currentId is None:
      # If ConfigAgent crashes between the CLI sending a request to the PD agent and
      # the PD agent sending a response the request/response will be left dangling in
      # Sysdb. To avoid this, we clear all requests each time we start up. This helps
      # us avoid re-using a previously used id.
      clientConfig.request.clear()
      currentId = 0
   currentId += 1
   return RequestId( clientName=CLI_CLIENT_NAME, id=currentId )

def generateRequest():
   requestId = getNewRequestId()
   clientConfig.newRequest( requestId )

   return clientConfig.request[ requestId ]

def sendRequest( request ):
   requestId = request.requestId
   request.valid = True

   try:
      Tac.waitFor( lambda: ( requestId in clientStatus.result and
                             clientStatus.result[ requestId ].valid ),
                   sleep=True, timeout=packetTracerHwStatus.timeout, warnAfter=None )
   except Tac.Timeout:
      result = None
   else:
      # Fetch result
      result = clientStatus.result[ requestId ]

   # Clear out the request
   del clientConfig.request[ requestId ]

   return result

def handleResult( mode, result ):
   if result.state == State.success:
      def filterFunc( intf ):
         if ( intf.startswith( 'Ethernet' ) or intf.startswith( 'Vlan' ) or
              intf.startswith( 'Port-Channel' ) ):
            return intfNameKey( intf )
         else:
            # Not all interfaces fit the format intfNameKey expects. As an example
            # L3FloodFap0.0 on Sand platforms.
            return intf

      # Generate a map of physical egress intf -> set of logical egress intfs
      egressIntfs = {}
      for resultEntry in result.egressIntf:
         if resultEntry.physicalIntf not in egressIntfs:
            egressIntfs[ resultEntry.physicalIntf ] = set()
         if ( resultEntry.logicalIntf and
              ( resultEntry.logicalIntf != resultEntry.physicalIntf ) ):
            egressIntfs[ resultEntry.physicalIntf ].add( resultEntry.logicalIntf )

      # Generate output of egress intfs and their associated logical egress intfs
      combined = []
      for egressIntf in sorted( egressIntfs, key=filterFunc ):
         logicalIntfs = sorted( egressIntfs[ egressIntf ], key=filterFunc )
         newLine = egressIntf
         if logicalIntfs:
            newLine += ' (' + ', '.join( logicalIntfs ) + ')'
         combined.append( newLine )

      print( 'Egress interface(s): ' + ', '.join( combined ) )
   elif result.state == State.invalidRequest:
      mode.addError( 'The request you made is invalid. Clearing packet '
                     'configuration.' )
      mode.session.sessionDataIs( 'PacketTracer.Packet', None )
      return None
   elif result.state == State.timeout:
      print( 'The packet was not forwarded' )
      return PacketDestination( status='timeout' )
   elif result.state == State.error:
      mode.addError( 'An error occurred with the request. Clearing packet '
                     'configuration.' )
      mode.session.sessionDataIs( 'PacketTracer.Packet', None )
      return None

   model = PacketDestination( status='success' )
   for resultEntry in result.egressIntf:
      ei = EgressInterface()
      ei.logicalEgressInterface = resultEntry.logicalIntf
      ei.physicalEgressInterface = resultEntry.physicalIntf
      model.egressInterfaces.append( ei )
   return model

def forwardingDestinationCliGuard( mode, token ):
   if packetTracerHwStatus.tracingSupported:
      return None
   return CliParser.guardNotThisPlatform

def rawPacketCliGuard( mode, token ):
   if packetTracerHwStatus.rawPacketSupported:
      return None
   return CliParser.guardNotThisPlatform

def keywordWithMaxMatches( keyword, helpdesc, maxMatches=1, guard=None ):
   return CliCommand.Node(
         CliMatcher.KeywordMatcher( keyword=keyword, helpdesc=helpdesc ),
         maxMatches=maxMatches,
         guard=guard )

# -----------------------------------------------------------------------------------
# show forwarding destination
# -----------------------------------------------------------------------------------
keywords = {
   'forwarding': 'Show forwarding information',
   'destination': keywordWithMaxMatches(
      'destination',
      helpdesc='Show packet forwarding destinations',
      guard=forwardingDestinationCliGuard ),
   'ingress-interface': keywordWithMaxMatches( 'ingress-interface',
                                               helpdesc='Ingress Interface' ),
   'edit': 'Edit the configured packet',
   'src-mac': keywordWithMaxMatches( 'src-mac',
                                     helpdesc='MAC address of the source' ),
   'dst-mac': keywordWithMaxMatches( 'dst-mac',
                                     helpdesc='MAC address for the destination' ),
   'eth-type': keywordWithMaxMatches( 'eth-type', helpdesc='Ethertype' ),
   'vlan': keywordWithMaxMatches( 'vlan', helpdesc='Identifier for a Virtual LAN' ),
   'inner-vlan': keywordWithMaxMatches( 'inner-vlan', helpdesc='Inner VLAN ID' ),
   'src-ipv4': keywordWithMaxMatches( 'src-ipv4',
                                      helpdesc='Source IPv4 Address' ),
   'dst-ipv4': keywordWithMaxMatches( 'dst-ipv4',
                                      helpdesc='Destination IPv4 Address' ),
   'ip-ttl': keywordWithMaxMatches( 'ip-ttl', helpdesc='IP TTL' ),
   'ip-protocol': keywordWithMaxMatches( 'ip-protocol', helpdesc='IP Protocol' ),
   'src-l4-port': keywordWithMaxMatches( 'src-l4-port',
                                         helpdesc='L4 Source Port' ),
   'dst-l4-port': keywordWithMaxMatches( 'dst-l4-port',
                                         helpdesc='L4 Destination Port' ),
   'src-ipv6': keywordWithMaxMatches( 'src-ipv6',
                                      helpdesc='Source IPv6 Address' ),
   'dst-ipv6': keywordWithMaxMatches( 'dst-ipv6',
                                      helpdesc='Destination IPv6 Address' ),
   'hop-limit': keywordWithMaxMatches( 'hop-limit', helpdesc='IPv6 Hop Limit' ),
   'next-header': keywordWithMaxMatches( 'next-header',
                                         helpdesc='IPv6 Next Header' ),
   'flow-label': keywordWithMaxMatches( 'flow-label',
                                        helpdesc='IPv6 Flow Label' ),
}

# The keyword matchers are defined in ForwardingDestinationPromptTree
keywords.update( ForwardingDestinationPromptTree.matchers )
del keywords[ '<rawPacket>' ]

def commandHandler( mode, args ):
   # Due to the use of iteration rules most arguments will be in list form. It's
   # not expected that any of them be an actual list longer than one element due
   # to using maxMatches=1. Go through each arg and change lists to be the value
   # itself.
   for key in args:
      if isinstance( args[ key ], ( list, tuple ) ):
         args[ key ] = args[ key ][ 0 ]

   treeDict, newConfiguration = \
         ForwardingDestinationHelper.fetchConfiguredPacket( mode )

   # Update packet fields with provided args. Also generate a user readable list
   # of all updated fields.
   updatedAttributes = []
   for key in args:
      if key in ForwardingDestinationHelper.ArgToLabel:
         updatedAttributes.append( ForwardingDestinationHelper.ArgToLabel[ key ] )
         treeDict[ key ] = args[ key ]

   # Determine packet type and L4 type
   packetType, l4Type = ForwardingDestinationHelper.generatePacketTypes(
         treeDict )
   treeDict[ '<packetType>' ] = packetType
   treeDict[ '<l4Type>' ] = l4Type

   # Check if we should prompt for a raw packet
   rawPacketPrompt = packetTracerHwStatus.rawPacketSupported
   if not isinstance( mode, BasicCliModes.EnableMode ):
      rawPacketPrompt = False

   if mode.session.isInteractive():
      # Prompt for all other required fields
      pt = ForwardingDestinationPromptTree.PacketPromptTree( rawPacketPrompt )
      originalTreeDict = treeDict.copy()
      reprompt = 'edit' in args
      try:
         pt.prompt( mode, treeDict, reprompt )
      except FieldException as e:
         mode.addError( e.message )
         return None

      for k in treeDict:
         if k in [ '<packetType>', '<l4Type>' ]:
            continue
         if ( k not in originalTreeDict or
              originalTreeDict[ k ] != treeDict[ k ] ):
            updatedAttributes.append(
                  ForwardingDestinationHelper.ArgToLabel[ k ] )
   else:
      if ForwardingDestinationHelper.checkForMissingFields(
            mode,
            ForwardingDestinationHelper.generateRequiredFields(
               packetType, l4Type ),
            treeDict ):
         return None

   if treeDict[ '<packetType>' ] == 'raw':
      # We don't store any information when it is a raw packet
      ForwardingDestinationHelper.clearConfiguredPacket( mode )
   else:
      # The user can opt to run the command without any parameters, which will
      # just re-use the existing config. In this case no attributes are updated.
      if updatedAttributes:
         if newConfiguration:
            print( 'Saved packet configuration' )
         else:
            print( 'Updated packet configuration with new field(s): ' +
                   ', '.join( updatedAttributes ) )

   # Update ethertype and L4 type based on packet type
   ForwardingDestinationHelper.updateEthertypeAndProtocol( treeDict )

   # Validate fields
   if not ForwardingDestinationHelper.validateFields( mode, treeDict,
                                                      packetTracerHwStatus,
                                                      packetTracerSwStatus ):
      # If validateFields returns False, it has added an error to the mode so we
      # do not need to return a model here.
      return None

   # Generate request in Sysdb and update it with provided fields, then print it out
   request = generateRequest()
   request = ForwardingDestinationHelper.updateRequest( request, treeDict )
   ForwardingDestinationHelper.printRequest( request )

   # Send request
   result = sendRequest( request )
   if not result:
      print( 'The packet was not forwarded' )
      return PacketDestination( status='timeout' )

   return handleResult( mode, result )

class ShowForwardingDestinationCommand( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show forwarding destination [ edit ] '
              '[ ( { ( ingress-interface <ingressIntf> ) |'
              '( src-mac <srcMac> ) | ( dst-mac <dstMac> ) | '
              '( eth-type <etherType> ) | ( vlan <vlan> ) | '
              '( inner-vlan <innerVlan> ) | ( src-ipv4 <srcIpv4> ) | '
              '( dst-ipv4 <dstIpv4> ) | ( ip-ttl <ipTtl> ) | '
              '( ip-protocol <ipProto> ) | ( src-ipv6 <srcIpv6> ) | '
              '( dst-ipv6 <dstIpv6> ) | ( hop-limit <hopLimit> ) | '
              '( next-header <nextHeader> ) | ( flow-label <flowLabel> ) | '
              '( src-l4-port <srcL4Port> ) | ( dst-l4-port <dstL4Port> ) '
              '} ) ]' )
   data = keywords
   cliModel = PacketDestination
   noMore = True

   @staticmethod
   def handler( mode, args ):
      return commandHandler( mode, args )

BasicCli.addShowCommandClass( ShowForwardingDestinationCommand )

class ShowRawForwardingDestinationCommand( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show forwarding destination ingress-interface <ingressIntf> '
              'raw <rawPacket>' )
   data = {
      'forwarding': 'Show forwarding information',
      'destination': keywordWithMaxMatches(
         'destination',
         helpdesc='Show packet forwarding destinations',
         guard=forwardingDestinationCliGuard ),
      'ingress-interface': keywordWithMaxMatches( 'ingress-interface',
                                                  helpdesc='Ingress Interface' ),
      '<ingressIntf>': ForwardingDestinationPromptTree.matchers[ '<ingressIntf>' ],
      'raw': keywordWithMaxMatches( 'raw', helpdesc='Raw Packet',
                                    guard=rawPacketCliGuard ),
      '<rawPacket>': ForwardingDestinationPromptTree.matchers[ '<rawPacket>' ],
   }

   cliModel = PacketDestination
   privileged = True

   @staticmethod
   def handler( mode, args ):
      return commandHandler( mode, args )

BasicCli.addShowCommandClass( ShowRawForwardingDestinationCommand )

# -----------------------------------------------------------------------------------
# clear forwarding destination packet
# -----------------------------------------------------------------------------------
class ClearForwardingDestinationPacketCommand( CliCommand.CliCommandClass ):
   syntax = 'clear forwarding destination packet'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'forwarding': 'Forwarding information',
      'destination': keywordWithMaxMatches(
         'destination',
         helpdesc='Packet forwarding destination',
         guard=forwardingDestinationCliGuard ),
      'packet': 'Saved packet configuration',
   }

   @staticmethod
   def handler( mode, args ):
      ForwardingDestinationHelper.clearConfiguredPacket( mode )
      print( 'Cleared packet configuration' )

BasicCli.UnprivMode.addCommandClass( ClearForwardingDestinationPacketCommand )

def Plugin( entityManager ):
   global clientConfig, clientStatus, packetTracerHwStatus, packetTracerSwStatus

   packetTracerHwStatus = LazyMount.mount(
      entityManager,
      'packettracer/hwstatus',
      'PacketTracer::HwStatus', 'r' )
   packetTracerSwStatus = LazyMount.mount(
      entityManager,
      'packettracer/swstatus',
      'PacketTracer::SwStatus', 'r' )
   clientStatus = LazyMount.mount(
      entityManager,
      'packettracer/status/{}'.format( CLI_CLIENT_NAME ),
      'PacketTracer::ClientStatus', 'r' )
   clientConfig = ConfigMount.mount(
      entityManager,
      'packettracer/config/{}'.format( CLI_CLIENT_NAME ),
      'PacketTracer::ClientConfig', 'w' )
