# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import CliMatcher
import CliParser
import CliParserCommon
from CliPlugin.IpGenAddrMatcher import IpGenPrefixMatcher

class RSMatcher( CliMatcher.Matcher ):
   '''Matcher for a flowspec nlri string.'''
   # The opVal components are rendered as an optional series of "<tag>:<value>;"
   # substrings.  These will be split out and matched against this regex.
   # Each component consists of a tag from a finite set of tags, and the value
   # can be an expression consisting of values and operators.  We won't attempt to
   # validate the grammer of this expression in the cli.
   opValComponentRe = re.compile( r'(IP|P|DP|SP|IT|IC|TCP|LEN|DSCP|FRAG|FL):\S+$' )
   # A valid rule string must start with a hex digit or *, since the first
   # part is the dest prefix.  If the user has started entering something
   # else, it is no longer a possible rule string.
   ruleStringRe = re.compile( r'[A-Fa-f0-9\*]+.*' )

   def __init__( self, helpdesc, **kargs ):
      super( RSMatcher, self ).__init__( helpdesc=helpdesc,
            **kargs )
      self.ipGenPrefixMatcher_ = IpGenPrefixMatcher( helpdesc )
      self.completion_ = CliParser.Completion( 'STRING',
                                               'BGP Flowspec rule string',
                                               False )

   def match( self, mode, context, token ):
      # Split out the prefix components so we can validate them appropriately.
      # We'll strip any trailing delimiter first so we don't have to deal wth
      # an empty string at the end of the list.
      components = token.rstrip( ';' ).split( ';' )
      if len( components ) < 2:
         # There must be at least two components (the dest & src prefixes).
         return CliParserCommon.noMatch
      # First two fields are either '*' or a valid IpGenPrefix
      for prefix in components[ 0 : 2 ]:
         if prefix == '*':
            continue
         if prefix == '':
            return CliParserCommon.noMatch
         if( self.ipGenPrefixMatcher_.match( mode, context, prefix ) is
               CliParserCommon.noMatch ):
            return CliParserCommon.noMatch
      # The remaining tokens should match the opval component regex
      for opValComponent in components[ 2 : ]:
         m = self.opValComponentRe.match( opValComponent )
         if m is None:
            return CliParserCommon.noMatch
      # Success!
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      # A valid rule string must start with a hex digit or *, since the first
      # part is the dest prefix.  If the user has started entering something
      # else, it is no longer a possible rule string.
      if not token:
         return [ self.completion_ ]
      m = re.match( self.ruleStringRe, token )
      if m is None:
         return []
      else:
         return [ self.completion_ ]

   def __str__( self ):
      return 'BGP Flowspec rule string'

ruleStringMatcher = RSMatcher( 'BGP NLRI string' )
