#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

''' CLI commands for configuring Flowspec. '''

import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import IntfCli
import os
import Tac
import Tracing
from FlowspecCliLib import flowspecCapable

t0 = Tracing.trace0

cliConfig = None

#-------------------------------------------------------------------------------
# Flowspec Interface configuration command
# config-if
#    [ no|default ] flow-spec { ipv4 [ ipv6 ] | ipv6 }
#-------------------------------------------------------------------------------

flowSpecMatcherForEnable = CliMatcher.KeywordMatcher(
      'flow-spec', helpdesc='Enable flow-spec' )

ipv4MatcherForConfig = CliMatcher.KeywordMatcher(
      'ipv4', helpdesc="Match on IPv4 traffic" )
ipv6MatcherForConfig = CliMatcher.KeywordMatcher(
      'ipv6', helpdesc="Match on IPv6 traffic" )

#-------------------------------------------------------------------------------
# Adds flowspec specific CLI commands to the "config-if" mode for routed ports.
#-------------------------------------------------------------------------------
class FlowspecIntfConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return flowspecCapable( mode.intf.name )

#-------------------------------------------------------------------------------
# Associate the FlowspecIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( FlowspecIntfConfigModelet )

modelet = FlowspecIntfConfigModelet

#-------------------------------------------------------------------------------
# Register a dependent class for all interfaces to cleanup on Intf deletion
#-------------------------------------------------------------------------------
class FlowspecIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "Intf deletion. Cleanup", self.intf_.name )
      del cliConfig.enabledIntf[ self.intf_.name ]

def configureFlowspec( intfName, ipv4=None, ipv6=None ):
   t0( "intf", intfName, "ipv4", ipv4, "ipv6", ipv6 )
   curIpVersion = cliConfig.enabledIntf.get( intfName )
   newIpVersion = Tac.Value( 'Flowspec::IpVersion' )
   # Initialize the newly configured ip version to what's currently
   # available and override the current config with the args of
   # ipv4/ipv6 provided they are not None.
   if curIpVersion:
      newIpVersion.ipv4 = curIpVersion.ipv4
      newIpVersion.ipv6 = curIpVersion.ipv6
   if ipv4 is not None:
      newIpVersion.ipv4 = ipv4
   if ipv6 is not None:
      newIpVersion.ipv6 = ipv6

   if newIpVersion:
      t0( "Enabling", newIpVersion, "on", intfName )
      cliConfig.enabledIntf[ intfName ] = newIpVersion
   else:
      t0( "All Flowspec::IpVersion attributes are false. Delete", intfName )
      del cliConfig.enabledIntf[ intfName ]

#-------------------------------------------------------------------------------
# config-if# [ no | default ] flow-spec ipv4 ipv6
# config-if# [ no | default ] flow-spec ipv4
# config-if# [ no | default ] flow-spec ipv6
#-------------------------------------------------------------------------------

# When ipv4 (or ipv6) is enabled, the user only wants ipv4 (or ipv6) to be turned on
# and not ipv6 (or ipv4).
# When ipv4 (or ipv6) is disabled, the user only wants to turn off ipv4 (or ipv6)
# and use the current operational state of ipv6 (or ipv4).
def handleFlowspecIpv4Ipv6( mode, args, enable=True ):
   # if enable=False, this function has been called as the noOrDefaultHandler:
   # when not in the args, the value will be None
   disable = False if enable else None
   ipv4 = enable if 'ipv4' in args else disable
   ipv6 = enable if 'ipv6' in args else disable
   configureFlowspec( mode.intf.name, ipv4=ipv4, ipv6=ipv6 )

def disableFlowspecIpv4Ipv6( mode, args ):
   handleFlowspecIpv4Ipv6( mode, args, enable=False )

flowSpecSyntax = 'flow-spec ( ipv4 ipv6 )'
# Selective toggle of ipv4 is not supported yet.
if os.environ.get( 'FLOWSPEC_ENABLE_IPV4' ):
   flowSpecSyntax += ' | ipv4'
# Selective toggle of ipv6 is not supported yet.
if os.environ.get( 'FLOWSPEC_ENABLE_IPV6' ):
   flowSpecSyntax += ' | ipv6'

# When FLOWSPEC_ENABLE_IPV{4,6} are in env, the full syntax would be
# flow-spec ( ipv4 ipv6 ) | ipv4 | ipv6
class FlowSpecIpv4Ipv6Cmd( CliCommand.CliCommandClass ):
   syntax = flowSpecSyntax
   noOrDefaultSyntax = syntax

   data = {
      'flow-spec' : flowSpecMatcherForEnable,
      'ipv4' : ipv4MatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
   }
   handler = handleFlowspecIpv4Ipv6
   noOrDefaultHandler = disableFlowspecIpv4Ipv6

modelet.addCommandClass( FlowSpecIpv4Ipv6Cmd )

#------------------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager,
                                  "flowspec/cliConfig",
                                  "Flowspec::CliConfig", "w" )
   IntfCli.Intf.registerDependentClass( FlowspecIntf )
