# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import LazyMount
import Tac
import Tracing
from CliToken.Flow import flowMatcherForShow
from FlowTrackerCliUtil import (
   ftrConfigType,
   ftrTypes,
   ftrTypeSampled,
   ftrTypeHardware,
   getFtrTypeFromArgs,
   ftrConfigPathPrefix,
   hwFtrConfigPathPrefix,
   hwFtrStatusPathPrefix,
   ftrCapabilitiesPathPrefix,
   hwFtrConfigType,
   hwFtrStatusType,
   ftrCapabilitiesType,
   collectorInactiveReason,
   trackerInactiveReason,
   AddressFamily,
   ftrStatusPathPrefix,
   ftrStatusType,
   exporterInactiveReasonEnum,
   getExpReasonEnumList,
   reservedGroupToSeqnoMap,
)
from FlowTrackerConst import (
   activeInterval,
   inactiveTimeout,
   mirrorSelectorAlgo,
   constants,
)
from FlowTrackingCliLib import (
   exporterNameMatcher,
   exporterKw,
   hardwareShowKw,
   flowTrackingAgentRunning,
   sflowHwStatus,
   trackingShowKw,
   sampledShowKw,
   telemetryShowKw,
   inbandShowKw,
   mirrorOnDropShowKw,
   trackerKw,
   trackerNameMatcher,
   groupShowKw,
   groupNameMatcher,
)
import FlowTrackingModel
import ShowCommand
from Toggles.InbandTelemetryCommonToggleLib import \
      toggleFeatureInbandTelemetryEnabled
from Toggles.FlowTrackerToggleLib import (
   toggleHwOffloadIpv4Enabled,
   toggleCopyToCollectorEnabled,
)

traceHandle = Tracing.Handle( 'FlowTrackingShowCli' )

activeAgentDir = None
entityManager = None
hwConfig = {}
hwTrackingStatus = {}
trackingConfig = {}
ftrCapabilities = {}
ftrStatus = {}

# SHOW COMMANDS
#------------------------------------------------------------
# show flow tracking ( sampled | hardware | ( telemetry inband ) )
#         [ tracker < name > [ exporter < name > ] ]
#------------------------------------------------------------
def showExporter( ftrType, trName, trackerModel, expName, tracker,
                  ftStatus ):
   exporter = tracker.expConfig.get( expName )
   expModel = FlowTrackingModel.ExpModel()
   trackerModel.exporters[ expName ] = expModel
   expReasonModel = FlowTrackingModel.ExpReasonModel()
   # Exporter configured but not yet present in hwConfig
   if exporter is None:
      expConfig = trackingConfig[ ftrType ].flowTrackerConfig[ trName ].expConfig[
                  expName ]
      expModel.active = False
      reason = FlowTrackingModel.ExpReason()
      reason.expReason = exporterInactiveReasonEnum.exporterNotProcessed
      expReasonModel.expReasons.append( reason )
      expModel.inactiveReasons[ AddressFamily.ipunknown ] = expReasonModel
      # We need to provide some default values for mandatory arguments in
      # ExpModel
      expModel.srcIpAddr = constants.ipAddrDefault
      expModel.srcIp6Addr = constants.ip6AddrDefault
      expModel.dscpValue = expConfig.dscpValue
      expModel.templateInterval = expConfig.templateInterval
      expModel.localIntf = expConfig.localIntfName
      expModel.vrfName = ''
      expFormatModel = FlowTrackingModel.ExpFormatModel()
      expFormatModel.name = "IPFIX"
      expFormatModel.version = expConfig.ipfixVersion
      expFormatModel.mtu = expConfig.ipfixMtu
      expModel.exportFormat = expFormatModel
      return

   expStatus = None
   if ftStatus:
      expStatus = ftStatus.expStatus.get( expName )

   # Things should not crash if expStatus doesn't exist for any reason (though
   # this is unlikely)
   if expStatus:
      expModel.active = expStatus.active
   else:
      reason = FlowTrackingModel.ExpReason()
      reason.expReason = exporterInactiveReasonEnum.exporterInactiveReasonUnknown
      expReasonModel.expReasons.append( reason )
      expModel.inactiveReasons[ AddressFamily.ipunknown ] = expReasonModel
      expModel.active = False

   if expStatus and not expStatus.active:
      # It might be possible that expStatus exists but expStatus.inactiveReason is
      # not set because tacc reactors have not yet run.
      if not expStatus.inactiveReason:
         reason = FlowTrackingModel.ExpReason()
         reason.expReason = exporterInactiveReasonEnum.exporterInactiveReasonUnknown
         expReasonModel.expReasons.append( reason )
         expModel.inactiveReasons[ AddressFamily.ipunknown ] = expReasonModel
      else:
         for ipVersion, inactiveReason in expStatus.inactiveReason.items():
            reasonEnumList = getExpReasonEnumList( inactiveReason )
            expReasonModel.expReasons = []
            for reasonEnum in reasonEnumList:
               reason = FlowTrackingModel.ExpReason()
               reason.expReason = reasonEnum
               expReasonModel.expReasons.append( reason )
            expModel.inactiveReasons[ ipVersion ] = expReasonModel

   expModel.vrfName = exporter.vrfName
   expModel.localIntf = Tac.Value( "Arnet::IntfId", "" )
   expConfig = None
   if trName in trackingConfig[ ftrType ].flowTrackerConfig:
      trackerConfig = trackingConfig[ ftrType ].flowTrackerConfig[ trName ]
      if expName in trackerConfig.expConfig:
         expConfig = trackerConfig.expConfig[ expName ]
         expModel.localIntf = Tac.Value( "Arnet::IntfId", expConfig.localIntfName )
   expModel.srcIpAddr = exporter.srcIpAddr
   expModel.srcIp6Addr = exporter.srcIp6Addr
   expFormatModel = FlowTrackingModel.ExpFormatModel()
   expFormatModel.name = "IPFIX"
   expFormatModel.version = exporter.ipfixVersion
   expFormatModel.mtu = exporter.ipfixMtu
   expModel.exportFormat = expFormatModel
   expModel.dscpValue = exporter.dscpValue
   expModel.templateInterval = exporter.templateInterval
   expModel.collectors = {}
   configCollector = []
   hwConfigCollector = []
   if expConfig:
      configCollector = expConfig.collectorHostAndPort.keys()
   hwConfigCollector = list( set( exporter.collectorIpAndPort.keys() ).union(
                             set( exporter.collectorIp6AndPort.keys() ) ) )
   collectors = list( set( configCollector ).union(
                      set( hwConfigCollector ) ) )

   for collectorName in collectors:
      collectorModel = FlowTrackingModel.CollectorModel()
      expModel.collectors[ collectorName ] = collectorModel
      collector = ( exporter.collectorIpAndPort.get( collectorName ) or
                    exporter.collectorIp6AndPort.get( collectorName ) )
      if collector:
         collectorModel.active = True
      else:
         collectorModel.active = False
         if( len( exporter.collectorIpAndPort ) +
             len( exporter.collectorIp6AndPort ) >=
             ftrCapabilities[ ftrType ].maxCollectorsPerExporter ):
            collectorModel.inactiveReason = \
                  collectorInactiveReason.maxCollectorsLimit
         else:
            collectorModel.inactiveReason = \
                  collectorInactiveReason.collectorNotProcessed
         if expConfig:
            collector = expConfig.collectorHostAndPort.get( collectorName )
      if collector:
         collectorModel.port = collector.port

def showGroup( ftrType, trName, trackerModel, fgName, tracker ):
   fgModel = FlowTrackingModel.FgModel()
   trackerModel.groups[ fgName ] = fgModel
   fgConfig = trackingConfig[ ftrType ].flowTrackerConfig[
               trName ].fgConfig.get( fgName )

   def getMirrorModel( mirrorConfig ):
      mirrorModel = FlowTrackingModel.MirrorModel()
      mirrorModel.sessionName = mirrorConfig.sessionName
      mirrorModel.initialCopy = mirrorConfig.initialCopy
      mirrorModel.mirrorSelectorAlgorithm = mirrorConfig.mirrorSelectorAlgo
      if mirrorConfig.mirrorSelectorAlgo == mirrorSelectorAlgo.fixed:
         mirrorModel.mirrorInterval = mirrorConfig.mirrorIntervalFixed
      elif mirrorConfig.mirrorSelectorAlgo == mirrorSelectorAlgo.random:
         mirrorModel.mirrorInterval = mirrorConfig.mirrorIntervalRandom
      else:
         mirrorModel.mirrorInterval = 0
      return mirrorModel

   if fgConfig is None:
      if fgName in reservedGroupToSeqnoMap:
         # For default reserved groups: use the info from hwFgConfig
         hwFgConfig = tracker.hwFgConfig.get( reservedGroupToSeqnoMap[ fgName ] )
         if hwFgConfig is None:
            # Shouldn't happen unless it is some transient corner case
            return

         fgModel.seqno = hwFgConfig.seqno
         # expNames
         for expName in hwFgConfig.expName:
            expModel = FlowTrackingModel.GroupExpModel()
            expModel.expName = expName
            fgModel.expNames[ expName ] = expModel

         # mirroring
         if ftrCapabilities[ ftrType ].mirroringSupported:
            mirrorConfig = hwFgConfig.mirrorConfig
            mirrorModel = getMirrorModel( mirrorConfig )
            fgModel.mirroring = mirrorModel

         # hwGroups
         hwFgModel = FlowTrackingModel.HwFgModel()
         hwFgModel.hwFgName = hwFgConfig.fgName
         fgModel.hwGroups[ long( hwFgConfig.seqno ) ] = hwFgModel
         return
      else:
         # If fgConfig got deleted while this command was being run but hwFgConfig
         # is not cleared yet, we cannot display this group in correct sequence
         # because we would have lost group seqno.
         # Even if we compute the seqno from hwFgConfig, there is no guarantee that
         # it would be in the correct sequence. We may get stale info in cases of
         # sequence number change due to resequencing or config change.
         # Futher, this would anyway be a transient case and things should be fine
         # from eventual consistency point of view.
         return

   # In regular case when a group exists in config, the attr value for each of the
   # corresponding hwGroups will be same. So we take it from config entity.
   # The status of things like exporter, mirroring etc will be taken later from
   # hwFgStatus (To Do Later)

   # seqno
   fgModel.seqno = fgConfig.seqno

   # Encapsulation and access lists are supported only for non-reserved groups
   if fgName not in reservedGroupToSeqnoMap:
      # encapTypes
      for encapType in fgConfig.encapType:
         encapTypeModel = FlowTrackingModel.EncapTypeModel()
         encapTypeModel.encapType = encapType
         fgModel.encapTypes.append( encapTypeModel )

      # ipAccessLists
      ipAccessList = fgConfig.ipAccessList
      if ipAccessList.aclName:
         ipAccessListModel = FlowTrackingModel.AccessListModel()
         ipAccessListModel.aclName = ipAccessList.aclName
         fgModel.ipAccessLists[ ipAccessList.seqno ] = ipAccessListModel

      # ip6AccessLists
      ip6AccessList = fgConfig.ip6AccessList
      if ip6AccessList.aclName:
         ip6AccessListModel = FlowTrackingModel.AccessListModel()
         ip6AccessListModel.aclName = ip6AccessList.aclName
         fgModel.ip6AccessLists[ ip6AccessList.seqno ] = ip6AccessListModel

   # expNames
   for expName in fgConfig.expName:
      expModel = FlowTrackingModel.GroupExpModel()
      expModel.expName = expName
      fgModel.expNames[ expName ] = expModel

   # mirroring
   if ftrCapabilities[ ftrType ].mirroringSupported:
      mirrorConfig = fgConfig.mirrorConfig
      mirrorModel = getMirrorModel( mirrorConfig )
      fgModel.mirroring = mirrorModel

   # hwGroups
   hwFgConfigMap = tracker.hwFgConfigMap.get( fgName )
   if hwFgConfigMap is not None:
      for hwFgSeqno, hwFgMap in hwFgConfigMap.hwFgNameAndSeqno.items():
         hwFgModel = FlowTrackingModel.HwFgModel()
         hwFgModel.hwFgName = hwFgMap.fgName
         fgModel.hwGroups[ long( hwFgSeqno ) ] = hwFgModel

def showIntfsForTracker( trackerModel, ftrType, trName, tracker ):
   trackerModel.activeIntfs = []
   trackerModel.inactiveIntfs = []
   # Look in hardware/flowtracking/config/...
   for intfName, hwIntfConfig in hwConfig[ ftrType ].hwIntfFtConfig.items():
      if hwIntfConfig.hwFtConfig == tracker:
         intfModel = FlowTrackingModel.IntfModel()
         intfModel.interface = Tac.Value( "Arnet::IntfId", intfName )
         # get inactive reason from sflow/hwstatus/sand/ for sampled tracker
         if ftrType == ftrTypeSampled and intfName in sflowHwStatus().intfOperStatus:
            intfModel.reason = \
               sflowHwStatus().intfOperStatus[ intfName ].reason
            trackerModel.inactiveIntfs.append( intfModel )
         else:
            trackerModel.activeIntfs.append( intfModel )
   # Look in flowtracking/config/...
   for intfName, intfConfig in trackingConfig[ 
                                 ftrType ].flowTrackerIntfConfig.items():
      if intfConfig.trackerName == trName:
         if not ( intfName in hwConfig[ ftrType ].hwIntfFtConfig and \
                  hwConfig[ ftrType ].hwIntfFtConfig[ intfName ].hwFtConfig ):
            intfModel = FlowTrackingModel.IntfModel()
            intfModel.interface = Tac.Value( "Arnet::IntfId", intfName )
            trackerModel.inactiveIntfs.append( intfModel )

def _showFlowTracking( mode, args ):
   ftrType = getFtrTypeFromArgs( args )
   trackerName = args.get( 'TRACKER_NAME' )
   exporterName = args.get( 'EXPORTER_NAME' )

   flowTrackingModel = FlowTrackingModel.FtrModel()
   flowTrackingModel.ftrType = ftrType

   if flowTrackingAgentRunning( ftrType, entityManager, activeAgentDir ):
      flowTrackingModel.running = True
      if ftrType == ftrTypeHardware:
         if ftrCapabilities[ ftrType ].swExportSupported:
            flowTrackingModel.standardFormatForCounters = \
                hwConfig[ ftrType ].swExport
            flowTrackingModel.standardFormatForTimeStamps = \
                hwConfig[ ftrType ].swExport
      if ftrType == ftrTypeSampled:
         flowTrackingModel.sampleRate = hwTrackingStatus[ ftrType ].sampleRate
         if toggleHwOffloadIpv4Enabled():
            if ftrCapabilities[ ftrType ].hwOffloadIpv4:
               flowTrackingModel.hwOffloadIpv4 = hwConfig[ ftrType ].hwOffloadIpv4
   else:
      flowTrackingModel.running = False
      return flowTrackingModel

   flowTrackingModel.trackers = {}
   # Take union of config + hwConfig and iterate over them.
   # Reason being, if for any reason agent is not running or transient
   # (agent didn't react yet), there can be discrepancy due to that.
   # For ex: 
   #    - tracker deleted in config but exists in hwConfig,
   #    - tracker exists in config but not in hwConfig
   # we should display the current functional state.
   trackers = list( set( trackingConfig[ ftrType ].flowTrackerConfig.keys() ).\
                    union(
                    set( hwConfig[ ftrType ].hwFtConfig.keys() ) ) )
   for trName in trackers:
      if trackerName and trName != trackerName:
         continue

      trackerModel = FlowTrackingModel.FtModel()
      ftStatus = ftrStatus[ ftrType ].ftStatus.get( trName )
      # Need to set default values for testing purposes
      trackerModel.inactiveTimeout = inactiveTimeout.timeoutDefault
      trackerModel.activeInterval = activeInterval.intervalDefault
      flowTrackingModel.trackers[ trName ] = trackerModel
      if trName in hwConfig[ ftrType ].hwFtConfig:
         tracker = hwConfig[ ftrType ].hwFtConfig[ trName ]
         trackerModel.active = ftStatus.active if ftStatus else False
      # Even if tracker is present in hwFtConfig and ftStatus, it is possible
      # to have ftStatus.active as False because of race between CLI and 
      # tacc agent code. So irrespective of whether tracker is present in hwFtConfig
      # or not, we need to get inactive reason when ftStatus.active flag is False.
      if ftStatus and not ftStatus.active:
         trackerModel.inactiveReason = ftStatus.inactiveReason
         trackerModel.active = ftStatus.active
         if trName not in hwConfig[ ftrType ].hwFtConfig:
            continue
      elif not ftStatus:
         trackerModel.inactiveReason = \
               trackerInactiveReason.trackerNotProcessed
         trackerModel.active = False
         continue

      if trName not in hwTrackingStatus[ ftrType ].hwFtStatus:
         trackerModel.inactiveReason = \
               trackerInactiveReason.trackerNotProcessed
         trackerModel.active = False
         continue

      hwStatus = hwTrackingStatus[ ftrType ].hwFtStatus[ trName ]
      trackerModel.activeInterval = hwStatus.activeInterval
      trackerModel.inactiveTimeout = hwStatus.inactiveTimeout

      configExporter = []
      trackerConfig = trackingConfig[ ftrType ].flowTrackerConfig.get( trName )
      if trackerConfig is not None:
         configExporter = trackerConfig.expConfig.keys()
      exporters = list( set( tracker.expConfig.keys() ).union(
                        set( configExporter ) ) )

      for expName in exporters:
         if exporterName and expName != exporterName:
            continue
         showExporter( ftrType, trName, trackerModel, expName, tracker,
                       ftStatus )

      if toggleCopyToCollectorEnabled() and \
         ftrCapabilities[ ftrType ].flowGroupConfigSupported:

         if trackerConfig is not None:
            configFlowGroups = trackerConfig.fgConfig.keys()
         groups = list( set( tracker.hwFgConfigMap ).union(
                        set( configFlowGroups ) ) )
         groupName = args.get( 'GROUP_NAME' )
         for fgName in groups:
            if groupName and groupName != fgName:
               continue
            showGroup( ftrType, trName, trackerModel, fgName, tracker )

         # Reserved groups
         # Each flow tracking feature can have different reserved groups with
         # seqno >= constants.reservedFgSeqnoBase
         reservedFgSeqnoBase = constants.reservedFgSeqnoBase
         for hwFgseqno in sorted( tracker.hwFgConfig, reverse=True ):
            if hwFgseqno < reservedFgSeqnoBase:
               break
            fgName = tracker.hwFgConfig[ hwFgseqno ].fgName
            if groupName and groupName != fgName or \
               fgName in configFlowGroups:
               continue
            showGroup( ftrType, trName, trackerModel, fgName, tracker )
      else:
         trackerModel.groups = {}
         for group in tracker.hwFgConfig.values():
            fgModel = FlowTrackingModel.FgModel()
            trackerModel.groups[ group.fgName ] = fgModel

      showIntfsForTracker( trackerModel, ftrType, trName, tracker )

   return flowTrackingModel

class ShowFlowTracking( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ( sampled | hardware | mirror-on-drop '
   if toggleFeatureInbandTelemetryEnabled():
      syntax += '| ( telemetry inband ) '
   syntax += ') [ tracker TRACKER_NAME [ exporter EXPORTER_NAME ]'
   if toggleCopyToCollectorEnabled():
      syntax += ' [ group GROUP_NAME ]'
   syntax += ' ]'

   data = {
         'flow' : flowMatcherForShow,
         'tracking' : trackingShowKw,
         'sampled' : sampledShowKw,
         'hardware' : hardwareShowKw,
         'mirror-on-drop' : mirrorOnDropShowKw,
         'tracker' : trackerKw,
         'TRACKER_NAME' : trackerNameMatcher,
         'exporter' : exporterKw,
         'EXPORTER_NAME' : exporterNameMatcher,
   }

   if toggleFeatureInbandTelemetryEnabled():
      data[ 'telemetry' ] = telemetryShowKw
      data[ 'inband' ] = inbandShowKw

   if toggleCopyToCollectorEnabled():
      data[ 'group' ] = groupShowKw
      data[ 'GROUP_NAME' ] = groupNameMatcher

   cliModel = FlowTrackingModel.FtrModel
   handler = _showFlowTracking

BasicCli.addShowCommandClass( ShowFlowTracking )

#--------------------------
def Plugin( em ):
   global activeAgentDir
   global entityManager
   global hwConfig
   global hwTrackingStatus
   global trackingConfig
   global ftrCapabilities
   global ftrStatus

   entityManager = em
   activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                     'Tac::Dir', 'ri' )
   for ftrType in ftrTypes:
      trackingConfig[ ftrType ] = LazyMount.mount( em,
                                    ftrConfigPathPrefix + ftrType,
                                    ftrConfigType, 'r' )
      hwConfig[ ftrType ] = LazyMount.mount( em,
                                    hwFtrConfigPathPrefix + ftrType,
                                    hwFtrConfigType, 'r' )
      hwTrackingStatus[ ftrType ] = LazyMount.mount( em,
                                       hwFtrStatusPathPrefix + ftrType,
                                       hwFtrStatusType, 'r' )
      ftrCapabilities[ ftrType ] = LazyMount.mount( em,
                                       ftrCapabilitiesPathPrefix + ftrType,
                                       ftrCapabilitiesType, 'r' )
      ftrStatus[ ftrType ] = LazyMount.mount( em,
                                       ftrStatusPathPrefix + ftrType,
                                       ftrStatusType, 'r' )
