# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Intf.IntfRange as IntfRange
import Tac
from ArnetModel import Ip4Address
from ArnetModel import Ip6Address
from CliModel import (
   Bool,
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from IntfModels import Interface
from FlowTrackerCliUtil import (
   ftrTypeShowStr,
   collectorInactiveReasonStr,
   exporterInactiveReasonStr,
   trackerInactiveReasonStr,
   getExpReasonFlagsFromExpReasonModel,
   exporterInactiveReasonEnum,
   encapTypeEnum,
   encapTypeShowStr,
   reservedGroupToSeqnoMap,
)
from collections import namedtuple
from Toggles.FlowTrackerToggleLib import (
   toggleHwOffloadIpv4Enabled,
   toggleCopyToCollectorEnabled,
)

CollectorPortAndReason = namedtuple( 'Collector', [ 'port', 'inactiveReason' ] )

collectorInactiveReason = Tac.Type( "FlowTracking::CollectorInactiveReason" )
trackerInactiveReason= Tac.Type( "FlowTracking::TrackerInactiveReason" )
mirrorSelectorAlgo = Tac.Type( "FlowTracking::MirrorSelectorAlgo" )

# Collector
class CollectorModel( Model ):
   port = Int( "Collector port" )
   active = Bool( help="Collector is ready for transmission" )
   inactiveReason = Enum( values=collectorInactiveReason.attributes, 
                          help="Collector inactive reason",
                          optional=True )

# Export Format
class ExpFormatModel( Model ):
   name = Str( help="Export format name" )
   mtu = Int( help="MTU of the export format", optional=True )
   version = Int( help="Version of the export format", optional=True )

class ExpReason( Model ):
   expReason = Enum( values=exporterInactiveReasonEnum.attributes,
                     help="Exporter inactive reason" )

class ExpReasonModel( Model ):
   expReasons = List( valueType=ExpReason,
                      help="List of reasons for inactive exporter" )


# Exporter
class ExpModel( Model ):
   vrfName = Str( help="VRF name" )
   localIntf = \
      Interface( help="Local interface to be used for IPFIX source IP4/6 addr" )
   srcIpAddr = Ip4Address(
      help="Local IPv4 address to be used for IPFIX source addr" )
   srcIp6Addr = Ip6Address(
      help="Local IPv6 address to be used for IPFIX source addr" )
   exportFormat = Submodel( valueType=ExpFormatModel,
                            help="Export format information" )
   dscpValue = Int( help="DSCP value in IP header" )
   templateInterval = Int( help="Template interval in ms" )
   active = Bool( help="Exporter is ready for transmission" )
   collectors = Dict( keyType=str, valueType=CollectorModel,
                help="A mapping between collector address and collector port" )
   inactiveReasons = Dict( keyType=str, valueType=ExpReasonModel,
                           help="A mapping between type of reason and "
                                "list of reasons for inactive exporter",
                           optional=True )

class EncapTypeModel( Model ):
   encapType = Enum( values=encapTypeEnum.attributes,
                     help="Encapsulation" )

# For now it has only one attribute, but in future it will also have
# active/inactive state
class GroupExpModel( Model ):
   expName = Str( help="Exporter name" )

# For now it has only one attribute, but in future it will have more
class AccessListModel( Model ):
   aclName = Str( help="Access list name" )

class MirrorModel( Model ):
   sessionName = Str( help="Mirror session name" )
   initialCopy = Int( help="Initial packets to be mirrored",
                      optional=True )
   mirrorInterval = Int( help="Sampling interval for packets to be mirrored",
                         optional=True )
   mirrorSelectorAlgorithm = Enum( values=mirrorSelectorAlgo.attributes,
                                   help="Mirror sampling algorithm",
                                   optional=True )

# For now it has only one attribute, but in future it will have more
class HwFgModel( Model ):
   hwFgName = Str( help="Hardware flow group name" )

# Flow group
class FgModel( Model ):
   seqno = Int( help="Flow group sequence number",
                optional=True )
   encapTypes = List( valueType=EncapTypeModel,
                      help="List of encapsulations",
                      optional=True )
   ipAccessLists = Dict( keyType=int, valueType=AccessListModel,
                         help="IPv4 ACLs, keyed by their sequence number",
                         optional=True )
   ip6AccessLists = Dict( keyType=int, valueType=AccessListModel,
                          help="IPv6 ACLs, keyed by their sequence number",
                          optional=True )
   expNames = Dict( keyType=str, valueType=GroupExpModel,
                    help="Exporters, keyed by exporter name",
                    optional=True )
   mirroring = Submodel( valueType=MirrorModel,
                         help="Mirroring details",
                         optional=True )
   hwGroups = Dict( keyType=long, valueType=HwFgModel,
                    help="Flow groups programmed in hardware,"
                         " keyed by their sequence number",
                    optional=True )

# Flow interface
class IntfModel( Model ):
   interface = Interface( help="Name of the interface" )
   reason = Str( help="Reason for interface inactive", optional=True )

# Flow tracker
class FtModel( Model ):
   inactiveTimeout = Int( help="Inactive timeout in ms" )
   activeInterval = Int( help="Active interval in ms" )
   activeIntfs = List( valueType=IntfModel, help="Active interfaces" )
   inactiveIntfs = List( valueType=IntfModel, help="Inactive interfaces" )
   # FgModel is per user defined group
   # Hardware groups corresponding to user defined groups are within FgModel
   groups = Dict( keyType=str, valueType=FgModel, help="Flow groups" )
   exporters = Dict( keyType=str, valueType=ExpModel,
                     help="A mapping between exporter name and exporter" )
   active = Bool( help="Tracker is active" )
   inactiveReason = Enum( values=trackerInactiveReason.attributes,
                          help="Tracker inactive reason",
                          optional=True )

FLOWTRACKING_TYPES = Tac.Type( "FlowTracking::FlowTrackingType" )

# Flow Tracking
class FtrModel( Model ):
   trackers = Dict( keyType=str, valueType=FtModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Flow tracking is running" )
   sampleRate = Int( help="Flow tracking sample every x packets", optional=True )
   ftrType = Enum( values=FLOWTRACKING_TYPES.attributes, help="Flow tracking type" )
   standardFormatForCounters = Bool( help="Flow tracking standard record format for"
                                     " counters", optional=True )
   standardFormatForTimeStamps = Bool( help="Flow tracking standard record format "
                                       "for timestamps", optional=True )
   if toggleHwOffloadIpv4Enabled():
      hwOffloadIpv4 = Bool( help="Flow tracking hardware offload IPv4 traffic",
                            optional=True )

   def renderCollector( self, ip, collector ):
      tab = " " * 8
      activeStr = ""
      if not collector.active:
         activeStr = collectorInactiveReasonStr( collector.inactiveReason )
      print tab + ip + " port " + str( collector.port ) + activeStr

   def renderExporter( self, expName, exporter ):
      tab = " " * 4
      if not exporter.active:
         reasonUnknown = True
         for reason in exporter.inactiveReasons.values():
            expInactiveReason = getExpReasonFlagsFromExpReasonModel( reason )
            if expInactiveReason.value != 0:
               reasonUnknown = False
               break

         padding = tab + 'Exporter: ' + expName
         for reasonType, reason in sorted( exporter.inactiveReasons.items() ):
            expInactiveReason = getExpReasonFlagsFromExpReasonModel( reason )
            print padding + exporterInactiveReasonStr( expInactiveReason,
                                                       reasonType )
            padding = ' ' * len( padding )

            # We should print the following reasons only once
            if expInactiveReason.maxExportersLimit or \
               expInactiveReason.exporterNotProcessed:
               return
            if reasonUnknown:
               break
      else:
         print tab + "Exporter: " + expName

      tab = " " * 6
      print tab + "VRF: " + exporter.vrfName
      if exporter.localIntf:
         ips = []
         if exporter.srcIpAddr.stringValue != exporter.srcIpAddr.ipAddrZero:
            ips.append( exporter.srcIpAddr.stringValue )
         if not exporter.srcIp6Addr.isZero:
            ips.append( exporter.srcIp6Addr.stringValue )
         print tab + "Local interface: " + exporter.localIntf.stringValue + \
            ( ( " (" + ", ".join( ips ) + ")" ) if ips else "" )
      formatInfo = exporter.exportFormat
      formatDetails = []
      if formatInfo.version:
         formatDetails.append( "version " + str( formatInfo.version ) )
      if formatInfo.mtu:
         formatDetails.append( "MTU " + str( formatInfo.mtu ) )
      print tab + "Export format: " + exporter.exportFormat.name + " " + \
            ", ".join( formatDetails )
      print tab + "DSCP: " + str( exporter.dscpValue )
      print tab + "Template interval: " + str( exporter.templateInterval ) + \
            "ms"
      # First print all active collectors,
      # then inactive collectors which are within max collector limit,
      # then inactive collectors due to max collector limit.
      inactiveCollector = {}
      beyondLimitCollector = {}
      if exporter.collectors:
         print tab + "Collectors:"
         for ip, collector in sorted( exporter.collectors.items() ): 
            if not collector.active:
               if collector.inactiveReason == \
                  collectorInactiveReason.maxCollectorsLimit:
                  beyondLimitCollector[ ip ] = collector
               else:
                  inactiveCollector[ ip ] = collector
               continue
            self.renderCollector( ip, collector )

         for ip in sorted( inactiveCollector ):
            self.renderCollector( ip, inactiveCollector[ ip ] )

         for ip in sorted( beyondLimitCollector ):
            self.renderCollector( ip, beyondLimitCollector[ ip ] )

   def renderGroup( self, groupName, group ):
      tab = " " * 4
      print tab + "Group: " + groupName

      tab = " " * 6
      # Encapsulation and access list are only supported for non-reserved groups
      if groupName not in reservedGroupToSeqnoMap:
         encaps = []
         for encap in group.encapTypes:
            if encap.encapType == encapTypeEnum.encapNone:
               continue
            encaps.append( encapTypeShowStr[ encap.encapType ] )
         encapStr = ", ".join( sorted( encaps ) ) if encaps else "none"
         print tab + "Encapsulation: " + encapStr

         ip4Acls = []
         for aclSeqno in sorted( group.ipAccessLists ):
            ip4Acls.append( group.ipAccessLists[ aclSeqno ].aclName )
         # ip4Acls are sorted by the seqno, do NOT sort here by name
         ip4AclStr = ", ".join( ip4Acls ) if ip4Acls else "none"
         print tab + "IPv4 access list: " + ip4AclStr

         ip6Acls = []
         for aclSeqno in sorted( group.ip6AccessLists ):
            ip6Acls.append( group.ip6AccessLists[ aclSeqno ].aclName )
         # ip6Acls are sorted by the seqno, do NOT sort here by name
         ip6AclStr = ", ".join( ip6Acls ) if ip6Acls else "none"
         print tab + "IPv6 access list: " + ip6AclStr

      expStr = ", ".join( sorted( group.expNames ) ) if group.expNames else "none"
      print tab + "Exporters: " + expStr

      if group.mirroring:
         groupMirroring = group.mirroring
         if groupMirroring.sessionName:
            print tab + "Mirroring:"
            tabMirroring = " " * 8
            sessionStr = groupMirroring.sessionName
            initialCopyStr = str( groupMirroring.initialCopy ) if \
                             groupMirroring.initialCopy else "none"
            if groupMirroring.mirrorSelectorAlgorithm != \
               mirrorSelectorAlgo.sampleModeNone:
               if groupMirroring.mirrorSelectorAlgorithm == \
                  mirrorSelectorAlgo.fixed:
                  selectorAlgoStr = " (fixed)"
               elif groupMirroring.mirrorSelectorAlgorithm == \
                    mirrorSelectorAlgo.random:
                  selectorAlgoStr = " (random)"
               mirrorIntervalStr = str( groupMirroring.mirrorInterval ) + \
                                   selectorAlgoStr
            else:
               mirrorIntervalStr = "none"

            print tabMirroring + "Monitor session: " + sessionStr
            print tabMirroring + "Number of initial packets: " + initialCopyStr
            print tabMirroring + "Sample interval: " + mirrorIntervalStr
         else:
            print tab + "Mirroring: none"

      hwGroupsStr = "Hardware groups:" if group.hwGroups else \
                    "Hardware groups: none"
      print tab + hwGroupsStr
      tabHwGroups = " " * 8
      for hwFgSeqno in sorted( group.hwGroups ):
         print tabHwGroups + group.hwGroups[ hwFgSeqno ].hwFgName

   def renderTracker( self, trackerName, tracker ):
      tab = " " * 2
      if not tracker.active:
         print tab + "Tracker: " + trackerName + \
               trackerInactiveReasonStr( tracker.inactiveReason )
         return
      print tab + "Tracker: " + trackerName
      tab = " " * 4
      print tab + "Active interval: " + str( tracker.activeInterval ) + "ms"
      print tab + "Inactive timeout: " + str( tracker.inactiveTimeout ) + "ms"

      userConfiguredflowGroups = False
      if toggleCopyToCollectorEnabled():
         for group in tracker.groups.values():
            if group.seqno is not None:
               userConfiguredflowGroups = True
               break
      if not userConfiguredflowGroups:
         print tab + "Groups: " + ", ".join( sorted( tracker.groups ) )
      # First print all active exporters,
      # then inactive exporters which are within max exporter limit,
      # then inactive exporters due to max exporter limit.
      inactiveExporter = {}
      beyondLimitExporter = {}
      for expName, exporter in sorted( tracker.exporters.items() ):
         if not exporter.active:
            maxExportersLimit = False
            for reason in exporter.inactiveReasons.values():
               expInactiveReason = getExpReasonFlagsFromExpReasonModel( reason )
               if expInactiveReason.maxExportersLimit:
                  maxExportersLimit = True
                  break
            if maxExportersLimit:
               beyondLimitExporter[ expName ] = exporter
            else:
               inactiveExporter[ expName ] = exporter
            continue
         self.renderExporter( expName, exporter )

      for expName in sorted( inactiveExporter ):
         self.renderExporter( expName, inactiveExporter[ expName ] )

      for expName in sorted( beyondLimitExporter ):
         self.renderExporter( expName, beyondLimitExporter[ expName ] )

      if userConfiguredflowGroups:
         seqnoToGroupMap = {}
         for groupName, group in tracker.groups.items():
            seqnoToGroupMap[ group.seqno ] = groupName
         for groupSeqno in sorted( seqnoToGroupMap ):
            groupName = seqnoToGroupMap[ groupSeqno ]
            self.renderGroup( groupName, tracker.groups[ groupName ] )

      if tracker.activeIntfs:
         tab = " " * 4
         print tab + "Active interfaces:"
         tab = " " * 6
         activeIntfs = [ i.interface.shortName for i in tracker.activeIntfs ]
         activeIntfs = IntfRange.intfListToCanonical( activeIntfs )
         print tab + ", ".join( sorted( activeIntfs ) )
      if tracker.inactiveIntfs:
         tab = " " * 4
         print tab + "Inactive interfaces:"
         tab = " " * 6
         inactiveIntfs = [ i.interface.shortName \
                           for i in tracker.inactiveIntfs if not i.reason ]
         inactiveIntfs = IntfRange.intfListToCanonical( inactiveIntfs )
         inactiveIntfs2 = [ i.interface.shortName + "(" + i.reason + ")" \
                           for i in tracker.inactiveIntfs if i.reason ]
         print tab + ", ".join( sorted( inactiveIntfs + inactiveIntfs2 ) )

   def render( self ):
      print "Flow Tracking Status"
      tab = " " * 2
      print tab + "Type: " + ftrTypeShowStr[ self.ftrType ]

      def renderYesNo( attr=None, text=None ):
         if attr is not None:
            print tab + text + ": " + ( "yes" if attr else "no" )

      renderYesNo( attr=self.running, text="Running" )

      if not self.running:
         return

      renderYesNo( attr=self.standardFormatForCounters,
            text="Standard record format for counters" )
      renderYesNo( attr=self.standardFormatForTimeStamps,
            text="Standard record format for timestamps" )

      if self.sampleRate is not None:
         print tab + "Sample rate: " + str( self.sampleRate )

      if toggleHwOffloadIpv4Enabled():
         renderYesNo( attr=self.hwOffloadIpv4, text="Hardware offload IPv4 traffic" )

      # First print all active trackers,
      # then inactive trackers which are within max tracker limit (TO DO),
      # then inactive trackers due to max tracker limit.
      inactiveTracker = {}
      beyondLimitTracker = {}
      for trackerName, tracker in sorted( self.trackers.items() ):
         if not tracker.active:
            if tracker.inactiveReason == trackerInactiveReason.maxTrackersLimit:
               beyondLimitTracker[ trackerName ] = tracker
            else:
               inactiveTracker[ trackerName ] = tracker
            continue
         self.renderTracker( trackerName, tracker )

      # This is a no-op for now, existing behavior since beginning.
      for trackerName in sorted( inactiveTracker ):
         self.renderTracker( trackerName, inactiveTracker[ trackerName ] )

      for trackerName in sorted( beyondLimitTracker ):
         self.renderTracker( trackerName, beyondLimitTracker[ trackerName ] )
