# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import LazyMount
import SmashLazyMount
import Tac
import Tracing
import CliCommand
import CliMatcher
from CliToken.Flow import flowMatcherForShow
from CliPlugin.FlowTrackingCliLib import (
   getIfIndexMap,
   flowTrackingAgentRunning,
   groupNameMatcher,
   groupKw,
   hardwareShowKw,
   sampledShowKw,
   telemetryShowKw,
   inbandShowKw,
   trackerNameMatcher,
   trackerKw,
   trackingShowKw,
   FtConsts,
   IpfixRecordType,
   SelectorAlgorithmType,
)
from CliPlugin.FlowTrackingIpfixModel import (
      FtTemplateModel,
      FtOptTableModel,
      OptTableModel,
      TemplateEntryModel,
      TemplateModel,
      TrackerInfoModel,
      TrackerOptTableModel,
      TrackerTemplateModel,
)

from FlowTrackerCliUtil import (
      ftrConfigPathPrefix,
      ftrConfigType,
      ftrTypeHardware,
      ftrTypeSampled,
      ftrTypeInbandTelemetry,
      getFtrTypeFromArgs,
      hwFtrConfigPathPrefix,
      hwFtrConfigType,
      hwFtrStatusPathPrefix,
      hwFtrStatusType,
      ftrCapabilitiesPathPrefix,
      ftrCapabilitiesType,
)

import ShowCommand
from TypeFuture import TacLazyType
import Toggles.InbandTelemetryCommonToggleLib

traceHandle = Tracing.Handle( 'FlowTrackingIpfixShowCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

StaticTemplateIdEnum = TacLazyType( 'FlowTracking::StaticTemplateId' )
VrfIdType = TacLazyType( 'Vrf::VrfIdMap::VrfId' )

activeAgentDir = None
arpVrfIdMap = None
entityManager = None
ethIntfStatusDir = None
tunnelIntfStatusDir = None
trackingConfig = {}
hwConfig = {}
hwTrackingStatus = {}
ftrCapabilities = {}

ElementIdColl = Tac.newInstance( 'Ipfix::ElementIdColl' )
ElementIdColl.doInitialize()

#----------------------------------------------------------------------
# show flow tracking ( sampled | hardware | ( telemetry inband ) )
# ipfix template [ tracker < name > ] [[data [group <name>]]|[options]]
#----------------------------------------------------------------------
def showDataTemplate( trackerModel, ftStatus, groupName ):
   trackerModel.dataTemplates = {}
   for templateStatus in ftStatus.templateStatus.values():
      if groupName is not None and groupName != templateStatus.templateName:
         continue
      templateModel = TemplateModel()
      templateModel.name = templateStatus.templateName
      templateModel.templateId = templateStatus.templateId
      templateModel.fields = []
      trackerModel.dataTemplates[ templateModel.templateId ] = templateModel
      for entry in templateStatus.templateEntry.values():
         field = TemplateEntryModel()
         field.fieldId = entry.fieldId
         field.fieldLen = entry.fieldLen
         element = ElementIdColl.elementIdColl[ field.fieldId ]
         field.fieldType = element.str
         field.enterpriseIe = entry.enterpriseIe
         templateModel.fields.append( field )
      if groupName is not None:
         break

def showTempateField( fields, fieldId ):
   field = TemplateEntryModel()
   field.fieldId = fieldId
   element = ElementIdColl.elementIdColl[ fieldId ]
   field.fieldLen = element.len
   field.fieldType = element.str
   fields.append( field )

def showOptTemplateIntfTable( trackerModel ):
   intfTable = TemplateModel()
   intfTable.name = "Interface Mapping"
   intfTable.templateId = FtConsts.reservedTemplateId(
      StaticTemplateIdEnum.optionsTemplateIntfTable )
   showTempateField( intfTable.scopeFields, IpfixRecordType.ingressInterface )
   showTempateField( intfTable.fields, IpfixRecordType.interfaceName )
   trackerModel.optionsTemplates[ intfTable.templateId ] = intfTable

def showOptTemplateVrfTable( trackerModel ):
   vrfTable = TemplateModel()
   vrfTable.name = "VRF Mapping"
   vrfTable.templateId = FtConsts.reservedTemplateId(
      StaticTemplateIdEnum.optionsTemplateVrfTable )
   showTempateField( vrfTable.scopeFields, IpfixRecordType.vrfId )
   showTempateField( vrfTable.fields, IpfixRecordType.vrfName )
   trackerModel.optionsTemplates[ vrfTable.templateId ] = vrfTable

def showOptTemplateFlowKey( trackerModel ):
   flowKey = TemplateModel()
   flowKey.name = "Flow Key"
   flowKey.templateId = FtConsts.reservedTemplateId(
      StaticTemplateIdEnum.optionsTemplateFlowKeys )
   showTempateField( flowKey.scopeFields, IpfixRecordType.templateId )
   showTempateField( flowKey.fields, IpfixRecordType.flowKeyIndicator )
   trackerModel.optionsTemplates[ flowKey.templateId ] = flowKey

def showOptTemplateFlowTrackerTable( trackerModel ):
   trTable = TemplateModel()
   trTable.name = "Tracker"
   trTable.templateId = FtConsts.reservedTemplateId(
      StaticTemplateIdEnum.optionsTemplateFlowTrackerTable )
   showTempateField( trTable.scopeFields, IpfixRecordType.observationDomainId )
   showTempateField( trTable.fields, IpfixRecordType.observationDomainName )
   showTempateField( trTable.fields, IpfixRecordType.flowActiveTimeout )
   showTempateField( trTable.fields, IpfixRecordType.flowIdleTimeout )
   showTempateField( trTable.fields, IpfixRecordType.selectorAlgorithm )
   showTempateField( trTable.fields, IpfixRecordType.samplingSize )
   showTempateField( trTable.fields, IpfixRecordType.samplingPopulation )
   showTempateField( trTable.fields, IpfixRecordType.flowTrackingType )
   trackerModel.optionsTemplates[ trTable.templateId ] = trTable

def showOptionsTemplate( trackerModel ):
   trackerModel.optionsTemplates = {}
   showOptTemplateIntfTable( trackerModel )
   showOptTemplateVrfTable( trackerModel )
   showOptTemplateFlowKey( trackerModel )
   showOptTemplateFlowTrackerTable( trackerModel )

def showTemplate( mode, args ):
   ftTemplateModel = FtTemplateModel()

   ftrType = getFtrTypeFromArgs( args )
   if not flowTrackingAgentRunning( ftrType, entityManager, activeAgentDir ):
      return ftTemplateModel

   trackerName = None
   groupName = None
   if args.get( 'TRACKER_NAME' ):
      trackerName = args.get( 'TRACKER_NAME' )
      t0( 'tracker Name: ', trackerName )

   ftTemplateModel.trackers = {}
   for trName, ftStatus in hwTrackingStatus[ ftrType ].hwFtStatus.items():
      if trackerName is not None and trackerName != trName:
         continue
      trackerModel = TrackerTemplateModel()
      ftTemplateModel.trackers[ trName ] = trackerModel
      if args.get( 'options' ):
         showOptionsTemplate( trackerModel )
      elif args.get( 'data' ):
         if args.get( 'GROUP_NAME' ):
            groupName = args.get( 'GROUP_NAME' )[ 0 ]
         showDataTemplate( trackerModel, ftStatus, groupName )
      else:
         showDataTemplate( trackerModel, ftStatus, groupName )
         showOptionsTemplate( trackerModel )
      if trackerName is not None:
         break
   return ftTemplateModel

sharedObj = object()

class ShowIpfixTemplate( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ( sampled | hardware '
   if Toggles.InbandTelemetryCommonToggleLib.toggleFeatureInbandTelemetryEnabled():
      syntax += '| ( telemetry inband ) '
   syntax += ') ipfix template [ { ( tracker TRACKER_NAME ) | ' + \
         '( ( data [ group GROUP_NAME ] ) | options ) } ]'

   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'hardware' : hardwareShowKw,
      'ipfix' : 'IPFIX format',
      'template' : 'Flow data and options template',
      'tracker' : CliCommand.Node( matcher=trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'data' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                             'data', 'Flow data template' ),
                  maxMatches=1,
                  sharedMatchObj=sharedObj ),
      'group' : groupKw,
      'GROUP_NAME' : groupNameMatcher,
      'options' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                                'options', 'Flow options template' ),
                  maxMatches=1,
                  sharedMatchObj=sharedObj ),
   }
   if Toggles.InbandTelemetryCommonToggleLib.toggleFeatureInbandTelemetryEnabled():
      data[ 'telemetry' ] = telemetryShowKw
      data[ 'inband' ] = inbandShowKw

   handler = showTemplate
   cliModel = FtTemplateModel

BasicCli.addShowCommandClass( ShowIpfixTemplate )

#--------------------------------------------------------------------------
# show flow tracking ( sampled | hardware | ( telemetry inband ) ) ipfix
# options - table [ tracker < name > ] [vrf|interface|flow-key|flow-tracker]
#--------------------------------------------------------------------------
def showOptVrfTable( trackerModel ):
   vrfTable = OptTableModel()
   trackerModel.tables[ 'VRF' ] = vrfTable
   vrfTable.entries = {}
   vrfTable.templateId = FtConsts.reservedTemplateId(
      StaticTemplateIdEnum.optionsTemplateVrfTable )
   element = ElementIdColl.elementIdColl[ IpfixRecordType.vrfId ]
   vrfTable.scopeFields = []
   vrfTable.scopeFields.append( element.str )
   for key, entry in arpVrfIdMap.vrfIdToName.items():
      vrfTable.entries[ long( key ) ] = entry.vrfName
   # Add nullVrf
   vrfTable.entries[ long( VrfIdType.null ) ] = FtConsts.nullVrfName

def showOptInterfaceTable( trackerModel, ftrType ):
   interfaceTable = OptTableModel()
   trackerModel.tables[ 'Interface' ] = interfaceTable
   interfaceTable.entries = {}
   interfaceTable.templateId = FtConsts.reservedTemplateId(
      StaticTemplateIdEnum.optionsTemplateIntfTable )
   element = ElementIdColl.elementIdColl[ IpfixRecordType.ingressInterface ]
   interfaceTable.scopeFields = []
   interfaceTable.scopeFields.append( element.str )
   ifIndexMap = getIfIndexMap( ethIntfStatusDir )
   for ifIndex, intfName in ifIndexMap.items():
      interfaceTable.entries[ long( ifIndex ) ] = intfName

   if ftrCapabilities[ ftrType ].tunnelIntfSupported:
      ifIndexMap = getIfIndexMap( tunnelIntfStatusDir )
      for ifIndex, intfName in ifIndexMap.items():
         interfaceTable.entries[ long( ifIndex ) ] = intfName

def showOptFlowKeyTable( trackerModel, ftStatus ):
   flowKeyTable = OptTableModel()
   trackerModel.tables[ 'Flow Keys' ] = flowKeyTable
   flowKeyTable.entries = {}
   flowKeyTable.templateId = FtConsts.reservedTemplateId(
      StaticTemplateIdEnum.optionsTemplateFlowKeys )
   element = ElementIdColl.elementIdColl[ IpfixRecordType.templateId ]
   flowKeyTable.scopeFields = []
   flowKeyTable.scopeFields.append( element.str )
   for templateStatus in ftStatus.templateStatus.values():
      flowKey = 0
      for seqno, entry in templateStatus.templateEntry.items():
         if entry.flowKey:
            flowKey |= ( 1 << seqno )
      flowKeyTable.entries[
            long( templateStatus.templateId ) ] = str( hex( flowKey ) )

def showOptFlowTrackerTable( ftrType, trackerModel, trackerName ):
   hwFtConfig = hwConfig[ ftrType ].hwFtConfig.get( trackerName )
   hwFtStatus = hwTrackingStatus[ ftrType ].hwFtStatus.get( trackerName )
   if hwFtConfig and hwFtStatus:
      trackerInfoModel = TrackerInfoModel()
      trackerModel.trackerInfo = trackerInfoModel
      trackerInfoModel.observationDomain = trackerName
      trackerInfoModel.observationDomainId = hwFtConfig.ftId
      trackerInfoModel.activeInterval = int( hwFtStatus.activeInterval / 1000 )
      trackerInfoModel.inactiveTimeout = int( hwFtStatus.inactiveTimeout / 1000 )
      trackerInfoModel.selectorAlgorithm = SelectorAlgorithmType.randomNofNSampling
      trackerInfoModel.sampleRate = hwTrackingStatus[ ftrType ].sampleRate
      trackerInfoModel.flowTrackingType = ftrType

def showOptionsTable( mode, args ):
   ftOptTableModel = FtOptTableModel()

   ftrType = getFtrTypeFromArgs( args )
   if not flowTrackingAgentRunning( ftrType, entityManager, activeAgentDir ):
      return ftOptTableModel

   trackerName = args.get( 'TRACKER_NAME' )

   ftOptTableModel.trackers = {}
   for trName, ftStatus in hwTrackingStatus[ ftrType ].hwFtStatus.items():
      if trackerName is not None and trackerName != trName:
         continue
      trackerModel = TrackerOptTableModel()
      ftOptTableModel.trackers[ trName ] = trackerModel
      trackerModel.tables = {}
      if args.get( 'vrf' ):
         showOptVrfTable( trackerModel )
      elif args.get( 'interface' ):
         showOptInterfaceTable( trackerModel, ftrType )
      elif args.get( 'flow-key' ):
         showOptFlowKeyTable( trackerModel, ftStatus )
      elif args.get( 'flow-tracker' ):
         showOptFlowTrackerTable( ftrType, trackerModel, trName )
      else:
         showOptVrfTable( trackerModel )
         showOptInterfaceTable( trackerModel, ftrType )
         showOptFlowKeyTable( trackerModel, ftStatus )
         showOptFlowTrackerTable( ftrType, trackerModel, trName )
      if trackerName is not None:
         break
   return ftOptTableModel

class ShowIpfixOptionsTable( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking ( sampled | hardware '
   if Toggles.InbandTelemetryCommonToggleLib.toggleFeatureInbandTelemetryEnabled():
      syntax += '| ( telemetry inband ) '
   syntax += ') ipfix options-table [ tracker TRACKER_NAME ] ' + \
         '[ vrf | interface | flow-key | flow-tracker ]'

   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'hardware' : hardwareShowKw,
      'ipfix' : 'IPFIX format',
      'options-table' : 'Flow options table',
      'tracker' : trackerKw,
      'TRACKER_NAME' : trackerNameMatcher,
      'vrf' : 'VRF options table',
      'interface' : 'Interface options table',
      'flow-key' : 'Flow keys options table',
      'flow-tracker' : 'Flow tracker options table'
   }
   if Toggles.InbandTelemetryCommonToggleLib.toggleFeatureInbandTelemetryEnabled():
      data[ 'telemetry' ] = telemetryShowKw
      data[ 'inband' ] = inbandShowKw

   handler = showOptionsTable
   cliModel = FtOptTableModel

BasicCli.addShowCommandClass( ShowIpfixOptionsTable )

#--------------------------
def Plugin( em ):
   global activeAgentDir
   global arpVrfIdMap
   global entityManager
   global ethIntfStatusDir, tunnelIntfStatusDir
   global trackingConfig, hwConfig, hwTrackingStatus, ftrCapabilities

   entityManager = em
   activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                     'Tac::Dir', 'ri' )
   ethIntfStatusDir = LazyMount.mount( em, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
   tunnelIntfStatusDir = LazyMount.mount( em, "interface/status/tunnel/intf",
                                       "Interface::TunnelIntfStatusDir", "r" )

   for ftrType in [ ftrTypeSampled, ftrTypeHardware, ftrTypeInbandTelemetry ]:
      trackingConfig[ ftrType ] = LazyMount.mount( em,
                                       ftrConfigPathPrefix + ftrType,
                                       ftrConfigType, 'r' )
      hwConfig[ ftrType ] = LazyMount.mount( em,
                                       hwFtrConfigPathPrefix + ftrType,
                                       hwFtrConfigType, 'r' )
      hwTrackingStatus[ ftrType ] = LazyMount.mount( em,
                                       hwFtrStatusPathPrefix + ftrType,
                                       hwFtrStatusType, 'r' )
      ftrCapabilities[ ftrType ] = LazyMount.mount( em,
                                       ftrCapabilitiesPathPrefix + ftrType,
                                       ftrCapabilitiesType, 'r' )

   arpVrfIdMap = SmashLazyMount.mount( em, "vrf/vrfIdMapStatus",
                                       "Vrf::VrfIdMap::Status",
                                       SmashLazyMount.mountInfo( 'reader' ) )
