# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import AgentDirectory
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.TunnelIntfCli as TunnelIntfCli
import Tracing
import SflowUtil
from if_ether_arista import (
   SFLOW_FORMAT_MULTIPLE_INTFS,
   SFLOW_FORMAT_PKT_DISCARD,
   SFLOW_INTERNAL_INTERFACE,
   SFLOW_OUTPUT_UNKNOWN,
)
from FlowTrackerCliUtil import (
   ftrCapabilitiesPathPrefix,
   ftrCapabilitiesType,
   ftrConfigPathPrefix,
   ftrConfigType,
   ftrTypes,
   ftrTypeHardware,
   ftrTypeSampled,
   ftrTypeInbandTelemetry,
   ftrTypeMirrorOnDrop,
   ftrTypeByName,
   ftrTypeKwStr,
   getFtrTypeFromArgs,
   hwFtrConfigPathPrefix,
   hwFtrConfigType,
   hwFtrStatusPathPrefix,
   hwFtrStatusType,
)

from CliMode.FlowTracking import FlowTrackingModeBase
from CliMode.FlowTracker import TrackerModeBase
from CliMode.FlowExporter import ExporterModeBase
from TypeFuture import TacLazyType
import LazyMount
from Toggles.FlowTrackerToggleLib import toggleCopyToCollectorEnabled

traceHandle = Tracing.Handle( 'FlowTrackingCliLib' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

ftrCapabilities = {}
sflowHwStatusDir = None
trackingConfig = {}
hwConfig = {}
hwStatus = {}

IP_GROUP = "IPv4"
IP6_GROUP = "IPv6"

FtConsts = TacLazyType( 'FlowTracking::Constants' )
# pkgdeps: rpm IntfSnmp-lib
IntfDetails = TacLazyType( 'IntfSnmp::IntfDetails' )
IntfIndexMapType = TacLazyType( 'FlowTracking::IntfIndexMap' )
IpfixRecordType = TacLazyType( 'Ipfix::RecordType' )
SelectorAlgorithmType = TacLazyType( 'Ipfix::SelectorAlgorithm' )

def isSftAgentRunning( entityManager, activeAgentDir ):
   return ( 'sft' in activeAgentDir and
         AgentDirectory.agent( entityManager.sysname(), 'SftAgent' ) is not None )

def isHwAccelAgentRunning( entityManager, activeAgentDir ):
   return ( 'hwAccel' in activeAgentDir and
         AgentDirectory.agent( entityManager.sysname(), 'StftAgent' ) is not None )

def isSfeAgentRunning( entityManager, activeAgentDir ):
   return ( 'bess' in activeAgentDir and
            AgentDirectory.agent( entityManager.sysname(), 'Sfe' ) is not None )

def isInbandTelemetryAgentRunning( entityManager, activeAgentDir ):
   return ( 'inbandTelemetry' in activeAgentDir and
         AgentDirectory.agent( entityManager.sysname(),
                               'InbandTelemetry' ) is not None )

def isMirrorOnDropAgentRunning( entityManager, activeAgentDir ):
   return ( 'StrataMirror' in activeAgentDir and
         AgentDirectory.agent( entityManager.sysname(),
                               'StrataMirror' ) is not None )

def flowTrackingAgentRunning( ftrType, entityManager, activeAgentDir ):
   if ftrType == ftrTypeSampled:
      return isSftAgentRunning( entityManager, activeAgentDir )
   if ftrType == ftrTypeHardware:
      return ( isHwAccelAgentRunning( entityManager, activeAgentDir ) or
               isSfeAgentRunning( entityManager, activeAgentDir ) )
   if ftrType == ftrTypeInbandTelemetry:
      return isInbandTelemetryAgentRunning( entityManager, activeAgentDir )
   if ftrType == ftrTypeMirrorOnDrop:
      return isMirrorOnDropAgentRunning( entityManager, activeAgentDir )
   return False

def sflowHwStatus():
   return SflowUtil.sflowHwStatus( sflowHwStatusDir )

def sampleRateRangeFn( mode ):
   return ( sflowHwStatus().minSampleRate, sflowHwStatus().maxSampleRate )

def swFtrSupported():
   return ftrCapabilities[ ftrTypeSampled ].supported

def hwFtrSupported():
   return ftrCapabilities[ ftrTypeHardware ].supported

def intFtrSupported():
   return ftrCapabilities[ ftrTypeInbandTelemetry ].supported

def modFtrSupported():
   return ftrCapabilities[ ftrTypeMirrorOnDrop ].supported

def swExportSupported( ftrType ):
   return ftrCapabilities[ ftrType ].swExportSupported

def hwOffloadSupported( ftrType ):
   return ftrCapabilities[ ftrType ].hwOffloadIpv4

def hwOffloadIpv4Supported( ftrType ):
   return ftrCapabilities[ ftrType ].hwOffloadIpv4

def flowGroupConfigSupported( ftrType ):
   return ( toggleCopyToCollectorEnabled() and
            ftrCapabilities[ ftrType ].flowGroupConfigSupported )

def mirroringSupported( ftrType ):
   return ( toggleCopyToCollectorEnabled() and 
            ftrCapabilities[ ftrType ].mirroringSupported )

def tunnelIntfFtrSupported( ftrType ):
   return ftrCapabilities[ ftrType ].tunnelIntfSupported

def getIfIndexMap( ethIntfStatusDir ):
   ifIndexMap = {}
   intfDetails = IntfDetails()
   for intf, intfStatus in ethIntfStatusDir.intfStatus.items():
      intfDetails.status = intfStatus
      ifIndexMap[ intfDetails.ifIndex ] = intf
   # also add mapping for 4 known intfs
   ifIndexMap[ SFLOW_FORMAT_PKT_DISCARD ] = IntfIndexMapType.discardIntfName
   ifIndexMap[ SFLOW_OUTPUT_UNKNOWN ] = IntfIndexMapType.unknownIntfName
   ifIndexMap[ SFLOW_INTERNAL_INTERFACE ] = IntfIndexMapType.cpuIntfName
   ifIndexMap[ SFLOW_FORMAT_MULTIPLE_INTFS ] = IntfIndexMapType.multicastIntfName
   return ifIndexMap

#-------------------------------------------------------------------------------
# Guard functions to prevent configuration on systems without flow tracker support
#-------------------------------------------------------------------------------
def guardFlowTracking( mode, token ):
   if swFtrSupported() or hwFtrSupported() or intFtrSupported() or modFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFlowTracker( mode, token ):
   return None

def guardIntfFtrSampled( mode, token ):
   if isinstance( mode.intf, TunnelIntfCli.TunnelIntf ):
      # tunnelIntfFtrSupported capability flag is ignored
      # as sampled ftr is not supported for Tunnel interface
      return CliParser.guardNotThisPlatform

   if swFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardIntfFtrHardware( mode, token ):
   if ( isinstance( mode.intf, TunnelIntfCli.TunnelIntf ) and
        not tunnelIntfFtrSupported( ftrTypeHardware ) ):
      return CliParser.guardNotThisPlatform

   if hwFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardSampleRate( mode, token ):
   if mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeSampled ]:
      return None
   return CliParser.guardNotThisPlatform

def guardFtrSampled( mode, token ):
   if swFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrTelemetry( mode, token ):
   if intFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrHardware( mode, token ):
   if hwFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardFtrMirrorOnDrop( mode, token ):
   if modFtrSupported():
      return None
   return CliParser.guardNotThisPlatform

def guardExportOnTcpState( mode, token ):
   t0( 'guardExportOnTcpState:', mode, mode.ftrTypeStr )
   if ftrCapabilities[ ftrTypeByName[ mode.ftrTypeStr ] ].exportOnTcpStateSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardSwExport( mode, token ):
   if swExportSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardSampledHwOffload( mode, token ):
   if hwOffloadSupported( ftrTypeSampled ):
      return None
   return CliParser.guardNotThisPlatform

def guardHwOffload( mode, token ):
   if hwOffloadSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardHwOffloadIpv4( mode, token ):
   if hwOffloadIpv4Supported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardFlowGroupConfig( mode, token ):
   if flowGroupConfigSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def guardShowFlowGroupConfig( mode, token ):
   # During show commands, the mode is enable mode by default and it does not
   # have access to ftrType
   if flowGroupConfigSupported( ftrTypeHardware ):
      return None
   return CliParser.guardNotThisPlatform

def guardMirroring( mode, token ):
   if mirroringSupported( ftrTypeByName[ mode.ftrTypeStr ] ):
      return None
   return CliParser.guardNotThisPlatform

def getTrackerNamesAndIds( ftrType ):
   trackers = [ ( f.name, f.ftId ) for f in
                  hwConfig[ ftrType ].hwFtConfig.values() ]
   return trackers

def getFlowGroups( ftrType, trackerName ):
   if trackerName not in hwConfig[ ftrType ].hwFtConfig:
      return []
   groups = [ g.fgName for g in
               hwConfig[ ftrType ].hwFtConfig[ trackerName ].hwFgConfig.values() ]
   return groups

def getFlowGroupsAndIds( ftrType, trackerName ):
   if trackerName not in hwConfig[ ftrType ].hwFtConfig:
      return []
   groups = [ ( g.fgName, g.templateId ) for g in
               hwConfig[ ftrType ].hwFtConfig[ trackerName ].hwFgConfig.values() ]
   return groups

def getHwFlowGroupIds( ftrType, trackerName, groupName ):
   hwFtStatus = hwStatus[ ftrType ].hwFtStatus.get( trackerName )
   if hwFtStatus is None:
      return []

   hwFgStatus = hwFtStatus.hwFgStatus.get( groupName )
   if hwFgStatus is None:
      return []

   return hwFgStatus.hwFgId.members()

def getConfigToHwFgConfigMap( ftrType, trackerName ):
   hwFtConfig = hwConfig[ ftrType ].hwFtConfig.get( trackerName )
   if hwFtConfig is None:
      return []
   groups = [ ( g.fgName, g.seqno, g.hwFgNameAndSeqno ) for g in
              hwFtConfig.hwFgConfigMap.values() ]
   return groups

# Config Keywords
trackingConfigKw = CliCommand.guardedKeyword( 'tracking',
                        helpdesc='Configure flow tracking',
                        guard=guardFlowTracking )

sampledConfigKw = CliCommand.guardedKeyword(
                        'sampled',
                        helpdesc='Configure sampled flow tracking',
                        guard=guardFtrSampled )

telemetryConfigKw = CliCommand.guardedKeyword( 'telemetry',
                        helpdesc='Configure sampled telemetry',
                        guard=guardFtrTelemetry )

inbandConfigKw = CliCommand.guardedKeyword( 'inband',
                        helpdesc='Configure sampled inband telemetry',
                        guard=guardFtrTelemetry )

hardwareConfigKw = CliCommand.guardedKeyword(
                        'hardware',
                        helpdesc='Configure hardware flow tracking',
                        guard=guardFtrHardware )

mirrorOnDropConfigKw = CliCommand.guardedKeyword(
                           'mirror-on-drop',
                           helpdesc='Configure mirror on drop feature',
                           guard=guardFtrMirrorOnDrop )

matcherRecord = CliMatcher.KeywordMatcher( 'record',
   helpdesc='Set characteristics of flow tracking record' )

# Show Keywords
trackingShowKw = CliCommand.guardedKeyword( 'tracking',
                     helpdesc='Show flow tracking',
                     guard=guardFlowTracking )

sampledShowKw = CliCommand.guardedKeyword( 'sampled',
                        helpdesc='Show sampled flow tracking information',
                        guard=guardFtrSampled,
                        storeSharedResult=True )

hardwareShowKw = CliCommand.guardedKeyword( 'hardware',
                        helpdesc='Show hardware flow tracking information',
                        guard=guardFtrHardware,
                        storeSharedResult=True )

telemetryShowKw = CliCommand.guardedKeyword( 'telemetry',
                        helpdesc='Show inband telemetry flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

inbandShowKw = CliCommand.guardedKeyword( 'inband',
                        helpdesc='Show inband telemetry flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

mirrorOnDropShowKw = CliCommand.guardedKeyword( 'mirror-on-drop',
                        helpdesc='Show mirror on drop flow tracking information',
                        guard=guardFtrMirrorOnDrop,
                        storeSharedResult=True )

groupShowKw = CliCommand.guardedKeyword( 'group',
                        helpdesc='Show flow group',
                        guard=guardShowFlowGroupConfig )

trackerKw = CliMatcher.KeywordMatcher( 'tracker', helpdesc='Flow tracker' )

groupKw = CliMatcher.KeywordMatcher( 'group', helpdesc='Flow group' )

exporterKw = CliMatcher.KeywordMatcher( 'exporter', helpdesc='Flow exporter' )

flowTableKw = CliMatcher.KeywordMatcher( 'flow-table', helpdesc='Flow table' )

countersKw = CliMatcher.KeywordMatcher( 'counters', helpdesc='Counters' )

interfaceKw = CliMatcher.KeywordMatcher( 'interface',
                                         helpdesc='Per interface flow count' )

# Clear 
trackingClearKw = CliCommand.guardedKeyword( 'tracking',
                        helpdesc='Clear flow tracking',
                        guard=guardFlowTracking )

sampledClearKw = CliCommand.guardedKeyword( 'sampled',
                        helpdesc='Clear sampled flow tracking information',
                        guard=guardFtrSampled,
                        storeSharedResult=True )

hardwareClearKw = CliCommand.guardedKeyword( 'hardware',
                        helpdesc='Clear hardware flow tracking information',
                        guard=guardFtrHardware,
                        storeSharedResult=True )

mirrorOnDropClearKw = CliCommand.guardedKeyword( 'mirror-on-drop',
                        helpdesc='Clear mirror on drop flow tracking information',
                        guard=guardFtrMirrorOnDrop,
                        storeSharedResult=True )

telemetryClearKw = CliCommand.guardedKeyword( 'telemetry',
                        helpdesc='Clear telemetry inband flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

inbandClearKw = CliCommand.guardedKeyword( 'inband',
                        helpdesc='Clear telemetry inband flow tracking information',
                        guard=guardFtrTelemetry,
                        storeSharedResult=True )

# Tracker Name
def getTrackerNamesFromContext( mode, context ):
   ftrType = getFtrTypeFromArgs( context.sharedResult )
   return trackingConfig[ ftrType ].flowTrackerConfig

trackerNameMatcher = CliCommand.Node( 
                        CliMatcher.DynamicNameMatcher( getTrackerNamesFromContext,
                                                       "Flow tracker name",
                                                       passContext=True ),
                         storeSharedResult=True, maxMatches=1 )

# Group
def getGroupNamesFromContext( mode, context ):
   ftrType = getFtrTypeFromArgs( context.sharedResult )
   trackerName = context.sharedResult[ 'TRACKER_NAME' ]
   if trackerName in trackingConfig[ ftrType ].flowTrackerConfig:
      return trackingConfig[ ftrType ].flowTrackerConfig[
                                 trackerName ].fgConfig
   else:
      return []

allGroupNameMatcher = CliCommand.Node(
                        CliMatcher.DynamicNameMatcher( [ IP_GROUP, IP6_GROUP ],
                                                       "Flow group name",
                                                       passContext=True ),
                         storeSharedResult=True, maxMatches=1 )

groupNameMatcher = CliCommand.Node( 
                        CliMatcher.DynamicNameMatcher( getGroupNamesFromContext,
                                                       "Flow group name",
                                                       passContext=True ),
                         storeSharedResult=True )

# Exporter
def getExporterNamesFromContext( mode, context ):
   ftrType = getFtrTypeFromArgs( context.sharedResult )
   trackerName = context.sharedResult[ 'TRACKER_NAME' ]
   if trackerName in trackingConfig[ ftrType ].flowTrackerConfig:
      return trackingConfig[ ftrType ].flowTrackerConfig[
                                 trackerName ].expConfig
   else:
      return []

exporterNameMatcher = CliMatcher.DynamicNameMatcher( getExporterNamesFromContext,
                                                   "Exporter name",
                                                   passContext=True )

class FlowTrackingMode( FlowTrackingModeBase, BasicCli.ConfigModeBase ):
   name = 'Flow tracking configuration'
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, context ):
      self.context_ = context
      self.session_ = session
      FlowTrackingModeBase.__init__( self, ftrTypeKwStr[ context.ftrType ] )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'FlowTrackingMode onExit ...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def commitContext( self ):
      if self.context_ is None:
         t1( 'FlowTrackingMode.commitContext has no context' )
         return
      context = self.context_
      self.context_ = None
      context.commit()

   def abort( self ):
      t1( 'FlowTrackingMode.abort' )
      self.context_ = None
      self.session_.gotoParentMode()

   def context( self ):
      return self.context_

class TrackerMode( TrackerModeBase, BasicCli.ConfigModeBase ):
   name = 'Tracker configuration'
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, ftrCtx, trCtx ):
      self.ftrCtx_ = ftrCtx
      self.context_ = trCtx
      self.session_ = session
      TrackerModeBase.__init__( self, ( ftrTypeKwStr[ trCtx.ftrType() ],
                                        trCtx.trackerName() ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'TrackerMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      t1( 'FlowTracker.abort' )
      trName = self.trackerName()
      self.context_ = None
      self.session_.gotoParentMode()
      ftrCtx = self.ftrContext()
      del ftrCtx.trCtx[ trName ]
      self.ftrCtx_ = None

   def commitContext( self ):
      if self.context_ is None:
         t1( 'FlowTracker.commitContext has no context' )
         return
      context = self.context_
      self.context_ = None
      context.commit()
      self.ftrCtx_ = None

   def trConfig( self ):
      return self.context_.trConfig

   def ftrContext( self ):
      return self.ftrCtx_

   def context( self ):
      return self.context_

   def trackerName( self ):
      return self.context_.trackerName()

class ExporterMode( ExporterModeBase, BasicCli.ConfigModeBase ):
   name = 'Exporter configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context, exporterName ):
      self.session_ = session
      self.context_ = context
      ExporterModeBase.__init__( self,
                           ( ftrTypeKwStr[ context.ftrType() ],
                             context.trackerName(), exporterName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def context( self ):
      return self.context_

def Plugin( em ):
   global ftrCapabilities
   global sflowHwStatusDir
   global trackingConfig
   global hwConfig
   global hwStatus

   sflowHwStatusDir = LazyMount.mount( em, 'sflow/hwstatus', 'Tac::Dir', 'ri' )

   for ftrType in ftrTypes:
      ftrCapabilities[ ftrType ] = LazyMount.mount( em,
                                    ftrCapabilitiesPathPrefix + ftrType,
                                    ftrCapabilitiesType, 'r' )
      hwConfig[ ftrType ] = LazyMount.mount( em, 
                                    hwFtrConfigPathPrefix + ftrType,
                                    hwFtrConfigType, 'r' )
      hwStatus[ ftrType ] = LazyMount.mount( em,
                                    hwFtrStatusPathPrefix + ftrType,
                                    hwFtrStatusType, 'r' )
      trackingConfig[ ftrType ] = LazyMount.mount( em,
                                    ftrConfigPathPrefix + ftrType,
                                    ftrConfigType, 'r' )
