# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import sys
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
from CliToken.Flow import flowMatcherForConfig, flowMatcherForClear
from CliToken.Clear import clearKwNode
import Tracing
from FlowTrackerConst import (
      sampleRate,
      spaceConst,
)
from FlowTrackerCliUtil import (
      ftrConfigPathPrefix,
      ftrConfigReqPathPrefix,
      ftrConfigType,
      ftrConfigReqType,
      ftrTypes,
      ftrTypeKwStr,
      getFtrTypeFromArgs,
      showExporter,
      showFlowTracking,
      showFlowTrackingTrailer,
      showFtr,
      showGroup,
)
from FlowTrackingCliLib import (
      FlowTrackingMode,
      trackingConfigKw,
      sampledConfigKw,
      telemetryConfigKw,
      inbandConfigKw,
      mirrorOnDropConfigKw,
      matcherRecord,
      hardwareConfigKw,
      guardSwExport,
      guardHwOffload,
      guardHwOffloadIpv4,
      trackingClearKw,
      sampledClearKw,
      hardwareClearKw,
      telemetryClearKw,
      inbandClearKw,
      mirrorOnDropClearKw,
      trackerKw,
      trackerNameMatcher,
      exporterKw,
      exporterNameMatcher,
      guardSampleRate,
      sampleRateRangeFn,
)
from FlowTrackerCli import TrackerContext, AbortCmd
from ShowCommand import ShowCliCommandClass
from Toggles.InbandTelemetryCommonToggleLib import \
                  toggleFeatureInbandTelemetryEnabled
from Toggles.FlowTrackerToggleLib import toggleHwOffloadIpv4Enabled

traceHandle = Tracing.Handle( 'FlowTrackingCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2
t3 = traceHandle.trace3

trackingConfig = {}
trackingConfigReq = {}

class FlowTrackingContext( object ):
   def __init__( self, config ):
      self.config = config
      self.ftrType = config.name
      self.enabled = self.config.enabled
      self.sampleRate = self.config.sampleRate
      self.swExport = self.config.swExport
      if toggleHwOffloadIpv4Enabled():
         self.hwOffloadIpv4 = self.config.hwOffloadIpv4
      self.trCtx = {}

   def commit( self ):
      self.config.sampleRate = self.sampleRate
      self.config.enabled = self.enabled
      self.config.swExport = self.swExport
      if toggleHwOffloadIpv4Enabled():
         self.config.hwOffloadIpv4 = self.hwOffloadIpv4

   def newTrackerContext( self, trName ):
      trConfig = None
      if trName in self.config.flowTrackerConfig:
         # Config changed after entering flow-tracking mode.
         # Create context with existing Sysdb config.
         trConfig = self.config.flowTrackerConfig[ trName ]
      self.trCtx[ trName ] = TrackerContext(
                              self.config, self.config.name, trName, trConfig )
      return self.trCtx[ trName ]

   def trackerContext( self, trName ):
      if trName in self.trCtx:
         return self.trCtx[ trName ]
      else:
         return None

   def trackingConfig( self ):
      return self.config

#-------------------------------------------------------------------------------
# "[no|default] flow tracking (sampled|hardware|(telemetry inband)|mirror-on-drop)"
# in config mode
#-------------------------------------------------------------------------------

def gotoFlowTrackingMode( mode, ftrType ):
   t1( 'gotoFlowTrackingMode', ftrType )
   context = FlowTrackingContext( trackingConfig[ ftrType ] )
   childMode = mode.childMode( FlowTrackingMode, context=context )
   mode.session_.gotoChildMode( childMode )

def noFlowTrackingMode( mode, ftrType ):
   t1( 'noFlowTrackingMode', ftrType )
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, FlowTrackingMode ) and lastMode.context() ):
      # If no flow tracking is issued in flow-tracking mode then delete
      # context to avoid committing it again
      lastMode.context_ = None
   trackingConfig[ ftrType ].enabled = False
   trackingConfig[ ftrType ].sampleRate = sampleRate.rateDefault
   trackingConfig[ ftrType ].flowTrackerConfig.clear()

class FlowTrackingConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'flow tracking ( sampled | hardware '
   if toggleFeatureInbandTelemetryEnabled():
      syntax += '| ( telemetry inband ) '
   syntax += '| mirror-on-drop )'
   
   noOrDefaultSyntax = syntax

   data = {
         'flow' : flowMatcherForConfig,
         'tracking' : trackingConfigKw,
         'sampled' : sampledConfigKw,
         'hardware' : hardwareConfigKw,
         'mirror-on-drop' : mirrorOnDropConfigKw,
   }
   
   if toggleFeatureInbandTelemetryEnabled():
      data[ 'telemetry' ] = telemetryConfigKw
      data[ 'inband' ] = inbandConfigKw

   @staticmethod
   def handler( mode, args ):
      gotoFlowTrackingMode( mode, getFtrTypeFromArgs( args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noFlowTrackingMode( mode, getFtrTypeFromArgs( args ) )

BasicCli.GlobalConfigMode.addCommandClass( FlowTrackingConfigCmd ) 

#-------------------------------------------------------------------------------
# "clear flow tracking (sampled | hardware | (telemetry inband) | 
#  mirror-on-drop) counters [tracker <name> [ exporter <name> ] ]"
#-------------------------------------------------------------------------------

def clearFlowTrackingCounters( mode, args ):
   ftrType = getFtrTypeFromArgs( args )
   config = trackingConfig[ ftrType ]
   configReq = trackingConfigReq[ ftrType ]

   tracker = args.get( 'TRACKER_NAME' )
   exporter = args.get( 'EXPORTER_NAME' )
   if tracker:
      trConfig = config.flowTrackerConfig.get( tracker )
      if not trConfig:
         return
      if exporter:
         expConfig = trConfig.expConfig.get( exporter )
         if not expConfig:
            return
      else:
         exporter = ""
   else:
      tracker = ""
      exporter = ""

   configReq.clearCounters = Tac.Value( "FlowTracking::ClearCountersReq",
                                        tracker=tracker,
                                        exporter=exporter,
                                        reqTime=Tac.now() )

class FlowTrackingClearCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear flow tracking ( sampled | hardware '
   if toggleFeatureInbandTelemetryEnabled():
      syntax += '| ( telemetry inband ) '
   syntax += '| mirror-on-drop ) counters '\
             '[ tracker TRACKER_NAME [ exporter EXPORTER_NAME ] ]'
   data = {
      'clear' : clearKwNode,
      'flow' : flowMatcherForClear,
      'tracking' : trackingClearKw,
      'sampled' : sampledClearKw,
      'hardware' : hardwareClearKw,
      'mirror-on-drop' : mirrorOnDropClearKw,
      'counters' : 'Clear flow tracking counters',
      'tracker' : trackerKw,
      'TRACKER_NAME' : trackerNameMatcher,
      'exporter' : exporterKw,
      'EXPORTER_NAME' : exporterNameMatcher,
   }
   if toggleFeatureInbandTelemetryEnabled():
      data[ 'telemetry' ] = telemetryClearKw
      data[ 'inband' ] = inbandClearKw

   @staticmethod
   def handler( mode, args ):
      clearFlowTrackingCounters( mode, args )

BasicCli.EnableMode.addCommandClass( FlowTrackingClearCountersCmd )

#-----------------------------------------------------------------
# "[no|default] sample <rate>" command in "config-flow-tracking" mode
#-----------------------------------------------------------------

class FtrSampleRateCmd( CliCommand.CliCommandClass ):
   syntax = 'sample RATE'
   noOrDefaultSyntax = 'sample ...'

   data =  {
      'sample' : CliCommand.guardedKeyword( 'sample',
                        helpdesc='Set sample characteristics for flow tracking',
                        guard=guardSampleRate ),
      'RATE' : CliMatcher.DynamicIntegerMatcher( sampleRateRangeFn,
                        helpdesc='Sampling rate' ),
   }
   @staticmethod
   def handler( mode, args ):
      mode.context().sampleRate = args.get( 'RATE', sampleRate.rateDefault )

   noOrDefaultHandler = handler

FlowTrackingMode.addCommandClass( FtrSampleRateCmd )

#-----------------------------------------------------------------
# "[no|defalt] shutdown" command in "config-flow-tracking" mode
#-----------------------------------------------------------------

class FtrShutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax

   data = {
      'shutdown' : CliMatcher.KeywordMatcher( 
                     'shutdown',
                     helpdesc="Enable or disable the flow tracking feature" )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().enabled = False

   defaultHandler = handler
       
   @staticmethod
   def noHandler( mode, args ):
      mode.context().enabled = True

FlowTrackingMode.addCommandClass( FtrShutdown )

#-----------------------------------------------------------------
# "[no|default] record format ipfix standard timestamps counters" command in
# "config-flow-tracking" mode
#-----------------------------------------------------------------

matcherFormat = CliMatcher.KeywordMatcher( 'format',
   helpdesc='Configure flow record format' )
matcherIpfix = CliMatcher.KeywordMatcher( 'ipfix',
   helpdesc='Configure IPFIX record format' )
matcherStandard = CliMatcher.KeywordMatcher( 'standard',
   helpdesc='Configure IPFIX record format standard' )
matcherTimestamps = CliMatcher.KeywordMatcher( 'timestamps',
   helpdesc='Configure IPFIX standard record format for timestamps' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
   helpdesc='Configure IPFIX standard record format for counters' )

class RecordFormatIpfixCmd( CliCommand.CliCommandClass ):
   syntax = 'record format ipfix standard timestamps counters'
   noOrDefaultSyntax = 'record format ipfix standard timestamps counters'
   data = {
      'record': CliCommand.Node( matcherRecord,
                                 guard=guardSwExport ),
      'format': matcherFormat,
      'ipfix': matcherIpfix,
      'standard': matcherStandard,
      'timestamps': matcherTimestamps,
      'counters': matcherCounters
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().swExport = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().swExport = False

FlowTrackingMode.addCommandClass( RecordFormatIpfixCmd )

if toggleHwOffloadIpv4Enabled():
   matchHardware = CliMatcher.KeywordMatcher( 'hardware',
      helpdesc='Configure hardware parameters' )

   matchOffload = CliMatcher.KeywordMatcher( 'offload',
      helpdesc='Configure hardware offload' )

   matchHwOffloadIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
      helpdesc='Configure hardware offload for IPv4 traffic' )

   class HwOffloadIpv4Cmd( CliCommand.CliCommandClass ):
      syntax = 'hardware offload ipv4'
      noOrDefaultSyntax = 'hardware offload ipv4'
      data = {
         'hardware' : CliCommand.Node( matchHardware, guard=guardHwOffload ),
         'offload' : matchOffload,
         'ipv4' : CliCommand.Node( matchHwOffloadIpv4, guard=guardHwOffloadIpv4 ),
      }

      @staticmethod
      def handler( mode, args ):
         mode.context().hwOffloadIpv4 = True

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         mode.context().hwOffloadIpv4 = False

   FlowTrackingMode.addCommandClass( HwOffloadIpv4Cmd )

# Simply abort configuring the current mode.
FlowTrackingMode.addCommandClass( AbortCmd )

#-----------------------------------------------------------------
# show active/pending/diff
#-----------------------------------------------------------------

def _showPending( mode, output=None ):
   ftrCtx = mode.context()
   if ftrCtx is None:
      return
   if output is None:
      output = sys.stdout
   t3( 'ftrCtx.enabled', ftrCtx.enabled )
   ftrLines = showFlowTracking( ftrCtx, ftrCtx.ftrType, cliSave=False )
   for line in ftrLines:
      output.write( line + '\n' )
   t3( 'ftrCtx.trCtx: ', ftrCtx.trCtx )
   # Create trackerCtx that are:
   # 1. Not visited by this CLI session.
   # 2. Added directly in Sysdb or by some other CLI
   config = mode.context().trackingConfig()
   for trName in config.flowTrackerConfig:
      if trName not in ftrCtx.trCtx:
         t0( 'Creating new tracker context', trName )
         ftrCtx.trCtx[ trName ] = TrackerContext(
                                    config,
                                    ftrCtx.ftrType,
                                    trName,
                                    config.flowTrackerConfig[ trName ] )
   
   if ftrCtx.trCtx and not ftrLines:
      output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrCtx.ftrType ] )

   for trName in sorted( ftrCtx.trCtx ):
      trCtx = ftrCtx.trCtx[ trName ]
      showFtr( output, trName, trCtx, cliSave=False,
               space=spaceConst )
      t3( 'trCtx.exporterCtx: ', trCtx.exporterCtx )
      for expName in sorted( trCtx.exporterCtx ):
         expCtx = trCtx.exporterCtx[ expName ]
         showExporter( output, expName, expCtx, cliSave=False,
                       space=spaceConst )
      if trCtx.groupCtx:
         groupsSpace = spaceConst * 2
         output.write( '%sgroups\n' % groupsSpace )
         for groupSeqno in sorted( trCtx.seqnoToGroupMap ):
            groupName = trCtx.seqnoToGroupMap[ groupSeqno ]
            showGroup( output, groupName, trCtx.groupCtx[ groupName ],
                       cliSave=False, space=groupsSpace )

   lines = showFlowTrackingTrailer( ftrCtx, ftrCtx.ftrType, cliSave=False )
   
   # Only Trailer is non-empty, so display flow tracking before trailer lines
   if not ftrCtx.trCtx and not ftrLines and lines:
      output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrCtx.ftrType ] )
   for line in lines:
      output.write( line + '\n' )

def _showActive( mode, output=None ):
   config = mode.context().trackingConfig()
   t3( 'config: ', config )
   t3( 'config.enabled', config.enabled )
   t3( 'ftrCtx.enabled', mode.context().enabled )
   if config is None:
      return
   if output is None:
      output = sys.stdout
   lines = showFlowTracking( config, config.name, cliSave=False )
   for line in lines:
      output.write( line + '\n' )
   t3( 'trConfigs: ', config.flowTrackerConfig.iterkeys() )

   printHeader = not lines

   for trName in sorted( config.flowTrackerConfig.iterkeys() ):
      if printHeader:
         output.write( 'flow tracking %s\n' % ftrTypeKwStr[ config.name ] )
         printHeader = False
      trConfig = config.flowTrackerConfig[ trName ]
      showFtr( output, trConfig.name, trConfig, cliSave=False,
               space=spaceConst )
      for expName in sorted( trConfig.expConfig.iterkeys() ):
         expConfig = trConfig.expConfig[ expName ]
         showExporter( output, expName, expConfig, cliSave=False,
                       space=spaceConst )
      if trConfig.fgConfig:
         seqnoToGroupMap = {}
         for groupName, groupConfig in trConfig.fgConfig.items():
            seqnoToGroupMap[ groupConfig.seqno ] = groupName
         groupsSpace = spaceConst * 2
         output.write( '%sgroups\n' % groupsSpace )
         for groupSeqno in sorted( seqnoToGroupMap ):
            groupName = seqnoToGroupMap[ groupSeqno ]
            showGroup( output, groupName, trConfig.fgConfig[ groupName ],
                       cliSave=False, space=groupsSpace )

   lines = showFlowTrackingTrailer( config, config.name, cliSave=False )
   # Only Trailer is non-empty, so display flow tracking before trailer lines
   if printHeader and lines:
      output.write( 'flow tracking %s\n' % ftrTypeKwStr[ config.name ] )
   for line in lines:
      output.write( line + '\n' )

def _showDiff( mode ):
   # generate diff between active and pending
   import difflib
   import cStringIO
   activeOutput = cStringIO.StringIO()
   _showActive( mode, output=activeOutput )
   pendingOutput = cStringIO.StringIO()
   _showPending( mode, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print '\n'.join( list( diff ) )

class ShowFtr( ShowCliCommandClass ):
   syntax = 'show ( pending | active | diff )'

   data = {
      'pending': 'Display the new flow tracking configuration to be applied',
      'active': 'Display the flow tracking configuration in the running-config',
      'diff': ( 'Display the diff between active flow tracking configuration'
                'and to be applied' ),
    }
 
   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         _showDiff( mode )
      elif 'active' in args:
         _showActive( mode )
      else:
         _showPending( mode )
 
FlowTrackingMode.addShowCommandClass( ShowFtr )

#--------------------------
def Plugin( em ):
   global trackingConfig
   global trackingConfigReq

   for ftrType in ftrTypes:
      trackingConfig[ ftrType ] = ConfigMount.mount( em,
                                     ftrConfigPathPrefix + ftrType,
                                     ftrConfigType, 'w' )
      trackingConfigReq[ ftrType ] = ConfigMount.mount( em,
                                        ftrConfigReqPathPrefix + ftrType,
                                        ftrConfigReqType, 'w' )
