# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for FlowTracker configuration commands

import sys
import Tac
import Tracing
import ConfigMount
import IntfCli
import CliCommand
import CliMatcher
import CliToken.Cli
from CliToken.Flow import flowMatcherForConfigIf
from ShowCommand import ShowCliCommandClass
from VlanCli import SwitchportModelet
from CliPlugin.TunnelIntfCli import TunnelIntfConfigModelet
from FlowTrackerConst import (
      activeInterval,
      constants,
      inactiveTimeout,
      spaceConst,
)
from FlowTrackerCliUtil import (
      ftrConfigType,
      ftrConfigPathPrefix,
      ftrTypes,
      ftrTypeSampled,
      ftrTypeHardware,
      getFtrTypeFromArgs,
      showExporter,
      showFtr,
      showGroup,
)
from TypeFuture import TacLazyType
from FlowTrackingCliLib import (
      FlowTrackingMode,
      TrackerMode,
      guardExportOnTcpState,
      guardIntfFtrHardware,
      guardIntfFtrSampled,
      trackerNameMatcher,
)
from FlowExporterCli import (
      ExporterContext,
      ExporterMode,
)
from FlowGroupsCli import (
      GroupContext,
      GroupMode,
      GroupsMode,
)

traceHandle = Tracing.Handle( 'FlowTrackerCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2
t3 = traceHandle.trace3

FtConst = TacLazyType( 'FlowTracking::Constants' )

trackingConfig = {}

#---------------------------------------------------------
# Flow Tracker Context
#  - Python object for Tracker config thats pending commit
#---------------------------------------------------------

class TrackerContext( object ):
   def __init__( self, config, ftrType, trName, trConfig=None ):
      self.ftrConfig_ = config
      self.ftrType_ = ftrType
      self.name_ = trName
      self.trConfig = trConfig
      self.groupCtx = {}
      self.exporterCtx = {}
      self.groupToSeqnoMap = {}
      self.seqnoToGroupMap = {}
      self.groupResequencingInProgress = False
      self.changed = False
      self.inactiveTimeout = None
      self.activeInterval = None
      self.tcpStateChangeExport = None
      t0( 'creating new tracker context', ftrType, trName, trConfig )

      if not trConfig:
         # new tracker
         self.initFromDefaults()
      else:
         self.initFromConfig()

   def ftrType( self ):
      return self.ftrType_

   def initFromDefaults( self ):
      self.inactiveTimeout = inactiveTimeout.timeoutDefault
      self.activeInterval = activeInterval.intervalDefault
      self.tcpStateChangeExport = constants.tcpStateChangeExportDefault
      self.setChanged()

   def initFromConfig( self ):
      self.inactiveTimeout = self.trConfig.inactiveTimeout
      self.activeInterval = self.trConfig.activeInterval
      self.tcpStateChangeExport = self.trConfig.tcpStateChangeExport
      expConfig = self.trConfig.expConfig
      for expName in expConfig:
         t0( 'Creating new exporter context', expName )
         self.exporterCtx[ expName ] = \
                  ExporterContext( self, expName, expConfig[ expName ] )
      fgConfig = self.trConfig.fgConfig
      self.groupToSeqnoMap = {}
      self.seqnoToGroupMap = {}
      for fgName in fgConfig:
         t0( 'creating new group context', fgName )
         self.groupCtx[ fgName ] = \
               GroupContext( self, fgName, fgConfig[ fgName ] )
         seqno = fgConfig[ fgName ].seqno
         self.groupToSeqnoMap[ fgName ] = seqno
         self.seqnoToGroupMap[ seqno ] = fgName
      if self.trConfig.groupResequencingInProgress:
         self.resequenceExistingFlowGroups()
      if self.trConfig.groupsCommitInProgress:
         # It is possible that group commit was in progress earlier such that all
         # the groups were actually committed in Sysdb but CLI crashes before
         # groupsCommitted flag was set to true. In this case when the system
         # comes up, everything will be there in Sysdb config, so
         # GroupContext.changed_ flag will not be set to true for any group.
         # For such cases we use the flag groupsCommitInProgress which will preserve
         # state through crashes. So when the system comes up after the crash,
         # if groupsCommitInProgress was true, it means that groupsCommitted was not
         # able to be set to true earlier, so we do it now.
         self.setChanged()

   def resequenceExistingFlowGroups( self ):
      self.setResequencingInProgress( True )
      self.groupToSeqnoMap = {}
      newSeqnoToGroupMap = {}
      # Re-assign seqno at default interval
      newSeqno = constants.fgSeqnoInterval
      for seqno in sorted( self.seqnoToGroupMap ):
         fgName = self.seqnoToGroupMap[ seqno ]
         self.groupContext( fgName ).seqno_ = newSeqno
         self.groupContext( fgName ).setChanged()
         self.groupToSeqnoMap[ fgName ] = newSeqno
         newSeqnoToGroupMap[ newSeqno ] = fgName
         newSeqno += constants.fgSeqnoInterval
      self.seqnoToGroupMap = newSeqnoToGroupMap

   def setChanged( self ):
      self.changed = True

   def setResequencingInProgress( self, state ):
      self.groupResequencingInProgress = state

   def commit( self ):
      if not self.changed:
         t0( 'commit: No changes found' )
         return
      if not self.trConfig:
         t0( 'commit: creating new FlowTrackerConfig', self.name_ )
         self.trConfig = trackingConfig[ 
                           self.ftrType_ ].newFlowTrackerConfig( self.name_ )

      self.trConfig.inactiveTimeout = self.inactiveTimeout
      self.trConfig.activeInterval = self.activeInterval
      self.trConfig.tcpStateChangeExport = self.tcpStateChangeExport

      expToDelete = []
      for expName, ctx in self.exporterCtx.items():
         ctx.commitExporter( self.trConfig )
         if ctx.deleted():
            expToDelete.append( expName )
      t0( 'commit: deleting Exporter context', expToDelete )
      for expName in expToDelete:
         del self.exporterCtx[ expName ]

      self.trConfig.groupsCommitted = False
      self.trConfig.groupResequencingDone = False
      self.trConfig.groupResequencingInProgress = self.groupResequencingInProgress
      fgToDelete = []

      for seqno in sorted( self.seqnoToGroupMap, reverse=True ):
         fgName = self.seqnoToGroupMap[ seqno ]
         ctx = self.groupCtx[ fgName ]
         ctx.commitGroup( self.trConfig )
         if ctx.deleted():
            fgToDelete.append( fgName )
         if ctx.deleted() or ctx.changed():
            self.trConfig.groupsCommitInProgress = True
      t0( 'commit: deleting FlowGroup context', fgToDelete )
      for fgName in fgToDelete:
         del self.groupCtx[ fgName ]
         seqno = self.groupToSeqnoMap[ fgName ]
         del self.seqnoToGroupMap[ seqno ]
         del self.groupToSeqnoMap[ fgName ]

      if self.groupResequencingInProgress:
         self.setResequencingInProgress( False )
         self.trConfig.groupResequencingDone = True
         self.trConfig.groupResequencingInProgress = self.groupResequencingInProgress

      if self.trConfig.groupsCommitInProgress:
         self.trConfig.groupsCommitted = True
         self.trConfig.groupsCommitInProgress = False

      return

   def newExporterContext( self, expName ):
      expConfig = None
      if self.trConfig is not None:
         if expName in self.trConfig.expConfig:
            # Config changed after entering flow-tracking mode.
            # Create context with existing Sysdb config.
            expConfig = self.trConfig.expConfig[ expName ]
      self.exporterCtx[ expName ] = ExporterContext( self, expName, expConfig )
      return self.exporterCtx[ expName ]

   def activeIntervalIs( self, _activeInterval ):
      if self.activeInterval != _activeInterval:
         self.activeInterval = _activeInterval
         self.setChanged()

   def inactiveTimeoutIs( self, timeout ):
      if self.inactiveTimeout != timeout:
         self.inactiveTimeout = timeout
         self.setChanged()

   def tcpStateChangeExportIs( self, enable ):
      if self.tcpStateChangeExport != enable:
         self.tcpStateChangeExport = enable
         self.setChanged()

   def trackerName( self ):
      return self.name_

   def ftrConfig( self ):
      return self.ftrConfig_

   def exporterConfig( self, expName ):
      if self.trConfig:
         if expName in self.trConfig.expConfig:
            return self.trConfig.expConfig[ expName ]
      return None

   def exporterContext( self, expName ):
      if expName in self.exporterCtx:
         return self.exporterCtx[ expName ]
      else:
         return None

   def newGroupContext( self, fgName ):
      fgConfig = None
      if self.trConfig is not None:
         if fgName in self.trConfig.fgConfig:
            # Config changed after entering flow-tracking mode.
            # Create context with existing Sysdb config.
            fgConfig = self.trConfig.fgConfig[ fgName ]
      self.groupCtx[ fgName ] = GroupContext( self, fgName, fgConfig )
      return self.groupCtx[ fgName ]

   def groupConfig( self, fgName ):
      if self.trConfig:
         if fgName in self.trConfig.fgConfig:
            return self.trConfig.fgConfig[ fgName ]
      return None

   def groupContext( self, fgName ):
      if fgName in self.groupCtx:
         return self.groupCtx[ fgName ]
      else:
         return None

#-------------------------------------------------------------------------------
# "[no|default] tracker <flow-tracker-name>" command
#-------------------------------------------------------------------------------

def gotoTrackerMode( mode, trackerName ):
   t1( 'gotoTrackerMode of', trackerName )
   if len( trackerName ) > FtConst.confNameMaxLen:
      mode.addError(
         'Tracker name is too long (maximum {})'.format( FtConst.confNameMaxLen ) )
      return

   ftrCtx = mode.context()
   trCtx = ftrCtx.trackerContext( trackerName )
   if trCtx is None:
      t0( 'Creating new tracker context for', trackerName )
      trCtx = ftrCtx.newTrackerContext( trackerName )
   elif trCtx.trConfig:
      # Re-enter tracker mode, update context with latest config.
      t0( 'Updating tracker context for', trackerName )
      trCtx.initFromConfig()
   childMode = mode.childMode( TrackerMode, ftrCtx=ftrCtx, trCtx=trCtx )
   mode.session_.gotoChildMode( childMode )

def noTrackerMode( mode, trackerName ):
   t1( 'noTrackerMode of', trackerName )
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, TrackerMode ) and lastMode.context()
         and lastMode.context().trackerName() == trackerName ):
      t0( 'lastMode is: TrackerMode' )
      # If no tracker is issued in ftr mode then delete trConfig from
      # context to avoid committing it again
      lastMode.context().trConfig = None
      lastMode.context_ = None
   elif ( isinstance( lastMode, ( ExporterMode, GroupMode, GroupsMode ) ) and
         lastMode.context() and
         lastMode.context().trackerName() == trackerName ):
      t0( 'lastMode is: ', lastMode )
      # If no tracker is issued in exporter/groups/group mode then delete
      # trConfig from ftrMode
      lastMode.context().trCtx_.trConfig = None

   del trackingConfig[ mode.context().ftrType ].flowTrackerConfig[ trackerName ]

   ftrCtx = mode.context()
   if trackerName in ftrCtx.trCtx:
      del ftrCtx.trCtx[ trackerName ]

trackerMatcherForConfig = CliMatcher.KeywordMatcher(
                                    'tracker',
                                    helpdesc='Configure flow tracker' )
def getTrackerName( mode ):
   return mode.context().trCtx.keys()

class TrackerCmd( CliCommand.CliCommandClass ):
   syntax = '''tracker TRACKER_NAME'''
   noOrDefaultSyntax = syntax

   data = {
      'tracker' : trackerMatcherForConfig,
      'TRACKER_NAME' : CliMatcher.DynamicNameMatcher( getTrackerName,
                                          "Flow tracker name" ),
   }

   @staticmethod
   def handler( mode, args ):
      gotoTrackerMode( mode, trackerName=args[ "TRACKER_NAME" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noTrackerMode( mode, trackerName=args[ "TRACKER_NAME" ] )

FlowTrackingMode.addCommandClass( TrackerCmd )

#-------------------------------------------------------------------------------
# "[no|default] record export on interval <interval>" command,
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

recordMatcher = CliMatcher.KeywordMatcher(
                  'record', helpdesc='Configure flow record export' )

exportMatcher = CliMatcher.KeywordMatcher(
                  'export', helpdesc='Configure flow record export' )

onMatcher = CliMatcher.KeywordMatcher(
                  'on', helpdesc='Configure flow record export' )

class ActiveIntervalCommand( CliCommand.CliCommandClass ):
   syntax = '''record export on interval INTERVAL'''
   noOrDefaultSyntax = '''record export on interval ...'''

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'on' : onMatcher,
      'interval' : 'Configure flow record export interval',
      'INTERVAL' : CliMatcher.IntegerMatcher(
                        activeInterval.minInterval,
                        activeInterval.maxInterval, 
                        helpdesc='Flow record export interval in milliseconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().activeIntervalIs(
            args.get( 'INTERVAL', activeInterval.intervalDefault ) )

   noOrDefaultHandler = handler

TrackerMode.addCommandClass( ActiveIntervalCommand )

#-------------------------------------------------------------------------------
# "[no|default] record export on inactive timeout <timeout>" command,
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

class InactiveTimeoutCommand( CliCommand.CliCommandClass ):
   syntax = '''record export on inactive timeout TIMEOUT'''
   noOrDefaultSyntax = '''record export on inactive timeout ...'''

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'on' : onMatcher,
      'inactive' : 'Configure flow record export inactive timeout',
      'timeout' : 'Configure flow record export inactive timeout',
      'TIMEOUT' : CliMatcher.IntegerMatcher(
                     inactiveTimeout.minTimeout,
                     inactiveTimeout.maxTimeout,
                     helpdesc='Flow record inactive export timeout in milliseconds' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().inactiveTimeoutIs(
            args.get( 'TIMEOUT', inactiveTimeout.timeoutDefault ) )

   noOrDefaultHandler = handler

TrackerMode.addCommandClass( InactiveTimeoutCommand )

#-------------------------------------------------------------------------------
# "[no|default] record export on tcp state change" command,
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

tcpMatcher = CliCommand.guardedKeyword( 'tcp',
                  helpdesc='Configure flow record export on TCP state',
                  guard=guardExportOnTcpState )

class ExportOnTcpStateCmd( CliCommand.CliCommandClass ):
   syntax = '''record export on tcp state change'''
   noOrDefaultSyntax = syntax

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'on' : onMatcher,
      'tcp' : tcpMatcher,
      'state' : 'Configure flow record export on TCP state',
      'change' : 'Configure flow record export on TCP state change',
   }

   @staticmethod
   def handler( mode, args ):
      t2( "setExportOnTcpState: ", mode, mode.ftrTypeStr )
      mode.context().tcpStateChangeExportIs( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().tcpStateChangeExportIs(
         constants.tcpStateChangeExportDefault )

TrackerMode.addCommandClass( ExportOnTcpStateCmd )

class AbortCmd( CliCommand.CliCommandClass ):
   syntax = '''abort'''
   data = {
         'abort': CliToken.Cli.abortMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

TrackerMode.addCommandClass( AbortCmd )

#----------------
# interface
#---------------

def setFlowTracker( mode, intfName, ftrType, no=None, trackerName=None ):
   intfId = Tac.Value( 'Arnet::IntfId', intfName )
   ftrIntf = None
   config = trackingConfig[ ftrType ].flowTrackerIntfConfig

   if intfId in config:
      ftrIntf = config[ intfId ]

   if no:
      if not trackerName or not ftrIntf or trackerName == ftrIntf.trackerName:
         del config[ intfId ]
         if ftrIntf and ftrIntf.trackerName:
            t1( 'delete tracker', ftrType, ftrIntf.trackerName, 'from', intfName )
   else:
      if not trackerName:
         return
      if ftrIntf and ftrIntf.trackerName != trackerName:
         del config[ intfId ]
         t1( 'delete tracker', ftrType, ftrIntf.trackerName, 'from', intfName )
         ftrIntf = None
      t1( 'set tracker', ftrType, trackerName, 'on', intfName )
      if ftrIntf:
         assert trackerName == ftrIntf.trackerName
      else:
         if trackerName not in trackingConfig[ ftrType ].flowTrackerConfig:
            mode.addWarning(
                  "Flow tracker: %s doesn't exist. The configuration will "
                  "not take effect until the flow tracker is configured." %
                  trackerName )
         ftrIntf = trackingConfig[
               ftrType ].newFlowTrackerIntfConfig( intfId, trackerName )
   return

sampledIfConfigKw = CliCommand.guardedKeyword( 'sampled',
                        helpdesc='Configure flow tracker sampled',
                        guard=guardIntfFtrSampled,
                        storeSharedResult=True )

hardwareIfConfigKw = CliCommand.guardedKeyword( 'hardware',
                        helpdesc='Configure flow tracker hardware',
                        guard=guardIntfFtrHardware,
                        storeSharedResult=True )

class FlowTrackerIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''flow tracker ( sampled | hardware ) TRACKER_NAME'''
   noOrDefaultSyntax = '''flow tracker ( sampled | hardware ) ...'''

   data = {
         'flow' : flowMatcherForConfigIf,
         'tracker' : trackerMatcherForConfig,
         'sampled' : sampledIfConfigKw,
         'hardware' : hardwareIfConfigKw,
         'TRACKER_NAME' : trackerNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      setFlowTracker( mode, mode.intf.name, getFtrTypeFromArgs( args ),
                      trackerName=args[ "TRACKER_NAME" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setFlowTracker( mode, mode.intf.name, getFtrTypeFromArgs( args ),
                      no=True )

SwitchportModelet.addCommandClass( FlowTrackerIntfConfigCmd )
TunnelIntfConfigModelet.addCommandClass( FlowTrackerIntfConfigCmd )

#-----------------------------------------------------------------
#              show active|pending|diff
#-----------------------------------------------------------------

def _showPending( mode, output=None ):
   trCtx = mode.context()
   if trCtx is None:
      return
   if output is None:
      output = sys.stdout
   showFtr( output, trCtx.name_, trCtx, cliSave=False )
   for expName in sorted( trCtx.exporterCtx ):
      showExporter( output, expName, trCtx.exporterCtx[ expName ],
                    cliSave=False )
   if trCtx.groupCtx:
      output.write( '%sgroups\n' % spaceConst )
      for groupSeqno in sorted( trCtx.seqnoToGroupMap ):
         groupName = trCtx.seqnoToGroupMap[ groupSeqno ]
         showGroup( output, groupName, trCtx.groupCtx[ groupName ], cliSave=False,
                    space=spaceConst )

def _showActive( mode, output=None ):
   trName = mode.trackerName()
   if trName not in mode.context().ftrConfig().flowTrackerConfig:
      return
   trConfig = mode.context().ftrConfig().flowTrackerConfig[ trName ]
   if output is None:
      output = sys.stdout
   showFtr( output, trName, trConfig, cliSave=False )
   for expName in sorted( trConfig.expConfig.iterkeys() ):
      showExporter( output, expName, trConfig.expConfig[ expName ],
                    cliSave=False )
   if trConfig.fgConfig:
      output.write( '%sgroups\n' % spaceConst )
      seqnoToGroupMap = {}
      for groupName, groupConfig in trConfig.fgConfig.items():
         seqnoToGroupMap[ groupConfig.seqno ] = groupName
      for groupSeqno in sorted( seqnoToGroupMap ):
         groupName = seqnoToGroupMap[ groupSeqno ]
         showGroup( output, groupName, trConfig.fgConfig[ groupName ],
                    cliSave=False, space=spaceConst )

def _showDiff( mode ):
   # generate diff between active and pending
   import difflib
   import cStringIO
   activeOutput = cStringIO.StringIO()
   _showActive( mode, output=activeOutput )
   pendingOutput = cStringIO.StringIO()
   _showPending( mode, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print '\n'.join( list( diff ) )

class ShowTracker( ShowCliCommandClass ):
   syntax = 'show ( pending | active | diff )'

   data = {
      'pending': 'Display the new tracker configuration to be applied',
      'active': 'Display the tracker configuration in the running-config',
      'diff': ( 'Display the diff between active tracker configuration'
                'to be applied' ),
    }
 
   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         _showDiff( mode )
      elif 'active' in args:
         _showActive( mode )
      else:
         _showPending( mode )
 
TrackerMode.addShowCommandClass( ShowTracker )

#-------------------------------------------------------------------------------
# The FlowTrackerIntf class is used to remove the FlowTracker IntfConfig object
# when an interface is deleted.
#-------------------------------------------------------------------------------
class FlowTrackerIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      for ftrType in [ ftrTypeSampled, ftrTypeHardware ]:
         setFlowTracker( None, self.intf_.name, ftrType, no=True )

#---------------------------------------------------------
def Plugin( em ):
   global trackingConfig

   for ftrType in ftrTypes:
      trackingConfig[ ftrType ] = ConfigMount.mount( em,
                                 ftrConfigPathPrefix + ftrType,
                                 ftrConfigType, 'wi' )

   IntfCli.Intf.registerDependentClass( FlowTrackerIntf )
