# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for FlowGroups configuration commands

import Tac
import Tracing
import CliMatcher
import CliCommand
import LazyMount
from TypeFuture import TacLazyType
from FlowTrackingCliLib import (
      TrackerMode,
      guardFlowGroupConfig,
      guardMirroring,
)
from FlowTrackerCliUtil import (
      encapTypeKwStr,
      reservedGroupToSeqnoMap,
      configSeqnoMax,
)
from CliMode.FlowGroups import (
      GroupsMode,
      GroupMode,
)
from Toggles.FlowTrackerToggleLib import toggleCopyToCollectorEnabled
from AclCli import (
      accessListMatcher,
      ipAclNameMatcher,
      ip6AclNameMatcher,
      getAclConfig,
)
from FlowTrackerConst import (
      accessListDefault,
      constants,
      reservedGroupNames,
      mirrorConfigDefault,
      initialCopy,
      mirrorIntervalFixed,
      mirrorIntervalRandom,
      mirrorSelectorAlgo,
)
from MirroringMonitorCli import matcherSessionName

traceHandle = Tracing.Handle( 'FlowGroupsCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2
t3 = traceHandle.trace3

AclTypes = TacLazyType( 'Acl::AclType' )
mirrorConfigDir = None

def getExporterName( mode ):
   return mode.context().trCtx_.exporterCtx.keys()

#-------------------------
# Flow Group Context
#--------------------------

class GroupContext( object ):
   def __init__( self, trContext, fgName, fgConfig=None ):
      self.trCtx_ = trContext
      self.fgName_ = fgName
      self.seqno_ = 0
      self.fgConfig_ = fgConfig
      self.changed_ = False
      self.deleted_ = False
      self.expName = None
      self.encapType = None
      self.ipAccessList = None
      self.ip6AccessList = None
      self.aclSeqno = 1
      self.mirrorConfig = None
      t0( "trCtx:", trContext, "fgName:", fgName, "fgConfig:", fgConfig )

      if not fgConfig:
         self.initFromDefaults()
      else:
         self.initFromConfig()

   def ftrType( self ):
      return self.trCtx_.ftrType()

   def trackerName( self ):
      return self.trCtx_.trackerName()

   def setChanged( self ):
      self.changed_ = True
      self.trCtx_.setChanged()

   def changed( self ):
      return self.changed_

   def initFromDefaults( self ):
      self.expName = []
      self.encapType = []
      self.ipAccessList = accessListDefault
      self.ip6AccessList = accessListDefault
      self.mirrorConfig =  mirrorConfigDefault
      self.setChanged()

   def initFromConfig( self ):
      t0( "initFromConfig" )
      self.seqno_ = self.fgConfig_.seqno
      self.expName = list( self.fgConfig_.expName ) 
      self.encapType = list( self.fgConfig_.encapType )
      self.ipAccessList = self.fgConfig_.ipAccessList
      self.ip6AccessList = self.fgConfig_.ip6AccessList
      self.mirrorConfig = self.fgConfig_.mirrorConfig

   def addExporter( self, mode, expName ):
      t1( 'FlowGroup addExporter', expName )
      if expName not in self.expName:
         self.expName.append( expName )
         self.setChanged()
      # The warning will be logged every time user enters the command
      # for an exporter which is not configured under tracker.
      if expName not in getExporterName( mode ):
         mode.addWarning(
               "Exporter: %s doesn't exist. The configuration will "
               "not take effect until the exporter is configured"
               " for flow tracker %s" % ( expName, self.trackerName() ) )

   def removeExporter( self, expName ):
      t1( 'FlowGroup removeExporter', expName )
      if expName in self.expName:
         self.expName.remove( expName )
         self.setChanged()

   def encapsulationIs( self, mode, args ):
      encapTypes = args[ 'ENCAP_TYPES' ]
      t1( 'FlowGroup encapsulationIs' )
      if self.fgName_ in reservedGroupToSeqnoMap:
         mode.addError( "Cannot modify encapsulation for reserved group %s"
                        % self.fgName_ )
         return
      for encapType, encapKwStr in encapTypeKwStr.items():
         if encapKwStr in encapTypes and encapType not in self.encapType:
            self.encapType.append( encapType )
            self.setChanged()

   def encapsulationDel( self, mode, args ):
      encapTypes = args[ 'ENCAP_TYPES' ]
      t1( 'FlowGroup encapsulationDel' )
      if self.fgName_ in reservedGroupToSeqnoMap:
         mode.addError( "Cannot modify encapsulation for reserved group %s"
                        % self.fgName_ )
         return
      for encapType, encapKwStr in encapTypeKwStr.items():
         if encapKwStr in encapTypes and encapType in self.encapType:
            self.encapType.remove( encapType )
            self.setChanged()

   def getAclSeqno( self ):
      return self.aclSeqno

   def accessListIs( self, mode, args ):
      aclType = AclTypes.ip if 'ip' in args else AclTypes.ipv6
      aclName = args.get( 'V4ACL' ) or args.get( 'V6ACL' )
      t1( 'FlowGroup update', aclType, 'access-list', aclName )
      if self.fgName_ in reservedGroupToSeqnoMap:
         mode.addError( "Cannot modify access-list for reserved group %s"
                        % self.fgName_ )
         return

      if aclName not in getAclConfig( aclType ):
         # The warning will be logged every time user enters the command
         # for an ACL which is not configured on the switch
         mode.addWarning(
               "%s access-list %s will not take effect until the ACL is configured"
               " under global config mode" % ( aclType, aclName ) )
      accessList = Tac.Value( "FlowTracking::AccessList",
                              self.getAclSeqno(), aclName, aclType )
      if aclType == AclTypes.ip and aclName != self.ipAccessList.aclName:
         self.ipAccessList = accessList
         self.setChanged()
      elif aclType == AclTypes.ipv6 and aclName != self.ip6AccessList.aclName:
         self.ip6AccessList = accessList
         self.setChanged()

   def accessListDel( self, mode, args ):
      aclType = AclTypes.ip if 'ip' in args else AclTypes.ipv6
      aclName = args.get( 'V4ACL' ) or args.get( 'V6ACL' )
      t1( 'FlowGroup remove', aclType, 'access-list', aclName )
      if self.fgName_ in reservedGroupToSeqnoMap:
         mode.addError( "Cannot modify access-list for reserved group %s"
                        % self.fgName_ )
         return

      if aclType == AclTypes.ip and aclName == self.ipAccessList.aclName:
         self.ipAccessList = accessListDefault
         self.setChanged()
      elif aclType == AclTypes.ipv6 and aclName == self.ip6AccessList.aclName:
         self.ip6AccessList = accessListDefault
         self.setChanged()

   def mirrorConfigIs( self, mode, args ):
      sessionName = args.get( 'MONITOR_SESSION_NAME' )
      t1( 'FlowGroup add mirror config', sessionName )
      if not mirrorConfigDir or \
         ( mirrorConfigDir and sessionName not in mirrorConfigDir.session ):
         mode.addWarning( "mirror config will not take effect until monitor session"
                          " %s is configured" % sessionName  )
      initialCopy_ = args.get( 'INITIAL_COPY', initialCopy.pktDefault )
      mirrorIntervalFixed_ = args.get( 'INTERVAL_FIXED',
                                       mirrorIntervalFixed.intervalDefault )
      mirrorIntervalRandom_ = args.get( 'INTERVAL_RANDOM',
                                        mirrorIntervalRandom.intervalDefault )
      if 'fixed' in args:
         mirrorSelectorAlgo_ = mirrorSelectorAlgo.fixed
      elif 'random' in args:
         mirrorSelectorAlgo_ = mirrorSelectorAlgo.random
      else:
         mirrorSelectorAlgo_ = mirrorSelectorAlgo.sampleModeNone
      mirrorConfig = Tac.Value( "FlowTracking::MirrorConfig",
                                sessionName,
                                initialCopy_,
                                mirrorSelectorAlgo_,
                                mirrorIntervalFixed_,
                                mirrorIntervalRandom_ )
      if self.mirrorConfig != mirrorConfig:
         self.mirrorConfig = mirrorConfig
         self.setChanged()

   def mirrorConfigDel( self, mode, args ):
      sessionName = args[ 'MONITOR_SESSION_NAME' ]
      t1( 'FlowGroup remove mirror config', sessionName )
      if self.mirrorConfig.sessionName == sessionName:
         self.mirrorConfig = mirrorConfigDefault
         self.setChanged()

   def commitGroup( self, trConfig ):
      if not self.changed_:
         t0( 'Nothing changed for', self.fgName_ )
         return None
      if self.deleted_:
         if self.fgConfig_ is not None:
            t0( 'Deleting group from flowTrackerConfig', self.fgName_ )
            del trConfig.fgConfig[ self.fgName_ ]
         else:
            t0( 'Group never created in flowTrackerConfig', self.fgName_ )
         return None

      if self.fgConfig_ is None:
         t0( 'Add new group to flowTrackerConfig', self.fgName_ )
         self.fgConfig_ = \
            trConfig.fgConfig.newMember( self.fgName_, self.seqno_ )
         t0( 'trConfig', trConfig, trConfig.fgConfig[ self.fgName_ ] )

      self.fgConfig_.groupCommitted = False
      staleExporters = set( self.fgConfig_.expName )
      for expName in self.expName:
         t0( 'Adding / updating exporter', expName  )
         self.fgConfig_.expName.add( expName )
         staleExporters.discard( expName )
      for expName in staleExporters:
         t0( 'Removing exporter', expName )
         self.fgConfig_.expName.remove( expName )

      staleEncap = set( self.fgConfig_.encapType )
      for encap in self.encapType:
         t0( 'Adding encapsulation', encap  )
         self.fgConfig_.encapType.add( encap )
         staleEncap.discard( encap )
      for encap in staleEncap:
         t0( 'Removing encapsulation', encap  )
         self.fgConfig_.encapType.remove( encap )

      self.fgConfig_.ipAccessList = self.ipAccessList
      self.fgConfig_.ip6AccessList = self.ip6AccessList
      self.fgConfig_.mirrorConfig = self.mirrorConfig
      self.fgConfig_.seqno = self.seqno_
      self.fgConfig_.groupCommitted = True

      return self.fgConfig_

   def deletedIs( self, deleted=False ):
      if self.deleted_ == deleted:
         return
      t0( 'group deletedIs: ', deleted )
      self.deleted_ = deleted
      self.setChanged()
      # deleted and recreated in same session
      if not deleted:
         self.initFromDefaults()

   def deleted( self ):
      return self.deleted_

   def groupConfig( self ):
      return self.fgConfig_

#-------------------------------------------------------------------------------
# "[no|default] groups" command, in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

groupsMatcher = CliMatcher.KeywordMatcher( 'groups',
                                           helpdesc='Configure flow groups' )

groupsNode = CliCommand.Node( groupsMatcher,
                              hidden=not toggleCopyToCollectorEnabled(),
                              guard=guardFlowGroupConfig )

class GroupsConfigCmd( CliCommand.CliCommandClass ):
   # Need to enable trailing garbage because of how Umbrella/NoOrDefaultCliTest.py
   # and EosImage/CliModeTests.py tests work with the hierarchy that we have
   # for flow tracking CLI commands. More details at https://bug/452471#comment2
   syntax = '''groups ...'''
   noOrDefaultSyntax = syntax
   data = { 'groups' : groupsNode }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( GroupsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if toggleCopyToCollectorEnabled():
         trContext = mode.context()
         for groupCtx in trContext.groupCtx.values():
            groupCtx.deletedIs( True )

TrackerMode.addCommandClass( GroupsConfigCmd )

if toggleCopyToCollectorEnabled():
   def getGroupName( mode ):
      return mode.context().groupCtx.keys()

   groupNameMatcher = CliMatcher.DynamicNameMatcher( getGroupName, "Group Name" )

   # Resequences the group sequence numbers and inserts the new group at
   # appropriate position.
   def resequenceFlowGroups( mode, args ):
      trContext = mode.context()
      position = args.get( 'before' ) or args.get( 'after' )
      refGroupName = args.get( "REF_GROUP_NAME" )

      if ( len( trContext.seqnoToGroupMap ) + 1 ) * constants.fgSeqnoInterval > \
         configSeqnoMax:
         mode.addError( "Unable to add group because maximum limit reached" )
         return -1 

      trContext.setResequencingInProgress( True )
      # Reset trContext.groupToSeqnoMap
      trContext.groupToSeqnoMap = {}

      newSeqnoToGroupMap = {}
      # Re-assign seqno at default interval
      newSeqno = constants.fgSeqnoInterval
      for seqno in sorted( trContext.seqnoToGroupMap ):
         fgName = trContext.seqnoToGroupMap[ seqno ]
         if fgName not in reservedGroupToSeqnoMap:
            if position == 'before' and fgName == refGroupName:
               newGroupSeqno = newSeqno
               newSeqno += constants.fgSeqnoInterval

            trContext.groupContext( fgName ).seqno_ = newSeqno
            trContext.groupContext( fgName ).setChanged()
            trContext.groupToSeqnoMap[ fgName ] = newSeqno
            newSeqnoToGroupMap[ newSeqno ] = fgName
            newSeqno += constants.fgSeqnoInterval

            if position == 'after' and fgName == refGroupName:
               newGroupSeqno = newSeqno
               newSeqno += constants.fgSeqnoInterval

      if not position:
         newGroupSeqno = newSeqno

      trContext.seqnoToGroupMap = newSeqnoToGroupMap
      return newGroupSeqno

   def getGroupSeqno( args, mode ):
      trContext = mode.context()
      position = args.get( 'before' ) or args.get( 'after' )
      refGroupName = args.get( 'REF_GROUP_NAME' )

      # Handle reservedGroups
      groupName = args[ 'GROUP_NAME' ]
      if groupName in reservedGroupToSeqnoMap:
         return reservedGroupToSeqnoMap[ groupName ]

      # While adding user defined groups at end, we will simply insert them with
      # seqno = highest user defined seqno + fgSeqnoInterval
      orderedGroupSeqnos = sorted( trContext.seqnoToGroupMap )
      for reservedSeqno in reservedGroupToSeqnoMap.values():
         if reservedSeqno in orderedGroupSeqnos:
            orderedGroupSeqnos.remove( reservedSeqno )

      # If no user defined group exists, seqno = fgSeqnoInterval = 100
      if not orderedGroupSeqnos:
         return constants.fgSeqnoInterval

      # Add the group in the end if:
      #         - before/after is not specified or
      #         - REF_GROUP_NAME does not exist or
      #         - group is added "before" reserved group "IPv4"
      if not position or refGroupName not in trContext.groupToSeqnoMap or \
         ( position == 'before' and \
           refGroupName == reservedGroupNames.groupHwIpv4 ):
         availableSeqno = orderedGroupSeqnos[ -1 ] + constants.fgSeqnoInterval
         if availableSeqno > configSeqnoMax:
            availableSeqno = resequenceFlowGroups( mode, args )
         return availableSeqno

      refGroupSeqno = trContext.groupToSeqnoMap[ refGroupName ]

      if position == 'before':
         lowIndex = orderedGroupSeqnos.index( refGroupSeqno ) - 1
         if lowIndex >= 0:
            availableSeqno = orderedGroupSeqnos[ lowIndex ] + \
                             ( refGroupSeqno - orderedGroupSeqnos[ lowIndex ] ) / 2
         else:
            availableSeqno = refGroupSeqno / 2

         if availableSeqno in orderedGroupSeqnos or \
            availableSeqno <= 0:
            availableSeqno = resequenceFlowGroups( mode, args )

      if position == 'after':
         highIndex = orderedGroupSeqnos.index( refGroupSeqno ) + 1
         if highIndex < len( orderedGroupSeqnos ):
            availableSeqno = refGroupSeqno + \
                             ( orderedGroupSeqnos[ highIndex ] - refGroupSeqno ) / 2
         else:
            # Adding in the end
            availableSeqno = refGroupSeqno + constants.fgSeqnoInterval
         if availableSeqno in orderedGroupSeqnos or \
            availableSeqno > configSeqnoMax:
            availableSeqno = resequenceFlowGroups( mode, args )

      return availableSeqno

   def gotoGroupMode( mode, args ):
      groupName = args[ 'GROUP_NAME' ]
      refGroupName = args.get( 'REF_GROUP_NAME' )
      trContext = mode.context()
      trackerName = mode.context().trackerName()
      t1( 'handler ', trackerName, groupName )

      if len( groupName ) > constants.confNameMaxLen:
         mode.addError(
               'Group name is too long (maximum {})'.format(
               constants.confNameMaxLen ) )
         return

      position = args.get( 'before' ) or args.get( 'after' )
      if groupName in reservedGroupToSeqnoMap and position:
         mode.addError( "Reserved group %s cannot be moved" % groupName )
         return

      if refGroupName in reservedGroupToSeqnoMap and \
         ( ( position == "before" and \
           refGroupName != reservedGroupNames.groupHwIpv4 ) or \
           position == "after" ):
         mode.addError( "User-defined group %s cannot be added after a reserved"
                        " group %s" % ( groupName, refGroupName ) )
         return

      groupCtx = trContext.groupContext( groupName )

      if groupCtx is None:
         t0( 'No groupContext found, creating new group context' )
         groupSeqno = getGroupSeqno( args, mode )
         if groupSeqno == -1:
            # resequenceFlowGroups is the only function which returns "-1" and it
            # has an error being logged there.
            return
         groupCtx = trContext.newGroupContext( groupName )
         trContext.groupToSeqnoMap[ groupName ] = groupSeqno
         trContext.seqnoToGroupMap[ groupSeqno ] = groupName
         groupCtx.seqno_ = groupSeqno

      else:
         if groupCtx.deleted() or groupCtx.groupConfig():
            t0( 'Re-entering a deleted groupContext' )
            groupCtx.deletedIs( False )
         # Handle re-ordering of existing groups
         # Ignore 'before/after' if:
         #      - reference group is the same as group being configured
         #      - reference group does not exist
         if position and groupName != refGroupName and \
            refGroupName in trContext.groupToSeqnoMap:
            groupSeqno = getGroupSeqno( args, mode )
            if groupSeqno == -1:
               # resequenceFlowGroups is the only function which returns "-1" and it
               # has an error being logged there.
               return
            oldSeqno = trContext.groupToSeqnoMap[ groupName ]
            trContext.groupToSeqnoMap[ groupName ] = groupSeqno
            # Cleanup old seqno for this group if needed
            if trContext.seqnoToGroupMap[ oldSeqno ] == groupName:
               del trContext.seqnoToGroupMap[ oldSeqno ]
            trContext.seqnoToGroupMap[ groupSeqno ] = groupName
            groupCtx.seqno_ = groupSeqno
            groupCtx.setChanged()

      childMode = mode.childMode( GroupMode,
                                  context=groupCtx,
                                  groupName=groupName )
      mode.session_.gotoChildMode( childMode )

   #-------------------------------------------------------------------------------
   # "[no|default] group <groupName> [ ( before | after ) <groupName> ]"
   # command, in "config-ftr-tr-grps" mode.
   #-------------------------------------------------------------------------------
   class GroupConfigCmd( CliCommand.CliCommandClass ):
      syntax = '''group GROUP_NAME [ ( before | after ) REF_GROUP_NAME ]'''
      noOrDefaultSyntax = '''group GROUP_NAME'''

      data = {
            'group' : 'Configure flow group',
            'GROUP_NAME' : groupNameMatcher,
            'before' : 'Add this group immediately before the specified group',
            'after' : 'Add this group immediately after the specified group',
            'REF_GROUP_NAME' : groupNameMatcher,
      }

      @staticmethod
      def handler( mode, args ):
         gotoGroupMode( mode, args )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         groupName = args[ 'GROUP_NAME' ]
         trContext = mode.context()
         trackerName = mode.context().trackerName()
         t1( 'noOrDefaultHandler', trackerName, groupName )

         groupCtx = trContext.groupContext( groupName )

         if groupCtx is None:
            return

         groupCtx.deletedIs( True )
         return

   GroupsMode.addCommandClass( GroupConfigCmd )

   #-------------------------------------------------------------------------------
   # "[no|default] exporter <exporterName>" command in "config-ftr-tr-grps-grp" mode
   #-------------------------------------------------------------------------------
   exporterNameMatcher = CliMatcher.DynamicNameMatcher( getExporterName,
                                                        "Exporter Name" )

   class GroupExporterCommand( CliCommand.CliCommandClass ):
      syntax = '''exporter EXPORTER_NAME'''
      noOrDefaultSyntax = syntax

      data = {
         'exporter' : 'Configure exporter for flow group',
         'EXPORTER_NAME' : exporterNameMatcher,
      }

      @staticmethod
      def handler( mode, args ):
         mode.context().addExporter( mode, args[ "EXPORTER_NAME" ] )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         mode.context().removeExporter( args[ "EXPORTER_NAME" ] )

   GroupMode.addCommandClass( GroupExporterCommand )

   #-------------------------------------------------------------------------------
   # "[ no | default ] encapsulation { ipv4 | ipv6 | vxlan }" command in
   # "config-ftr-tr-grps-grp" mode
   #-------------------------------------------------------------------------------
   class GroupEncapsulationCommand( CliCommand.CliCommandClass ):
      syntax = '''encapsulation ENCAP_TYPES'''
      noOrDefaultSyntax = syntax
      data = {
         'encapsulation' : 'Packet encapsulation',
         'ENCAP_TYPES' : CliCommand.SetEnumMatcher( {
            'ipv4' : 'IPv4 flows',
            'ipv6' : 'IPv6 flows',
            'vxlan' : 'VXLAN flows',
            } ),
         }
      @staticmethod
      def handler( mode, args ):
         mode.context().encapsulationIs( mode, args )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         mode.context().encapsulationDel( mode, args )

   GroupMode.addCommandClass( GroupEncapsulationCommand )

   #-------------------------------------------------------------------------------
   # "[ no | default ] ( ip | ipv6 ) access-list ACL_NAME" command in
   # "config-ftr-tr-grps-grp" mode
   #-------------------------------------------------------------------------------

   class GroupAccessListCommand( CliCommand.CliCommandClass ):
      syntax = '''( ( ip access-list V4ACL ) | ( ipv6 access-list V6ACL ) )'''
      noOrDefaultSyntax = syntax
      data = {
            'ip': 'IP config commands',
            'ipv6': 'IPv6 config commands',
            'access-list' : accessListMatcher,
            'V4ACL': ipAclNameMatcher,
            'V6ACL': ip6AclNameMatcher,
            } 

      @staticmethod
      def handler( mode, args ):
         mode.context().accessListIs( mode, args )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         mode.context().accessListDel( mode, args )

   GroupMode.addCommandClass( GroupAccessListCommand )

   #-------------------------------------------------------------------------------
   # "[ no | default ] mirror MONITOR_SESSION_NAME [ initial packets
   # INITIAL_COPY ] [ sample interval ( fixed | random ) INTERVAL ]" command in
   # "config-ftr-tr-grps-grp" mode
   #-------------------------------------------------------------------------------

   mirrorConfigKw = CliCommand.guardedKeyword( 'mirror',
                                               helpdesc='Mirror config commands',
                                               guard=guardMirroring )
   class GroupMirrorCommand( CliCommand.CliCommandClass ):
      syntax = '''mirror MONITOR_SESSION_NAME [ initial packets INITIAL_COPY ]
                  [ sample interval ( fixed INTERVAL_FIXED ) |
                  ( random INTERVAL_RANDOM ) ]'''
      noOrDefaultSyntax = '''mirror MONITOR_SESSION_NAME'''
      data = {
            'mirror' : mirrorConfigKw,
            'MONITOR_SESSION_NAME' : matcherSessionName,
            'initial' : 'Configure initial packets to be mirrored',
            'packets' : 'Configure initial packets to be mirrored',
            'INITIAL_COPY' : CliMatcher.IntegerMatcher(
               initialCopy.minPkts,
               initialCopy.maxPkts,
               helpdesc='Number of initial packets to be mirrored' ),
            'sample' : 'Configure sampling for packets to be mirrored',
            'interval' : 'Configure sampling interval for packets to be mirrored',
            'fixed' : 'Sample packets at fixed interval',
            'INTERVAL_FIXED' : CliMatcher.IntegerMatcher(
               mirrorIntervalFixed.minInterval,
               mirrorIntervalFixed.maxInterval,
               helpdesc='Sample interval N to sample every Nth packet' ),
            'random' : 'Sample packets at random interval',
            'INTERVAL_RANDOM' : CliMatcher.IntegerMatcher(
               mirrorIntervalRandom.minInterval,
               mirrorIntervalRandom.maxInterval,
               helpdesc='Sample interval N to sample packets with a probability' \
                        ' of 1/N' ),
            }

      @staticmethod
      def handler( mode, args ):
         mode.context().mirrorConfigIs( mode, args )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         mode.context().mirrorConfigDel( mode, args )

   GroupMode.addCommandClass( GroupMirrorCommand )

   def Plugin( em ):
      global mirrorConfigDir
      mirrorConfigDir = LazyMount.mount( em, "mirroring/config",
                                         "Mirroring::Config", "r" )

