# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for FlowExporter configuration commands

import Tac
import Tracing
import CliMatcher
import CliCommand
import IntfCli
import IpGenAddrMatcher
import IpAddrMatcher
from FlowTrackerConst import (
      constants,
      ipfixMtu,
      ipfixPort,
      ipfixVersion,
      templateInterval,
)
from TypeFuture import TacLazyType
from FlowTrackingCliLib import (
      ExporterMode,
      TrackerMode,
)

traceHandle = Tracing.Handle( 'FlowExporterCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2
t3 = traceHandle.trace3

FtConst = TacLazyType( 'FlowTracking::Constants' )

#-------------------------
# Flow Exporter Context
#--------------------------

class ExporterContext( object ):
   def __init__( self, trContext, expName, expConfig=None ):
      self.trCtx_ = trContext
      self.expName_ = expName
      self.expConfig_ = expConfig
      self.changed_ = False
      self.deleted_ = False
      self.collectorHostAndPort = None
      self.dscpValue = None
      self.ipfixFormatConfigured = None
      self.ipfixMtu = None
      self.localIntfName = None
      self.localAddress = None
      self.templateInterval = None
      self.ipfixVersion = None
      t0( "trCtx:", trContext, "expName:", expName, "expConfig:", expConfig )

      if not expConfig:
         self.initFromDefaults()
      else:
         self.initFromConfig()

   def ftrType( self ):
      return self.trCtx_.ftrType()

   def trackerName( self ):
      return self.trCtx_.trackerName()

   def setChanged( self ):
      self.changed_ = True
      self.trCtx_.setChanged()

   def initFromDefaults( self ):
      self.collectorHostAndPort = {}
      self.dscpValue = constants.dscpValueDefault
      self.templateInterval = templateInterval.intervalDefault
      self.ipfixFormatConfigured = False
      self.ipfixMtu = ipfixMtu.mtuDefault
      self.ipfixVersion = ipfixVersion.versionDefault
      self.localIntfName = constants.intfNameDefault
      self.localAddress = constants.ipAddrDefault
      self.setChanged()

   def initFromConfig( self ):
      self.collectorHostAndPort = {}
      for hnp in self.expConfig_.collectorHostAndPort.values():
         self.collectorHostAndPort[ hnp.hostname ] = hnp
      self.dscpValue = self.expConfig_.dscpValue
      self.templateInterval = self.expConfig_.templateInterval
      self.ipfixMtu = self.expConfig_.ipfixMtu
      self.ipfixVersion = self.expConfig_.ipfixVersion
      self.localIntfName = self.expConfig_.localIntfName
      self.localAddress = self.expConfig_.localAddress
      self.ipfixFormatConfigured = self.expConfig_.ipfixFormatConfigured

   def addCollector( self, hostAndPort ):
      t1( 'addCollector', hostAndPort )
      try:
         hnp = self.collectorHostAndPort[ hostAndPort.hostname ]
      except KeyError:
         hnp = None

      if hnp is None or hnp.port != hostAndPort.port:
         t1( 'addCollector hnp:', hnp )
         self.collectorHostAndPort[ hostAndPort.hostname ] = hostAndPort
         self.setChanged()

   def removeCollector( self, hnp ):
      t1( 'removeCollector hnp:', hnp )
      if hnp.hostname in self.collectorHostAndPort:
         del self.collectorHostAndPort[ hnp.hostname ]
         self.setChanged()

   def dscpIs( self, dscp ):
      t1( 'dscpIs:', dscp )
      if self.dscpValue != dscp:
         self.dscpValue = dscp
         self.setChanged()

   def templateIntervalIs( self, interval ):
      t1( 'templateIntervalIs:', interval )
      if self.templateInterval != interval:
         self.templateInterval = interval
         self.setChanged()

   def mtuIs( self, mtu ):
      if self.ipfixMtu != mtu:
         self.ipfixMtu = mtu
         self.setChanged()

   def ipfixVersionIs( self, _ipfixVersion ):
      if self.ipfixVersion != _ipfixVersion:
         self.ipfixVersion = _ipfixVersion
         self.setChanged()

   def localIntfNameIs( self, localIntfName ):
      if self.localIntfName != localIntfName:
         self.localIntfName = localIntfName
         self.setChanged()

   def localAddressIs( self, localAddress ):
      if self.localAddress != localAddress:
         self.localAddress = localAddress
         self.setChanged()

   def ipfixFormatConfiguredIs( self, ipfixFormatConfigured ):
      t3( 'ipfixFormatConfiguredIs', ipfixFormatConfigured )
      if self.ipfixFormatConfigured != ipfixFormatConfigured:
         self.ipfixFormatConfigured = ipfixFormatConfigured
         self.setChanged()

   def commitExporter( self, trConfig ):
      if not self.changed_:
         t0( 'Nothing changed for', self.expName_ )
         return None
      if self.deleted_:
         if self.expConfig_ is not None:
            t0( 'Deleting exporter from flowTrackerConfig', self.expName_ )
            del trConfig.expConfig[ self.expName_ ]
         else:
            t0( 'Exporter never created in flowTrackerConfig', self.expName_ )
         return None

      if self.expConfig_ is None:
         t0( 'Add new exporter to flowTrackerConfig', self.expName_ )
         self.expConfig_ = \
            trConfig.expConfig.newMember( self.expName_ )
         t0( 'trConfig', trConfig, trConfig.expConfig[ self.expName_ ] )

      staleCollector = set( self.expConfig_.collectorHostAndPort )
      for hostname, hnp in self.collectorHostAndPort.items():
         t0( 'Adding / updating collector configuration for', hostname )
         self.expConfig_.collectorHostAndPort.addMember( hnp )
         staleCollector -= set( [ hostname ] )
      for hostname in staleCollector:
         t0( 'Removing collector configuration for', hostname )
         del self.expConfig_.collectorHostAndPort[ hostname ]

      self.expConfig_.dscpValue = self.dscpValue
      self.expConfig_.templateInterval = self.templateInterval
      self.expConfig_.ipfixFormatConfigured = self.ipfixFormatConfigured
      self.expConfig_.ipfixVersion = self.ipfixVersion
      self.expConfig_.ipfixMtu = self.ipfixMtu
      self.expConfig_.localIntfName = self.localIntfName
      self.expConfig_.localAddress = self.localAddress

      return self.expConfig_

   def deletedIs( self, deleted=False ):
      if self.deleted_ == deleted:
         return
      t0( 'exporter deletedIs: ', deleted )
      self.deleted_ = deleted
      self.setChanged()
      # deleted and recreated in same session
      if not deleted:
         self.initFromDefaults()

   def deleted( self ):
      return self.deleted_

   def expConfig( self ):
      return self.expConfig_


#-------------------------------------------------------------------------------
# "[no|default] exporter <exporter-name>" command, in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

def getExporterName( mode ):
   return mode.context().exporterCtx.keys()

exporterNameMatcher = CliMatcher.DynamicNameMatcher( getExporterName,
                                                     "Exporter Name" )

def gotoExporterMode( mode, exporterName ):
   trContext = mode.context()
   trackerName = mode.context().trackerName()
   t1( 'gotoExporterMode of', trackerName, exporterName )

   if len( exporterName ) > FtConst.confNameMaxLen:
      mode.addError(
         'Exporter name is too long (maximum {})'.format( FtConst.confNameMaxLen ) )
      return

   exporterCtx = trContext.exporterContext( exporterName )

   if exporterCtx is None:
      t0( 'No exporterContext found, creating new exporter context' )
      exporterCtx = trContext.newExporterContext( exporterName )
   elif exporterCtx.deleted() or exporterCtx.expConfig():
      t0( 'Re-entering a deleted exporterContext' )
      exporterCtx.deletedIs( False )

   childMode = mode.childMode( ExporterMode,
                  context=exporterCtx, exporterName=exporterName )
   mode.session_.gotoChildMode( childMode )

def noExporterMode( mode, exporterName ):
   trContext = mode.context()
   trackerName = mode.context().trackerName()
   t1( 'noExporterMode of', trackerName, exporterName )

   exporterCtx = trContext.exporterContext( exporterName )

   if exporterCtx is None:
      return

   exporterCtx.deletedIs( True )
   return

class ExporterCmd( CliCommand.CliCommandClass ):
   syntax = '''exporter EXPORTER_NAME'''
   noOrDefaultSyntax = syntax

   data = {
      'exporter' : 'Configure flow exporter',
      'EXPORTER_NAME' : exporterNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      gotoExporterMode( mode, exporterName=args[ 'EXPORTER_NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noExporterMode( mode, exporterName=args[ 'EXPORTER_NAME' ] )

TrackerMode.addCommandClass( ExporterCmd )

#-------------------------------------------------------------------------------
# "[no|default] collector <ipv4/6-addr> [ port <udp-port>]" command in
# "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------
def getHostAndPort( mode, args ):
   ipGenAddr=args[ 'COLLECTOR' ]
   port=args.get( 'UDP_PORT', ipfixPort.ipfixPortDefault )
   addr = constants.ipAddrDefault if not ipGenAddr else str( ipGenAddr )
   trackerName = mode.context().trackerName()
   t1( 'getHostAndPort', trackerName, ipGenAddr, port, addr )
   return Tac.Value( 'Arnet::HostAndPort', hostname=addr, port=port )

class ExporterCollectorCmd( CliCommand.CliCommandClass ):
   syntax = '''collector COLLECTOR [ port UDP_PORT ]'''
   noOrDefaultSyntax = syntax

   data = {
      'collector' : 'Configure flow collector',
      'COLLECTOR' : IpGenAddrMatcher.IpGenAddrMatcher(
                        helpdesc='Collector IPv4 or IPv6 Address' ),
      'port' : 'Collector port',
      'UDP_PORT' : CliMatcher.IntegerMatcher(
                        ipfixPort.minPort,
                        ipfixPort.maxPort,
                        helpdesc='Collector port' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().addCollector( getHostAndPort( mode, args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().removeCollector( getHostAndPort( mode, args ) )

ExporterMode.addCommandClass( ExporterCollectorCmd )

#-------------------------------------------------------------------------------
# "[no|default] dscp <dscp>" command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------
class ExporterDscpCmd( CliCommand.CliCommandClass ):
   syntax = '''dscp DSCP'''
   noOrDefaultSyntax = '''dscp ...'''

   data = {
      'dscp' : 'Set the DSCP value',
      'DSCP' : CliMatcher.IntegerMatcher( 0, 63,
                  helpdesc='DSCP value between 0 and 63' )
   }
   @staticmethod
   def handler( mode, args ):
      mode.context().dscpIs( args.get( 'DSCP', constants.dscpValueDefault ) )

   noOrDefaultHandler = handler

ExporterMode.addCommandClass( ExporterDscpCmd )

#-------------------------------------------------------------------------------
# "[no|default] format ipfix version VERSION [ max-packet-size MTU ]" command
# in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------

class ExporterFormat( CliCommand.CliCommandClass ):
   syntax = '''format ipfix version VERSION [ max-packet-size MTU ]'''
   noOrDefaultSyntax = \
         '''format ipfix version [ ... | VERSION max-packet-size ... ] ) '''

   data = {
      'format' : 'Configure flow export format',
      'ipfix' : 'Configure flow export IPFIX format',
      'version' : 'Configure IPFIX version',
      'VERSION' : CliMatcher.IntegerMatcher(
                     ipfixVersion.minVersion, 
                     ipfixVersion.maxVersion,
                     helpdesc='IPFIX version' ),
      'max-packet-size' : 'Configure IPFIX maximum packet size',
      'MTU' : CliMatcher.IntegerMatcher(
                     ipfixMtu.minMtu,
                     ipfixMtu.maxMtu,
                     helpdesc='IPFIX maximum packet size' ),
   }

   @staticmethod
   def handler( mode, args ):
      version = args.get( 'VERSION', ipfixVersion.versionDefault )
      mtu = args.get( 'MTU', ipfixMtu.mtuDefault )
      t1( 'setFormatConfig version version', version, mtu )
      mode.context().ipfixVersionIs( version )
      mode.context().mtuIs( mtu )
      mode.context().ipfixFormatConfiguredIs( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().mtuIs( ipfixMtu.mtuDefault )
      if args.get( 'VERSION' ):
         # If version is specified then user wants to remove MTU setting.
         return
      else:
         # if verison is not specified, user unconfiguring export format
         mode.context().ipfixFormatConfiguredIs( False )
         mode.context().ipfixVersionIs( ipfixVersion.versionDefault )

ExporterMode.addCommandClass( ExporterFormat )

#-------------------------------------------------------------------------------
# "[no|default] local interface <intf>" command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------

localMatcher = CliMatcher.KeywordMatcher( 'local',
                        helpdesc='Configure the local interface' )

class ExporterLocalInterface( CliCommand.CliCommandClass ):
   syntax = '''local interface INTERFACE'''
   noOrDefaultSyntax = '''local interface ...'''

   data = {
      'local' : localMatcher,
      'interface' : 'Configure the local interface',
      'INTERFACE' : IntfCli.Intf.matcherWithIpSupport,
   }
   
   @staticmethod
   def handler( mode, args ):
      if mode.context().localAddress == constants.ipAddrDefault:
         mode.context().localIntfNameIs( args.get( 'INTERFACE' ).name )
      else:
         mode.addError( "Local address is already configured" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().localIntfNameIs( constants.intfNameDefault )

ExporterMode.addCommandClass( ExporterLocalInterface )

#-------------------------------------------------------------------------------
# "[no|default] local address <address>" command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------
class ExporterLocalAddress( CliCommand.CliCommandClass ):
   syntax = '''local address ADDRESS'''
   noOrDefaultSyntax = '''local address ...'''

   data = {
      'local' : localMatcher,
      'address' : CliCommand.Node( CliMatcher.KeywordMatcher( 'address',
                     helpdesc='Configure the local address' ), hidden=True ),
      'ADDRESS' : IpAddrMatcher.IpAddrMatcher( helpdesc='Local address' )
   }
   
   @staticmethod
   def handler( mode, args ):
      if mode.context().localIntfName == constants.intfNameDefault:
         mode.context().localAddressIs( args.get( 'ADDRESS' ) )
      else:
         mode.addError( "Local interface is already configured" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().localAddressIs( constants.ipAddrDefault )

ExporterMode.addCommandClass( ExporterLocalAddress )

#-------------------------------------------------------------------------------
# "[no|default] template interval <interval>" command in "config-ftr-tr-exp" mode
#-------------------------------------------------------------------------------

class ExporterTemplateCommand( CliCommand.CliCommandClass ):
   syntax = '''template interval INTERVAL'''
   noOrDefaultSyntax = '''template interval ...'''

   data = {
      'template' : 'Configure template parameters',
      'interval' : 'Configure template interval',
      'INTERVAL' : CliMatcher.IntegerMatcher(
                        templateInterval.minInterval,
                        templateInterval.maxInterval,
                        helpdesc='Template interval in milliseconds' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().templateIntervalIs(
               args.get( 'INTERVAL', templateInterval.intervalDefault ) )

   noOrDefaultHandler = handler

ExporterMode.addCommandClass( ExporterTemplateCommand )
