# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMode.FlexEncap
import CliMode.Intf
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import Tac
from TypeFuture import TacLazyType
from SubIntfCli import SubIntfModelet
from Toggles.EbraToggleLib import toggleMplsEvpnVpwsP2Enabled
from VlanCli import vlanIdMatcher

intfCapability = None
subIntfConfigDir = None

Dot1qEncap = TacLazyType( 'Bridging::Dot1qEncap' )
Dot1qEncapConfigMode = TacLazyType( 'Interface::SubIntfDot1qEncapConfigMode' )
FlexEncapRange = TacLazyType( 'Ebra::FlexEncapRange' )
FlexEncapField = TacLazyType( 'Ebra::FlexEncapField' )
FlexEncapSpec = TacLazyType( 'Ebra::FlexEncapSpec' )
FlexEncapEntry = TacLazyType( 'Ebra::FlexEncapEntry' )
VlanEncapReason = TacLazyType( 'Interface::SubIntfConfigDir::VlanEncapReason' )

ModeConflictStr = ( 'encapsulation submode cannot be used in conjunction '
                    'with encapsulation dot1q vlan command' )

def vlanIdFromMatcher( vlanObj ):
   return vlanObj.id if vlanObj else None

def field( value ):
   if value is None:
      return FlexEncapField.createAnyOrNone()
   else:
      # We only support single tagged value at the moment
      assert isinstance( value, int )
      fld = Tac.nonConst( FlexEncapField.createDot1qTagged() )
      fld.insertTaggedRange( FlexEncapRange( value, value ) )
      return fld

def parseClientEncapsulation( args ):
   """
   Parser for ClientExpression; the argument keys are a concatenation of
   of the key specified in ClientExpression and the corresponding
   SingleTagExprFactory/DoubleTagExprFactory
   """
   outerValue = vlanIdFromMatcher( args.get( 'CLIENT_SINGLE_TAG' ) or
                                   args.get( 'CLIENT_DOUBLE_OUTER_TAG' ) )
   outer = field( outerValue )
   inner = field( vlanIdFromMatcher( args.get( 'CLIENT_DOUBLE_INNER_TAG' ) ) )
   return FlexEncapSpec.createTagged( outer, inner )

def parseNetworkEncapsulation( args ):
   if 'NETWORK_CLIENT' in args:
      return FlexEncapSpec.createClient()
   else:
      outerValue = vlanIdFromMatcher( args.get( 'NETWORK_SINGLE_TAG' ) or
                                      args.get( 'NETWORK_DOUBLE_OUTER_TAG' ) )
      if outerValue is not None:
         outer = field( outerValue )
         inner = field( vlanIdFromMatcher( args.get( 'NETWORK_DOUBLE_INNER_TAG' ) ) )
         return FlexEncapSpec.createTagged( outer, inner )
      else:
         return FlexEncapSpec.createAnyOrNone()

def setEncap( mode, args ):
   """
   handler used by different variants of EncapsulationCmd(s) to parse
   the data and set the encapsulation
   """
   client = parseClientEncapsulation( args )
   network = parseNetworkEncapsulation( args )
   entry = FlexEncapEntry( client, network )
   flexEncapMode = Dot1qEncapConfigMode.subIntfDot1qEncapConfigFlexEncap
   status = subIntfConfigDir.setVlanEncap(
         mode.intf.name, entry, flexEncapMode, False )
   if status.reason == VlanEncapReason.modeConflict:
      mode.addError( ModeConflictStr )
   elif status.reason != VlanEncapReason.success:
      # sanity check
      #
      # We will not support CLI revertive behavior for dot1q conflict
      # because this logic will become extremely complicated once we
      # support VLAN ranges, and multiple flex encap entries.
      #
      # Even though it's easy to detect dot1q conflict in the initial phase,
      # we want to maintain a consistent user experience across phases.
      #
      # As a result, we should never get siblingConflict as a return value
      mode.addError( "Unexpected failure reason: {}".format( status.reason ) )

def noOrDefaultEncap( mode, args ):
   """
   no or default handler used by different variants of EncapsulationCmd(s) to parse
   the data and delete the encapsulation

   The deletion does not do wild card matching, so we will only delete the
   encapsulation entry if there's an exact match
   """
   client = parseClientEncapsulation( args )
   network = parseNetworkEncapsulation( args )
   entry = FlexEncapEntry( client, network )
   flexEncapMode = Dot1qEncapConfigMode.subIntfDot1qEncapConfigFlexEncap
   reason = subIntfConfigDir.deleteVlanEncap(
         mode.intf.name, entry, True, flexEncapMode )
   if reason == VlanEncapReason.modeConflict:
      mode.addError( ModeConflictStr )
   elif reason != VlanEncapReason.success:
      # sanity check
      mode.addError( "Unexpected failure reason: {}".format( reason ) )

def vlanXlateSupportedGuard( mode, token ):
   if intfCapability.flexEncapVlanXlateSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def vlanXlateFullySupportedGuard( mode, token ):
   if intfCapability.flexEncapVlanXlateFullySupported:
      return vlanXlateSupportedGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def raiseInvalidInputError( msg ):
   raise CliParser.InvalidInputError( ": " + msg )

class EncapsulationMode( CliMode.FlexEncap.EncapsulationBaseMode,
                         BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, intf ):
      CliMode.FlexEncap.EncapsulationBaseMode.__init__( self, intf.name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.intf = intf

#-------------------------------------------------------------------------------
# The "[no] encapsulation vlan" command
#-------------------------------------------------------------------------------
class EncapsulationsCmd( CliCommand.CliCommandClass ):
   syntax = "encapsulation vlan"
   noOrDefaultSyntax = syntax
   data = {
      "encapsulation": CliCommand.Node(
                          matcher=CliMatcher.KeywordMatcher(
                             "encapsulation", helpdesc="configure encapsulation" ),
                          canMerge=False ),
      "vlan": 'VLAN encapsulation',
   }

   @staticmethod
   def handler( mode, args ):
      if not isinstance( mode.session_.mode_, CliMode.Intf.IntfMode ):
         raiseInvalidInputError( "not supported in interface range mode" )
      childMode = mode.childMode( EncapsulationMode, intf=mode.intf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      flexEncapMode = Dot1qEncapConfigMode.subIntfDot1qEncapConfigFlexEncap
      reason = subIntfConfigDir.deleteVlanEncap(
            mode.intf.name, FlexEncapEntry(), False, flexEncapMode )
      if reason == VlanEncapReason.modeConflict:
         mode.addWarning( ModeConflictStr )
      elif reason == VlanEncapReason.success:
         childMode = mode.childMode( EncapsulationMode, intf=mode.intf )
         childMode.removeComment()
      else:
         # sanity check
         mode.addError( "Unexpected failure reason: {}".format( reason ) )

if toggleMplsEvpnVpwsP2Enabled():
   SubIntfModelet.addCommandClass( EncapsulationsCmd )

#-------------------------------------------------------------------------------
# Submode: encapsulation vlan
# TAG-SPEC: 1 - 4094
# TPID: dot1q
# SINGLE-TAG: TAG-SPEC
# DOUBLE-TAG: outer TAG-SPEC inner TAG-SPEC
# [no] client TPID ( SINGLE-TAG | DOUBLE-TAG )
#      [ network client | TPID ( SINGLE-TAG | DOUBLE-TAG ) ]
#
# flexEncapVlanXlateSupported hwCapability is required to enable the optional
# network commands
# flexEncapVlanXlateFullySupported hwCapability is further required to enable
# all VLAN translations
#-------------------------------------------------------------------------------
dot1qMatcher = CliMatcher.KeywordMatcher( 'dot1q', 'Dot1q encapsulation' )
nodeDot1q = CliCommand.Node( matcher=dot1qMatcher )
nodeDot1qWithGuard = CliCommand.Node( matcher=dot1qMatcher,
                                      guard=vlanXlateFullySupportedGuard )
outerMatcher = CliMatcher.KeywordMatcher( 'outer', 'Outer VLAN tags' )
nodeOuter = CliCommand.Node( matcher=outerMatcher, noResult=True )
nodeOuterWithGuard = CliCommand.Node( matcher=outerMatcher,
                                      guard=vlanXlateFullySupportedGuard,
                                      noResult=True )
nodeNetwork = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher(
         "network", "Network encapsulation" ),
         guard=vlanXlateSupportedGuard,
      noResult=True )
nodeNetworkSideClient = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher(
         "client", "Retain client encapsulation" ) )

class TpidExprFactory( CliCommand.CliExpressionFactory ):
   """
   A TPID expression that can be used by both the client and network portion

   This expression becomes more beneficial once we support multiple TPIDs
   (dot1q, dot1ad)

   Note: the TPID nodes are defined as a class variable (for instance, dot1qNode)
         because in some variants, we need the nodes to be guarded
   """
   dot1qNode = nodeDot1q

   def generate( self, name ):
      dot1qName = name + '_dot1q'
      class TpidExpression( CliCommand.CliExpression ):
         expression = '{}'.format( dot1qName )
         data = {
            dot1qName : self.dot1qNode
         }
      return TpidExpression

class GuardedTpidExprFactory( TpidExprFactory ):
   """ TPID tokens are guarded with vlanXlateFullySupportedGuard """
   dot1qNode = nodeDot1qWithGuard

class SingleTagExprFactory( CliCommand.CliExpressionFactory ):
   """
   A single tag expression that can be used by both the client and network portion
   ie.
      TAG-SPEC
   """
   def generate( self, name ):
      tagName = name + '_TAG'
      class SingleTagExpression( CliCommand.CliExpression ):
         expression = '{}'.format( tagName )
         data = {
            tagName : vlanIdMatcher,
         }
      return SingleTagExpression

class DoubleTagExprFactory( CliCommand.CliExpressionFactory ):
   """
   A double tag expression that can be used by both the client and network portion
   ie.
      outer TAG-SPEC inner TAG-SPEC

   Note: 'outer' node is defined as a class variable (outerNode) because in some
         variants, we need the node to be guarded
   """
   outerNode = nodeOuter

   def generate( self, name ):
      outerNodeKey = name + '_outer'
      outerTagName = name + '_OUTER_TAG'
      innerNodeKey = name + '_inner'
      innerTagName = name + '_INNER_TAG'
      class DoubleTagExpression( CliCommand.CliExpression ):
         expression = '{} {} {} {}'.format( outerNodeKey, outerTagName,
                                            innerNodeKey, innerTagName )
         data = {
            outerNodeKey : self.outerNode,
            outerTagName : vlanIdMatcher,
            innerNodeKey : CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher(
                                 'inner', 'Inner VLAN tags' ),
                              noResult=True ),
            innerTagName : vlanIdMatcher,
         }
      return DoubleTagExpression

class GuardedDoubleTagExprFactory( DoubleTagExprFactory ):
   """ outer node is guarded with vlanXlateFullySupportedGuard """
   outerNode = nodeOuterWithGuard

class ClientExpression( CliCommand.CliExpression ):
   expression = 'client TPID_EXPR'
   data = {
      'client' : 'Client encapsulation',
      'TPID_EXPR' : TpidExprFactory(),
   }

class SingleTagClientCmd( CliCommand.CliCommandClass ):
   """
   client dot1q TAG-SPEC [ network client |
                                   dot1q TAG-SPEC |
                                         ( outer TAG-SPEC inner TAG-SPEC )

   node network is guarded with vlanXlateSupportedGuard
   nodeOuter is guarded with vlanXlateFullySupportedGuarded
   """
   syntax = """CLIENT_EXPR CLIENT_SINGLE
               [ network NETWORK_CLIENT |
                         ( NETWORK_TPID NETWORK_SINGLE | NETWORK_DOUBLE ) ]"""
   noOrDefaultSyntax = syntax

   data = {
      'CLIENT_EXPR' : ClientExpression,
      'CLIENT_SINGLE' : SingleTagExprFactory(),
      "network" : nodeNetwork,
      "NETWORK_CLIENT" : nodeNetworkSideClient,
      "NETWORK_TPID" : TpidExprFactory(),
      "NETWORK_SINGLE" : SingleTagExprFactory(),
      "NETWORK_DOUBLE" : GuardedDoubleTagExprFactory(),
   }

   handler = setEncap
   noOrDefaultHandler = noOrDefaultEncap
EncapsulationMode.addCommandClass( SingleTagClientCmd )

class DoubleTagClientCmd( CliCommand.CliCommandClass ):
   """
   client dot1q outer TAG-SPEC inner TAG-SPEC
   [ network client | dot1q TAG-SPEC |
                            ( outer TAG-SPEC inner TAG-SPEC )

   node network is guarded with vlanXlateSupportedGuard
   network-side node dot1q is guarded with vlanXlateFullySupportedGuarded
   """
   syntax = """CLIENT_EXPR CLIENT_DOUBLE
               [ network NETWORK_CLIENT |
                         ( NETWORK_TPID NETWORK_SINGLE | NETWORK_DOUBLE ) ]"""
   noOrDefaultSyntax = syntax

   data = {
      'CLIENT_EXPR' : ClientExpression,
      'CLIENT_DOUBLE' : DoubleTagExprFactory(),
      "network" : nodeNetwork,
      "NETWORK_CLIENT" : nodeNetworkSideClient,
      "NETWORK_TPID" : GuardedTpidExprFactory(),
      "NETWORK_SINGLE" : SingleTagExprFactory(),
      "NETWORK_DOUBLE" : DoubleTagExprFactory(),
   }

   handler = setEncap
   noOrDefaultHandler = noOrDefaultEncap
EncapsulationMode.addCommandClass( DoubleTagClientCmd )

def Plugin( entityManager ):
   global intfCapability
   global subIntfConfigDir
   intfCapability = LazyMount.mount( entityManager,
                                     "interface/hardware/capability",
                                     "Interface::Hardware::Capability", "r" )
   subIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/subintf",
                                         "Interface::SubIntfConfigDir", "w" )
