#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import CliMode.Firewall
import CliMode.Classification
import CliParser
import ConfigMount
import EthIntfCli
import LazyMount
import Tracing
import CliToken
import CliMatcher
import CliCommand
import ShowCommand
from VlanCli import vlanSetMatcher
from FirewallModels import (
      FirewallInfo,
      Segment,
      FwPolicyInfo,
      FwServiceInfo,
      FwCounterInfo,
      VlanRangeModel,
      InterfaceVlanRange,
      FwHwStatusInfo,
      )
from FirewallCliLib import Context
from CliExtensions import CliHook
from CliPlugin.RouteMapCli import rtMapCommListPreListRe
from CliPlugin.VrfCli import DEFAULT_VRF, VrfExprFactory
from VirtualIntfRule import IntfMatcher, VirtualIntfMatcher
from LagIntfCli import EthLagIntf
from IntfRangePlugin.LagIntf import PortChannelNum

# pkgdeps: rpm FirewallLib-lib
# pkgdeps: rpm MatchList-lib

t0 = Tracing.Handle( "FirewallCli" ).trace0

# --------------------------------------------------------------------------
# Firewall globals
# --------------------------------------------------------------------------
entMan = None
firewallConfig = None
config = None
status = None
firewallCounters = None
hwCapability = None
matchListCapability = None
l3ConfigDir = None
ipStatus = None
matchListConfig = None
fwConsts = Tac.Type( "Firewall::FirewallConstants" )
VlanRange = Tac.Type( "Firewall::VlanRange" )
FirewallProtocol = Tac.Type( "Firewall::Protocol" )

# --------------------------------------------------------------------------
# Utility functions
# --------------------------------------------------------------------------
def segSecSupportedGuard( mode, token ):
   if hwCapability.segSecSupported:
      return None
   return CliParser.guardNotThisPlatform

def portChannelSegmentSupportedGuard( mode, token ):
   if hwCapability.portChannelSegmentSupported:
      return None
   return CliParser.guardNotThisPlatform

def vrfSupportedGuard( mode, token ):
   if hwCapability.vrfSupported:
      return None
   return CliParser.guardNotThisPlatform

def intfVlanSupportedGuard( mode, token ):
   if hwCapability.intfVlanSupported:
      return None
   return CliParser.guardNotThisPlatform

def statelessActionSupportedGuard( mode, token ):
   if hwCapability.statelessActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def clearVrfSessionsSupportedGuard( mode, token ):
   if hwCapability.clearVrfSessionsSupported and hwCapability.vrfSupported:
      return None
   return CliParser.guardNotThisPlatform

def segSecCountersSupportedGuard( mode, token ):
   if hwCapability.segSecCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

def segSecMatchInterfaceSupportedGuard( mode, token ):
   if hwCapability.segmentMatchInterfaceSupported:
      return None
   return CliParser.guardNotThisPlatform

def segSecMatchIpv6PrefixSupportedGuard( mode, token ):
   if matchListCapability.ipv6PrefixSupported:
      return None
   return CliParser.guardNotThisPlatform

def segSecConfigurablePolicySupportedGuard( mode, token ):
   if hwCapability.segSecConfigurablePolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def defaultPolicyDropGuard( mode, token ):
   if hwCapability.defaultPolicyDropSupported:
      return None
   return CliParser.guardNotThisPlatform

# --------------------------------------------------------------------------
# config mode definition
# --------------------------------------------------------------------------
class FirewallConfigMode( CliMode.Firewall.FirewallConfigMode,
                          BasicCli.ConfigModeBase ):
   name = "Global Segment Security configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.session_ = session
      CliMode.Firewall.FirewallConfigMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getClassName( self, segmentName ):
      return '__segment_%s' % ( segmentName )

   def getSegment( self, segmentName ):
      return firewallConfig.segmentDir.segment[ segmentName ]

   def getConfigChildModeType( self ):
      return ClassMapFirewallConfigMode

   def getPolicyChildModeType( self ):
      return SegmentPolicyMode

class FirewallPolicyConfigMode( CliMode.Firewall.FirewallPolicyConfigMode,
                                BasicCli.ConfigModeBase ):
   context = None
   name = "Segment Security Policy definition"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      t0( "PolicyConfigMode init" )
      self.context = context
      self.aborted = False

      CliMode.Firewall.FirewallPolicyConfigMode.__init__( self,
                                                          context.policyName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( "onExit called for PolicyConfigMode" )
      if not self.aborted:
         self.context.commit()

   def abort( self ):
      t0( "abort called for PolicyConfigMode" )
      self.context.abort()
      self.aborted = True
      self.session_.gotoParentMode()

class FirewallVrfConfigMode( CliMode.Firewall.FirewallVrfConfigMode,
                             BasicCli.ConfigModeBase ):
   name = "Segment Security definition"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      self.session_ = session
      self.vrfName = vrfName
      firewallConfig.vrf.newMember( vrfName )
      CliMode.Firewall.FirewallVrfConfigMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getClassName( self, segmentName ):
      return '__VRF_%s_segment_%s' % ( self.vrfName_, segmentName )

   def getSegment( self, segmentName ):
      return firewallConfig.vrf[ self.vrfName_ ].segmentDir.segment[ segmentName ]

   def getConfigChildModeType( self ):
      return ClassMapFirewallVrfConfigMode

   def getPolicyChildModeType( self ):
      return VrfSegmentPolicyMode

class FirewallVrfSegmentConfigMode( CliMode.Firewall.FirewallVrfSegmentConfigMode,
                                    BasicCli.ConfigModeBase ):
   name = "Segment definition"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, segmentName ):
      self.session_ = session
      firewallConfig.vrf[ parent.vrfName ].segmentDir.segment.newMember(
            segmentName )
      CliMode.Firewall.FirewallVrfSegmentConfigMode.__init__( self, segmentName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class FirewallSegmentConfigMode( CliMode.Firewall.FirewallSegmentConfigMode,
                                 BasicCli.ConfigModeBase ):
   name = "Segment definition"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, segmentName ):
      self.session_ = session
      if firewallConfig.segmentDir is None:
         firewallConfig.segmentDir = ( "l2SegDir", )
      firewallConfig.segmentDir.segment.newMember( segmentName )
      CliMode.Firewall.FirewallSegmentConfigMode.__init__( self, segmentName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ClassMapFirewallVrfConfigMode( CliMode.Firewall.ClassMapFirewallVrfConfigMode,
                                     BasicCli.ConfigModeBase ):
   name = "Configure Segment Security Definition"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, cmapName ):
      self.session_ = session
      self.cmapName = cmapName
      firewallConfig.classMap.newMember( self.cmapName )
      CliMode.Firewall.ClassMapFirewallVrfConfigMode.__init__( self, self.cmapName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ClassMapFirewallConfigMode( CliMode.Firewall.ClassMapFirewallConfigMode,
                                  BasicCli.ConfigModeBase ):
   name = "Configure Segment Security Definition"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, cmapName ):
      self.session_ = session
      self.cmapName = cmapName
      firewallConfig.classMap.newMember( self.cmapName )
      CliMode.Firewall.ClassMapFirewallConfigMode.__init__( self, self.cmapName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class VrfSegmentPolicyMode( CliMode.Firewall.VrfSegmentPolicyMode,
                         BasicCli.ConfigModeBase ):
   name = "Segment policies"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.session_ = session
      CliMode.Firewall.VrfSegmentPolicyMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class SegmentPolicyMode( CliMode.Firewall.SegmentPolicyMode,
                         BasicCli.ConfigModeBase ):
   name = "Segment policies"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.session_ = session
      CliMode.Firewall.SegmentPolicyMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------
# Helper Functions
# --------------------------------------------------------------------------
def getAllServiceName( mode ):
   return config.services.keys()

def getAllPoliciesName( mode ):
   return firewallConfig.policy.keys()

def getIpv4PrefixListNames( mode ):
   return matchListConfig.matchIpv4PrefixList

def getIpv6PrefixListNames( mode ):
   return matchListConfig.matchIpv6PrefixList

def getSegmentNamesInVrf( mode ):
   if hasattr( mode, 'vrfName_' ):
      vrfName = mode.vrfName_
   elif hasattr( mode.parent_.parent_, 'vrfName_' ):
      vrfName = mode.parent_.parent_.vrfName_
   else:
      return None

   return firewallConfig.vrf[ vrfName ].segmentDir.segment

# --------------------------------------------------------------------------
# Common nodes and Matchers
# --------------------------------------------------------------------------
firewallPolicyNameMatcher = CliMatcher.DynamicNameMatcher( getAllPoliciesName,
                                                           helpdesc='Policy name' )

ipv4PrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getIpv4PrefixListNames,
      pattern=rtMapCommListPreListRe, helpdesc='Ipv4 Prefix list' )

ipv6PrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getIpv6PrefixListNames,
      pattern=rtMapCommListPreListRe, helpdesc='Ipv6 Prefix list' )

vrfSegmentNameMatcher = CliMatcher.DynamicNameMatcher( getSegmentNamesInVrf,
      pattern='[a-zA-Z0-9_-]+', helpdesc='Segment name', helpname='WORD' )

segmentNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                               helpdesc='Segment name',
                                               helpname='WORD' )

applicationNameMatcher = CliMatcher.DynamicNameMatcher( getAllServiceName,
                                                       helpdesc='Application name' )
segmentSecurityNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'segment-security',
         helpdesc='Show segment security information' ),
      guard=segSecSupportedGuard )

segmentSecurityNodeForClear = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'segment-security',
         helpdesc='Clear segment security information' ),
      guard=segSecSupportedGuard )

policyNodeForDefaultPolicyDrop = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'policy',
         helpdesc='Configure policy default drop behaviour' ),
      guard=defaultPolicyDropGuard )

matchNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'match',
         helpdesc='Specify interfaces or subnets' ) )

segmentSecurityCountersNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'counters',
         helpdesc='Segment security counters' ),
      guard=segSecCountersSupportedGuard )

segmentSecurityInterfaceNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'interface',
         helpdesc='Name of the interface' ),
      guard=segSecMatchInterfaceSupportedGuard )

segmentSecurityIpv6PrefixNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'prefix-ipv6',
         helpdesc='Match Ipv6 match list' ),
      guard=segSecMatchIpv6PrefixSupportedGuard )

segmentSecurityConfigurablePolicyNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'policy',
         helpdesc='Configure Segment Security policy' ),
      guard=segSecConfigurablePolicySupportedGuard )

vrfExprFactoryForShow = VrfExprFactory(
      helpdesc='Show segment security information in VRF',
      guard=vrfSupportedGuard )

vrfExprFactoryForClear = VrfExprFactory(
      helpdesc='Clear segment security information in VRF',
      guard=vrfSupportedGuard )

# --------------------------------------------------------------------------
# 'router segment-security' to enter config-router-seg-sec mode
# --------------------------------------------------------------------------
class RouterSegmentSecurityModeCmd( CliCommand.CliCommandClass ):
   syntax = 'router segment-security'
   noOrDefaultSyntax = syntax
   data = {
         'router': CliToken.Router.routerMatcherForConfig,
         'segment-security': segmentSecurityNode,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( FirewallConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      def clearPolicies():
         for policy in list( firewallConfig.policy ):
            if not firewallConfig.policy[ policy ].readonly:
               del firewallConfig.policy[ policy ]
      if firewallConfig.enabled:
         firewallConfig.enabled = False
      clearPolicies()
      firewallConfig.vrf.clear()
      firewallConfig.classMap.clear()
      firewallConfig.segmentDir = None

BasicCli.GlobalConfigMode.addCommandClass( RouterSegmentSecurityModeCmd )

# --------------------------------------------------------------------------
# 'policy <policy-name>' to enter router-seg-sec-policy mode
# --------------------------------------------------------------------------
class PolicyModeCmd( CliCommand.CliCommandClass ):
   syntax = 'policy POLICY_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'policy': segmentSecurityConfigurablePolicyNode,
         'POLICY_NAME': firewallPolicyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      t0( "%s" % policyName )
      context = Context( policyName, firewallConfig.policy )
      childMode = mode.childMode( FirewallPolicyConfigMode, context=context )
      childMode.context.modeIs( childMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      t0( "Delete policy %s" % policyName )
      del firewallConfig.policy[ policyName ]
      # If the delete is done from within the policy config mode, check
      # that it isn't this instance; if it is, abort all changes here
      if isinstance( mode.session_.modeOfLastPrompt(), FirewallPolicyConfigMode ):
         if policyName == mode.session_.modeOfLastPrompt().context.policyName:
            mode.session_.modeOfLastPrompt().context.abort()
            mode.session_.modeOfLastPrompt().aborted = True

FirewallConfigMode.addCommandClass( PolicyModeCmd )

# --------------------------------------------------------------------------
# abort ( from FirewallPolicyConfigMode )
# --------------------------------------------------------------------------
class PolicyAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = { 'abort': 'Exit without committing changes' }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

FirewallPolicyConfigMode.addCommandClass( PolicyAbortCmd )

# --------------------------------------------------------------------------
# 'vrf VRF' to enter router-seg-sec-vrf mode
# Only 'default' is allowed.
# --------------------------------------------------------------------------
class RouterSegmentSecurityVrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
         'VRF': VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                guard=vrfSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( FirewallVrfConfigMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      del firewallConfig.vrf[ vrfName ]
      classNamePrefix = '__VRF_%s_segment' % vrfName
      for cmap in firewallConfig.classMap:
         if cmap[ : len( classNamePrefix ) ] == classNamePrefix:
            del firewallConfig.classMap[ cmap ]

FirewallConfigMode.addCommandClass( RouterSegmentSecurityVrfModeCmd )

def segmentAbsent( configDir, segmentName ):
   if not configDir.segmentDir:
      return True
   return segmentName not in configDir.segmentDir.segment

def getSegmentCount( segDir ):
   if segDir:
      return len( segDir.segment )
   else:
      return 0

# --------------------------------------------------------------------------
# 'segment <segment-name>' to enter router-seg-sec-vrf-segment mode
# --------------------------------------------------------------------------
class RouterSegmentSecurityVrfSegmentModeCmd( CliCommand.CliCommandClass ):
   syntax = 'segment SEGMENT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'segment': 'Configure Segment',
         'SEGMENT_NAME': vrfSegmentNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      segAbsent = segmentAbsent( firewallConfig.vrf[ mode.vrfName_ ], segmentName )
      segLimit = hwCapability.maxSegmentsSupported
      segmentCount = 0
      if hwCapability.maxSegmentsLimitsPerVrf:
         segDir = firewallConfig.vrf[ mode.vrfName_ ].segmentDir
         segmentCount = getSegmentCount( segDir )
      else:
         for vrf in firewallConfig.vrf:
            sDir = firewallConfig.vrf[ vrf ].segmentDir
            segmentCount += getSegmentCount( sDir )
      if segAbsent and segmentCount >= segLimit:
         mode.addError( 'Number of segments cannot exceed the '
               'limit of %d' % segLimit )
         return
      childMode = mode.childMode( FirewallVrfSegmentConfigMode,
                                  segmentName=segmentName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      del firewallConfig.vrf[ mode.vrfName_ ].segmentDir.segment[ segmentName ]
      className = mode.getClassName( segmentName )
      del firewallConfig.classMap[ className ]

FirewallVrfConfigMode.addCommandClass( RouterSegmentSecurityVrfSegmentModeCmd )

# --------------------------------------------------------------------------
# 'segment <segment-name>' to enter router-seg-sec-segment mode
# --------------------------------------------------------------------------
segmentNode = CliCommand.Node( matcher=segmentNameMatcher,
                               guard=intfVlanSupportedGuard )

class RouterSegmentSecuritySegmentModeCmd( CliCommand.CliCommandClass ):
   syntax = 'segment SEGMENT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'segment': 'Configure Segment',
         'SEGMENT_NAME': segmentNode,
   }

   @staticmethod
   def handler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      segAbsent = segmentAbsent( firewallConfig, segmentName )
      segLimit = hwCapability.maxSegmentsSupported
      segDir = firewallConfig.segmentDir
      if segAbsent and getSegmentCount( segDir ) >= segLimit:
         mode.addError( 'Number of segments cannot exceed the '
               'limit of %d' % segLimit )
         return
      childMode = mode.childMode( FirewallSegmentConfigMode,
                                  segmentName=segmentName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segmentName = args[ 'SEGMENT_NAME' ]
      if firewallConfig.segmentDir:
         del firewallConfig.segmentDir.segment[ segmentName ]
         # If all L2 segments are removed, remove the dir
         if not firewallConfig.segmentDir.segment:
            firewallConfig.segmentDir = None
      className = mode.getClassName( segmentName )
      del firewallConfig.classMap[ className ]

FirewallConfigMode.addCommandClass( RouterSegmentSecuritySegmentModeCmd )

# ---------------------------------------------------------------------------------
# 'definition' to enter 'config-router-seg-sec-vrf-segment-def' mode
#    or
# 'definition' to enter 'config-router-seg-sec-segment-def' mode
# ---------------------------------------------------------------------------------
class SegmentDefinitionModeCmd( CliCommand.CliCommandClass ):
   syntax = 'definition'
   noOrDefaultSyntax = syntax

   data = {
         'definition': 'Configure segment definition',
   }

   @staticmethod
   def handler( mode, args ):
      className = mode.parent_.getClassName( mode.segment_ )
      segment = mode.parent_.getSegment( mode.segment_ )
      childModeType = mode.parent_.getConfigChildModeType()
      firewallConfig.classMap.newMember( className )
      segment.classMap = className
      childMode = mode.childMode( childModeType, cmapName=className )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      className = mode.parent_.getClassName( mode.segment_ )
      segment = mode.parent_.getSegment( mode.segment_ )
      segment.classMap = ''
      segment.ipv4PrefixList = ''
      del firewallConfig.classMap[ className ]

FirewallVrfSegmentConfigMode.addCommandClass( SegmentDefinitionModeCmd )
FirewallSegmentConfigMode.addCommandClass( SegmentDefinitionModeCmd )

# --------------------------------------------------------------------------
# 'policies' to enter router-seg-sec-vrf-segment-policies mode
#   or
# 'policies' to enter router-seg-sec-segment-policies mode
# --------------------------------------------------------------------------
class SegmentPoliciesModeCmd( CliCommand.CliCommandClass ):
   syntax = 'policies'
   noOrDefaultSyntax = syntax

   data = {
         'policies': 'Segment policies',
   }

   @staticmethod
   def handler( mode, args ):
      childModeType = mode.parent_.getPolicyChildModeType()
      childMode = mode.childMode( childModeType )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segment = mode.parent_.getSegment( mode.segment_ )
      segment.policy.clear()

FirewallVrfSegmentConfigMode.addCommandClass( SegmentPoliciesModeCmd )
FirewallSegmentConfigMode.addCommandClass( SegmentPoliciesModeCmd )

# --------------------------------------------------------------------------
# commands in 'policy <policy-name>' sub-mode
# [ no | <sequence-no> ] application <app-name>
#                         action <forward|drop> [stateless] [ log ]
# no <sequence-no>
# --------------------------------------------------------------------------
# max rule sequence number 10-bit. We use only 10-bits out of 32-bit, as this
# will be combined with 10 bit service rule sequence number to generate a 20 bit
# ruleId.
# Platforms can use the remaining 12 bits to augment any PD specific things
MAX_SEQ = 0x3FF # 10-bit integer
sequenceNumMatcher = CliMatcher.IntegerMatcher( 1,
                                                MAX_SEQ,
                                                helpdesc='Index in the sequence' )

statelessActionNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'stateless',
         helpdesc='Perform action irrespective of state' ),
      guard=statelessActionSupportedGuard )

class ApplicationActionCmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQUENCE_NUM ] application APPLICATION_NAME' \
            ' ( action ( forward  | drop ) [ stateless ] ) [ log ]'
   noOrDefaultSyntax = '( SEQUENCE_NUM | ' \
        '( [ SEQUENCE_NUM ]application APPLICATION_NAME ) )' \
            ' [ ( action ( forward  | drop ) [ stateless ] ) [ log ] ]'

   data = {
      'SEQUENCE_NUM': sequenceNumMatcher,
      'application': 'Application name',
      'APPLICATION_NAME': applicationNameMatcher,
      'action': 'Specify an action',
      'forward': 'Allow an application',
      'stateless': statelessActionNode,
      'drop': 'Drop an application',
      'log': 'Enable logging',
   }

   @staticmethod
   def handler( mode, args ):
      serviceName = args[ 'APPLICATION_NAME' ]
      seqNum = args.get( 'SEQUENCE_NUM' )
      policy = mode.context.tmpPolicy

      def getLastSeqNum():
         return policy.lastSeqNum

      def setLastSeqNum():
         lastSeqnum = policy.lastSeqNum
         if lastSeqnum <= seqNum:
            policy.lastSeqNum = seqNum

      if seqNum == 'default':
         return
      assert not ( isinstance( seqNum, bool ) and seqNum is True )
      if seqNum is None:
         seqNum = getLastSeqNum() + 10
      if seqNum > MAX_SEQ or seqNum < 1:
         mode.addError( "Sequence number out of range [1..%s]" % str( MAX_SEQ ) )
         return

      if policy.rule.has_key( seqNum ) and \
            policy.rule[ seqNum ].serviceName != serviceName:
         mode.addError( "Sequence %d is already associated with application %s" %
               ( seqNum, policy.rule[ seqNum ].serviceName ) )
         return

      # cleanup old sequence number associated with serviceName from
      if policy.serviceToSeqnum.has_key( serviceName ):
         oldSeqnum = policy.serviceToSeqnum[ serviceName ]
         if policy.rule.has_key( oldSeqnum ):
            del policy.rule[ oldSeqnum ]
      if policy.rule.has_key( seqNum ):
         del policy.rule[ seqNum ]

      policy.serviceToSeqnum[ serviceName ] = seqNum
      action = 'permit' if 'forward' in args else 'deny'
      if action == 'permit' and 'stateless' in args:
         action = 'statelessPermit'
      if action == 'deny' and 'stateless' in args:
         action = 'statelessDeny'
      logValue = True if 'log' in args else False
      policy.newRule( seqNum, serviceName, action, logValue )
      setLastSeqNum()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      seqNum = args.get( 'SEQUENCE_NUM' )
      policy = mode.context.tmpPolicy
      if not seqNum:
         serviceName = args[ 'APPLICATION_NAME' ]
         del policy.serviceToSeqnum[ serviceName ]
         for seq in policy.rule.keys():
            if policy.rule[ seq ].serviceName == serviceName:
               del policy.rule[ seq ]
               break
         return
      if not policy.rule.has_key( seqNum ):
         return
      serviceName = policy.rule[ seqNum ].serviceName
      del policy.serviceToSeqnum[ serviceName ]
      del policy.rule[ seqNum ]

FirewallPolicyConfigMode.addCommandClass( ApplicationActionCmd )

# --------------------------------------------------------------------------
# '[ no | default ] shutdown' in 'vrf default'
# --------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax

   data = {
      'shutdown': 'Shutdown Segment Security'
   }

   @staticmethod
   def handler( mode, args ):
      firewallConfig.enabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      firewallConfig.enabled = True

FirewallConfigMode.addCommandClass( ShutdownCmd )

# --------------------------------------------------------------------------
# '[ no | default ] segment policy policy-drop-all default'
# --------------------------------------------------------------------------
class SegmentPolicyDefaultDrop( CliCommand.CliCommandClass ):
   syntax = 'segment policy policy-drop-all default'
   noOrDefaultSyntax = syntax

   data = {
      'segment': 'Configure Segment',
      'policy' : policyNodeForDefaultPolicyDrop,
      'policy-drop-all' : 'Configure policy drop',
      'default' : 'Default policy drop behaviour',
   }

   @staticmethod
   def handler( mode, args ):
      firewallConfig.defaultPolicyDrop = 'enable'

   @staticmethod
   def noHandler( mode, args ):
      firewallConfig.defaultPolicyDrop = 'disable'

   defaultHandler = handler

FirewallConfigMode.addCommandClass( SegmentPolicyDefaultDrop )

# --------------------------------------------------------------------------
# commands in
# segment <segment-name>
#     policies
#      [ no ] from <segment-name> policy <policy-name>
# --------------------------------------------------------------------------
class SegmentPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'from SEGMENT_NAME policy POLICY_NAME'
   noOrDefaultSyntax = 'from SEGMENT_NAME [ policy POLICY_NAME ]'

   data = {
         'from': 'Associate a policy from Segment',
         'SEGMENT_NAME': vrfSegmentNameMatcher,
         'policy': 'Policy name',
         'POLICY_NAME': firewallPolicyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      fromSegmentName = args[ 'SEGMENT_NAME' ]
      segmentName = mode.parent_.segment_
      if segmentName == fromSegmentName and (
            not hwCapability.intraSegmentPolicySupported ):
         mode.addError( "Intra segment policy is not allowed." )
         return
      segment = mode.parent_.parent_.getSegment( segmentName )
      policyName = args[ 'POLICY_NAME' ]
      segment.policy[ fromSegmentName ] = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segment = mode.parent_.parent_.getSegment( mode.parent_.segment_ )
      fromSegmentName = args[ 'SEGMENT_NAME' ]
      del segment.policy[ fromSegmentName ]

VrfSegmentPolicyMode.addCommandClass( SegmentPolicyCmd )
SegmentPolicyMode.addCommandClass( SegmentPolicyCmd )

# --------------------------------------------------------------------------
# commands in 'config-router-seg-sec-vrf-segment-def' sub-mode
# segment <segment-name>
#    definition
#       match interface <intf>+
# --------------------------------------------------------------------------
intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= VirtualIntfMatcher( 'Port-Channel',
                                   PortChannelNum.min,
                                   PortChannelNum.max,
                                   helpdesc="Link Aggregation Group (LAG)",
                                   value=lambda mode, intf: EthLagIntf( intf, mode ),
                                   guard=portChannelSegmentSupportedGuard )

def intfInOtherClassMap( intf, thisCmapName ):
   for cmapName, cmapConfig in firewallConfig.classMap.iteritems():
      if cmapName == thisCmapName:
         continue
      if intf.name in cmapConfig.intfVlanData:
         return True
   return False

class MatchInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'match interface { INTF_LIST }'
   noOrDefaultSyntax = syntax
   data = {
         'match': 'Specify interfaces or subnets',
         'interface': segmentSecurityInterfaceNode,
         'INTF_LIST' : intfMatcher
   }

   @staticmethod
   def handler( mode, args ):
      intfs = args[ 'INTF_LIST' ]

      # Validate command
      for intf in intfs:
         if intfInOtherClassMap( intf, mode.cmapName ):
            mode.addError( 'Interface is part of another segment' )
            return

      oldIntfColl = firewallConfig.classMap[ mode.cmapName ].intfVlanData
      for intf in intfs:
         if intf.name in oldIntfColl:
            if not oldIntfColl[ intf.name ].vlanRange:
               continue
            # Remove any stale vlan config
            oldIntfColl[ intf.name ].vlanRange.clear()
         else:
            firewallConfig.classMap[ mode.cmapName ].intfVlanData.newMember(
               intf.name )
         if isinstance( mode.parent_.parent_, FirewallVrfConfigMode ):
            vrfName = mode.parent_.parent_.vrfName
            if vrfName == 'default':
               continue
            l3c = l3ConfigDir.intfConfig.get( intf.name )
            if not l3c or l3c.vrf != vrfName:
               mode.addWarning(
                     "Interface %s not in VRF %s" % ( intf.name, vrfName ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfs = args[ 'INTF_LIST' ]
      for intf in intfs:
         del firewallConfig.classMap[ mode.cmapName ].intfVlanData[ intf.name ]

ClassMapFirewallVrfConfigMode.addCommandClass( MatchInterfaceCmd )
ClassMapFirewallConfigMode.addCommandClass( MatchInterfaceCmd )

def vlanSetToRanges( vlanSet ):
   # vlanSet will be a set of ints
   vlanRanges = list()
   if not vlanSet:
      return vlanRanges

   vlanList = sorted( vlanSet )

   curRangeStart = None
   previous = None
   for vlan in vlanList:
      if curRangeStart is None:
         curRangeStart = vlan
      else:
         if vlan != previous + 1:
            # Reached range boundary
            vlanRanges.append( VlanRange( curRangeStart, previous ) )
            curRangeStart = vlan
      previous = vlan
   vlanRanges.append( VlanRange( curRangeStart, previous ) )
   return vlanRanges

def vlanRangesToSet( intfVlanData ):
   vlanSet = set()
   for vlanRange in intfVlanData.vlanRange:
      vlanSet.update( xrange( vlanRange.vlanBegin, vlanRange.vlanEnd + 1 ) )
   return vlanSet

def intfVlanInOtherClassMap( intf, vlanSet, thisCmapName ):
   for cmapName, cmapConfig in firewallConfig.classMap.iteritems():
      if cmapName == thisCmapName:
         continue
      if intf.name in cmapConfig.intfVlanData:
         intfVlanData = cmapConfig.intfVlanData[ intf.name ]
         if not intfVlanData.vlanRange:
            # NULL vlanRange means that all vlans are part of the config
            return True
         else:
            otherVlanSet = vlanRangesToSet( intfVlanData )
            if vlanSet.intersection( otherVlanSet ):
               # Overlapping vlan range
               return True
   return False

# --------------------------------------------------------------------------
# commands in 'config-router-seg-sec-segment' sub-mode
# segment <segment-name>
#    definition
#       match interface <intf> vlan [ ( add | remove ) ] <vlan_set>
# --------------------------------------------------------------------------
class MatchInterfaceVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'match interface INTF_NAME vlan [ (add | remove ) ] VLAN_SET'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Specify interfaces or subnets',
      'interface': segmentSecurityInterfaceNode,
      'INTF_NAME': intfMatcher,
      'vlan': 'Specify VLAN',
      'add': 'Add VLAN',
      'remove': 'Remove VLAN',
      'VLAN_SET': vlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF_NAME' ]
      updateSet = args[ 'VLAN_SET' ].ids

      if 'remove' not in args and \
         intfVlanInOtherClassMap( intf, updateSet, mode.cmapName ):
         mode.addError( 'Interface VLAN range is part of another segment' )
         return

      classMap = firewallConfig.classMap[ mode.cmapName ]
      intfVlanDataConfig = None
      curSet = set()
      if intf.name in classMap.intfVlanData:
         intfVlanDataConfig = classMap.intfVlanData[ intf.name ]
         curSet = vlanRangesToSet( intfVlanDataConfig )

      newSet = set()
      if 'add' in args:
         newSet = curSet.union( updateSet )
      elif 'remove' in args:
         newSet = curSet.difference( updateSet )
      else:
         newSet = updateSet

      if newSet:
         newVlanRanges = vlanSetToRanges( newSet )
         if not intfVlanDataConfig:
            intfVlanDataConfig = classMap.newIntfVlanData( intf.name )
         else:
            for vlanRange in intfVlanDataConfig.vlanRange:
               if vlanRange not in newVlanRanges:
                  # Remove stale range entry
                  intfVlanDataConfig.vlanRange.remove( vlanRange )

         for vlanRange in newVlanRanges:
            # New range entry, add
            intfVlanDataConfig.vlanRange.add( vlanRange )
      # If all vlans have been removed, remove the interface itself from config
      # Only remove the interface, if the original vlan set was not empty.
      # Running vlan remove on an interface originally configured without vlans
      # shouldn't remove the interface from config
      elif intfVlanDataConfig and curSet:
         del classMap.intfVlanData[ intf.name ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      classMap = firewallConfig.classMap[ mode.cmapName ]
      intf = args[ 'INTF_NAME' ]
      if intf.name in classMap.intfVlanData:
         del classMap.intfVlanData[ intf.name ]

ClassMapFirewallConfigMode.addCommandClass( MatchInterfaceVlanCmd )

# --------------------------------------------------------------------------
# commands in 'config-router-seg-sec-vrf-segment-def' sub-mode
# segment <segment-name>
#    definition
#       match prefix-ipv4 <match-list prefix-ipv4 name >
# --------------------------------------------------------------------------
class MatchIpv4PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'match prefix-ipv4 IPV4_PREFIX_LIST'
   noOrDefaultSyntax = syntax

   data = {
         'match': matchNode,
         'prefix-ipv4': 'Match Ipv4 match list',
         'IPV4_PREFIX_LIST': ipv4PrefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      listName = args[ 'IPV4_PREFIX_LIST' ]
      fwConfig = firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      fwConfig.segmentDir.segment[ mode.parent_.segment_ ].ipv4PrefixList = listName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      listName = args[ 'IPV4_PREFIX_LIST' ]
      fwConfig = firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      if segment.ipv4PrefixList == listName:
         segment.ipv4PrefixList = ''

ClassMapFirewallVrfConfigMode.addCommandClass( MatchIpv4PrefixCmd )

# --------------------------------------------------------------------------
# commands in 'config-router-seg-sec-vrf-segment-def' sub-mode
# segment <segment-name>
#    definition
#       match prefix-ipv6 <match-list prefix-ipv6 name >
# --------------------------------------------------------------------------
class MatchIpv6PrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'match prefix-ipv6 IPV6_PREFIX_LIST'
   noOrDefaultSyntax = syntax

   data = {
         'match': matchNode,
         'prefix-ipv6': segmentSecurityIpv6PrefixNode,
         'IPV6_PREFIX_LIST': ipv6PrefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      listName = args[ 'IPV6_PREFIX_LIST' ]
      fwConfig = firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      fwConfig.segmentDir.segment[ mode.parent_.segment_ ].ipv6PrefixList = listName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      listName = args[ 'IPV6_PREFIX_LIST' ]
      fwConfig = firewallConfig.vrf[ mode.parent_.parent_.vrfName ]
      segment = fwConfig.segmentDir.segment[ mode.parent_.segment_ ]
      if segment.ipv6PrefixList == listName:
         segment.ipv6PrefixList = ''

ClassMapFirewallVrfConfigMode.addCommandClass( MatchIpv6PrefixCmd )

# --------------------------------------------------------------------------
# 'show segment-security [ VRF ] [ segment <segment-name> ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurity( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security [ VRF ] [ segment SEGMENT_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'VRF': vrfExprFactoryForShow,
         'segment': 'Show segment security segment information',
         'SEGMENT_NAME': segmentNameMatcher,
   }
   cliModel = FirewallInfo

   @staticmethod
   def handler( mode, args ):
      firewallModel = FirewallInfo()
      vrfName = args.get( 'VRF' )
      segmentName = args.get( 'SEGMENT_NAME' )
      vrfModel = None
      if vrfName:
         vrfModel = firewallModel.Vrf()
         vrfModel.ipv6PrefixSupported = \
            matchListCapability.ipv6PrefixSupported
         vrfModel.enabled = firewallConfig.enabled
         # Check if the specified VRF exists. If not then bailout.
         firewallModel.vrfs[ vrfName ] = vrfModel
         if vrfName not in config.vrf:
            vrfModel.vrfExist = False
            return firewallModel
         else:
            vrfModel.vrfExist = True
      else:
         if not hwCapability.intfVlanSupported:
            vrfModel = firewallModel.Vrf()
            vrfModel.ipv6PrefixSupported = \
               matchListCapability.ipv6PrefixSupported
            vrfModel.enabled = firewallConfig.enabled
            firewallModel.vrfs[ 'default' ] = vrfModel
            vrfModel.vrfExist = True
            vrfName = 'default'
            if vrfName not in config.vrf:
               vrfModel.vrfExist = False
               return firewallModel

      if not firewallConfig.enabled:
         # Output is not displayed if firewall is not enabled.
         return firewallModel

      def populateSegmentModel( segmentName ):
         if vrfModel:
            segment = config.vrf[ vrfName ].segmentDir.segment[ segmentName ]
         else:
            if not config.l2Seg:
               return
            segment = config.l2Seg.segment[ segmentName ]
         segmentModel = Segment()
         segmentModel.classMap = segment.className
         for intfVlanRanges in segment.intfVlanRangeSet.values():
            if intfVlanRanges.vlanRange:
               intfVlanModel = InterfaceVlanRange()
               intfVlanModel.interface = intfVlanRanges.intfId
               for vlanRange in intfVlanRanges.vlanRange:
                  vlanRangeModel = VlanRangeModel()
                  vlanRangeModel.startVlanId = vlanRange.vlanBegin
                  vlanRangeModel.endVlanId = vlanRange.vlanEnd
                  intfVlanModel.vlanRanges.append( vlanRangeModel )
               segmentModel.interfaceVlans.append( intfVlanModel )
            else:
               segmentModel.interfaces.append( intfVlanRanges.intfId )

         if vrfModel:
            if segmentName in firewallConfig.vrf[ vrfName ].segmentDir.segment:
               segmentModel.ipv4PrefixListName = \
                  firewallConfig.vrf[ vrfName ].segmentDir.segment[
                        segmentName ].ipv4PrefixList
               segmentModel.ipv6PrefixListName = \
                  firewallConfig.vrf[ vrfName ].segmentDir.segment[
                        segmentName ].ipv6PrefixList
            else:
               segmentModel.ipv4PrefixListName = ''
               segmentModel.ipv6PrefixListName = ''
         else:
            segmentModel.ipv4PrefixListName = ''
            segmentModel.ipv6PrefixListName = ''
         for fromSegment in segment.fromSegment:
            fromSegmentName = segment.fromSegment[ fromSegment ].segment.name
            policyName = segment.fromSegment[ fromSegment ].policy.name
            segmentModel.fromSegments[ fromSegmentName ] = policyName
         if vrfModel:
            vrfModel.segments[ segmentName ] = segmentModel
         else:
            firewallModel.segments[ segmentName ] = segmentModel

      if vrfModel:
         if vrfName in config.vrf:
            # populate vrfModel
            if not segmentName:
               for segmentName in config.vrf[ vrfName ].segmentDir.segment:
                  populateSegmentModel( segmentName )
            elif segmentName in config.vrf[ vrfName ].segmentDir.segment:
               # populate vrfModel only specified segment
               populateSegmentModel( segmentName )
      else:
         if not segmentName and config.l2Seg:
            for segmentName in config.l2Seg.segment:
               populateSegmentModel( segmentName )
         else:
            populateSegmentModel( segmentName )
      return firewallModel

BasicCli.addShowCommandClass( ShowSegmentSecurity )

# --------------------------------------------------------------------------
# 'show segment-security policy [ policy-name ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurityPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security policy [ POLICY_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'policy': 'Show segment security policy',
         'POLICY_NAME': firewallPolicyNameMatcher,
   }
   cliModel = FwPolicyInfo

   @staticmethod
   def handler( mode, args ):
      policyName = args.get( 'POLICY_NAME' )
      policyInfo = FwPolicyInfo()

      def populatePolicyModel( policyName ):
         policyObj = firewallConfig.policy[ policyName ]
         policyModel = policyInfo.Policy()
         policyModel.readonly = policyObj.readonly
         for seqnum in policyObj.rule.keys():
            policyDef = policyModel.PolicyDef()
            policyDef.serviceName = policyObj.rule[ seqnum ].serviceName
            policyDef.action = policyObj.rule[ seqnum ].action
            policyDef.log = policyObj.rule[ seqnum ].log
            policyModel.policyDefs[ seqnum ] = policyDef
         policyInfo.policies[ policyName ] = policyModel

      if not policyName:
         # if no policyName is specified
         for policy in firewallConfig.policy:
            populatePolicyModel( policy )
      elif policyName in firewallConfig.policy:
         # if policyName is specified
         populatePolicyModel( policyName )
      return policyInfo

BasicCli.addShowCommandClass( ShowSegmentSecurityPolicy )

# --------------------------------------------------------------------------
# 'show segment-security application [ application-name ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurityApplication( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security application [ APPLICATION_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'application': 'Show segment security application',
         'APPLICATION_NAME': applicationNameMatcher,
   }
   cliModel = FwServiceInfo

   @staticmethod
   def handler( mode, args ):
      serviceName = args.get( 'APPLICATION_NAME' )
      serviceInfo = FwServiceInfo()

      def populateServiceModel( serviceName ):
         service = config.services[ serviceName ]
         serviceModel = serviceInfo.Service()

         for protocol in service.protocol:
            serviceDef = serviceModel.ServiceDef()
            if protocol == FirewallProtocol.all:
               serviceDef.allProtocols = True
            else:
               serviceDef.allProtocols = False
               serviceDef.ipProtocol = protocol

            portList = []
            for start, end in service.dstPort.items():
               port = serviceDef.PortRange()
               port.start = start
               port.end = end
               portList.append( port )
            serviceDef.dstPort = portList
            portList = []
            for start, end in service.srcPort.items():
               port = serviceDef.PortRange()
               port.start = start
               port.end = end
               portList.append( port )
            serviceDef.srcPort = portList
            serviceModel.serviceDef.append( serviceDef )
         serviceModel.srcPrefix = service.srcIpPrefix.keys()
         serviceModel.dstPrefix = service.dstIpPrefix.keys()
         serviceInfo.services[ serviceName ] = serviceModel

      if not serviceName:
         for serviceName in config.services:
            populateServiceModel( serviceName )
      else:
         if serviceName in config.services:
            populateServiceModel( serviceName )
      return serviceInfo

BasicCli.addShowCommandClass( ShowSegmentSecurityApplication )

# Hooks for platform to register its function to handle the cli command
showFirewallSessionHook = CliHook()
clearFirewallSessionHook = CliHook()
clearFirewallCountersHook = CliHook()

# --------------------------------------------------------------------------
# clear segment-security sessions [ VRF ]
# --------------------------------------------------------------------------
class ClearSegmentSecuritySessions( CliCommand.CliCommandClass ):
   syntax = 'clear segment-security sessions [ VRF ]'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'segment-security': segmentSecurityNodeForClear,
         'sessions': 'Segment security sessions',
         'VRF': VrfExprFactory(
                       helpdesc='Clear sessions belonging to a particular VRF',
                       guard=clearVrfSessionsSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      for hook in clearFirewallSessionHook.extensions():
         hook( mode, vrfName )

BasicCli.EnableMode.addCommandClass( ClearSegmentSecuritySessions )


# --------------------------------------------------------------------------
# show segment-security sessions [ VRF ]
# --------------------------------------------------------------------------
class ShowSegmentSecuritySessions( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security sessions [ VRF ]'
   data = {
         'segment-security': segmentSecurityNode,
         'sessions': 'Segment security sessions',
         'VRF': vrfExprFactoryForShow,
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      if not vrfName and not hwCapability.intfVlanSupported:
         vrfName = DEFAULT_VRF
      for hook in showFirewallSessionHook.extensions():
         hook( mode, vrfName )

BasicCli.addShowCommandClass( ShowSegmentSecuritySessions )

# --------------------------------------------------------------------------
# 'show segment-security counters [ VRF ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurityCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security counters [ VRF ]'
   data = {
         'segment-security': segmentSecurityNode,
         'counters': segmentSecurityCountersNode,
         'VRF': vrfExprFactoryForShow,
   }
   cliModel = FwCounterInfo

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      fwCounterModel = FwCounterInfo()
      if not vrfName and not hwCapability.intfVlanSupported:
         vrfName = DEFAULT_VRF

      if vrfName:
         if vrfName not in firewallCounters.vrf:
            return fwCounterModel
         fwCounter = firewallCounters.vrf[ vrfName ].countersDir
         fwCounterModel.vrfExist = True
         fwCounterModel.vrf = vrfName
      else:
         fwCounter = firewallCounters.l2CountersDir
         fwCounterModel.vrfExist = False

      if fwCounter.globalCounters:
         fwCounterModel.flowsCreated = fwCounter.globalCounters.flowCreated
         fwCounterModel.invalidPackets = fwCounter.globalCounters.invalidPkts
         fwCounterModel.bypassPackets = fwCounter.globalCounters.firewallBypass
      else:
         fwCounterModel.flowsCreated = 0
         fwCounterModel.invalidPackets = 0
         fwCounterModel.bypassPackets = 0

      for policyName in fwCounter.policy:
         policy = fwCounter.policy[ policyName ]
         policyCounters = fwCounterModel.PolicyCounter()
         policyCounters.hits = policy.hitCount
         policyCounters.drops = policy.dropCount
         policyCounters.defaultDrops = policy.defaultDrop
         fwCounterModel.policies[ policyName ] = policyCounters

      for dstSeg in fwCounter.segment:
         dstSegModel = fwCounterModel.DestSegment()
         segCounter = fwCounter.segment[ dstSeg ]
         for srcSeg in segCounter.segmentCounters:
            srcSegModel = dstSegModel.SourceSegment()
            srcSegModel.policyName = segCounter.segmentCounters[ srcSeg ].policy
            srcSegModel.hits = segCounter.segmentCounters[ srcSeg ].hitCount
            dstSegModel.srcSegments[ srcSeg ] = srcSegModel
         fwCounterModel.dstSegments[ dstSeg ] = dstSegModel
      return fwCounterModel

BasicCli.addShowCommandClass( ShowSegmentSecurityCounters )


# -------------------------------------------------------------------------------
# 'show segment-security status [ VRF ] [ segment <segment-name> ]'
# -------------------------------------------------------------------------------
class ShowSegmentSecurityStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security status [ VRF ] [ segment SEGMENT_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'status': 'Show status of segment security',
         'VRF': vrfExprFactoryForShow,
         'segment': 'Show status of segment',
         'SEGMENT_NAME': segmentNameMatcher,
   }
   cliModel = FwHwStatusInfo

   @staticmethod
   def handler( mode, args ):
      firewallModel = FwHwStatusInfo()
      vrfName = args.get( 'VRF' )
      segmentName = args.get( 'SEGMENT_NAME' )
      firewallModel.ipv6PrefixSupported = \
            matchListCapability.ipv6PrefixSupported

      def populateVrfModule( vrfName, segmentName ):
         if vrfName not in status.status:
            return
         vrfModel = firewallModel.Vrf()
         vrfSegmentStatus = status.status[ vrfName ]
         vrfConfigDir = firewallConfig.vrf[ vrfName ].segmentDir

         def populateDstSegmentModel( dstSegmentName ):
            if dstSegmentName in vrfSegmentStatus.segmentDir.segment:
               dstSegmentStatus = vrfSegmentStatus.segmentDir.segment[
                     dstSegmentName ]
               dstSegmentModel = vrfModel.DestSegment()
               # Populate matchlist names for current segment
               if dstSegmentName in vrfConfigDir.segment:
                  v4List = vrfConfigDir.segment[ dstSegmentName ].ipv4PrefixList
                  v6List = vrfConfigDir.segment[ dstSegmentName ].ipv6PrefixList
               dstSegmentModel.ipv4PrefixListName = v4List
               dstSegmentModel.ipv6PrefixListName = v6List
               # Populate status of source segments
               for srcSegmentName in dstSegmentStatus.fromSegment:
                  srcSegmentModel = dstSegmentModel.SourceSegment()
                  srcSegmentModel.status = dstSegmentStatus.fromSegment[
                        srcSegmentName ]
                  dstSegmentModel.srcSegments[ srcSegmentName ] = srcSegmentModel
               if dstSegmentModel.srcSegments:
                  vrfModel.dstSegments[ dstSegmentName ] = dstSegmentModel

         if segmentName:
            populateDstSegmentModel( segmentName )
         else:
            for dstSegmentName in vrfSegmentStatus.segmentDir.segment:
               populateDstSegmentModel( dstSegmentName )

         firewallModel.vrfs[ vrfName ] = vrfModel

      if vrfName:
         populateVrfModule( vrfName, segmentName )
      else:
         for vrfName in status.status:
            populateVrfModule( vrfName, None )

      return firewallModel

BasicCli.addShowCommandClass( ShowSegmentSecurityStatus )

# --------------------------------------------------------------------------
# clear segment-security counters [ VRF ]
# --------------------------------------------------------------------------
class ClearSegmentSecurityCounters( CliCommand.CliCommandClass ):
   syntax = 'clear segment-security counters [ VRF ]'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'segment-security': segmentSecurityNodeForClear,
         'counters': segmentSecurityCountersNode,
         'VRF': vrfExprFactoryForClear,
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      if vrfName is None and not hwCapability.intfVlanSupported:
         vrfName = DEFAULT_VRF
      for hook in clearFirewallCountersHook.extensions():
         hook( mode, vrfName )

BasicCli.EnableMode.addCommandClass( ClearSegmentSecurityCounters )

def Plugin( entityManager ):
   global entMan, firewallConfig, config, hwCapability, l3ConfigDir, ipStatus, \
      firewallCounters, matchListConfig, matchListCapability, status

   entMan = entityManager
   firewallConfig = ConfigMount.mount( entityManager, 'firewall/config/cli',
                                       'Firewall::Config', 'w' )
   config = LazyMount.mount( entityManager, 'firewall/hw/config',
                             'Firewall::HwConfig', 'r' )
   status = LazyMount.mount( entityManager, 'firewall/hw/status',
                             'Firewall::HwStatus', 'r' )
   firewallCounters = LazyMount.mount( entityManager, 'firewall/counters',
                                       'Firewall::Counters', 'r' )
   hwCapability = LazyMount.mount( entityManager, 'firewall/hw/capability',
                                   'Firewall::HwCapability', 'r' )
   matchListCapability = LazyMount.mount( entityManager,
                                          'matchlist/hw/capability',
                                          'MatchList::HwCapability', 'r' )
   l3ConfigDir = LazyMount.mount(
      entityManager, 'l3/intf/config', 'L3::Intf::ConfigDir', 'r' )
   ipStatus = LazyMount.mount(
      entityManager, 'ip/status', 'Ip::Status', 'r' )
   matchListConfig = LazyMount.mount( entityManager, "matchlist/config/cli",
                                      "MatchList::Config", "r" )
