# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import Tac
import TableOutput
import CliModel

FileReplicationStatus = Tac.Type( "FileReplication::FileStatus::Status" )

class FileStatus( CliModel.Model ):
   status = CliModel.Enum( values=FileReplicationStatus.attributes,
                           help="Status of file synchronization" )
   lastSyncTime = CliModel.Float( help="Last synchronized time", optional=True )

class RequesterStatus( CliModel.Model ):
   files = CliModel.Dict( keyType=str, valueType=FileStatus,
                          help=( "A mapping of file names to the replication "
                                 "status of each file" ) )

   def render( self ):
      self._render()

   def _render( self, indentLevel=0 ):
      indent = '  ' * indentLevel

      def pluralize( count, word ):
         return word if count == 1 else word + 's'

      # Print the summary
      summary = ""
      countByStatus = {}
      for state in FileReplicationStatus.attributes:
         countByStatus[ state ] = 0
      for f in self.files:
         countByStatus[ self.files[ f ].status ] += 1
      for state in FileReplicationStatus.attributes:
         count = countByStatus[ state ]
         summary += "%d %s %s, " % ( count, pluralize( count, 'file' ), state )
      summary += "%d files total." % len( self.files )
      print '%s%s' % ( indent, summary )
      print

      # Print the table
      headings = ( "File", "Status", "Last Synchronized" )
      table = TableOutput.createTable( headings, indent=indentLevel * 2 )
      f = TableOutput.Format( justify="left" )
      f.noPadLeftIs( True )
      table.formatColumns( *( [ f ] * len( headings ) ) )
      for sf in sorted( self.files ):
         syncFileObj = self.files[ sf ]
         if syncFileObj.lastSyncTime:
            lastSyncTimeStr = Ark.timestampToStr( syncFileObj.lastSyncTime )
         else:
            lastSyncTimeStr = "Never"
         table.newRow( sf, syncFileObj.status.title(), lastSyncTimeStr )
      print table.output()

class Status( CliModel.Model ):
   requesters = CliModel.Dict( keyType=str, valueType=RequesterStatus,
                               help=( "A mapping from replicated file set names to "
                                      "the the replication status of "
                                      "the file set" ) )

   def render( self ):
      self._render()

   def _render( self, indentLevel=0 ):
      indent = '  ' * indentLevel
      for reqName in sorted( self.requesters.keys() ):
         print '%sFile set %s' % ( indent, reqName )
         # pylint: disable-msg=W0212
         self.requesters[ reqName ]._render( indentLevel=indentLevel + 1 )
