#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import print_function

from CliModel import Enum
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Bool
from FileCliUtil import sizeFmt
from TableOutput import createTable, Format

class FileSystem( Model ):
   """Model representing a file system and it's related information like total size
   free space, permissions, etc. This model is used to build a list that is
   returned on execution of command 'show file systems'
   """
   currentFs = Bool( default=False, help="When set it signifies this file system is "
                     "the current one" )
   size = Int( help="Size of the file system in KB" )
   free = Int( help="Free space in the file system in KB" )
   fsType = Enum( values=( "extension", "flash", "network", "peer", "session",
                           "simple", "ssl", "system", "checkpoint", "alert-base",
                           "ssh" ),
                  help="file system type" )
   linuxFs = Enum( values=( "vfat", "ext4" ), optional=True,
                   help="Actual filesystem type used on the physical medium. "
                        "It will be omitted if there's no physical medium "
                        "attached to the filesystem" )
   permission = Enum( values=( "r", "w", "rw" ),
                      help="File permissions of read-only, write-only and "
                      "read-write" )
   prefix = Enum( values=( "certificate:", "extension:", "file:", "flash:", "ftp:", 
                           "http:", "https:", "scp:", "session:", "sftp:", 
                           "simple:", "sslkey:", "supervisor-peer:", "system:",
                           "tftp:", "usb1:", "usb2:", "drive:", "platform:", 
                           "checkpoint:", "crash:", "alert-base:",
                           "ssh-ca-key:", "ssh-host-cert:", "ssh-revoke-list:" ),
                    help="file system prefix" )

class FileSystems( Model ):
   """Model representing a list of file systems each of which is represented by a
   model FileSystem. This model will contain all the file systems with their 
   information represented in individual file system model
   """
   fileSystems = List( valueType=FileSystem, help="List of file systems" )

   def render( self ):
      def makeTable( header ):
         t = createTable( header )
         f = Format( justify='left' )
         f.noPadLeftIs( True )
         f.padLimitIs( True )
         t.formatColumns( *[ f ] * len( header ) )
         return t

      table1 = makeTable( ( 'Prefixes', ' ', 'Size(KB)', 'Free(KB)' ) )
      table2 = makeTable( ( 'Prefixes', 'Flags', 'Type', 'Fs' ) )

      for fs in self.fileSystems:
         if fs.currentFs:
            star = '*'
         else:
            star = ' '
         # Convert size back to bytes from KB returned in the model
         ( size, free ) = ( fs.size * 1024, fs.free * 1024 )
         if size == 0:
            ( size, free ) = ( '-', '-' )
            ( fmtsize, fmtfree ) = ( '', '' )
         else:
            fmtsize = '(' + sizeFmt( size ) + ')'
            fmtfree = '(' + sizeFmt( free ) + ')'
            size /= 1024
            free /= 1024
         finalSize = '%12s %10s' % ( size, fmtsize )
         finalFree = '%12s %10s' % ( free, fmtfree )
         table1.newRow( fs.prefix, star, finalSize, finalFree )
         table2.newRow( fs.prefix, fs.permission, fs.fsType, fs.linuxFs or '-' )

      print( "File Systems:" )
      print()
      print( table1.output() )
      print( table2.output(), end='' )

class FileInformation( Model ):
   """A model that represents information about a given file path. This is returned
   on execution of command 'show file information <path>'
   """
   path = Str( help="Path name for which this information is generated" )
   isDir = Bool( default=False, help="When this flag is set to True it signifies"
                 "that the given path is a directory. All other fields are empty"
                 " and not relevant when this flag is set" )
   isExecutable = Bool( help="A flag when set to True"
                        " signifies that the given file path is executable",
                        optional=True )
   isEmpty = Bool( help="A flag when set to True signifies that the given"
                   " path is an empty file", optional=True )
   fileType = Enum( values=( "elf", "swi", "swix", "zip", "script",
                             "pythonScript" ),
                    help="File types are:\n"
                    " elf: ELF binary\n"
                    " swi: Arista software image with version in version field\n"
                    " swix: Arista EOS extension\n"
                    " zip: Zip archive data\n"
                    " script: A script file\n"
                    " pythonScript: A Python script file",
                    optional=True )
   fileName = Str( help="File name if path is a local file else empty",
                   optional=True )
   version = Str( help="When fileType is swi this field is the Arista EOS version",
                  optional=True )
   
   def render( self ):
      if self.isDir:
         filetype = "directory"
      else:
         filetype = []
         if self.isExecutable:
            filetype.append( "executable" )
         if self.isEmpty:
            filetype.append( "empty file" )
         elif self.fileName:
            if self.fileType == "elf":
               filetype.append( "ELF binary" )
            elif self.fileType == "swi":
               filetype.append( 'Arista software image, version' )
               filetype.append( self.version )
            elif self.fileType == "swix":
               filetype.append( 'Arista EOS extension' )
            elif self.fileType == "zip":
               filetype.append( 'Zip archive data' )
            elif self.fileType == "pythonScript":
               filetype.append( "Python script" )
            elif self.fileType == "script":
               filetype.append( 'script' )
            else:
               filetype.append( 'file' )
         else:
            filetype.append( 'file' )
         filetype = ' '.join( filetype )
      print( self.path + ":" )
      print( "  type is %s" % filetype )

