# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Fabric interface type.  In particular, it provides
# the FabricIntf class.
#-------------------------------------------------------------------------------
import BasicCli
import CliMode.Fabric
import CliParser
import CliPlugin.IntfCli as IntfCli
import CliPlugin.VirtualIntfRule as VirtualIntfRule
import CliCommand
import LazyMount
import Tac

# The following is needed to introduce a dependency between the cli and the main rpm.
# pkgdeps: rpm FabricIntf-lib

fabricIntfConfigSysdbPath = "interface/config*/fabric/all"
fabricIntfStatusSysdbPath = "interface/status*/fabric/all"

fabricIntfConfigDir = None
fabricIntfStatusDir = None
cliConfig = None
systemName = None
entityManager = None

# guard for tokenFabric
def isFabricSystem( mode, token ):
   if fabricIntfStatusDir.enabled:
      return None
   return CliParser.guardNotThisPlatform
#-------------------------------------------------------------------------------
# A subclass of the base IntfCli.Intf class for Fabric interfaces.
#-------------------------------------------------------------------------------
class FabricIntf( IntfCli.VirtualIntf ):

   #----------------------------------------------------------------------------
   # Creates a new FabricIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      IntfCli.VirtualIntf.__init__( self, name, mode )
      self.intfConfigDir = fabricIntfConfigDir
      self.intfStatuses = fabricIntfStatusDir.intfStatus
      self.intfStatus = None

   #----------------------------------------------------------------------------
   # The rule for matching Fabric interface names.  When this pattern matches, it
   # returns an instance of the FabricIntf class.
   #----------------------------------------------------------------------------
   matcher = VirtualIntfRule.IntfMatcher()

   #----------------------------------------------------------------------------
   # Creates the Interface::FabricIntfConfig object for this interface if it does
   # not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      pass

   #----------------------------------------------------------------------------
   # Determines whether the Interface::FabricIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      self.intfStatus = self.intfStatuses.get( self.name )
      return self.intfStatus is not None

   #----------------------------------------------------------------------------
   # Destroys the Interface::FabricIntfConfig object for this interface if it
   # already exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      pass

   #----------------------------------------------------------------------------
   # Returns the FabricIntfConfig object for this interface.
   #----------------------------------------------------------------------------
   def config( self ):
      return self.intfConfigDir.get( self.name )

   #----------------------------------------------------------------------------
   # Returns the FabricIntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      if not self.intfStatus:
         self.intfStatus = self.intfStatuses.get( self.name )
      return self.intfStatus

   #----------------------------------------------------------------------------
   # Returns the FabricIntfCounters object for this interface.
   #----------------------------------------------------------------------------
   def counter( self ):
      return IntfCli.Intf.counter( self )

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def showPhysical( self, mode, intfStatusModel ):
      return 

   def getIntfCounterDir( self ):
      return
   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addrStr( self ):
      return None

   def addr( self ):
      return None

   def burnedInAddr( self ):
      return None
   
   def hardware( self ):
      return "fabric"
   
   def bandwidth( self ):
      return 0

   def countersSupported( self ):
      return False
   
                                                       
   #----------------------------------------------------------------------------
   # Returns empty list for Fabric interfaces
   #----------------------------------------------------------------------------
   def getAllPhysical( mode ):
      return []


   def setDefault( self ):
      IntfCli.VirtualIntf.setDefault( self )

   getAllPhysical = staticmethod( getAllPhysical )

#--------------------------------------------------------------------------------
# Define FabricIntf Modelet to add the autostate cli command
#--------------------------------------------------------------------------------
class FabricIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      # pylint: disable-msg=W0231
      return isinstance( mode.intf, FabricIntf )

#--------------------------------------------------------------------------------
# [ no | default ] autostate
#--------------------------------------------------------------------------------
class AutostateCmd( CliCommand.CliCommandClass ):
   syntax = 'autostate'
   noOrDefaultSyntax = syntax
   data = {
      'autostate' : 'Automatically manage the interface link state',
   }


   @staticmethod
   def handler( mode, args ):
      mode.intf.config().autoState = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.config().autoState = False

   defaultHandler = handler

FabricIntfModelet.addCommandClass( AutostateCmd )
IntfCli.IntfConfigMode.addModelet( FabricIntfModelet )

#--------------------------------------------------------------------------------
# [ no | default ] interface Fabric ...
#--------------------------------------------------------------------------------
fabricIntfKw = CliCommand.guardedKeyword( 'Fabric',
                                          'Fabric interface',
                                          isFabricSystem )

class InterfaceFabricCmd( CliCommand.CliCommandClass ):
   syntax = 'interface Fabric ...'
   noOrDefaultSyntax = syntax
   data = {
      'interface' : 'Interface Configuration',
      'Fabric' : fabricIntfKw,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CliMode.Fabric.FabricMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for name in fabricIntfStatusDir.intfStatus:
         intf = FabricIntf( name, mode )
         intf.setDefault()

BasicCli.GlobalConfigMode.addCommandClass( InterfaceFabricCmd )

__countersMounted = False
mountCounters = None

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( em ):
   global fabricIntfConfigDir, fabricIntfStatusDir, cliConfig
   global entityManager

   entityManager = em
   mg = entityManager.mountGroup()
   mg.close( callback=None )

   fabricIntfConfigDir = LazyMount.mount( entityManager, 
         fabricIntfConfigSysdbPath,
         "Interface::FabricIntfConfigDir", "r" )
   fabricIntfStatusDir = LazyMount.mount( entityManager, 
         fabricIntfStatusSysdbPath,
         "Interface::FabricIntfStatusDir", "r" )
   global systemName
   cliConfig = None
   systemName = entityManager.sysname()
