#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentDirectory
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Switch
import Cell
import ConfigMount
import Tac

fabricConfig = None

def isModular():
   return Cell.cellType() != "fixed"

def isStrataRunning( sysname ):
   return AgentDirectory.agent( sysname, 'StrataCentral' )

def insufficientBandwidthActionGuard( mode, token ):
   if ( isStrataRunning( mode.sysname ) and isModular() ):
      return None
   else:
      return CliParser.guardNotThisPlatform

matcherAction = CliMatcher.KeywordMatcher( 'action',
      helpdesc='Configure insufficient-bandwidth fabric action' )
matcherFabric = CliMatcher.KeywordMatcher( 'fabric',
      helpdesc='Configure fabric parameters' )
matcherInsufficientBandwidth = CliMatcher.KeywordMatcher( 'insufficient-bandwidth',
      helpdesc='Configure insufficient-bandwidth fabric parameter' )
nodeInsufficientBandwidth = CliCommand.guardedKeyword( 'insufficient-bandwidth',
      helpdesc='Configure insufficient-bandwidth fabric parameter',
      guard=insufficientBandwidthActionGuard )

#--------------------------------------------------------------------------------
# [ no | default ] switch fabric insufficient-bandwidth action linecard-shutdown
#                                                                threshold THRESHOLD
#--------------------------------------------------------------------------------
class InsufficientBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switch fabric insufficient-bandwidth action linecard-shutdown '
                                                              'threshold THRESHOLD' )
   noOrDefaultSyntax = 'switch fabric insufficient-bandwidth action ...'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'fabric' : matcherFabric,
      'insufficient-bandwidth' : nodeInsufficientBandwidth,
      'action' : matcherAction,
      'linecard-shutdown' : ( 'Power off linecard as a result of fabric '
                              'bandwidth loss' ),
      'threshold' : ( 'Set the insufficient-bandwidth threshold corresponding to '
                      'the linecard-shutdown action' ),
      'THRESHOLD' : CliMatcher.IntegerMatcher( 5, 95,
         helpdesc='% of bandwidth loss allowed' ),
   }

   @staticmethod
   def handler( mode, args ):
      fabricConfig.linkFailureActionIgnore = False
      fabricConfig.linkFailureThresholdPercent = args.get( 'THRESHOLD', 80 )

   @staticmethod
   def noHandler( mode, args ):
      fabricConfig.linkFailureActionIgnore = True

   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( InsufficientBandwidthCmd )

#--------------------------------------------------------------------------------
# switch fabric insufficient-bandwidth action ignore
#--------------------------------------------------------------------------------
class InsufficientBandwidthIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'switch fabric insufficient-bandwidth action ignore'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'fabric' : matcherFabric,
      'insufficient-bandwidth' : nodeInsufficientBandwidth,
      'action' : matcherAction,
      'ignore' : 'Take no action as a result of fabric bandwidth loss',
   }

   @staticmethod
   def handler( mode, args ):
      fabricConfig.linkFailureActionIgnore = True

BasicCli.GlobalConfigMode.addCommandClass( InsufficientBandwidthIgnoreCmd )

def Plugin( em ):
   global fabricConfig
   fabricConfig = ConfigMount.mount( em, "hardware/fabric/fabricGlobalConfig/", 
                                     'Fabric::Config', 'rw' )
