# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliParser
import ConfigMgmtMode
import BasicCli
import ConfigMount
import CliPlugin.HttpService as HttpService
import CliPlugin.VrfCli as VrfCli
from CliMode.ExternalHttpServices import ExternalHttpServicesVrfConfigModeBase
import ExternalServicesConstants

capiConfig = None

class ExternalHttpServicesConfigMode( ConfigMgmtMode.ConfigMgmtMode ):

   name = "External HTTP services configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session,
                                              "api-external-services" )
      self.config_ = capiConfig

   def enterCmd( self ):
      return "management api external-services"

#--------------------------------------------------------------------
# The "[no|default] management api external-services" mode command.
#--------------------------------------------------------------------
class EnterExternalHttpServicesMode( CliCommand.CliCommandClass ):
   syntax = """management api external-services"""
   noOrDefaultSyntax = syntax

   data = { "management": ConfigMgmtMode.managementKwMatcher,
            "api": ConfigMgmtMode.apiKwMatcher,
            "external-services": ( "Configure the external(customer specific) "
                                   "web services on HTTP server" )}

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ExternalHttpServicesConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      shutdownExternalServices( mode )
      removeExternalServiceVrfs( mode, {} )

BasicCli.GlobalConfigMode.addCommandClass( EnterExternalHttpServicesMode )

#-------------------------------------------------------------------
# The "[no|default] shutdown" command in "management api external-services" mode
#-------------------------------------------------------------------
def shutdownExternalServices( mode ):
   capiConfig.service[ ExternalServicesConstants.SERVICE_NAME ].enabled = False

class ConfigureExternalHttpServices( CliCommand.CliCommandClass ):
   syntax = """shutdown"""
   noOrDefaultSyntax = syntax

   data = { "shutdown": "Disable external HTTP services access" }

   @staticmethod
   def handler( mode, args ):
      shutdownExternalServices( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      shutdownExternalServices( mode ) # by default we are off

   @staticmethod
   def noHandler( mode, args ): 
      if ( not capiConfig.httpsConfig.enabled and 
           not capiConfig.httpConfig.enabled and 
           not capiConfig.localHttpConfig.enabled and 
           not capiConfig.unixConfig.enabled ):
         # No protocol is enabled
         mode.addWarning( "No protocols are enabled" )
      capiConfig.service[ ExternalServicesConstants.SERVICE_NAME ].enabled = True

ExternalHttpServicesConfigMode.addCommandClass( ConfigureExternalHttpServices )

#-------------------------------------------------------------------------------
# The "[no | default] shutdown" command,
# in "vrf <vrf name>" mode
# under "management api external-services" mode.
#-------------------------------------------------------------------------------
class ExternalHttpServicesVrfConfigMode( ExternalHttpServicesVrfConfigModeBase, 
                                         BasicCli.ConfigModeBase ):
   name = "External services VRF configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      ExternalHttpServicesVrfConfigModeBase.__init__( self, ( vrfName, capiConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class VrfModeShutdown( CliCommand.CliCommandClass ):
   syntax = '''shutdown'''
   noOrDefaultSyntax = '''shutdown ...'''
   data = { 'shutdown': HttpService.vrfShutdownHelpdesc }

   @staticmethod
   def handler( mode, args ):
      mode.shutdown()

   @staticmethod
   def noHandler( mode, args ):
      mode.noShutdown()

   defaultHandler = handler

ExternalHttpServicesVrfConfigMode.addCommandClass( VrfModeShutdown )

def enterExternalHttpServicesVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   if not capiConfig.service[ ExternalServicesConstants.SERVICE_NAME ].enabled:
      mode.addWarning( "Cannot configure external-services in VRF %s because "
                       "the service is globally disabled. Please enable service in "
                       "'management api external-services' mode first" % vrfName )
   childMode = mode.childMode( ExternalHttpServicesVrfConfigMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def removeExternalServiceVrfs( mode, args ):
   HttpService.removeVrfFromService( capiConfig,
                                     ExternalServicesConstants.SERVICE_NAME,
                                     args.get( 'VRF' ) )

class EnterVrfMode( CliCommand.CliCommandClass ):
   syntax = '''vrf VRF'''
   noOrDefaultSyntax = '''vrf [ VRF ]'''
   data = { 'vrf': HttpService.vrfHelpdesc,
            'VRF': VrfCli.VrfNameExprFactory( inclAllVrf=True ) }
   handler = enterExternalHttpServicesVrfConfigMode
   noOrDefaultHandler = removeExternalServiceVrfs

ExternalHttpServicesConfigMode.addCommandClass( EnterVrfMode )

def Plugin( entityManager ):
   global capiConfig
   capiConfig = ConfigMount.mount( entityManager,
                                   "mgmt/capi/config",
                                   "HttpService::Config",
                                   "w" )
