# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import EvpnCli
import Tac

import CliPlugin.ArBgpCli as ArBgpCli
from CliPlugin.EvpnCliModels import EvpnRoutes
from CliPlugin.BgpVpnCli import VpnCliHelperCommand
from CliPlugin.RoutingBgpShowCli import (
        arBgpShowCmdDict,
)
from BgpLib import routeTargetToExtCommU64Value


class EvpnExtCommParser( object ):
   # constants and bit shifts for constructing extended communities
   TYPE_AND_SUBTYPE_BGP_ENCAP = 0x030c

   TYPE_EVPN = 0x06
   TYPE_EVPN_SUBTYPE_MAC_MOBILITY = 0x00
   TYPE_EVPN_SUBTYPE_ESI_LABEL = 0x01
   TYPE_EVPN_SUBTYPE_ROUTER_MAC = 0x03
   ENCAP_TUNNEL_TYPE = 0x8

   SHIFT_TYPE = 56
   SHIFT_SUBTYPE = 48
   SHIFT_STICKY_BIT = 40
   SHIFT_SINGLE_ACTIVE = 40

   @staticmethod
   def tunnelEncapExtComm( tunnelTypeStr ):
      # The format of the Tunnel Encapsulation Extended Community can be found in
      # RFC5512 section 4.5.
      # The tunnel types are defined in draft-ietf-bess-evpn-overlay-05 section 13.
      #
      # TODO - Currently we only support "vxlan" tunnel type. Need to add a matcher
      # with other possible tokens when we decide to support more types.
      assert tunnelTypeStr == 'vxlan'
      C = EvpnExtCommParser
      c = ( C.TYPE_AND_SUBTYPE_BGP_ENCAP << C.SHIFT_SUBTYPE ) | C.ENCAP_TUNNEL_TYPE
      return c

   @staticmethod
   def macMobilityExtComm( seqNoStr ):
      # The format of the Mac Mobility extended community can be found in
      # RFC7432 section 7.7
      C = EvpnExtCommParser
      c = ( C.TYPE_EVPN << C.SHIFT_TYPE )
      c |= ( C.TYPE_EVPN_SUBTYPE_MAC_MOBILITY << C.SHIFT_SUBTYPE )
      if seqNoStr == 'sticky':
         c |= ( 0x1 << C.SHIFT_STICKY_BIT )
      else:
         c |= int( seqNoStr )
      return c

   @staticmethod
   def esiLabelExtComm( esiLabel ):
      # The format of the ESI Label extended community can be found in
      # RFC7432 section 7.5
      C = EvpnExtCommParser
      c = ( C.TYPE_EVPN << C.SHIFT_TYPE ) | \
         ( C.TYPE_EVPN_SUBTYPE_ESI_LABEL << C.SHIFT_SUBTYPE )
      if esiLabel[ 'esiLabelRedundancyMode' ] == 'single-active':
         c |= ( 0x1 << C.SHIFT_SINGLE_ACTIVE )
      c |= int( esiLabel[ 'esiLabelValue' ] )
      return c

   @staticmethod
   def routerMacExtComm( macStr ):
      # The format of the Router Mac extended community from:
      # draft-sajassi-l2vpn-evpn-inter-subnet-forwarding-05 sec 6.1, mentioned below.
      """ A new EVPN BGP Extended Community called Router's MAC is introduced
      here. This new extended community is a transitive extended community
      with the Type field of 0x06 (EVPN) and the Sub-Type of 0x03. It may
      be advertised along with BGP Encapsulation Extended Community define
      in section 4.5 of [RFC5512].
      The Router's MAC Extended Community is encoded as an 8-octet value as
      follows:
      0                   1                   2                   3
      0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
      +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      | Type=0x06     | Sub-Type=0x03 |        Router's MAC           |
      +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      |                      Router's MAC Cont'd                      |
      +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ """

      C = EvpnExtCommParser
      mac = Tac.Value( 'Arnet::EthAddr', stringValue=macStr )
      c = ( C.TYPE_EVPN << C.SHIFT_TYPE )
      c |= ( C.TYPE_EVPN_SUBTYPE_ROUTER_MAC << C.SHIFT_SUBTYPE )
      c |= ( mac.word0 << 32 ) | ( mac.word1 << 16 ) | mac.word2
      return c


class EvpnCliHelperCommand( VpnCliHelperCommand ):
   def __init__( self, mode, command, **kwargs ):
      super( EvpnCliHelperCommand, self ).__init__(
         mode, command, "l2vpnEvpn", **kwargs )

   @staticmethod
   def flattenExtCommArgs( fromKwargs, toKwargs ):
      """ Input: fromKwargs: cli input params in following format:
      fromKwargs[ "extCommunityValues" ]["rt"]= 1:1
      fromKwargs[ "extCommunityValues" ]["tunnelEncap"]="vxlan" etc.

      convertExtCommunityValues: populates toKwargs[ "extCommValues" ] with a string:
      '|' separated, U64 values of extcommunities passed in cli. such as:
      toKwargs[ "extCommValues" ]='43189991171031040|21550481834311688|5629542483886'
      toKwargs[ "exact" ] = fromKwargs["exact" ], if it is set.
      """
      exact = fromKwargs.get( "exact", None )
      if exact:
         toKwargs[ "extendedCommunitiesExactMatch" ] = exact
      commValues = fromKwargs[ "extCommunityValues" ]
      extCommunities = ""
      C = EvpnExtCommParser
      for ( extCommType, extCommValue ) in commValues:
         if extCommunities != "":
            extCommunities += "|"
         if extCommType == 'rt':
            extCommunities += "|".join( [ str( routeTargetToExtCommU64Value( i ) )
               for i in extCommValue ] )
         elif extCommType == 'tunnelEncap':
            extCommunities += str( C.tunnelEncapExtComm( extCommValue ) )
         elif extCommType == 'macMobility':
            extCommunities += str( C.macMobilityExtComm( extCommValue ) )
         elif extCommType == 'esiLabel':
            extCommunities += str( C.esiLabelExtComm( extCommValue ) )
         elif extCommType == 'routerMac':
            extCommunities += str( C.routerMacExtComm( extCommValue ) )
         else:
            assert False, "invalid extComType: {}".format( extCommType )
      toKwargs[ "extCommValues" ] = extCommunities

   @staticmethod
   def flattenMulticastArgs( fromKwargs, toKwargs ):
      """ Input: fromKwargs: cli input params in following format:
      fromKwargs[ "multicast" ]["first"]= IPv4 or IPv6 addr
      fromKwargs[ "multicast" ]["second"]= Optional IPv4 or IPv6 addr

      Only applies to Multicast routes ( Type 6,7,8, and 10 )
      if only first is specified, output routes with source or group that match
      - toKwargs[ "multicast" ] = '<IP addr>'
      if second IP addr also exists, output routes where the source,group matches
      first,second or second,first depending on which one is the unicast or
      multicast address.
      - toKwargs[ "multicast" ] = '<first IP>|<second IP>'
      """
      first = fromKwargs.get( "first" )
      second = fromKwargs.get( "second" )
      if second:
         toKwargs[ "multicast" ] = "%s|%s" % ( first, second )
      else:
         toKwargs[ "multicast" ] = first

   @staticmethod
   def flattenArgs( fromKwargs, toKwargs ):
      k = 'extCommValuesAndExact'
      extComm = fromKwargs.get( k, None )
      if extComm:
         EvpnCliHelperCommand.flattenExtCommArgs( fromKwargs[ k ], toKwargs )

      multicast = fromKwargs.pop( 'multicast', None )
      if multicast:
         EvpnCliHelperCommand.flattenMulticastArgs( multicast, toKwargs )

      VpnCliHelperCommand.flattenArgs( fromKwargs, toKwargs )

   def run( self, **kwargs ):
      super( EvpnCliHelperCommand, self ).run( **kwargs )
      return EvpnRoutes

def doShowBgpEvpn( mode, **kwargs ):
   argsValue = kwargs.pop( 'args', None )
   if argsValue:
      # copy argsValue back to kwargs
      kwargs.update( argsValue )

   kwargs[ 'vniDottedNotation' ] = EvpnCli.vxlanCtrlConfig.vniInDottedNotation

   # Flatten the arguments
   setKwargs = {}
   EvpnCliHelperCommand.flattenArgs( kwargs, setKwargs )

   if setKwargs.get( "prefixValue", False ):
      # Set a disctinct prefix value to not colide with MplsVpn prefix value
      setKwargs[ "evpnPrefixValue" ] = setKwargs.pop( "prefixValue" )

   if setKwargs.get( "internal", False ):
      prefix = setKwargs.pop( "evpnPrefixValue", None )
      ArBgpCli.ArBgpAsyncCliCommand( mode, "show bgp <af> internal",
                                     prefix=prefix, **setKwargs ).run()
      return EvpnRoutes

   return EvpnCliHelperCommand( mode, 'show bgp evpn', **setKwargs ).run()

arBgpShowCmdDict[ 'doShowBgpEvpn' ] = doShowBgpEvpn
