# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from CliModel import Model, Int, Bool, Dict, Enum
from ArnetModel import IpGenericAddress
import TableOutput
from TypeFuture import TacLazyType

class EvpnGatewayVrf( Model ):
   vlan = Int( help="Internal VLAN ID for this VRF" )

class EvpnGatewayVrfs( Model ):
   vrfs = Dict( help="Mapping of VRFs to their VLAN IDs", keyType=str,
                valueType=EvpnGatewayVrf )
   _detail = Bool( help="Detailed PIM gateway info", optional=True )

   def detailIs( self, detail ):
      self._detail = detail

   def render( self ):
      if self.vrfs:
         if self._detail:
            headers = ( "VRF Name", "VLAN" )
            formatLeft = TableOutput.Format( justify="left" )
            formatLeft.noPadLeftIs( True )
            table = TableOutput.createTable( headers )
            table.formatColumns( formatLeft, formatLeft )
            for vrf, vlan in sorted( self.vrfs.items() ):
               table.newRow( vrf, vlan.vlan )
            print( table.output() )
         else:
            print( "VRF Name" )
            print( "--------------" )
            for vrf in sorted( self.vrfs ):
               print( vrf )

class EvpnGatewayDr( Model ):
   role = Enum( values=TacLazyType( "Routing::Multicast::PegDrRole" ).attributes,
                help="Router role on this VLAN" )
   addr = IpGenericAddress( help="Address of Designated Router" )

class EvpnGatewayDrs( Model ):
   vlans = Dict( help="Mapping of VLAN to external gateway DR information",
                 keyType=int,
                 valueType=EvpnGatewayDr )

   def render( self ):
      if self.vlans:
         headers = ( "VLAN", "Role", "DR Address" )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )
         table = TableOutput.createTable( headers )
         table.formatColumns( formatLeft, formatLeft, formatLeft )
         for vlan, dr in sorted( self.vlans.items() ):
            role = 'DR' if dr.role == \
                  TacLazyType( "Routing::Multicast::PegDrRole" ).dr else 'Non-DR'
            table.newRow( vlan, role, dr.addr )
         print( table.output() )
