# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, DeferredModel, Submodel, Str, Int, Float, List, Enum, \
GeneratorDict, Bool, Dict
from ArnetModel import IpGenericAddress, Ip4Address, IpGenericPrefix, MacAddress
from CliPlugin.BgpCliModels import BgpRoutePeerEntry, EncapLabel, \
   REASON_NOT_BEST_LIST, _degradeToIpGenAddress, PmsiTunnel
from IntfModels import Interface
from prettytable import PLAIN_COLUMNS, PrettyTable
import Tac
import time

class EvpnRouteASPathEntry( DeferredModel ):
   asPath = Str( optional=True, help='AS path string (if absent,  \
                 then the route was originated locally)' )
   asPathType = Enum( values=( 'Internal', 'External', 'Confed-External', 'Local', 
                               'Invalid' ), 
                      help='AS path type: \
                            Internal - originated by I-BGP \
                            External - originated by E-BGP \
                            Confed-External - originated by a E-BGP confederation \
                            Local - originated locally \
                            Invalid - AS path is invalid' )

class EvpnRouteTypeEntry ( DeferredModel ):
   stale = Bool( default=False, help='Route is stale' )
   valid = Bool( default=False, help='Route is valid' )
   suppressed = Bool( default=False, 
                      help='Route is suppressed from entering the Rib' )
   active = Bool( default=False, help='Route is active' )
   backup = Bool( default=False, help='Route is backup' )
   ecmpHead = Bool( default=False, help='Route is the ECMP head' )
   ecmp = Bool( default=False, help='Route is an ECMP route' )
   ucmp = Bool( optional=True, default=False, help='Route is an UCMP route' )
   ecmpContributor = Bool( default=False, help='Route contributes to ECMP' )
   atomicAggregator = Bool( default=False, help='Route is an atomic-aggregate' )
   queued = Bool( default=False, help='Route is queued for advertisement' )
   localAgg = Bool( optional=True, help='Route is locally aggregated' )
   notInstalledReason = Enum( values=( 'routeBestInactive', 'routeInstallDenied',
                                       'labeledRoutePresent' ),
                              optional=True,
                              help="Reason for route not being installed" )
   origin = Enum( optional=True, values=( 'Igp', 'Egp', 'Incomplete' ),
                  help = 'Route origin' )
   waitForConvergence = Bool( optional=True,
                              help='Route is pending BGP convergence' )

class TunnelEncap( DeferredModel ):
   asn = Int( help='Autonomous System Number' )
   endpoint = IpGenericAddress( help='Tunnel egress endpoint' )

class EvpnRouteDetailEntry( DeferredModel ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         if 'peerEntry' in dictRepr:
            dictRepr[ 'peerEntry' ] = \
                  BgpRoutePeerEntry().degrade( dictRepr[ 'peerEntry' ], 1 )
      return dictRepr

   aggregator = Str( optional=True, help='Aggregator of the route' )
   communities = List( valueType=str, help='Route community' )
   extCommunities = List( valueType=str, help='Extended community for route' )
   largeCommunities = List( valueType=str, help='Large community for route' )
   rxPathId = Int( optional=True, help='Received path ID of this route' )
   txPathId = Int( optional=True, help='Advertised path ID for this route' )
   rxSafi = Str( optional=True, help='Received SAFI of this route' )
   origin = Enum( values=( 'Igp', 'Egp', 'Incomplete' ), help = 'Route origin' )
   originator = Ip4Address( optional=True, 
                            help='Router ID of the originator of this route' )
   clusterList = Str( optional=True, 
                       help='Cluster list for the route' )
   peerEntry = Submodel( valueType=BgpRoutePeerEntry, optional=True, 
                         help='Peer information for the route' )
   fromRRClient = Bool( default=False,
                        help='Route received from route reflector client' )
   seqNumber = Int( optional=True, help='Route sequence number' )
   pendingTimeToAdv = Float( optional=True, help='Timestamp of route advertisement' )
   # the redistibution protocol strings here need to be kept in sync
   # with the strings returned by
   # gated/gated-ctk/src/bgp/bgp_dget_route.c:redist_proto_str()
   # note these are usually transformed by maybeCamelize()
   redistributionProtocol = Enum( values=( 'Connected', 'Static', 'Ospf3', 'Ospf',
                                           'Rip', 'Is-Is', 'unknown' ),
                                  optional=True,
                                  help='Protocol from which route got \
                                     redistributed into BGP' )
   labelStack = List( optional=True, valueType=int, 
                      help='MPLS label stack information')
   tunnelRibEligible = Bool( default=False,
                             help='Route eligible to be installed in Tunnel RIB' )
   label = Submodel( optional=True, valueType=EncapLabel,
                     help='Local or remote label' )
   l3Label = Submodel( optional=True, valueType=EncapLabel,
                       help='Local or remote L3 label' )
   esi = Str( optional=True, help="Ethernet segment identifier" )
   domainPath = List( optional=True, valueType=str, help="Domain path attribute" )
   pmsiTunnel = Submodel( optional=True, valueType=PmsiTunnel,
                          help="P-Multicast Serivce Interface Tunnel Attribute" )
   tunnelEncap = GeneratorDict( optional=True, valueType=TunnelEncap,
         help="Tunnel encapsulation attribute" )

class EvpnRoutePath( DeferredModel ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2 :
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         if 'nextHop' in dictRepr:
            dictRepr[ 'nextHop' ] = _degradeToIpGenAddress( dictRepr[ 'nextHop' ] )
         if 'routeDetail' in dictRepr:
            dictRepr[ 'routeDetail' ] = \
                  EvpnRouteDetailEntry().degrade( dictRepr[ 'routeDetail' ], 1 )
      return dictRepr

   nextHop = Str( optional=True, help='Route next hop address' )
   asPathEntry = Submodel( valueType=EvpnRouteASPathEntry, 
                           help='AS path information' )
   importedEvpnPathRd = Str( optional=True,
                             help='Imported EVPN path RouteDistinguisher' )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=EvpnRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received.\
                          Only returned with 'show bgp evpn detail'" )
   routeDetail = Submodel( valueType=EvpnRouteDetailEntry, optional=True, 
                           help='Route details' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class EvpnRouteKeyDetail( DeferredModel ):
   rd = Str( optional=True, help='Route distinguisher' )
   nlriType = Enum( values=( 'auto-discovery', 'mac-ip', 'imet', 'ethernet-segment',
                             'ip-prefix', 'smet', 'spmsi', 'join-sync',
                             'leave-sync' ), help='NLRI type' )
   eti = Int( optional=True,
         help='For auto-discovery, mac-ip, imet, and ip-prefix routes, the ethernet \
              tag identifier' )
   esi = Str( optional=True,
         help='For auto-discovery and ethernet-segment routes, the ethernet segment \
              identifier' )
   mac = Str( optional=True, help='For mac-ip routes, the ethernet address' )
   ipGenAddr = IpGenericAddress( optional=True,
         help='For mac-ip routes, the IP address corresponding to the ethernet \
               address. \
               For imet and ethernet-segment routes, the originating router IP \
               address.' )
   ipGenPrefix = IpGenericPrefix( optional=True,
         help='For ip-prefix routes, the IPv4 or IPv6 address prefix' )
   source = IpGenericAddress( optional=True, help='Multicast source address' )
   group = IpGenericAddress( optional=True, help='Multicast group address' )
   syncnum = Int( optional=True, help='Leave group synchronization number' )

class EvpnRouteEntry( DeferredModel ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         evpnRoutePaths = [ EvpnRoutePath().degrade( rtPath, 1 ) \
                           for rtPath in dictRepr[ 'evpnRoutePaths' ] ]
         dictRepr[ 'evpnRoutePaths' ] = evpnRoutePaths
      return dictRepr

   routeKeyDetail = Submodel( valueType=EvpnRouteKeyDetail, help='NLRI details' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   evpnRoutePaths = List( valueType=EvpnRoutePath, 
                          help='List of BGP EVPN route ECMP paths' )

class EvpnRoutes( DeferredModel ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr[ 'evpnRoutes' ] = { \
               key: EvpnRouteEntry().degrade( rtEntry, 1 ) for key, rtEntry in \
               dictRepr[ 'evpnRoutes' ].iteritems() }
      return dictRepr

   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   lastProcessedSeqNum = Int( optional=True, 
         help='Last route sequence number acknowledged' )
   currentSeqNum = Int( optional=True, 
         help='Current route sequence number' )
   _detail = Bool( optional=True, help='Detailed output is requested' )
   _advRoutes = Bool( optional=True, help='Advertised routes output is requested' )
   evpnRoutes = GeneratorDict( keyType=str, valueType=EvpnRouteEntry, 
         help='Dictionary of BGP EVPN route entries indexed by the route key' )

class EsiPeer( DeferredModel ):
   ip = IpGenericAddress( help='IP address of peer' )
   esiLabel = Int( help='MPLS ESI label advertised by peer', 
                   optional=True )

class EthernetSegment( DeferredModel ):
   intf = Interface( help='Interface name' )
   redundancyMode = Enum( values=( 'singleActive', 'allActive' ),
                          help='Redundancy mode' )
   state = Enum( values=( 'up', 'down' ), help='Ethernet segment state' )
   esiLabel = Int( help='MPLS ESI label used for split-horizon filtering',
                   optional=True )
   sharedEsiLabel = Bool( help="MPLS ESI label shared with other PEs",
                          optional=True )
   importRT = Str( help='ES import route target' )
   dFElectionState = Enum( values=( 'pending', 'started', 'finished' ),
                           help='DF election state', optional=True )
   dFPeer = Submodel( valueType=EsiPeer, help='Designated forwarder' )
   nonDFPeers = List( valueType=EsiPeer, help='Non-designated forwarders' )

class BgpEvpnVpwsPseudowire( DeferredModel ):
   status = Enum( values=( "pending", "up", "noLocalId", "noRemoteId",
                           "localIdConflict", "remoteIdConflict", "noRequest",
                           "noLocalLabel" ),
                  help="Pseudowire status" )
   vpwsLabel = Int( help="Advertised pseudowire label", optional=True )
   vpwsIdLocal = Int( help="Advertised virtual private wire service (VPWS) "
                      "instance identifier", optional=True )
   vpwsIdRemote = Int( help="Peer's virtual private wire service (VPWS) "
                       "instance identifier", optional=True )

class BgpEvpnVpwsDetail( DeferredModel ):
   l2Mtu = Int( help="L2 MTU value used for validation, if enabled", optional=True )
   controlWord = Bool( help="Control word required when sending EVPN packets to "
                       "this device" )
   pseudowires = Dict( valueType=BgpEvpnVpwsPseudowire,
                       help="Mapping of pseudowire name to detailed information" )

class BgpEvpnInstance( DeferredModel ):
   rd = Str( help='EVPN instance route distinguisher' )
   importRts = List( valueType=long, help="Route targets to import" )
   exportRts = List( valueType=long, help="Route targets to export" )
   serviceIntf = Enum( values=( 'vlanBased', 'vlanAwareBundle' ),
                       help='EVPN instance service interface' )
   localIp = IpGenericAddress( help='Local IP address' )
   encapType = Enum( values=( 'vxlan', 'mpls' ), help='Encapsulation type' )
   labelAllocMode = Enum( values=( 'perVlan', 'perInstance', 'perPseudowire' ),
                          help='Label allocation mode', optional=True )
   macLabel = Int( help='MPLS label used in mac-ip routes', optional=True )
   imetLabel = Int( help='MPLS label used in imet routes', optional=True )
   adLabel = Int( help='MPLS label used in auto-discovery routes',
                  optional=True )
   evpnVpwsDetail = Submodel( valueType=BgpEvpnVpwsDetail,
         help="EVPN virtual private wire service (VPWS) detailed information",
         optional=True )
   ethernetSegments = Dict( valueType=EthernetSegment,
                            help='Local ethernet segments indexed by ESI',
                            optional=True )

class BgpEvpnInstances( DeferredModel ):
   bgpEvpnInstances = Dict( valueType=BgpEvpnInstance,
                            help='BGP EVPN instances indexed by instance name' )

class BgpEvpnHostFlapEntry( Model ):
   vlan = Int( help="VLAN ID of blacklisted MAC Address" )
   macAddr = MacAddress( help="Blacklisted MAC Address due to duplication" )
   time = Float( help="Time when the MAC Address was added to the blacklist" )

class BgpEvpnHostFlapEntries( Model ):
   duplicationBlacklistedMacs = List( valueType=BgpEvpnHostFlapEntry,
         help='List of routes that contain MAC Addresses '\
         'that are blacklisted due to duplication' )

   def render( self ):
      table = PrettyTable( [ 'VLAN', 'MAC Address', 'time' ] )
      table.set_style( PLAIN_COLUMNS )
      for entry in self.duplicationBlacklistedMacs:
         table.add_row( [ entry.vlan, entry.macAddr,
                          time.ctime( entry.time ) ] )
      print table
